;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: matrix.lisp
;;;  Author:  Simoncelli/Heeger/Chichilnisky
;;;  Description:  Non-pointwise floating point matrix operations 
;;;                (pointwise matrix operations are in array-ops.lisp).
;;;  Creation Date:  Fall, 1988
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)
(export '(matrix-mul  matrix-transpose-mul  matrix-mul-transpose outer-product
	  matrix-transpose normalize identity-matrix diagonal-matrix
	  cross-product dot-product vector-length vector-distance
	  diagonal make-matrix matrix-trace matrix-inverse
	  square-p diagonal-p identity-p unitary-p symmetric-p orthogonal-p
	  singular-p))

;;; NOTE: This file provides operations that only work on floating point 
;;; simple-vectors and simple-arrays of rank 2.

#|
;;; Declare matrices to be 2-D simple-arrays of type float.  Declare the vectors 
;;; to be 1-D simple-arrays of type float.
;;; Note: Symbolics doesn't allow array-register declarations on arguments.
(defmacro declare-matrices (array-list vector-list)
  #-Symbolics
  `(declare (type (simple-array single-float (* *))
	     ,@(loop for arr in array-list collect arr))
            (type (simple-array single-float (*))
	     ,@(loop for vect in vector-list collect vect))))

;;; Check that the arrays and vectors are as defined by the above declarations.
(defmacro checktype-matrices (array-list vector-list)
  #-Symbolics
  `(when (not (and  ,@(loop for arr in array-list collect
			    `(typep ,arr '(simple-array single-float (* *))))
	            ,@(loop for vect in vector-list collect
			    `(typep ,vect '(simple-array single-float (* ))))))
    (error "Matrix operations only defined for floating point simple-arrays."))
  #+Symbolics
  `(when (not (and  ,@(loop for arr in array-list collect
			    `(typep ,arr '(array t (* *))))
	            ,@(loop for vect in vector-list collect
			    `(typep ,vect '(array t (*))))))
    (error "Matrix operations only defined for arrays.")))
|#

;;; Declare matrices to be 2-D arrays of type float.  Declare the vectors 
;;; to be 1-D arrays of type float.
;;; Note: Symbolics doesn't allow array-register declarations on arguments.
(defmacro declare-matrices (array-list vector-list)
  `(declare (type (array single-float (* *))
	     ,@(loop for arr in array-list collect arr))
            (type (array single-float (*))
	     ,@(loop for vect in vector-list collect vect))))

#|
;;; Check that the arrays and vectors are as defined by the above declarations.
(defmacro checktype-matrices (array-list vector-list)
  `(when (not (and  ,@(loop for arr in array-list collect
			    `(typep ,arr '(array single-float (* *))))
	       ,@(loop for vect in vector-list collect
		       `(typep ,vect '(array single-float (* ))))))
    (error "Matrix operations only defined for floating point arrays.")))
|#

;;; Check that the arrays and vectors are as defined by the above declarations.
(defmacro checktype-matrices (array-list)
  `(unless (and ,@(loop for arr in array-list collect
			`(typep ,arr '(array single-float))))
    (error "Matrix operations only defined for floating point arrays.")))

(defmacro check-matrix-mul-compatibility (row1 col1 row2 col2 row3 col3)
  `(unless (and (= ,col1 ,row2) (= ,row1 ,row3) (= ,col2 ,col3))
    (error "Arrays are incompatible for multiplication")))


;;; Note: can matrix-mul back into one of the original arrays, but that conses.

(defmethod matrix-mul ((m1 array) (m2 array)
		       &key ((:-> res) (make-array (list (array-dimension m1 0)
							 (array-dimension m2 1))
						   :element-type 'single-float)))
  (check-matrix-mul-compatibility (row-dim m1) (col-dim m1)
				  (row-dim m2) (col-dim m2)
				  (row-dim res) (col-dim res))
  (checktype-matrices (m1 m2 res))
  (if (or (eq res m1) (eq res m2))
      (progn (warn "matrix multiplication: source and destination are the same...consing a copy" )
	     (copy (matrix-mul m1 m2) :-> res))
      (internal-matrix-mul m1 (row-dim m1) (col-dim m1)
			   m2 (row-dim m2) (col-dim m2)
			   res (row-dim res) (col-dim res)))
  
  res)


(defmethod matrix-mul-transpose ((m1 array) (m2 array) 
				 &key ((:-> res) (make-array (list (array-dimension m1 0) 
								   (array-dimension m2 0))
							     :element-type 'single-float)))
  (check-matrix-mul-compatibility (row-dim m1) (col-dim m1)
				  (col-dim m2) (row-dim m2)
				  (row-dim res) (col-dim res))
  (checktype-matrices (m1 m2 res))
  (if (or (eq res m1) (eq res m2))
      (progn (warn "matrix multiplication: source and destination are the same...consing a copy")
	     (copy (matrix-mul-transpose m1 m2) :-> res))
      (internal-matrix-mul-transpose m1 (row-dim m1) (col-dim m1)
				     m2 (row-dim m2) (col-dim m2)
				     res (row-dim res) (col-dim res)))
      
  res)

(defmethod matrix-transpose-mul ((m1 array) (m2 array) 
				 &key ((:-> res) (make-array (list (array-dimension m1 1) 
								   (array-dimension m2 1))
							     :element-type 'single-float)))
  (check-matrix-mul-compatibility (col-dim m1) (row-dim m1)
				  (row-dim m2) (col-dim m2)
				  (row-dim res) (col-dim res))
  (checktype-matrices (m1 m2 res))
  (if (or (eq res m1) (eq res m2))
      (progn (warn "matrix multiplication: source and destination are the same...consing a copy")
	     (copy (matrix-transpose-mul m1 m2) :-> res))
      (internal-matrix-transpose-mul m1 (row-dim m1) (col-dim m1)
				     m2 (row-dim m2) (col-dim m2)
				     res (row-dim res) (col-dim res)))
  res)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Fixes to handling of matrix multiplication with vectors
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;; Old vector-array code does not work right.
;;; For example, matrix-transpose-mul with 2 vectors failed.
;;; Basically, need a uniform convention for vectors.
;;; It seems that the right convention is that all vectors are row vectors.
;;; Only by transposing can they be treated as column vectors.
;;; Note that only *some* array-vector operations should be allowed!!
;;; For now at least, doing incorrect operations cause a continuable error.
;;; When a one of these is done, what gets returned is a vector with the same interpretation.


;;; These methods make good sense for (row) vectors

(defmethod matrix-mul
    ((v vector) (m array)
     &key ((:-> res) (make-array (col-dim m) :element-type (array-element-type v))))
  (check-matrix-mul-compatibility (row-dim v) (col-dim v)
				  (row-dim m) (col-dim m)
				  (row-dim res) (col-dim res))
  (checktype-matrices (m v res))
  (if (eq res v)
      (progn (warn "matrix multiplication: source and destination are the same...consing a copy")
	     (copy (matrix-mul v m) :-> res))
      (internal-matrix-mul v (row-dim v) (col-dim v)
			   m (row-dim m) (col-dim m)
			   res (row-dim res) (col-dim res)))
  res)

(defmethod matrix-mul-transpose
    ((m array) (v vector)
     &key ((:-> res) (make-array (list (row-dim m) 1) :element-type (array-element-type v))))
  (check-matrix-mul-compatibility (row-dim m) (col-dim m)
				  (col-dim v) (row-dim v)
				  (row-dim res) (col-dim res))
  (checktype-matrices (m res v))
  (internal-matrix-mul-transpose m (row-dim m) (col-dim m)
				 v (row-dim v) (col-dim v)
				 res (row-dim res) (col-dim res))
  res)

(defmethod matrix-mul-transpose
    ((v vector) (m array)
     &key ((:-> res) (make-array (row-dim m) :element-type (array-element-type v))))
  (check-matrix-mul-compatibility (row-dim v) (col-dim v)
				  (col-dim m) (row-dim m)
				  (row-dim res) (col-dim res))
  (checktype-matrices (m v res))
  (if (eq res v)
      (progn (warn "matrix multiplication: source and destination are the same...consing a copy")
	     (copy (matrix-mul v m) :-> res))
      (internal-matrix-mul-transpose v (row-dim v) (col-dim v)
				     m (row-dim m) (col-dim m)
				     res (row-dim res) (col-dim res)))
  res)


;;; These methods cause continuable errors, and do the right thing with transposes

(defmethod matrix-mul
    ((mat array) (vec vector)
     &key ((:-> res) (make-array (row-dim mat) :element-type (array-element-type vec))))
  (cerror "(matrix-mul mat (columnize vec) :-> (columnize res))"
	  "matrix times row-vector doesn't make sense")
  (matrix-mul mat (columnize vec) :-> (columnize res))
  res)

(defmethod matrix-transpose-mul
    ((vec vector) (mat array)
     &key ((:-> res) (make-array (col-dim mat) :element-type (array-element-type vec))))
  (cerror "(matrix-mul vec mat :-> (vectorize res))"
	  "column-vector times matrix doesn't make sense")
  (matrix-mul vec mat :-> (vectorize res)))

(defmethod matrix-transpose-mul
    ((mat array) (vec vector)
     &key ((:-> res) (make-array (col-dim mat) :element-type (array-element-type vec))))
  (cerror "(matrix-transpose-mul mat (columnize vec) :-> (columnize res))"
	  "matrix times row-vector doesn't make sense")
  (matrix-transpose-mul mat (columnize vec) :-> (columnize res))
  res)


(defmethod matrix-transpose
    ((v vector) &key ((:-> res) (make-array (list (col-dim v) 1)
					    :element-type (array-element-type v))))
  (unless (and (= (row-dim v) (col-dim res)) (= (col-dim v) (row-dim res)))
    (error "Arrays are incompatible for transposing"))
  (copy v :-> (vectorize res))
  res)

;;; Put in this new method 5.22.92 since earlier one failed with
;;; non-float matrices. EJC.
(defmethod matrix-transpose
    ((m array) &key ((:-> res) (make-array (list (col-dim m) (row-dim m))
					   :element-type (array-element-type m))))
  (cond ((lucid-float-arrays-p m res)
	 (unless (and (= (row-dim m) (col-dim res)) (= (col-dim m) (row-dim res)))
	   (error "Arrays are incompatible for transposing"))
	 (internal-matrix-transpose m (array-dimension m 0) (array-dimension m 1) res))
	(t (if (eq res m)
	       (progn (warn "Transpose: source and destination are the same...consing a copy")
		      (copy (matrix-transpose m) :-> res))
	       (let ((element-type (array-element-type res)))
		 (dotimes (row (row-dim m))
		   (declare (fixnum row))
		   (dotimes (col (col-dim m))
		     (declare (fixnum col))
		     (setf (aref res col row) (coerce (aref m row col) element-type))))))))
  res)

#|
(defmethod matrix-transpose
    ((m array) &key ((:-> res) (make-array (list (col-dim m) (row-dim m))
					   :element-type 'single-float)))
  (declare-matrices (m res) ())
  (checktype-matrices (m res))
  (unless (and (= (row-dim m) (col-dim res)) (= (col-dim m) (row-dim res)))
    (error "Arrays are incompatible for transposing"))
  (internal-matrix-transpose m (array-dimension m 0) (array-dimension m 1) res)
  res)
|#

(defmethod matrix-mul-transpose
    ((v1 vector) (v2 vector)
     &key ((:-> res) (make-array 1 :element-type (array-element-type v1))))
  (matrix-mul v1 (columnize v2) :-> res))

(defmethod matrix-transpose-mul
    ((v1 vector) (v2 vector)
     &key ((:-> res) (make-array (list (length v1) (length v2))
				 :element-type (array-element-type v1))))
  (declare-matrices (res) (v1 v2))
  (checktype-matrices (res v1 v2))
  (check-matrix-mul-compatibility (col-dim v1) (row-dim v1)
				  (row-dim v2) (col-dim v2)
				  (row-dim res) (col-dim res))
  (internal-matrix-transpose-mul v1 (row-dim v1) (col-dim v1)
				 v2 (row-dim v2) (col-dim v2)
				 res (row-dim res) (col-dim res))
  res)

(defmethod outer-product
    ((v1 vector) (v2 vector)  
     &key ((:-> res) (make-array (list (length v1) (length v2))
				 :element-type 'single-float)))
  (matrix-transpose-mul v1 v2 :-> res))


;;; Returns the norm as a second value.
(defmethod normalize ((v vector) &key (norm 1.0) ((:-> res) (similar v)))
  (declare-matrices () (v res))
  (checktype-matrices (v res))
  (loop with length = (sqrt (loop for i from 0 below (length v)
				summing (sqr (aref v i))))
	for i from 0 below (length v)
	do (setf (aref res i) (/-0 (aref v i) (/ length norm) 0.0))
	finally (return (values res (/ length norm)))))

(defmethod normalize ((arr array) &key ((:-> result) (similar arr)))
  (checktype-matrices (arr result))
  (multiple-value-bind (vec mag) (normalize (vectorize arr) :-> (vectorize result))
    (declare (ignore vec))
    (values result mag)))

(defmethod cross-product ((v1 vector) (v2 vector) 
		      &key ((:-> res) 
			    (make-array 3 :element-type 'single-float)))
  (declare-matrices () (v1 v2 res))
  (checktype-matrices (v1 v2 res))
  (when (/= 3 (car (dimensions v1))
	      (car (dimensions v2))
	      (car (dimensions res)))
    (error "Vectors must be three-dimensional"))
  (setf (aref res 0) (- (* (aref v1 1) (aref v2 2))
			(* (aref v1 2) (aref v2 1))))
  (setf (aref res 1) (- (* (aref v1 2) (aref v2 0))
			(* (aref v1 0) (aref v2 2))))
  (setf (aref res 2) (- (* (aref v1 0) (aref v2 1))
			(* (aref v1 1) (aref v2 0))))
  res)

(defmethod vector-length ((arr array))
  (sqrt (dot-product arr arr)))

(defmethod vector-distance ((v1 array) (v2 array))
  (sqrt (* (mean-square-error v1 v2) (total-size v1))))

(defmethod dot-product ((arr-1 array) (arr-2 array) &key ->)
  (declare (ignore ->))
  (check-size arr-1 arr-2)
  (cond ((lucid-float-arrays-p arr-1 arr-2)
	 (internal-dot-product arr-1 arr-2 (total-size arr-1)))
	((lucid-1bit-arrays-p arr-1 arr-2)
	 (internal1-dot-product arr-1 arr-2 (total-size arr-1)))
	(t (error "Dot-product only implemented for single-float or bit arrays"))))

;; *** should be C code 
(defun internal1-dot-product (arr-1 arr-2 size)
  (declare (ignore size))
  (let ((result 0))
    (with-displaced-vectors ((vec-1 arr-1)
			     (vec-2 arr-2))
      (declare (type (array bit (*)) vec-1 vec-2))
      (dotimes (i (length vec-1))
	(declare (fixnum i))
	(incf result (* (aref vec-1 i) (aref vec-2 i)))))
    (float result)))

#|
;;; Relies on the smarts of (fill!) and (dimensions)
(defmethod make-matrix ((list cons) &key ((:-> result)))
  (unless result
    (let ((dimensions (cond ((or (numberp (first list)) (listp (first list))) (dimensions list))
			    ((vectorp (first list)) (list (length list) (length (first list)))))))
      (setf result (make-array dimensions :element-type 'single-float))))
  (fill! result list))

(defmethod make-matrix ((x number) &rest numbers &key)
  (make-matrix (cons x (remove-if-not 'numberp numbers))))
|#
;;; *** replaced all the (matrix) methods 5.22.92 to make them smarter.

(defmethod make-matrix ((x vector) &rest vectors)
  (setq vectors (cons x vectors))
  (apply 'check-size vectors)
  (let* ((dimensions (list (length vectors) (length (first vectors))))
	 (result (make-array dimensions :element-type 'single-float)))
    (loop for vec in vectors
	  for row from 0
	  do (copy vec :-> (displaced-row row result)))
    result))

(defmethod make-matrix ((x viewable) &rest viewables)
  (apply 'check-size viewables)
  (apply 'make-matrix (mapcar 'data (cons x viewables))))

(defmethod make-matrix ((x array) &rest arrays)
  (setq arrays (cons x arrays))
  (apply 'row-check-size arrays)
  (let* ((rows (* (row-dim (first arrays)) (length arrays)))
	 (cols (col-dim (first arrays)))
	 (result (make-array (list rows cols) :element-type 'single-float)))
    (loop for arr in arrays
	  with y-offset = 0
	  do
	  (paste arr result :-> result :y y-offset)
	  (incf y-offset (row-dim arr)))
    result))

(defmethod make-matrix ((x number) &rest numbers)
  (setq numbers (cons x numbers))
  (fill! (make-array (length numbers) :element-type 'single-float)  numbers))

;; *** There is a hack here to deal with numbers,
;; since the more elegant method fails when the list is longer
;; than (apply) can work with
(defmethod make-matrix ((x list) &rest lists)
  (if lists
      (apply 'make-matrix (mapcar 'make-matrix (cons x lists)))
      (if (numberp (first x))
	  (fill! (make-array (length x) :element-type 'single-float)  x)
	  (apply 'make-matrix x))))

#|
;; test code
(make-matrix '(1 2 3))
(make-matrix 1 2 3)
(make-matrix '((1 2) (3 4)))
(make-matrix (list (make-matrix 1 2) (make-matrix 3 4)))
|#

;;; Return a size x size  identity matrix 
(defun identity-matrix (size &key ((:-> res)
				   (make-array (list size size) 
					       :element-type 'single-float)))
  (declare-matrices (res) ())
  (checktype-matrices (res))
  (zero! res)
  (dotimes (i (min (row-dim res) (col-dim res)))
    (declare (fixnum i))
    (setf (aref res i i) 1.0))
  res)

(defmethod diagonal-matrix ((diagonals list)
			&key ((:-> result) (make-array (list (length diagonals)
							     (length diagonals))
						       :element-type 'single-float)))
  (fill! result 0.0)
  (dotimes (index (length diagonals))
    (setf (aref result index index) (float (elt diagonals index))))
  result)

(defmethod diagonal-matrix ((diagonals array)
			    &key ((:-> result) (make-array (list (total-size diagonals)
								 (total-size diagonals))
							   :element-type 'single-float)))
  (fill! result 0.0)
  (let ((vec (vectorize diagonals)))
    (dotimes (index (length vec))
      (setf (aref result index index) (float (aref vec index)))))
  result)


;;; Return a vector of the diagonal elements
(defmethod diagonal ((matrix array) &key
		     ((:-> result) (similar matrix :dimensions (min (x-dim matrix) (y-dim matrix)))))
  (dotimes (index (length result))
    (setf (aref result index) (aref matrix index index)))
  result)

(defmethod matrix-trace ((arr array))
  (assert (= (row-dim arr) (col-dim arr)))
  (loop for i from 0 below (col-dim arr)
	sum (aref arr i i)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Useful predicates for handling matrices.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod square-p ((matrix array))
  (when (= (row-dim matrix) (col-dim matrix))
    matrix))

(defmethod diagonal-p ((vector vector) &key (tolerance *tolerance*))
  (declare (ignore tolerance))
  (square-p vector))

(defmethod diagonal-p ((matrix array) &key (tolerance *tolerance*))
  (with-static-arrays ((diagonal (diagonal matrix))
		       (diagonal-matrix (diagonal-matrix diagonal)))
    (when (almost-equal diagonal-matrix matrix :tolerance tolerance)
      matrix)))

(defmethod identity-p ((matrix array) &key (tolerance *tolerance*))
  (when (and (square-p matrix)
	     (diagonal-p matrix :tolerance tolerance)
	     (with-static-arrays ((diagonal (diagonal matrix)))
	       (almost-equal 1.0 diagonal)))
    matrix))

(defmethod unitary-p ((matrix array) &key (tolerance *tolerance*))
  (with-static-arrays ((product (matrix-mul-transpose matrix matrix)))
    (when (identity-p product :tolerance tolerance)
      matrix)))

(defmethod symmetric-p ((matrix array) &key (tolerance *tolerance*))
  (with-static-arrays ((transpose (matrix-transpose matrix)))
    (when (almost-equal matrix transpose :tolerance tolerance)
      matrix)))
	
(defmethod symmetric-p ((vector vector) &key (tolerance *tolerance*))
  (declare (ignore tolerance))
  (square-p vector))


(defmethod orthogonal-p ((v1 vector) (v2 vector) &key (tolerance *tolerance*))
  (when (almost-equal 0.0 (dot-product v1 v2) :tolerance tolerance)
    v1))

(defmethod orthogonal-p ((arr array) (arr2 array) &key (tolerance *tolerance*))
  (with-static-arrays ((product (matrix-mul-transpose arr arr2)))
    (when (almost-equal 0.0 product :tolerance tolerance)
      arr)))

(defmethod orthogonal-p ((arr array) (vec vector) &key (tolerance *tolerance*))
  (with-static-arrays ((product (matrix-mul arr vec)))
    (when (almost-equal 0.0 product :tolerance tolerance)
      arr)))

(defmethod orthogonal-p ((vec vector) (arr array) &key (tolerance *tolerance*))
  (with-static-arrays ((product (matrix-mul arr vec)))
    (when (almost-equal 0.0 product :tolerance tolerance)
      vec)))

(defmethod singular-p ((matrix array) &key (tolerance *tolerance*))
  (when (< (condition-number matrix) tolerance)
    matrix))


#|
;;; Replaced by svd code (see svd.lisp).  -DH 8/91

;;; Compute the Moore-Penrose pseudoinverse of the matrix m.  
;;; Expects a 2D floating point matrix!!
(defun matrix-inverse (m &key ((:-> res)
			       (make-array (reverse (array-dimensions m))
					   :element-type 'single-float)))
  #-Lucid (error "MATRIX-INVERSE not defined.")
  #+Lucid
  (progn
    (checktype-matrices (m res))
    (internal-pinverse m res (array-dimension m 0) (array-dimension m 1)))
  res)
|#



;;; Local Variables:
;;; buffer-read-only: t 
;;; End:
