;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: color-picture.lisp
;;;  Author: heeger
;;;  Description: color pictures, dithered, 2 bits each R,G,B
;;;  Creation Date: 9/91
;;; ----------------------------------------------------------------
;;; This file is part of the Object-Oriented Picture System (OBVIUS),
;;; (C) Vision Science Group,  Media Laboratory,  
;;; Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Color Picture Object and display methods

(export '(color-picture))

(def-simple-class color-picture (picture)
  ((pedestal :type (or number (eql :auto))
	     :documentation "Value added to floating point viewable values to map them
to [0,1]")
   (scale :type (or number (eql :auto))
	  :documentation "Value used as a divisor for floating point viewable values to map them
to [0,1]."))
  (:default-initargs :pedestal :auto :scale :auto))

(defmethod settable-parameters ((class-name (eql 'color-picture)))    
  (append '(pedestal scale)  (call-next-method)))

(defmethod set-not-current ((pic color-picture))
  (call-next-method)
  (reinitialize-instance pic
			 :scale (get-default 'color-picture 'scale)
			 :pedestal (get-default 'color-picture 'pedestal)))

(defmethod title-bar-string ((pic color-picture))
  (format nil "(~S - ~,2,-2G) / ~,3,-2G" 
	  (name (viewable pic)) (pedestal pic) (scale pic)))

(defmethod position-message ((pic color-picture) (im color-image)
			     pane pane-y pane-x)
  (declare (ignore pane))
  (multiple-value-bind (y x)
      (pane-coord-to-viewable-coord pic pane-y pane-x)
    (let* ((vbl-list (viewable-list im))
	   (data1 (data (first vbl-list)))
	   (data2 (data (second vbl-list)))
	   (data3 (data (first vbl-list))))
      (if (array-in-bounds-p data1 y x)
	  (status-message "(~d, ~d): (~g,~g,~g)"
			  y x (aref data1 y x) (aref data2 y x) (aref data3 y x))
	  (status-message "(~d, ~d): out of bounds" y x)))))

(defmethod drag-picture ((pic color-picture) dy dx)
  (with-slots (y-offset x-offset pane-of zoom) pic
    (if (and dy dx)
	(setf y-offset (+ y-offset dy)  x-offset (+ x-offset dx))
	(setf y-offset 0  x-offset 0))
    (clear pane-of)
    (render pane-of (system-dependent-frob pic) y-offset x-offset zoom)))

(defmethod compute-picture ((pic color-picture) (im color-image))
  (with-slots (system-dependent-frob pane-of) pic
    (setf system-dependent-frob		;remake the bltable!
	  (make-bltable (screen-of pane-of) (dimensions im)
			:bltable system-dependent-frob
			:depth (depth (screen-of pane-of))))
    (clear system-dependent-frob :color (background pane-of))
    (let* ((vbl-list (viewable-list im))
	   (data1 (data (first vbl-list)))
	   (data2 (data (second vbl-list)))
	   (data3 (data (third vbl-list))))
      (draw-color-float-arrays system-dependent-frob data1 data2 data3
			       (pedestal pic) (scale pic) (zoom pic)
			       0 0))	;x and y offsets
    ))

(defmethod reset-picture-defaults ((pic color-picture) (vbl viewable) &rest initargs
				   &key
				   (pane-of (slot-value pic 'pane-of))
				   (current (slot-value pic 'pane-of))
				   (zoom nil zoom-supplied-p)
				   (scale nil scale-supplied-p)
				   (pedestal nil pedestal-supplied-p))
  (when scale-supplied-p
    (cond ((eq scale :auto) (setf (getf initargs :scale) (range vbl)))
	  ((not (numberp scale)) (setf (getf initargs :scale) 1)))
    (when (current-p pic) (setf (getf initargs :current) nil))) ;for set-not-current
  (when pedestal-supplied-p
    (cond ((eq pedestal :auto) (setf (getf initargs :pedestal) (minimum vbl)))
	  ((not (numberp pedestal)) (setf (getf initargs :pedestal) 0)))
    (when (current-p pic) (setf (getf initargs :current) nil)))
  (when zoom-supplied-p
    (setq zoom
	  (cond ((numberp zoom) zoom)
		((eq zoom :auto)
		 (apply 'min (mapcar #'(lambda (pane-dim pic-dim) (/ pane-dim pic-dim))
				     (dimensions pane-of) (dimensions vbl))))
		((num-list-2-p zoom)
		 (apply 'min (mapcar #'(lambda (zoom-dim pic-dim) (/ zoom-dim pic-dim))
				     zoom (dimensions vbl))))
		(t 1)))
    (setf (getf initargs :zoom) (if (> zoom 1) (round zoom) (/ (round (/ zoom))))))
  (apply #'call-next-method pic vbl initargs))

;;; *** what to do about gamma correction?
;;; Write internal-dither-into-8bit (without the lut).  Then:
;;;   (add r-8bit (mul red-levels g-8bit) (mul (* red-levels green-levels) b-8bit))

;;; Zoom is ignored here.  Zooming happens in render.
(defmethod draw-color-float-arrays
    ((bltable X-bltable)
     r-float-array g-float-array b-float-array
     pedestal scale zoom
     x-offset y-offset)
  (declare (ignore zoom))
  (setq scale (/-0 1.0 scale 1.0))
  (let* ((screen (screen-of bltable))
	 (bltable-data (data bltable))
	 (red-levels (expt 2 (first (rgb-bits screen))))
	 (green-levels (expt 2 (second (rgb-bits screen))))
	 (blue-levels (expt 2 (third (rgb-bits screen))))
	 (r-lut (make-array red-levels :element-type '(unsigned-byte 8)))
	 (g-lut (make-array green-levels :element-type '(unsigned-byte 8)))
	 (b-lut (make-array blue-levels :element-type '(unsigned-byte 8))))
    (loop for i from 0 below red-levels do (setf (aref r-lut i) i))
    (loop for i from 0 below green-levels do (setf (aref g-lut i) (* red-levels i)))
    (loop for i from 0 below blue-levels do (setf (aref b-lut i) (* red-levels green-levels i)))
    (zero! bltable-data)
    (with-status-message "Dithering colors"
      (with-static-arrays ((tmp-8bit-array (allocate-array (dimensions bltable-data)
							   :element-type '(unsigned-byte 8))))
	(loop for float-array in (list r-float-array g-float-array b-float-array)
	      for lut in (list r-lut g-lut b-lut)
	      do
	      (zero! tmp-8bit-array)
	      (internal-dither-into-8bit-lut
	       float-array (x-dim float-array) (y-dim float-array)
	       tmp-8bit-array (x-dim tmp-8bit-array)
	       (float pedestal) (float (* scale (length lut)))
	       (round x-offset) (round y-offset) lut (length lut))
	      (add tmp-8bit-array bltable-data :-> bltable-data))))
    (with-status-message "Converting to color map values"
      (loop for j from 0 below (y-dim bltable-data) do
	    (loop for i from 0 below (x-dim bltable-data) do
		  (setf (aref bltable-data j i)
			(aref (color-lut screen) (aref bltable-data j i)))))))
  bltable)

#|
(load-image "/home/heeger/images/clown")
(setq little-clown (gauss-out clown))
(setq bltable (make-bltable (current-screen) (dimensions little-clown)))

(setq bltable (make-bltable (current-screen) (dimensions little-clown) :bltable bltable))
(draw-color-float-arrays bltable
			 (data (frame 0 little-clown))
			 (data (frame 1 little-clown))
			 (data (frame 2 little-clown))
			 0.0 256.0 1 0 0)
(render *current-pane* bltable 0 0 1)
|#
