/******************************************************************************\
********************************************************************************
********                                                                ********
******      Visual robot simulator                                        ******
****                                                                        ****
**          Author: Craig Dillon, Andrew Conway                               **
**                                                                            **
**          NEWPS.C                                                           **
****                                                                        ****
******                                                                    ******
********                                                                ********
********************************************************************************
\******************************************************************************/

#include<stdio.h>
#ifdef R_SGI
#include<gl/gl.h>
#include<gl/device.h>
#endif
#include<math.h>
#include<string.h>
#include<stdlib.h>
#include<malloc.h>
#include"windowlib.h"
#include"schematic.h"
#include"rob.h"

static FILE *psfile;
static int linemode;

void ps_start(FILE *handle,int x1,int y1,int x2,int y2)
{
	int width=x2-x1;
	int height=y2-y1;

	if(width>((7*height)/10)) {	/* too wide */
		height=(10*width)/7;
	} else {
		width=(7*height)/10;
	}
	psfile=handle;
	fprintf(psfile,"%%! Robot Simulator output\n");
	fprintf(psfile,"/leftmargin 20 def\n");
	fprintf(psfile,"/rightmargin 20 def\n");
	fprintf(psfile,"/topmargin 20 def\n");
	fprintf(psfile,"/bottommargin 20 def\n");
	fprintf(psfile,"/in {72 mul} def\n");
	fprintf(psfile,"/width 7 in def\n");
	fprintf(psfile,"/height 10 in def\n");
	fprintf(psfile,"leftmargin rightmargin translate\n");
	fprintf(psfile,"/logwidth %d def\n",width);
	fprintf(psfile,"/logheight %d def\n",height);
	fprintf(psfile,"/physwidth width leftmargin rightmargin add sub def\n");
	fprintf(psfile,"/physheight height topmargin leftmargin add sub def\n");
	fprintf(psfile,"physwidth logwidth div physheight logheight div scale\n");
	fprintf(psfile,"/Times-Roman findfont 10 scalefont setfont\n");
	fprintf(psfile,"%d %d translate\n",-x1,-y1);
}

void ps_end()
{
	fprintf(psfile,"showpage\n");
	fclose(psfile);
}

void ps_bgnline()
{
	fprintf(psfile,"newpath\n");
	linemode=0;
}

void ps_endline()
{
	fprintf(psfile,"stroke\n");
}

void ps_v2i(long *pos)
{
	if (linemode) fprintf(psfile,"%d %d lineto\n",pos[0],pos[1]); 
	else fprintf(psfile,"%d %d moveto\n",pos[0],pos[1]); 
	linemode=1;
}

void ps_cmov2i(int x,int y)
{
	fprintf(psfile,"%d %d moveto\n",x,y);
}

void ps_charstr(char *s)
{
	fprintf(psfile,"(%s) show\n",s);
}

void ps_recti(int x1,int y1,int x2,int y2)
{
	fprintf(psfile,"newpath %d %d moveto %d %d lineto %d %d lineto\n",
		x1,y1,x2,y1,x2,y2);
	fprintf(psfile,"%d %d lineto closepath stroke\n",x1,y2);
}

void ps_arci(int x,int y,int r,int from,int to)
{
	fprintf(psfile,"newpath %d %d %d %f %f arc stroke\n",x,y,r,from/10.0,
					to/10.0);
}

void ps_circi(int x,int y,int r)
{
	ps_arci(x,y,r,0,3600);
}

void ps_linestyle(int style)
{
	static char *(styles[])={"","1","2","3","1 3","2 5","3 5 1 5","2 1 2"};
	int numstyles=8;
	style%=numstyles;
	fprintf(psfile,"[%s] 0 setdash\n",styles[style]);
}
	
void print_inputnode(eptr)
ENTITY *eptr;
{
	INPUTNODE *iptr;
	long v[2];

#if(1)
	iptr=(INPUTNODE *)(eptr->entity);
	ps_linestyle(0);
	ps_arci(eptr->x+eptr->height/2,eptr->y+eptr->height/2,eptr->height/2,900,2700);
	ps_arci(eptr->x+eptr->width-eptr->height/2,eptr->y+eptr->height/2,eptr->height/2,2700,900);
	v[0]=eptr->x+eptr->height/2;
	v[1]=eptr->y+eptr->height;
	ps_bgnline();
	ps_v2i(v);
	v[0]+=eptr->width-eptr->height;
	ps_v2i(v);
	ps_endline();
	v[1]-=eptr->height;
	ps_bgnline();
	ps_v2i(v);
	v[0]-=(eptr->width-eptr->height);
	ps_v2i(v);
	ps_endline();
	ps_cmov2i(eptr->x+eptr->height/2,eptr->y+eptr->height/2);
	ps_charstr(eptr->name);
#else
#ifdef R_SGI
    lrectwrite(-schematic_x+eptr->x,-schematic_y+eptr->y,-schematic_x+eptr->x+ep
tr->width-1,-schematic_y+eptr->y+eptr->height-1,eptr->a);
    cpack(ENTITY_COLOUR);
    cmov2i(eptr->x+10,eptr->y+7);
    charstr(eptr->name);
#endif
#endif
}

void print_outputnode(eptr)
ENTITY *eptr;
{
	OUTPUTNODE *optr;
	long v[2];

#if(1)
    optr=(OUTPUTNODE *)(eptr->entity);
	ps_linestyle(0);
    ps_arci(eptr->x+eptr->height/2,eptr->y+eptr->height/2,eptr->height/2,900,2700);
    ps_arci(eptr->x+eptr->width-eptr->height/2,eptr->y+eptr->height/2,eptr->height/2,2700,900);
    v[0]=eptr->x+eptr->height/2;
    v[1]=eptr->y+eptr->height;
    ps_bgnline();
    ps_v2i(v);
    v[0]+=eptr->width-eptr->height;
    ps_v2i(v);
    ps_endline();
    v[1]-=eptr->height;
    ps_bgnline();
    ps_v2i(v);
    v[0]-=(eptr->width-eptr->height);
    ps_v2i(v);
	ps_endline();
    ps_cmov2i(eptr->x+eptr->height/2,eptr->y+eptr->height/2);
    ps_charstr(eptr->name);
#else
#ifdef R_SGI
    lrectwrite(-schematic_x+eptr->x,-schematic_y+eptr->y,-schematic_x+eptr->x+eptr->width-1,-schematic_y+eptr->y+eptr->height-1,eptr->a);
    cpack(ENTITY_COLOUR);
    cmov2i(eptr->x+10,eptr->y+7);
    charstr(eptr->name);
#endif
#endif
}

void print_transformation(eptr)
ENTITY *eptr;
{
    TRANSFORMATION *tptr;
	long v[2];

#if(1)
    tptr=(TRANSFORMATION *)(eptr->entity);
    ps_linestyle(0);
    ps_recti(eptr->x,eptr->y,eptr->x+eptr->width,eptr->y+eptr->height);
	v[0]=eptr->x+tptr->lx;
	v[1]=eptr->y+tptr->ly;
	ps_bgnline();
	ps_v2i(v);
	v[0]+=tptr->ll;
	ps_v2i(v);
	ps_endline();
	ps_cmov2i(eptr->x+tptr->tx,eptr->y+tptr->ty);
	ps_charstr(tptr->top);
	ps_cmov2i(eptr->x+tptr->bx,eptr->y+tptr->by);
	ps_charstr(tptr->bot);
#else
#ifdef R_SGI
	tptr=(TRANSFORMATION *)(eptr->entity);
    lrectwrite(-schematic_x+eptr->x,-schematic_y+eptr->y,-schematic_x+eptr->x+eptr->width-1,-schematic_y+eptr->y+eptr->height-1,eptr->a);
    cpack(ENTITY_COLOUR);
    v[0]=eptr->x+tptr->lx;
    v[1]=eptr->y+tptr->ly;
    bgnline();
    v2i(v);
    v[0]+=tptr->ll;
    v2i(v);
    endline();
    cmov2i(eptr->x+tptr->tx,eptr->y+tptr->ty);
    charstr(tptr->top);
    cmov2i(eptr->x+tptr->bx,eptr->y+tptr->by);
    charstr(tptr->bot);
#endif
#endif
}

void print_summation(eptr)
ENTITY *eptr;
{
	SUMMATION *sptr;

#if(1)
    sptr=(SUMMATION *)(eptr->entity);
    ps_linestyle(0);
    ps_circi(eptr->x+eptr->width/2,eptr->y+eptr->height/2,eptr->width/2);
#else
#ifdef R_SGI
    lrectwrite(-schematic_x+eptr->x,-schematic_y+eptr->y,-schematic_x+eptr->x+eptr->width-1,-schematic_y+eptr->y+eptr->width-1,eptr->a);
    cpack(ENTITY_COLOUR);
    cmov2i(eptr->x+5,eptr->y+5);
    charstr("+");
#endif
#endif
}

void print_difference(eptr)
ENTITY *eptr;
{
	DIFFERENCE *dptr;

#if(1)
    dptr=(DIFFERENCE *)(eptr->entity);
    ps_linestyle(0);
	ps_circi(eptr->x+eptr->width/2,eptr->y+eptr->height/2,eptr->width/2);
#else
#ifdef R_SGI
    lrectwrite(-schematic_x+eptr->x,-schematic_y+eptr->y,-schematic_x+eptr->x+eptr->width-1,-schematic_y+eptr->y+eptr->width-1,eptr->a);
    cpack(ENTITY_COLOUR);
    cmov2i(eptr->x+5,eptr->y+5);
    charstr("-");
#endif
#endif
}

void print_bias(eptr)
ENTITY *eptr;
{
    BIAS *bptr;

#if(1)
#else
#ifdef R_SGI
    bptr=(BIAS *)(eptr->entity);
    lrectwrite(-schematic_x+eptr->x,-schematic_y+eptr->y,-schematic_x+eptr->x+eptr->width-1,-schematic_y+eptr->y+eptr->height-1,eptr->a);
    cpack(ENTITY_COLOUR);
    cmov2i(eptr->x+10,eptr->y+7);
    charstr(bptr->str);
#endif
#endif
}

#define PORT_LENGTH 10
#define PORT_ARROW 5

void print_connections()	/* draws schematic connections */
{
	int i,j,sp,dp;
	ENTITY *septr,*deptr;
	long v[2];
	int ex,ey,sx,sy,mx,my,sd,dd;
	static int dx[]={0,1,0,-1};
	static int dy[]={1,0,-1,0};

	for(i=0;i<numentities;i++) {
		septr=entity[i];
		for(j=0;j<septr->numports;j++) {
			if(septr->mode[j]==PORT_INPUT) {
				sp=j;
				deptr=entity[septr->c_entity[j]];
				dp=septr->c_port[j];
				sd=septr->position[sp];
				dd=deptr->position[dp];
	
	
				/* draw arrow head */
				sx=septr->x+septr->dx[sd];
				sy=septr->y+septr->dy[sd];
				v[0]=sx;
				v[1]=sy;
				ps_bgnline();
				ps_v2i(v);
				v[0]+=dx[sd]*PORT_ARROW;
				v[1]+=dy[sd]*PORT_ARROW;
				v[0]+=dx[(sd+5)%4]*PORT_ARROW;
				v[1]+=dy[(sd+5)%4]*PORT_ARROW;
				ps_v2i(v);
				ps_endline();
				
				v[0]=sx;
                v[1]=sy;
                ps_bgnline();
                ps_v2i(v);
                v[0]+=dx[sd]*PORT_ARROW;
                v[1]+=dy[sd]*PORT_ARROW;
                v[0]+=dx[(sd+3)%4]*PORT_ARROW;
                v[1]+=dy[(sd+3)%4]*PORT_ARROW;
                ps_v2i(v);
                ps_endline();

				/* draw from septr,sp to deptr,dp */
				sx=septr->x+septr->dx[sd];
                sy=septr->y+septr->dy[sd];
				ex=deptr->x+deptr->dx[dd];
				ey=deptr->y+deptr->dy[dd];

				v[0]=sx;
				v[1]=sy;
				ps_bgnline();
				ps_v2i(v);
				sx+=dx[sd]*PORT_LENGTH;	
				sy+=dy[sd]*PORT_LENGTH;
				v[0]=sx;
                v[1]=sy;
				ps_v2i(v);
				ps_endline();

				v[0]=ex;
                v[1]=ey;
                ps_bgnline();
                ps_v2i(v);
                ex+=dx[dd]*PORT_LENGTH;     
                ey+=dy[dd]*PORT_LENGTH;
                v[0]=ex;
                v[1]=ey;
                ps_v2i(v);
                ps_endline();
				
				mx=sx;
				my=ey;		/* swap these to go other way */
				if(schematic_blocked(sx,sy,mx,my)||
				   schematic_blocked(mx,my,ex,ey)) {
					mx=ex;
					my=sy;
				}
				v[0]=sx;
				v[1]=sy;
				ps_bgnline();
				ps_v2i(v);
				v[0]=mx;
				v[1]=my;
				ps_v2i(v);
				v[0]=ex;
				v[1]=ey;
				ps_v2i(v);
				ps_endline();
			}
		}
	}
}

void print_schematic()
{
	int i;
	ENTITY *eptr;
	char s[80];
	long sx,sy,lx,ly;
	FILE *out;

	sprintf(s,"%s.ps",schematic_print_name);
	out=fopen(s,"w");
	if(!out) {
		printf("Cannot open file: %s\n",s);
		exit(0);
	}

	eptr=entity[0];
	lx=sx=eptr->x;
	ly=sy=eptr->y;
	for(i=0;i<numentities;i++) {
		eptr=entity[i];
		if(eptr->x<sx) sx=eptr->x;
		if(eptr->y<sy) sy=eptr->y;
		if(eptr->x+eptr->width>lx) lx=eptr->x+eptr->width;
		if(eptr->y+eptr->height>ly) ly=eptr->y+eptr->height;
	}
	sx-=50;
	sy-=50;
	lx+=50;
	ly+=50;
	ps_start(out,sx,sy,lx,ly);
	for(i=0;i<numentities;i++) {
		eptr=entity[i];
		switch(eptr->type) {
			case ENTITY_INPUTNODE:
				print_inputnode(eptr);
				break;

			case ENTITY_OUTPUTNODE:
				print_outputnode(eptr);
				break;

			case ENTITY_TRANSFORMATION:
				print_transformation(eptr);
				break;

			case ENTITY_SUMMATION:
				print_summation(eptr);
				break;

			case ENTITY_DIFFERENCE:
				print_difference(eptr);
				break;

			case ENTITY_BIAS:
				print_bias(eptr);
				break;

			default:
				printf("Unknown entity type: %d\n",eptr->type);
				exit(0);
		}
	}
	print_connections();
	ps_end();
}

void print_one_graph(gptr)	/* does whole graph from scratch */
GRAPH *gptr;
{	
	int i,j,sx,ex;
	GPARAMETER *ggptr;
	float miny,maxy,extra;
	int vx,vy;
	long vert[2];
	float st,et,v;
	char s[40];

	if(step_count==0) return;
	ps_recti(gptr->x+55,gptr->y+25,gptr->x+gptr->width-1-5,gptr->y+gptr->height-1-25);
	if(gptr->horizontal_mode==GMODE_AUTO) {
		sx=0;
		ex=step_count;
		st=0.0;
		et=actual_time[step_count-1];
	} else {
		printf("Graph Mode Fixed not supported yet\n");
		exit(0);
	}
	miny=maxy=(gptr->parameter[0])->graph[sx];
	for(i=0;i<(gptr->num_parameters);i++) {
		ggptr=gptr->parameter[i];
		for(j=sx;j<ex;j++) {
			if((ggptr->graph[j])>maxy) maxy=ggptr->graph[j];
			if((ggptr->graph[j])<miny) miny=ggptr->graph[j];
		}
	}	
	extra=(maxy-miny)*0.1;
	maxy+=extra;
	miny-=extra;
	ps_linestyle(0);
	for(i=0;i<gptr->vertical_labels;i++) {
		v=miny+(maxy-miny)*(float)i/(float)(gptr->vertical_labels-1);
		sprintf(s,"%2.2f",v);
		s[5]=0;
		ps_cmov2i(gptr->x+7,gptr->y+28+(i*(gptr->height-66))/(gptr->vertical_labels-1));
		ps_charstr(s);
	}
	for(i=0;i<gptr->horizontal_labels;i++) {
        v=st+(et-st)*(float)i/(float)(gptr->horizontal_labels-1);
        sprintf(s,"%2.2f",v);
        s[5]=0;
        ps_cmov2i(gptr->x+57+(i*(gptr->width-110))/(gptr->horizontal_labels-1),gptr->y+9);
        ps_charstr(s);
    }
	for(i=0;i<gptr->num_parameters;i++) {
        ggptr=gptr->parameter[i];
		ps_linestyle(i);
		ps_bgnline();
		for(j=sx;j<ex;j++) {
			vx=gptr->x+59+(int)
				(((actual_time[j]-st)*(float)(gptr->width-67))/(et-st));
			vy=gptr->y+30+(int)
				(((float)(gptr->height-60)*(ggptr->graph[j]-miny))/(maxy-miny));
			vert[0]=vx;
			vert[1]=vy;
			ps_v2i(vert);
		}
		ps_endline();
	}
}

void print_graph()
{
    long ow;
    int i,j;
    GRAPH *gptr;
	GPARAMETER *ggptr;
	char *s,ss[256];
	int pos;
	FILE *out;
	long sx,sy,lx,ly;

	sprintf(ss,"%s.ps",save_name);
	out=fopen(ss,"w");
	if(!out) {
		printf("File could not be opened: %s\n",ss);
		exit(0);
	}
	gptr=graph[0];
    lx=sx=gptr->x;
    ly=sy=gptr->y;
    for(i=0;i<numgraphs;i++) {
        gptr=graph[i];
        if(gptr->x<sx) sx=gptr->x;
        if(gptr->y<sy) sy=gptr->y;
        if(gptr->x+gptr->width>lx) lx=gptr->x+gptr->width;
        if(gptr->y+gptr->height>ly) ly=gptr->y+gptr->height;
    }
    sx-=50;
    sy-=50;
    lx+=50;
    ly+=50;
	ps_start(out,sx,sy,lx,ly);
    for(i=0;i<numgraphs;i++) {
        gptr=graph[i];
		ps_recti(gptr->x,gptr->y,gptr->x+gptr->width-1,gptr->y+gptr->height-1);
		pos=0;
		for(j=0;j<gptr->num_parameters;j++) {
			ggptr=gptr->parameter[j];
			if(ggptr->source==GS_JOINT) {
				switch(ggptr->type) {
					case PAR_X: s="x"; break;
					case PAR_Y: s="y"; break;
					case PAR_Z: s="z"; break;
					case PAR_XD: s="xd"; break;
					case PAR_YD: s="yd"; break;
					case PAR_ZD: s="zd"; break;
					case PAR_DX: s="dx"; break;
					case PAR_DY: s="dy"; break;
					case PAR_DZ: s="dz"; break;
					case PAR_DXD: s="dxd"; break;
					case PAR_DYD: s="dyd"; break;
					case PAR_DZD: s="dzd"; break;
					case PAR_A: s="a"; break;
					case PAR_D: s="d"; break;
					case PAR_ALPHA: s="alpha"; break;
					case PAR_THETA: s="theta"; break;
					case PAR_AD: s="ad"; break;
					case PAR_DD: s="dd"; break;
					case PAR_ALPHAD: s="alphad"; break;
					case PAR_THETAD: s="thetad"; break;
					case PAR_ADD: s="add"; break;
					case PAR_DDD: s="ddd"; break;
					case PAR_ALPHADD: s="alphadd"; break;
					case PAR_THETADD: s="thetadd"; break;
					case PAR_FA: s="fa"; break;
					case PAR_FD: s="fd"; break;
					case PAR_FALPHA: s="falpha"; break;
					case PAR_FTHETA: s="ftheta"; break;
					default: printf("Unknown mode\n"); exit(0);
				}
				sprintf(ss,"%s:%s",joint[ggptr->num]->name,s);
			} else {
				sprintf(ss,"%s",entity[ggptr->num]->name);
			}
			ps_cmov2i(gptr->x+60+pos,gptr->y+gptr->height-20);
            ps_charstr(ss);
            pos+=8+10*strlen(ss);
		}
    }
	for(i=0;i<numgraphs;i++) print_one_graph(graph[i]);
	ps_end();
}
