/******************************************************************************\
********************************************************************************
********                                                                ********
******      Visual robot simulator                                        ******
****                                                                        ****
**          Author: Craig Dillon, Andrew Conway                               **
**                                                                            **
****        FILE.C                                                          ****
******                                                                    ******
********                                                                ********
********************************************************************************
\******************************************************************************/

#include<stdio.h>
#include<math.h>
#include<malloc.h>
#include<sys/types.h>
#include<string.h>
#include<stdlib.h>
#ifdef R_SGI
#include<gl.h>
#include<gl/device.h>
#endif
#include"windowlib.h"
#include"rob.h"
#include"schematic.h"

#define PRINT_SCHEMATIC	0		/* use for debugging */

int find_box(name)
char *name;
{
    int i;
    for(i=0;i<numboxes;i++) {
        if(!strcmp(name,box[i]->name)) return(i);
    }
    return(-1);
}

int find_variable(name)	
char *name;
{
	int i;

	for(i=0;i<numvariables;i++) {
		if(!strcmp(name,variable[i]->name)) return(i);
	}
	return(-1);
}

int find_graph(name)
char *name;
{
    int i;

    for(i=0;i<numgraphs;i++) {
        if(!strcmp(name,graph[i]->name)) return(i);
    }
    return(-1);
}

int find_joint(name)
char *name;
{
    int i;

    for(i=0;i<numjoints;i++) {
        if(!strcmp(name,joint[i]->name)) return(i);
    }
    return(-1);
}

int find_schematic(name)
char *name;
{
    int i;

    for(i=0;i<numentities;i++) {
        if(!strcmp(name,entity[i]->name)) return(i);
    }
    return(-1);
}

int find_port(e,name)
int e;	/* entity type */
char *name;
{
	int p;

	p= -1;
	switch(e) {
		case ENTITY_INPUTNODE:
			if(!strcmp(name,"output")) p=INPUTNODE_OUT;
			break;

		case ENTITY_OUTPUTNODE:
			if(!strcmp(name,"input")) p=OUTPUTNODE_IN;
			break;

		case ENTITY_TRANSFORMATION:
			if(!strcmp(name,"input")) p=TRANSFORMATION_IN;
			if(!strcmp(name,"output")) p=TRANSFORMATION_OUT;
			break;

		case ENTITY_SUMMATION:
			if(!strcmp(name,"inputa")) p=SUMMATION_INA;
			if(!strcmp(name,"inputb")) p=SUMMATION_INB;
			if(!strcmp(name,"output")) p=SUMMATION_OUT;
			break;

		case ENTITY_DIFFERENCE:
			if(!strcmp(name,"inputa")) p=DIFFERENCE_INA;
			if(!strcmp(name,"inputb")) p=DIFFERENCE_INB;
			if(!strcmp(name,"output")) p=DIFFERENCE_OUT;
			break;

		case ENTITY_BIAS:
			if(!strcmp(name,"output")) p=BIAS_OUT;
			break;

		default:
			printf("Unknown entity type\n");
			exit(0);
	}
	return(p);
}

void define_angle(s)
char *s;
{
	char *name;
	char ss[80];
	VARIABLE *vptr;
	int num;

	num=sscanf(s," %s %s",ss,ss);	
	if(num!=2) {
		printf("File format error: %s\n",s);
		printf("Usage: angle <name>\n");
		exit(0);
	}
	if(find_variable(ss)!= -1) {
		printf("Variable %s defined more than once\n",ss);
		exit(0);
	}
	name=(char *)calloc(strlen(ss)+1,sizeof(char));
	strcpy(name,ss);
	vptr=(VARIABLE *)calloc(1,sizeof(VARIABLE));
	variable[numvariables]=vptr;
	numvariables++;
	vptr->name=name;
	vptr->type=ROB_ANGLE;
	vptr->value=0.0;
	vptr->min=0.0;
	vptr->max=2.0*M_PI;
}

void define_distance(s)
char *s;
{
    char *name;
    char ss[80];
	VARIABLE *vptr;
	int num;

    num=sscanf(s," %s %s",ss,ss);
	if(num!=2) {
        printf("File format error: %s\n",s);
        printf("Usage: distance <name>\n");
        exit(0);
    }
	if(find_variable(ss)!= -1) {
        printf("Variable %s defined more than once\n",ss);
        exit(0);
    }
    name=(char *)calloc(strlen(ss)+1,sizeof(char));
    strcpy(name,ss);
    vptr=(VARIABLE *)calloc(1,sizeof(VARIABLE));
    variable[numvariables]=vptr;
    numvariables++;
    vptr->name=name;
    vptr->type=ROB_DIST;
    vptr->value=0.0;
    vptr->min=0.0;
    vptr->max=1.0;
}

void bound_variable(s)
char *s;
{
	char name[80];
	double min,max;
	int vnum,num;
	VARIABLE *vptr;

	num=sscanf(s," %s %s %lf %lf",name,name,&min,&max);
	if(num!=4) {
		printf("File format error: %s\n",s);
		printf("Usage: bound <name> <min> <max>\n");
		exit(0);
	}
	vnum=find_variable(name);
	if(vnum== -1) {
		printf("Variable %s not defined\n",name);
		exit(0);
	}
	vptr=variable[vnum];
	vptr->max=max;
	vptr->min=min;
}

void set_variable(s)
char *s;
{
	char name[80];
    double value;
	VARIABLE *vptr;
	int num,vnum;

    num=sscanf(s," %s %s %lf",name,name,&value);
	if(num!=3) {
		printf("File format error: %s\n",s);
		printf("Usage: set <name> <value>\n");
		exit(0);
	}
	vnum=find_variable(name);
	if(vnum== -1) {
		printf("Variable %s undefined\n",name);
		exit(0);
	}
	vptr=variable[vnum];
    vptr->value=value;
}

void define_joint(s)
char *s;
{
	char name[80],sa[80],sd[80],salpha[80],stheta[80],mode[80];
	int num,ca,cd,calpha,ctheta,i;
	JOINT *jptr;
	float *material;

	if(robot_created) {
		printf("File order error: %s\n",s);
		printf("You cannot create joints after the robot has been\n");
		printf("created. Ensure that all from and to commands are\n");
		printf("after the define joint commands.\n");
		exit(0);
	}
	num=sscanf(s," %s %s %s %s %s %s %s",name,name,sa,sd,salpha,stheta,mode);
	if(num!=7) {
		printf("File format error: %s\n",s);
		printf("Usage: joint <name> <a> <d> <alpha> <theta> <simulation mode>\n");
		exit(0);
	}
	jptr=(JOINT *)calloc(1,sizeof(JOINT));
	ca=find_variable(sa);
	cd=find_variable(sd);
	calpha=find_variable(salpha);
	ctheta=find_variable(stheta);
	
	if(ca<0) {
		sscanf(sa,"%lf",&(jptr->a));
		jptr->ca=ROB_SET;
	} else {
		jptr->a_var=ca;
		jptr->ca=ROB_CON;
	}
    if(cd<0) {
        sscanf(sd,"%lf",&(jptr->d));
        jptr->cd=ROB_SET;
    } else {
        jptr->d_var=cd;
        jptr->cd=ROB_CON;
    }
    if(calpha<0) {
        sscanf(salpha,"%lf",&(jptr->alpha));
        jptr->calpha=ROB_SET;
    } else {
        jptr->alpha_var=calpha;
        jptr->calpha=ROB_CON;
    }
    if(ctheta<0) {
        sscanf(stheta,"%lf",&(jptr->theta));
        jptr->ctheta=ROB_SET;
    } else {
        jptr->theta_var=ctheta;
        jptr->ctheta=ROB_CON;
    }
	jptr->sim_mode= -1;
	if(!strcmp(mode,"none")) jptr->sim_mode=SIM_NONE;
	if(!strcmp(mode,"force")) jptr->sim_mode=SIM_F;
	if(!strcmp(mode,"acceleration")) jptr->sim_mode=SIM_VDD;
	if(!strcmp(mode,"velocity")) jptr->sim_mode=SIM_VD;
	if(!strcmp(mode,"position")) jptr->sim_mode=SIM_V;
	if(jptr->sim_mode<0) {
		printf("Simulation mode: %s unknown\n",mode);
		exit(0);
	}
	jptr->name=strdup(name);
	jptr->radius=0.1;
	jptr->numpanels=4;
	material=(float *)calloc(15,sizeof(float));
#ifdef R_SGI
	for(i=0;i<15;i++) material[i]=mat_panel[i];
#endif
	jptr->material=material;
	joint[numjoints]=jptr;
	numjoints++;
}

void set_tensor(s)
char *s;
{
	int n,num;
	char name[40];
	double xx,yy,zz,xy,xz,yz,x,y,z,mass;
	JOINT *jptr;

	num=sscanf(s," %s %s %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf",name,name,
		&xx,&yy,&zz,&xy,&xz,&yz,&x,&y,&z,&mass);
	if(num!=12) {
		printf("File format error: %s\n",s);
		printf("Usage: tensor name xx yy zz xy xz yz x y z mass\n");
		exit(0);
	}
	n=find_joint(name);
	if(n== -1) {
		printf("Joint %s not known\n",name);
		exit(0);
	}
	jptr=joint[n];
	jptr->J[0][0]=xx;
	jptr->J[0][1]=xy;
	jptr->J[0][2]=xz;
	jptr->J[0][3]=x;
	jptr->J[1][0]=xy;
	jptr->J[1][1]=yy;
	jptr->J[1][2]=yz;
	jptr->J[1][3]=y;
	jptr->J[2][0]=xz;
	jptr->J[2][1]=yz;
	jptr->J[2][2]=zz;
	jptr->J[2][3]=z;
	jptr->J[3][0]=x;
	jptr->J[3][1]=y;
	jptr->J[3][2]=z;
	jptr->J[3][3]=mass;
}	

void set_friction(s)
char *s;
{
    int n,p,num;
    char name[40],pname[40];
    double v,vs,min;
    JOINT *jptr;

    num=sscanf(s," %s %s %s %lf %lf %lf",name,name,pname,&v,&vs,&min);
    if(num!=6) {
        printf("File format error: %s\n",s);
        printf("Usage: friction name parameter vel velsquared minimum\n");
        exit(0);
    }
    n=find_joint(name);
    if(n== -1) {
        printf("Joint %s not known\n",name);
        exit(0);
    }
	jptr=joint[n];
	p= -1;
	if(!strcmp(pname,"a")) p=PAR_A;
	if(!strcmp(pname,"d")) p=PAR_D;
	if(!strcmp(pname,"alpha")) p=PAR_ALPHA;
	if(!strcmp(pname,"theta")) p=PAR_THETA;
	if(p== -1) {
		printf("Unknown Parameter: %s\n",pname);
		exit(0);
	}
	switch(p) {
	
		case PAR_A:
			jptr->fric_ad=v;
			jptr->fric_ads=vs;
			jptr->fric_amin=min;
			break;

		case PAR_D:
            jptr->fric_dd=v;
            jptr->fric_dds=vs;
            jptr->fric_dmin=min;
            break;

		case PAR_ALPHA:
            jptr->fric_alphad=v;
            jptr->fric_alphads=vs;
            jptr->fric_alphamin=min;
            break;

		case PAR_THETA:
            jptr->fric_thetad=v;
            jptr->fric_thetads=vs;
            jptr->fric_thetamin=min;
            break;

		default:
			printf("Illegal parameter type\n");
			exit(0);
	}
}   

void set_boundtime(s)
char *s;
{
	char ss[40];
	int num;

	num=sscanf(s," %s %lf %lf",ss,&time_bound_min,&time_bound_max);
	if(num!=3) {
		printf("File format error: %s\n",s);
		printf("Usage: boundtime <min> <max>\n");
		exit(0);
	}
}

void set_numsteps(s)
char *s;
{
    char ss[40];
    int num;

    num=sscanf(s," %s %d",ss,&step_num);
    if(num!=2) {
        printf("File format error: %s\n",s);
        printf("Usage: numsteps <number>\n");
        exit(0);
    }
	if((step_num<0)||(step_num>9999)) {
		printf("Number of steps requested outside permitted range\n");
		printf("[0-9999]\n");
		exit(0);
	}
}   

void set_timestep(s)
char *s;
{
    char ss[40];
    int num;

    num=sscanf(s," %s %lf",ss,&time_step);
    if(num!=2) {
        printf("File format error: %s\n",s);
        printf("Usage: time_step <value>\n");
        exit(0);
    }
}

void setup_joints()
{
	int i;
	JOINT *sjptr,*djptr;

	if(robot_created) return;
	for(i=0;i<numjoints;i++) {
		sjptr=joint[i];

		djptr=(JOINT *)calloc(1,sizeof(JOINT));
		from_joint[i]=djptr;
		memcpy(djptr,sjptr,sizeof(JOINT));

		djptr=(JOINT *)calloc(1,sizeof(JOINT));
        to_joint[i]=djptr;
        memcpy(djptr,sjptr,sizeof(JOINT));

		djptr=(JOINT *)calloc(1,sizeof(JOINT));
        pjoint[i]=djptr;
        memcpy(djptr,sjptr,sizeof(JOINT));
	}
	robot_created=1;
}

void update_joints()
{
	int i;
	JOINT *sptr,*daptr,*dbptr;

	for(i=0;i<numjoints;i++) {
		sptr=joint[i];
		daptr=from_joint[i];
		dbptr=to_joint[i];

		daptr->ca=dbptr->ca=sptr->ca;
		daptr->cd=dbptr->cd=sptr->cd;
		daptr->calpha=dbptr->calpha=sptr->calpha;
		daptr->ctheta=dbptr->ctheta=sptr->ctheta;

        daptr->a_var=dbptr->a_var=sptr->a_var;
        daptr->d_var=dbptr->d_var=sptr->d_var;
        daptr->alpha_var=dbptr->alpha_var=sptr->alpha_var;
        daptr->theta_var=dbptr->theta_var=sptr->theta_var;

		daptr->s_a=dbptr->s_a=sptr->s_a;
        daptr->s_d=dbptr->s_d=sptr->s_d;
        daptr->a_alpha=dbptr->a_alpha=sptr->a_alpha;
        daptr->a_theta=dbptr->a_theta=sptr->a_theta;
	}
}

void graph_activate(s)
char *s;
{
	graph_active=1;
}

void graph_create(s)
char *s;
{
	char name[40];
	int num,x,y,width,height;
	int i,j;
	GRAPH *gptr;

	num=sscanf(s," %s %s %s %d %d %d %d",name,name,name,&x,&y,&width,&height);
	if(num!=7) {
		printf("File format error: %s\n",s);
		printf("Usage: graph create <name> <x> <y> <width> <height>\n");
		exit(0);
	}
	num=find_graph(name);
	if(num!= -1) {
		printf("Graph already exists: %s\n",name);
		exit(0);
	}
	gptr=(GRAPH *)calloc(1,sizeof(GRAPH));
	gptr->x=x;
	gptr->y=y;
	gptr->width=width;
	gptr->height=height;
	gptr->name=strdup(name);

	gptr->wout=(long *)calloc(width*height,sizeof(long));
	fill_array(width,height,gptr->wout,0x00808080);
	gptr->win=(long *)calloc((width-60)*(height-50),sizeof(long));
	fill_invarray(width-60,height-50,gptr->win,0x00808080);
	for(j=10;j<height-60;j+=20) {
		for(i=4;i<width-63;i++) {
			gptr->win[i+j*(width-60)]=0x0000ff00;
		}
	}
	for(i=10;i<width-65;i+=20) {
		for(j=3;j<height-54;j++) {
			gptr->win[i+j*(width-60)]=0x0000ff00;
		}
	}

	graph[numgraphs]=gptr;
	numgraphs++;
	
	if((x+width+10)>graph_width) graph_width=(x+width+10);
	if((y+height+10)>graph_height) graph_height=(y+height+10);
}

void graph_mode(s)
char *s;
{
	char name[40],hmode[40],vmode[40];
	int num,g,h,v;

	num=sscanf(s," %s %s %s %s %s",name,name,name,hmode,vmode);
	if(num!=5) {
		printf("File format error: %s\n",s);
		printf("Usage: graph mode <name> <horizontal> <vertical>\n");
		exit(0);
	}
	g=find_graph(name);
	if(g== -1) {
		printf("Uknown graph: %s\n",name);
		exit(0);
	}
	h= -1;
	v= -1;
	if(!strcmp(hmode,"fixed")) h=GMODE_FIXED;
	if(!strcmp(hmode,"auto")) h=GMODE_AUTO;
	if(!strcmp(vmode,"fixed")) v=GMODE_FIXED;
	if(!strcmp(vmode,"auto")) v=GMODE_AUTO;
	if(h== -1) {
		printf("Unknown mode: %s\n",hmode);
		exit(0);
	}
	if(v== -1) {
		printf("Unknown mode: %s\n",vmode);
		exit(0);
	}
	graph[g]->vertical_mode=v;
	graph[g]->horizontal_mode=h;
}

void graph_horizontal(s)
char *s;
{
	char name[40];
	float scale;
	int num,g;

	num=sscanf(s," %s %s %s %f",name,name,name,&scale);
	if(num!=4) {
		printf("File format error: %s\n",s);
		printf("Usage: graph horizontal <name> <scale>\n");
		exit(0);
	}
	g=find_graph(name);
	if(g== -1) {
		printf("Unknown graph: %s\n",name);
		exit(0);
	}
	graph[g]->horizontal_scale=scale;
}

void graph_vertical(s)
char *s;
{
    char name[40];
    float min,max;
    int num,g;

    num=sscanf(s," %s %s %s %f %f",name,name,name,&min,&max);
    if(num!=5) {
        printf("File format error: %s\n",s);
        printf("Usage: graph vertical <name> <min> <max>\n");
        exit(0);
    }
    g=find_graph(name);
    if(g== -1) {
        printf("Unknown graph: %s\n",name);
        exit(0);
    }
    graph[g]->vertical_min=min;
    graph[g]->vertical_max=max;
}   

void graph_labels(s)
char *s;
{
	char name[40];
	int num,g,h,v;

	num=sscanf(s," %s %s %s %d %d",name,name,name,&h,&v);
    if(num!=5) {
        printf("File format error: %s\n",s);
        printf("Usage: graph labels <name> <horizontal> <vertical>\n");
        exit(0);
    }
    g=find_graph(name);
    if(g== -1) {
        printf("Unknown graph: %s\n",name);
        exit(0);
    }
    graph[g]->horizontal_labels=h;
    graph[g]->vertical_labels=v;
}   

void graph_trace(s)
char *s;
{
	char gname[40],jname[40],pname[40],cname[40];
	int num,g,p,j,i;
	long rgb;
	GRAPH *gptr;
	GPARAMETER **gpptr,*pptr;
	JOINT *jptr;
	double *vptr;

	num=sscanf(s," %s %s %s %s %s %s",gname,gname,gname,jname,pname,cname);
	if(num!=6) {
		printf("File format error: %s\n",s);
		printf("Usage: graph trace <graph name> <joint name>\n");
		printf("       <parameter name> <colour>\n");
		exit(0);
	}
	g=find_graph(gname);
	if(g== -1) {
        printf("Unknown graph: %s\n",gname);
        exit(0);
    }
	gptr=graph[g];
	j=find_joint(jname);
	if(j== -1) {
		printf("Unknown joint: %s\n",jname);
		exit(0);
	}
	p= -1;
	jptr=joint[j];
	if(!strcmp(pname,"x")) {
		p=PAR_X;
		vptr= &(jptr->x);
	}
	if(!strcmp(pname,"y")) {
		p=PAR_Y;
		vptr= &(jptr->y);
	}
	if(!strcmp(pname,"z")) {
		p=PAR_Z;
		vptr= &(jptr->z);
	}
	if(!strcmp(pname,"xd")){
		p=PAR_XD;
		vptr= &(jptr->xd);
	}
	if(!strcmp(pname,"yd")){
		p=PAR_YD;
		vptr= &(jptr->yd);
	}
	if(!strcmp(pname,"zd")){
		p=PAR_ZD;
		vptr= &(jptr->zd);
	}
	if(!strcmp(pname,"dx")){
		p=PAR_DX;
		vptr= &(jptr->dx);
	}
	if(!strcmp(pname,"dy")) {
		p=PAR_DY;
		vptr= &(jptr->dy);
	}
	if(!strcmp(pname,"dz")) {
		p=PAR_DZ;
		vptr= &(jptr->dz);
	}
	if(!strcmp(pname,"dxd")) {
		p=PAR_DXD;
		vptr= &(jptr->dxd);
	}
	if(!strcmp(pname,"dyd"))  {
		p=PAR_DYD;
		vptr= &(jptr->dyd);
	}
	if(!strcmp(pname,"dzd")) {
		p=PAR_DZD;
		vptr= &(jptr->dzd);
	}
	if(!strcmp(pname,"a"))  {
		p=PAR_A;
		vptr= &(jptr->a);
	}
	if(!strcmp(pname,"d")) {
		p=PAR_D;
		vptr= &(jptr->d);
	}
	if(!strcmp(pname,"alpha")) {
		p=PAR_ALPHA;
		vptr= &(jptr->alpha);
	}
	if(!strcmp(pname,"theta")) {
		p=PAR_THETA;
		vptr= &(jptr->theta);
	}
	if(!strcmp(pname,"ad"))  {
		p=PAR_AD;
		vptr= &(jptr->ad);
	}
	if(!strcmp(pname,"dd")) {
		p=PAR_DD;
		vptr= &(jptr->dd);
	}
	if(!strcmp(pname,"alphad")) {
		p=PAR_ALPHAD;
		vptr= &(jptr->alphad);
	}
	if(!strcmp(pname,"thetad")) {
		p=PAR_THETAD;
		vptr= &(jptr->thetad);
	}
	if(!strcmp(pname,"add"))  {
		p=PAR_ADD;
		vptr= &(jptr->add);
	}
	if(!strcmp(pname,"ddd")) {
		p=PAR_DDD;
		vptr= &(jptr->ddd);
	}
	if(!strcmp(pname,"alphadd")) {
		p=PAR_ALPHADD;
		vptr= &(jptr->alphadd);
	}
	if(!strcmp(pname,"thetadd")) {
		p=PAR_THETADD;
		vptr= &(jptr->thetadd);
	}
	if(!strcmp(pname,"fa")) {
		p=PAR_FA;
		vptr= &(jptr->fa);
	}
	if(!strcmp(pname,"fd")) {
		p=PAR_FD;
		vptr= &(jptr->fd);
	}
	if(!strcmp(pname,"falpha")) {
		p=PAR_FALPHA;
		vptr= &(jptr->falpha);
	}
	if(!strcmp(pname,"ftheta")) {
		p=PAR_FTHETA;
		vptr= &(jptr->ftheta);
	}
	if(p== -1) {
		printf("Unknown parameter name: %s\n",pname);
		exit(0);
	}
	rgb= -1;
	if(!strcmp(cname,"black")) rgb=0x00000000;
	if(!strcmp(cname,"red")) rgb=0x000000ff;
	if(!strcmp(cname,"green")) rgb=0x0000ff00;
	if(!strcmp(cname,"yellow")) rgb=0x0000ffff;
	if(!strcmp(cname,"blue")) rgb=0x00ff0000;
	if(!strcmp(cname,"magenta")) rgb=0x00ff00ff;
	if(!strcmp(cname,"cyan")) rgb=0x00ffff00;
	if(!strcmp(cname,"white")) rgb=0x00ffffff;
	if(rgb== -1) {	
		printf("Unknown colour: %s\n",cname);
		exit(0);
	}
	gpptr=(GPARAMETER **)calloc(1+gptr->num_parameters,sizeof(GPARAMETER *));
	for(i=0;i<gptr->num_parameters;i++) {
     	gpptr[i]=gptr->parameter[i];
	}
	pptr=(GPARAMETER *)calloc(1,sizeof(GPARAMETER));
	gpptr[gptr->num_parameters]=pptr;
	pptr->source=GS_JOINT;
	pptr->num=j;
	pptr->type=p;
	pptr->value=vptr;
	pptr->graph=(float *)calloc(10000,sizeof(float));
	pptr->rgb=rgb;
	if(gptr->num_parameters) {
		free(gptr->parameter);
	}
	gptr->parameter=gpptr;
	(gptr->num_parameters)++;
}

void graph_probe(s)
char *s;
{
    char gname[40],ename[40],cname[40];
    int num,g,e,i;
    long rgb;
    GRAPH *gptr;
    GPARAMETER **gpptr,*pptr;
    ENTITY *eptr;
    double *vptr;

    num=sscanf(s," %s %s %s %s %s",gname,gname,gname,ename,cname);
    if(num!=5) {
        printf("File format error: %s\n",s);
        printf("Usage: graph probe <graph name> <entity name> <colour>\n");
        exit(0);
    }
    g=find_graph(gname);
    if(g== -1) {
        printf("Unknown graph: %s\n",gname);
        exit(0);
    }
    gptr=graph[g];
    e=find_schematic(ename);
    if(e== -1) {
        printf("Unknown schematic: %s\n",ename);
        exit(0);
    }
	eptr=entity[e];
    rgb= -1;
    if(!strcmp(cname,"black")) rgb=0x00000000;
    if(!strcmp(cname,"red")) rgb=0x000000ff;
    if(!strcmp(cname,"green")) rgb=0x0000ff00;
    if(!strcmp(cname,"yellow")) rgb=0x0000ffff;
    if(!strcmp(cname,"blue")) rgb=0x00ff0000;
    if(!strcmp(cname,"magenta")) rgb=0x00ff00ff;
    if(!strcmp(cname,"cyan")) rgb=0x00ffff00;
    if(!strcmp(cname,"white")) rgb=0x00ffffff;
    if(rgb== -1) {
        printf("Unknown colour: %s\n",cname);
        exit(0);
    }
    gpptr=(GPARAMETER **)calloc(1+gptr->num_parameters,sizeof(GPARAMETER *));
    for(i=0;i<gptr->num_parameters;i++) {
        gpptr[i]=gptr->parameter[i];
    }
    pptr=(GPARAMETER *)calloc(1,sizeof(GPARAMETER));
    gpptr[gptr->num_parameters]=pptr;
	pptr->source=GS_ENTITY;
    pptr->num=e;
    pptr->type=0;	/* type not used for entities */
    pptr->value= &(eptr->output);
    pptr->graph=(float *)calloc(10000,sizeof(float));
    pptr->rgb=rgb;
    if(gptr->num_parameters) {
        free(gptr->parameter);
    }
    gptr->parameter=gpptr;
    (gptr->num_parameters)++;
}

void read_graph(s)
char *s;
{
	char ss[40];
	
	sscanf(s," %s %s",ss,ss);
	if(!strcmp(ss,"activate")) graph_activate(s);
	if(!strcmp(ss,"create")) graph_create(s);
	if(!strcmp(ss,"mode")) graph_mode(s);
	if(!strcmp(ss,"horizontal")) graph_horizontal(s);
	if(!strcmp(ss,"vertical")) graph_vertical(s);
	if(!strcmp(ss,"labels")) graph_labels(s);
	if(!strcmp(ss,"trace")) graph_trace(s);
	if(!strcmp(ss,"probe")) graph_probe(s);
}

void read_from(s)
char *s;
{
	int num,jnum;
	char jname[40],pname[40];
	double value;
	JOINT *jptr;

	num=sscanf(s," %s %s %s %lf",jname,jname,pname,&value);
	if(num!=4) {
		printf("File format error: %s\n",s);
		printf("Usage: from joint parameter value\n");
		exit(0);
	}
	jnum=find_joint(jname);
	if(jnum== -1) {
		printf("Joint %s not defined\n",jname);
		exit(0);
	}
	if(!robot_created) {
		setup_joints();
	}
	jptr=from_joint[jnum];
	if(!strcmp(pname,"a")) {
		if(jptr->ca==ROB_CON) jptr->a=value;
		return;
	}
	if(!strcmp(pname,"d")) {
		if(jptr->cd==ROB_CON)jptr->d=value;
		return;
	}
	if(!strcmp(pname,"alpha")) {
		if(jptr->calpha==ROB_CON)jptr->alpha=value;
		return;
	}
	if(!strcmp(pname,"theta")) {
		if(jptr->ctheta==ROB_CON)jptr->theta=value;
		return;
	}
	printf("Parameter name %s not known\n",pname);
	exit(0);
}

void read_to(s)
char *s;
{
    int num,jnum;
    char jname[40],pname[40];
    double value;
	JOINT *jptr;

    num=sscanf(s," %s %s %s %lf",jname,jname,pname,&value);
    if(num!=4) {
        printf("File format error: %s\n");
        printf("Usage: to joint parameter value\n");
        exit(0);
    }
    jnum=find_joint(jname);
    if(jnum== -1) {
        printf("Joint %s not defined\n",jname);
        exit(0);
    }
    if(!robot_created) {
        setup_joints();
    }
    jptr=to_joint[jnum];
    if(!strcmp(pname,"a")) {
        if(jptr->ca==ROB_CON)jptr->a=value;
        return;
    }
    if(!strcmp(pname,"d")) {
        if(jptr->cd==ROB_CON)jptr->d=value;
        return;
    }
    if(!strcmp(pname,"alpha")) {
        if(jptr->calpha==ROB_CON)jptr->alpha=value;
        return;
    }
    if(!strcmp(pname,"theta")) {
        if(jptr->ctheta==ROB_CON)jptr->theta=value;
        return;
    }
    printf("Parameter name %s not known\n",pname);
    exit(0);
}

#define MAX_FILES 16

FILE *read_include(s,fptr,nptr)
char *s;
FILE **fptr;
int *nptr;
{
	int num;
	char fname[40];
	FILE *nin;

	num=sscanf(s," %s %s",fname,fname);
	if(num!=2) {
		printf("File format error: %s\n",s);
		printf("Usage: include file\n");
		exit(0);
	}
	nin=fopen(fname,"r");
	if(!nin) {
		printf("Cannot open file: %s\n",fname);
		exit(0);
	}
	if((*nptr)>=MAX_FILES) {
		printf("Include files too deeply nested\n");
		exit(0);
	}
	fptr[*nptr]=nin;
	(*nptr)++;
	return(nin);
}

INPUTNODE *create_inputnode(name,eptr)
char *name;
ENTITY *eptr;
{
	INPUTNODE *iptr;

	iptr=(INPUTNODE *)calloc(1,sizeof(INPUTNODE));
	eptr->height=20;
	eptr->width=20+10*strlen(name);
	eptr->a=(long *)calloc((eptr->width)*(eptr->height),sizeof(long));
	rfill_array(eptr->width,eptr->height,eptr->a,0x00808080,0x00804040);
	iptr->noise=NOISE_OFF;

	eptr->numports=1;
	eptr->mode[INPUTNODE_OUT]=PORT_OUTPUT;
	eptr->pname[INPUTNODE_OUT]=(char *)NULL;
	eptr->dx[POS_UP]=eptr->height;
	eptr->dy[POS_UP]=(eptr->width)/2;
	eptr->dx[POS_DOWN]=eptr->dx[POS_UP];
	eptr->dy[POS_DOWN]=0;
	eptr->dx[POS_RIGHT]=eptr->width;
	eptr->dy[POS_RIGHT]=(eptr->height)/2;
	eptr->dx[POS_LEFT]=0;
	eptr->dy[POS_LEFT]=eptr->dy[POS_RIGHT];
	return(iptr);
}

OUTPUTNODE *create_outputnode(name,eptr)
char *name;
ENTITY *eptr;
{
    OUTPUTNODE *optr;

    optr=(OUTPUTNODE *)calloc(1,sizeof(OUTPUTNODE));
    eptr->height=20;
    eptr->width=20+10*strlen(name);
	eptr->a=(long *)calloc((eptr->width)*(eptr->height),sizeof(long));
	rfill_array(eptr->width,eptr->height,eptr->a,0x00808080,0x00804040);
	
	eptr->numports=1;
    eptr->mode[OUTPUTNODE_IN]=PORT_INPUT;
    eptr->pname[OUTPUTNODE_IN]=(char *)NULL;
    eptr->dx[POS_UP]=eptr->height;
    eptr->dy[POS_UP]=(eptr->width)/2;
    eptr->dx[POS_DOWN]=eptr->dx[POS_UP];
    eptr->dy[POS_DOWN]= -eptr->dy[POS_UP];
    eptr->dx[POS_RIGHT]=eptr->width;
    eptr->dy[POS_RIGHT]=(eptr->height/2);
    eptr->dx[POS_LEFT]=0;
    eptr->dy[POS_LEFT]=eptr->dy[POS_RIGHT];
    return(optr);
}

TRANSFORMATION *create_transformation(name,eptr)
char *name;
ENTITY *eptr;
{
	TRANSFORMATION *tptr;

	tptr=(TRANSFORMATION *)calloc(1,sizeof(TRANSFORMATION));
	eptr->height=20;
	eptr->width=50;
	eptr->a=(long *)calloc(eptr->width*eptr->height,sizeof(long));
    fill_array(eptr->width,eptr->height,eptr->a,0x00808080);

	eptr->numports=2;
	eptr->mode[TRANSFORMATION_IN]=PORT_INPUT;
	eptr->mode[TRANSFORMATION_OUT]=PORT_OUTPUT;	
	eptr->pname[TRANSFORMATION_IN]="input";
	eptr->pname[TRANSFORMATION_OUT]="output";
	eptr->dx[POS_UP]=(eptr->width)/2;
	eptr->dy[POS_UP]=eptr->height;
	eptr->dx[POS_DOWN]=eptr->dx[POS_UP];
	eptr->dy[POS_DOWN]=0;
	eptr->dx[POS_RIGHT]=eptr->width;
	eptr->dy[POS_RIGHT]=(eptr->height)/2;
	eptr->dx[POS_LEFT]=0;
	eptr->dy[POS_LEFT]=eptr->dy[POS_RIGHT];
	return(tptr);
}

SUMMATION *create_summation(name,eptr)
char *name;
ENTITY *eptr;
{
	SUMMATION *sptr;

	sptr=(SUMMATION *)calloc(1,sizeof(SUMMATION));
	eptr->width=20;
	eptr->height=20;
	eptr->a=(long *)calloc(eptr->width*eptr->height,sizeof(long));
    rfill_array(eptr->width,eptr->width,eptr->a,0x00808080,0x00804040);
	
	eptr->numports=3;
	eptr->mode[SUMMATION_INA]=PORT_INPUT;
	eptr->mode[SUMMATION_INB]=PORT_INPUT;
	eptr->mode[SUMMATION_OUT]=PORT_OUTPUT;
	eptr->pname[SUMMATION_INA]="ina";
	eptr->pname[SUMMATION_INB]="inb";
	eptr->pname[SUMMATION_OUT]="out";
	eptr->dx[POS_UP]=(eptr->width)/2;	
	eptr->dy[POS_UP]=eptr->width;
	eptr->dx[POS_DOWN]=eptr->dx[POS_UP];
	eptr->dy[POS_DOWN]=0;
	eptr->dx[POS_RIGHT]=eptr->width;
	eptr->dy[POS_RIGHT]=(eptr->width)/2;
	eptr->dx[POS_LEFT]=0;
	eptr->dy[POS_LEFT]=eptr->dy[POS_RIGHT];
	return(sptr);
}

DIFFERENCE *create_difference(name,eptr)
char *name;
ENTITY *eptr;
{
    DIFFERENCE *dptr;

    dptr=(DIFFERENCE *)calloc(1,sizeof(DIFFERENCE));
    eptr->width=20;
    eptr->height=20;
	eptr->a=(long *)calloc(eptr->width*eptr->height,sizeof(long));
    rfill_array(eptr->width,eptr->width,eptr->a,0x00808080,0x00804040);

    eptr->numports=3;
    eptr->mode[DIFFERENCE_INA]=PORT_INPUT;
    eptr->mode[DIFFERENCE_INB]=PORT_INPUT;
    eptr->mode[DIFFERENCE_OUT]=PORT_OUTPUT;
    eptr->pname[DIFFERENCE_INA]="ina";
    eptr->pname[DIFFERENCE_INB]="inb";
    eptr->pname[DIFFERENCE_OUT]="out";
	eptr->dx[POS_UP]=(eptr->width)/2;   
    eptr->dy[POS_UP]=eptr->width;
    eptr->dx[POS_DOWN]=eptr->dx[POS_UP];
    eptr->dy[POS_DOWN]=0;
    eptr->dx[POS_RIGHT]=eptr->width;
    eptr->dy[POS_RIGHT]=(eptr->width)/2;
    eptr->dx[POS_LEFT]=0;
    eptr->dy[POS_LEFT]=eptr->dy[POS_RIGHT];
    return(dptr);
}

BIAS *create_bias(name,eptr)
char *name;
ENTITY *eptr;
{
	BIAS *bptr;

	bptr=(BIAS *)calloc(1,sizeof(BIAS));
	eptr->height=20;
	eptr->width=20;
	eptr->a=(long *)calloc(eptr->width*eptr->height,sizeof(long));
    fill_array(eptr->width,eptr->height,eptr->a,0x00808080);

	eptr->numports=1;
	eptr->mode[BIAS_OUT]=PORT_OUTPUT;
	eptr->pname[BIAS_OUT]="out";
    eptr->dx[POS_UP]=(eptr->width)/2;
    eptr->dy[POS_UP]=eptr->height;
    eptr->dx[POS_DOWN]=eptr->dx[POS_UP];
    eptr->dy[POS_DOWN]=0;
    eptr->dx[POS_RIGHT]=eptr->width;
    eptr->dy[POS_RIGHT]=(eptr->height)/2;
    eptr->dx[POS_LEFT]=0;
    eptr->dy[POS_LEFT]=eptr->dy[POS_RIGHT];
	return(bptr);
}

void schematic_create(s)
char *s;
{
	int num,snum,x,y;
	char name[40],type[40];
	ENTITY *eptr;
	void *vptr;

	num=sscanf(s," %s %s %s %s %d %d",name,name,name,type,&x,&y);
	if(num!=6) {
		printf("File format error: %s\n",s);
		printf("Usage: schematic create name type x y\n");
		exit(0);
	}
	snum=find_schematic(name);
	if(snum!= -1) {
		printf("Schematic %s already defined\n",name);
		exit(0);
	}
	eptr=(ENTITY *)calloc(1,sizeof(ENTITY));
    eptr->name=strdup(name);
    eptr->x=x;
    eptr->y=y;
	eptr->type= -1;
	if(!strcmp(type,"inputnode")) {
		eptr->type=ENTITY_INPUTNODE;
		eptr->entity=(void *)create_inputnode(name,eptr);
	}
	if(!strcmp(type,"outputnode")) {
		eptr->type=ENTITY_OUTPUTNODE;
		eptr->entity=(void *)create_outputnode(name,eptr);
	}
	if(!strcmp(type,"transformation")) {
		eptr->type=ENTITY_TRANSFORMATION;
		eptr->entity=(void *)create_transformation(name,eptr);
	}
	if(!strcmp(type,"summation")) {
		eptr->type=ENTITY_SUMMATION;
		eptr->entity=(void *)create_summation(name,eptr);
	}
	if(!strcmp(type,"difference")) {
		eptr->type=ENTITY_DIFFERENCE;
		eptr->entity=(void *)create_difference(name,eptr);
	}
	if(!strcmp(type,"bias")) {
		eptr->type=ENTITY_BIAS;
		eptr->entity=(void *)create_bias(name,eptr);
	}
	if(eptr->type== -1) {
		printf("Unknown entity type: %s\n",type);
		exit(0);
	}
	entity[numentities]=eptr;
	numentities++;
}

void schematic_connect(s)
char *s;
{
	int num;	
	char name1[40],port1[40],name2[40],port2[40];
	int n1,n2,p1,p2;

	num=sscanf(s," %s %s %s %s %s %s",name1,name1,name1,port1,name2,port2);
	if(num!=6) {
		printf("File format error: %s\n",s);
		printf("Usage: schematic connect name1 port1 name2 port2\n");
		exit(0);
	}
	n1=find_schematic(name1);
	if(n1== -1) {
		printf("Unknown schematic: %s\n",name1);
		exit(0);
	}
	n2=find_schematic(name2);
    if(n2== -1) {
        printf("Unknown schematic: %s\n",name2);
        exit(0);
    }
	p1=find_port(entity[n1]->type,port1);
	if(p1== -1) {
		printf("Unknown port: %s\n",port1);
		exit(0);
	}
	p2=find_port(entity[n2]->type,port2);
    if(p2== -1) {
        printf("Unknown port: %s\n",port2);
        exit(0);
    }
	if(entity[n1]->mode[p1]==PORT_INPUT) {	/* first port is input */
		entity[n1]->c_entity[p1]=n2;
		entity[n1]->c_port[p1]=p2;
	} else {								/* second port is input */
		entity[n2]->c_entity[p2]=n1;
		entity[n2]->c_port[p2]=p1;
	}
}

void schematic_position(s)
char *s;
{
	char name[40],port[40],position[40];
	int num,p,n,pp;

	num=sscanf(s," %s %s %s %s %s",name,name,name,port,position);
	if(num!=5) {
		printf("File format error: %s\n",s);
		printf("Usage: schematic position name port position\n");
		exit(0);
	}
	n=find_schematic(name);
	if(n== -1) {
        printf("Unknown schematic: %s\n",name);
        exit(0);
    }
	p=find_port(entity[n]->type,port);
    if(p== -1) {
        printf("Unknown port: %s\n",port);
        exit(0);
    }
	pp= -1;
	if(!strcmp(position,"up")) pp=POS_UP;
	if(!strcmp(position,"down")) pp=POS_DOWN;
	if(!strcmp(position,"left")) pp=POS_LEFT;
	if(!strcmp(position,"right")) pp=POS_RIGHT;
	if(pp== -1) {
		printf("Position unkown: %s\n",position);
		exit(0);
	}
	entity[n]->position[p]=pp;
}

POLYNOMIAL *process_polynomial(s)
char *s;
{
	POLYNOMIAL *pptr;
	char ss[256];
	char c,oc;
	int ps,i,skip;
	char *p,word[256];
	int numwords;
	int type,ntype,wtype[256];
	int power,coeff_loaded,power_loaded,finished;
	double coeff;
	
	pptr=(POLYNOMIAL *)calloc(1,sizeof(POLYNOMIAL));
	pptr->numterms=0;

#if(PRINT_SCHEMATIC)
	printf("\n%s\n",s);
#endif
	ps=1;
	ss[0]=s[0];
	oc=s[0];
	numwords=1;
	for(i=1;i<strlen(s);i++) {
		c=s[i];
		skip=0;	/* skip flag off */
		if(c=='^') skip=1;
		if(c=='z') skip=1;
		if(oc=='z') skip=1;
		if(oc=='^') skip=1;
		if((c=='+')&&( (oc!='e')&&(oc!='E') )) skip=1;
		if((c=='-')&&( (oc!='e')&&(oc!='E') )) skip=1;
		if(skip) {
			ss[ps]=' ';
			ps++;
			numwords++;
		}
		ss[ps]=c;
		oc=c;
		ps++;
	}
	ss[ps]=0;

	p=ss;
	for(i=0;i<numwords;i++) {	
		sscanf(p," %s",word);
		p=strchr(p+1,' ');
		type= SCH_N;
		if(!strcmp(word,"z")) type=SCH_Z;
		if(!strcmp(word,"^")) type=SCH_P;
		wtype[i]=type;
#if(PRINT_SCHEMATIC)
		printf("word: %s\n",word);
#endif
	}
	p=ss;
	coeff=1.0;
	coeff_loaded=0;
	power=0;
	power_loaded=0;
	for(i=0;i<numwords;i++) {
		sscanf(p," %s",word);
		p=strchr(p+1,' ');
		type=wtype[i];	
		if(type==SCH_N) {	/* a number */
			if(!strcmp(word,"+")) {
				if(!coeff_loaded) { 
					coeff=1.0; 	
					coeff_loaded=1; 
				} else {
					power=1;
					power_loaded=1; 
				}
			} else {
				if(!strcmp(word,"-")) {
					if(!coeff_loaded) {
						coeff= -1.0;
						coeff_loaded=1;
					} else {
						power=1;
						power_loaded=1;
					}
				} else {
					if(!coeff_loaded) {
						sscanf(word,"%lf",&coeff);
						coeff_loaded=1;
					} else {
						sscanf(word,"%d",&power);
						power_loaded=1;
					}
				}
			}
		}
		if(type==SCH_Z) power=1;
		if(i==(numwords-1)) {
			finished=1;
		} else {
			finished=0;
			ntype=wtype[i+1];
			if((type==SCH_N)&&(ntype==SCH_N)) finished=1;
			if((type==SCH_Z)&&(ntype==SCH_N)) finished=1;
		}
		if(finished) {
			pptr->zpower[pptr->numterms]=power;
			pptr->coefficient[pptr->numterms]=coeff;
			pptr->numterms++;
		    coeff=1.0;
   			coeff_loaded=0;
    		power=0;
    		power_loaded=0;
		}
	}
#if(PRINT_SCHEMATIC)
	for(i=0;i<pptr->numterms;i++) {
		printf("%lf z ^ %d\n",pptr->coefficient[i],pptr->zpower[i]);
	}
#endif
	return(pptr);
}

void schematic_program(s)
char *s;
{
	int num,n;
	char name[40],ss[256];
	char top[256],bot[256];
	ENTITY *eptr;
	TRANSFORMATION *tptr;
	char *p;
	int pp,maxp,i;
	POLYNOMIAL *toppoly;
	POLYNOMIAL *botpoly;
	int top_width,bot_width;

	num=sscanf(s," %s %s %s %s",name,name,name,ss);
	if(num!=4) {
		printf("File format error: %s\n",s);
		printf("Usage: schematic program name function\n");
		exit(0);
	}
	n=find_schematic(name);
	if(n== -1) {
		printf("Unknown schematic: %s\n",name);
		exit(0);
	}
	eptr=entity[n];
	if(eptr->type!=ENTITY_TRANSFORMATION) {
		printf("Entity must be a transformation for program\n");
		exit(0);
	}
	tptr=(TRANSFORMATION *)(eptr->entity);
	/* first thing to do is split the string by the '/' symbol */
	p=strchr(ss,'/');
	if(!p) {
		printf("Polynomial must be given in fraction form - \n");
		exit(0);	/* fix this later */
	}
	pp=(int)(p-ss);
	strcpy(top,ss);
	top[pp]=0;
	strcpy(bot,ss+pp+1);
	toppoly=process_polynomial(top);
	botpoly=process_polynomial(bot);
	tptr->ptop=toppoly;
	tptr->pbot=botpoly;

	maxp=0;
	for(i=0;i<toppoly->numterms;i++) {
		if(toppoly->zpower[i]>maxp) {
			maxp=toppoly->zpower[i];
		}
	}
	tptr->lhb=maxp+1;
	tptr->hb=(double *)calloc(maxp+1,sizeof(double));

	maxp=0;
    for(i=0;i<botpoly->numterms;i++) {
        if(botpoly->zpower[i]>maxp) {
            maxp=botpoly->zpower[i];
        }
    }
	tptr->lha=maxp+1;
    tptr->ha=(double *)calloc(maxp+1,sizeof(double));

	top_width=10*strlen(top);
	bot_width=10*strlen(bot);

	eptr->height=34;
	tptr->lx=10;
	tptr->ly=16;
	tptr->ty=20;
	tptr->by=4;
	tptr->top=strdup(top);
	tptr->bot=strdup(bot);
	if(top_width<bot_width) {
		eptr->width=bot_width+20;
		tptr->bx=10;
		tptr->tx=10+(bot_width-top_width)/2;
		tptr->ll=bot_width;
	} else {
		tptr->bx=10+(top_width-bot_width)/2;
		tptr->tx=10;
		eptr->width=top_width+20;
		tptr->ll=top_width;
	}
	if(eptr->a) free(eptr->a);
	eptr->a=(long *)calloc(eptr->width*eptr->height,sizeof(long));
    fill_array(eptr->width,eptr->height,eptr->a,0x00808080);

    eptr->dx[POS_UP]=(eptr->width)/2;
    eptr->dy[POS_UP]=eptr->height;
    eptr->dx[POS_DOWN]=eptr->dx[POS_UP];
    eptr->dy[POS_DOWN]=0;
    eptr->dx[POS_RIGHT]=eptr->width;
    eptr->dy[POS_RIGHT]=(eptr->height)/2;
    eptr->dx[POS_LEFT]=0;
    eptr->dy[POS_LEFT]=eptr->dy[POS_RIGHT];
}

void schematic_bias(s)
char *s;
{
	int num,e;
	char name[40];
	double value;
	ENTITY *eptr;
	BIAS *bptr;

	num=sscanf(s,"%s %s %s %lf\n",name,name,name,&value);
	if(num!=4) {
		printf("File format error: %s\n");
		printf("Usage: schematic bias name value\n");
		exit(0);
	}
	e=find_schematic(name);
	if(e== -1) {
		printf("Unknown schematic: %s\n",name);
		exit(0);
	}
	eptr=entity[e];
	bptr=(BIAS *)(eptr->entity);
	eptr->output=value;
	sprintf(name,"%2.2lf\n",value);
	name[5]=0;
	bptr->str=strdup(name);
	eptr->width=20+10*strlen(name);
	if(eptr->a) free(eptr->a);
	eptr->a=(long *)calloc(eptr->width*eptr->height,sizeof(long));
    fill_array(eptr->width,eptr->height,eptr->a,0x00808080);

    eptr->dx[POS_UP]=(eptr->width)/2;
    eptr->dy[POS_UP]=eptr->height;
    eptr->dx[POS_DOWN]=eptr->dx[POS_UP];
    eptr->dy[POS_DOWN]=0;
    eptr->dx[POS_RIGHT]=eptr->width;
    eptr->dy[POS_RIGHT]=(eptr->height)/2;
    eptr->dx[POS_LEFT]=0;
    eptr->dy[POS_LEFT]=eptr->dy[POS_RIGHT];
}

void schematic_noise(s)
char *s;
{
    int num,e;
    char name[40];
    double value;
    ENTITY *eptr;
    INPUTNODE *iptr;

    num=sscanf(s,"%s %s %s %lf\n",name,name,name,&value);
    if(num!=4) {
        printf("File format error: %s\n");
        printf("Usage: schematic noise name value\n");
        exit(0);
    }
    e=find_schematic(name);
    if(e== -1) {
        printf("Unknown schematic: %s\n",name);
        exit(0);
    }
    eptr=entity[e];
    iptr=(INPUTNODE *)(eptr->entity);
	iptr->noise=NOISE_ON;
	iptr->noise_scale=value;
}

void schematic_bind(s)
char *s;
{
	int num;
	char name[40],jname[40],pname[40],source[40];
	double *vptr;
	JOINT *jptr;
	JOINT **ljptr;
	ENTITY *eptr;
	int e,p,j,bm;

	num=sscanf(s," %s %s %s %s %s %s",name,name,source,name,jname,pname);
	if(num!=6) {
		printf("File format error: %s\n");
		printf("Usage: schematic bind source name joint parameter\n");
		exit(0);
	}
	ljptr=(JOINT **)NULL;
	if(!strcmp(source,"current")) { 
		ljptr=joint; 
		bm=BIND_CURRENT;
	}
	if(!strcmp(source,"from")) {
    	ljptr=from_joint; 
		bm=BIND_FROM; 
	}
	if(!strcmp(source,"to")) { 
		ljptr=to_joint; 
		bm=BIND_TO;
	}
	if(ljptr==(JOINT **)NULL) {
		printf("Unknown joint source: %s\n",source);
		exit(0);
	}
	e=find_schematic(name);
	if(e== -1) {
		printf("Unknown entity: %s\n",name);
		exit(0);
	}
	eptr=entity[e];
	if(!((eptr->type==ENTITY_INPUTNODE)||(eptr->type==ENTITY_OUTPUTNODE))) {
		printf("Entity must be input or output node\n");
		exit(0);
	}
	j=find_joint(jname);
	if(j== -1) {
		printf("Unknown joint name: %s\n",jname);
		exit(0);
	}
	jptr=ljptr[j];
	p= -1;
    if(!strcmp(pname,"x")) {
        p=PAR_X;
        vptr= &(jptr->x);
    }
    if(!strcmp(pname,"y")) {
        p=PAR_Y;
        vptr= &(jptr->y);
    }
    if(!strcmp(pname,"z")) {
        p=PAR_Z;
        vptr= &(jptr->z);
    }
    if(!strcmp(pname,"xd")){
        p=PAR_XD;
        vptr= &(jptr->xd);
    }
    if(!strcmp(pname,"yd")){
        p=PAR_YD;
        vptr= &(jptr->yd);
    }
    if(!strcmp(pname,"zd")){
        p=PAR_ZD;
        vptr= &(jptr->zd);
    }
    if(!strcmp(pname,"dx")){
        p=PAR_DX;
        vptr= &(jptr->dx);
    }
    if(!strcmp(pname,"dy")) {
        p=PAR_DY;
        vptr= &(jptr->dy);
    }
    if(!strcmp(pname,"dz")) {
        p=PAR_DZ;
        vptr= &(jptr->dz);
    }
    if(!strcmp(pname,"dxd")) {
        p=PAR_DXD;
        vptr= &(jptr->dxd);
    }
    if(!strcmp(pname,"dyd"))  {
        p=PAR_DYD;
        vptr= &(jptr->dyd);
    }
    if(!strcmp(pname,"dzd")) {
        p=PAR_DZD;
        vptr= &(jptr->dzd);
    }
    if(!strcmp(pname,"a"))  {
        p=PAR_A;
        vptr= &(jptr->a);
    }
    if(!strcmp(pname,"d")) {
        p=PAR_D;
        vptr= &(jptr->d);
    }
    if(!strcmp(pname,"alpha")) {
        p=PAR_ALPHA;
        vptr= &(jptr->alpha);
    }
	if(!strcmp(pname,"theta")) {
        p=PAR_THETA;
        vptr= &(jptr->theta);
    }
    if(!strcmp(pname,"ad"))  {
        p=PAR_AD;
        vptr= &(jptr->ad);
    }
    if(!strcmp(pname,"dd")) {
        p=PAR_DD;
        vptr= &(jptr->dd);
    }
    if(!strcmp(pname,"alphad")) {
        p=PAR_ALPHAD;
        vptr= &(jptr->alphad);
    }
    if(!strcmp(pname,"thetad")) {
        p=PAR_THETAD;
        vptr= &(jptr->thetad);
    }
    if(!strcmp(pname,"add"))  {
        p=PAR_ADD;
        vptr= &(jptr->add);
    }
    if(!strcmp(pname,"ddd")) {
        p=PAR_DDD;
        vptr= &(jptr->ddd);
    }
    if(!strcmp(pname,"alphadd")) {
        p=PAR_ALPHADD;
        vptr= &(jptr->alphadd);
    }
    if(!strcmp(pname,"thetatdd")) {
        p=PAR_THETADD;
        vptr= &(jptr->thetadd);
    }
    if(!strcmp(pname,"fa")) {
        p=PAR_FA;
        vptr= &(jptr->fa);
    }
    if(!strcmp(pname,"fd")) {
        p=PAR_FD;
        vptr= &(jptr->fd);
    }
    if(!strcmp(pname,"falpha")) {
        p=PAR_FALPHA;
        vptr= &(jptr->falpha);
    }
    if(!strcmp(pname,"ftheta")) {
        p=PAR_FTHETA;
        vptr= &(jptr->ftheta);
    }
    if(p== -1) {
        printf("Unknown parameter name: %s\n",pname);
        exit(0);
    }
	if(eptr->type==ENTITY_INPUTNODE) {
		((INPUTNODE *)(eptr->entity))->value=vptr;
		((INPUTNODE *)(eptr->entity))->joint=j;
		((INPUTNODE *)(eptr->entity))->parameter=p;
		((INPUTNODE *)(eptr->entity))->bind_mode=bm;
	}
	if(eptr->type==ENTITY_OUTPUTNODE) {
		((OUTPUTNODE *)(eptr->entity))->value=vptr;
		((OUTPUTNODE *)(eptr->entity))->joint=j;
		((OUTPUTNODE *)(eptr->entity))->parameter=p;
		((OUTPUTNODE *)(eptr->entity))->bind_mode=bm;
	}
}

void schematic_origin(s)
char *s;
{
	int num,x,y;
	char ss[40];

	num=sscanf(s," %s %s %d %d",ss,ss,&x,&y);
	if(num!=4) {
		printf("File format error: %s\n",s);
		exit(0);
	}
	schematic_x=x;
	schematic_y=y;
}

void schematic_print(s)
char *s;
{
    int num,x,y;

    num=sscanf(s," %s %s %s",schematic_print_name,schematic_print_name,schematic_print_name);
    if(num!=3) {
        printf("File format error: %s\n",s);
        exit(0);
    }
#ifndef R_SGI
	print_schematic();
#endif
}

void read_schematic(s)
char *s;
{
	char ss[40];

	sscanf(s," %s %s",ss,ss);
	if(!strcmp(ss,"create")) schematic_create(s);
	if(!strcmp(ss,"connect")) schematic_connect(s);
	if(!strcmp(ss,"position")) schematic_position(s);
	if(!strcmp(ss,"program")) schematic_program(s);
	if(!strcmp(ss,"bind")) schematic_bind(s);
	if(!strcmp(ss,"bias")) schematic_bias(s);
	if(!strcmp(ss,"noise")) schematic_noise(s);
	if(!strcmp(ss,"origin")) schematic_origin(s);
	if(!strcmp(ss,"print")) schematic_print(s);
}

void read_runmode(s)
char *s;
{
	int num;
	char mode[40];

	num=sscanf(s," %s %s",mode,mode);
	if(num!=2) {
		printf("File format error: %s\n",s);
		printf("Usage: runmode mode\n");
		exit(0);
	}
	runmode= -1;
	if(!strcmp(mode,"none")) runmode=RMODE_NONE;
	if(!strcmp(mode,"user")) runmode=RMODE_USER;
	if(!strcmp(mode,"schematic")) runmode=RMODE_SCHEMATIC;
	if(!strcmp(mode,"userschematic")) runmode=RMODE_USERSCHEMATIC;
	if(runmode== -1) {
		printf("Unknown runmode: %s\n",mode);
		exit(0);
	}
}

void read_worldmode(s)
char *s;
{
    int num;
    char mode[40];

    num=sscanf(s," %s %s",mode,mode);
    if(num!=2) {
        printf("File format error: %s\n",s);
        printf("Usage: worldmode mode\n");
        exit(0);
    }
    world_mode= -1;
    if(!strcmp(mode,"simple")) world_mode=WORLD_SIMPLE;
    if(!strcmp(mode,"complex")) world_mode=WORLD_COMPLEX;
    if(world_mode== -1) {
        printf("Unknown worldmode: %s\n",mode);
        exit(0);
    }
}

void read_picture(s)
char *s;
{
	char fn[80];
	int num;
	
	num=sscanf(s," %s %s %s",fn,fn,fn);
	if(num!=3) {
		printf("File format error: %s\n",s);
		printf("Usage: visual picture file\n");
		exit(0);
	}
	load_picture(fn);
}

void read_shape(s)
char *s;
{
	char name[40];
	int num,j;
	int numpanels;
	double radius;

	num=sscanf(s,"%s %s %s %d %lf",name,name,name,&numpanels,&radius);
	if(num!=5) {
		printf("File format error:%s\n",s);
		printf("Usage: visual shape name numpanels radius\n");
		exit(0);
	}
	j=find_joint(name);
	if(j== -1) {
		printf("Joint %s not known\n",name);
		exit(0);
	}
	joint[j]->radius=radius;
	joint[j]->numpanels=numpanels;
}

void read_ambient(s)
char *s;
{
	int num,j;
	char name[40];
	float r,g,b;

	num=sscanf(s,"%s %s %s %f %f %f",name,name,name,&r,&g,&b);
	if(num!=6) {
		printf("File format error: %s\n",s);
		printf("Usage: visual ambient name r g b\n");
		exit(0);
	}
	j=find_joint(name);
	if(j== -1) {
		printf("Joint %s not known\n",name);
		exit(0);
	}
	joint[j]->material[1]=r;
	joint[j]->material[2]=g;
	joint[j]->material[3]=b;
}

void read_diffuse(s)
char *s;
{
    int num,j;
    char name[40];
    float r,g,b;

    num=sscanf(s,"%s %s %s %f %f %f",name,name,name,&r,&g,&b);
    if(num!=6) {
        printf("File format error: %s\n",s);
        printf("Usage: visual diffuse name r g b\n");
        exit(0);
    }
    j=find_joint(name);
    if(j== -1) {
        printf("Joint %s not known\n",name);
        exit(0);
    }
	joint[j]->material[5]=r;
	joint[j]->material[6]=g;
	joint[j]->material[7]=b;
}

void read_specular(s)
char *s;
{
    int num,j;
    char name[40];
    float r,g,b;

    num=sscanf(s,"%s %s %s %f %f %f",name,name,name,&r,&g,&b);
    if(num!=6) {
        printf("File format error: %s\n",s);
        printf("Usage: visual specular name r g b\n");
        exit(0);
    }
    j=find_joint(name);
    if(j== -1) {
        printf("Joint %s not known\n",name);
        exit(0);
    }
	joint[j]->material[9]=r;
	joint[j]->material[10]=g;
	joint[j]->material[11]=b;
}

void read_shininess(s)
char *s;
{
    int num,j;
    char name[40];
    float v;

    num=sscanf(s,"%s %s %s %f",name,name,name,&v);
    if(num!=6) {
        printf("File format error: %s\n",s);
        printf("Usage: visual shininess name v\n");
        exit(0);
    }
    j=find_joint(name);
    if(j== -1) {
        printf("Joint %s not known\n",name);
        exit(0);
    }
	joint[j]->material[13]=v;
}

void read_visual(s)
char *s;
{
	char name[40];

	sscanf(s,"%s %s",name,name);
	if(!strcmp(name,"picture")) read_picture(s);
	if(!strcmp(name,"shape")) read_shape(s);
	if(!strcmp(name,"ambient")) read_ambient(s);
	if(!strcmp(name,"diffuse")) read_diffuse(s);
	if(!strcmp(name,"specular")) read_specular(s);
	if(!strcmp(name,"shininess")) read_shininess(s);
}

void read_create_box(s)
char *s;
{
	char name[40];
	float x,y,z,dx,dy,dz;
	int num,b,i;
	BOX *bptr;
	float *material;

	num=sscanf(s," %s %s %s %f %f %f %f %f %f",name,name,name,&x,&y,&z,&dx,&dy,&dz);
	if(num!=9) {
		printf("File format error: %s\n",s);
		printf("box create name x y z dx dy dz\n");
		exit(0);
	}
	b=find_box(name);
	if(b!= -1) {
		printf("Box already defined: %s\n",name);
		exit(0);
	}
	bptr=(BOX *)calloc(1,sizeof(BOX));
	bptr->name=strdup(name);
	bptr->x=x;
	bptr->y=y;
	bptr->z=z;
	bptr->dx=dx;
	bptr->dy=dy;
	bptr->dz=dz;
	material=(float *)calloc(15,sizeof(float));
#ifdef R_SGI
    for(i=0;i<15;i++) material[i]=mat_panel[i];
#endif
    bptr->material=material;
	box[numboxes]=bptr;
	numboxes++;
}

void read_box_ambient(s)
char *s;
{
    int num,j;
    char name[40];
    float r,g,b;

    num=sscanf(s,"%s %s %s %f %f %f",name,name,name,&r,&g,&b);
    if(num!=6) {
        printf("File format error: %s\n",s);
        printf("Usage: box ambient name r g b\n");
        exit(0);
    }
    j=find_box(name);
    if(j== -1) {
        printf("Box %s not known\n",name);
        exit(0);
    }
    box[j]->material[1]=r;
    box[j]->material[2]=g;
    box[j]->material[3]=b;
}

void read_box_diffuse(s)
char *s;
{
    int num,j;
    char name[40];
    float r,g,b;

    num=sscanf(s,"%s %s %s %f %f %f",name,name,name,&r,&g,&b);
    if(num!=6) {
        printf("File format error: %s\n",s);
        printf("Usage: box diffuse name r g b\n");
        exit(0);
    }
    j=find_box(name);
    if(j== -1) {
        printf("Box %s not known\n",name);
        exit(0);
    }
    box[j]->material[5]=r;
    box[j]->material[6]=g;
    box[j]->material[7]=b;
}

void read_box_specular(s)
char *s;
{
    int num,j;
    char name[40];
    float r,g,b;

    num=sscanf(s,"%s %s %s %f %f %f",name,name,name,&r,&g,&b);
    if(num!=6) {
        printf("File format error: %s\n",s);
        printf("Usage: box specular name r g b\n");
        exit(0);
    }
    j=find_box(name);
    if(j== -1) {
        printf("Box %s not known\n",name);
        exit(0);
    }
    box[j]->material[9]=r;
    box[j]->material[10]=g;
    box[j]->material[11]=b;
}

void read_box_shininess(s)
char *s;
{
    int num,j;
    char name[40];
    float v;

    num=sscanf(s,"%s %s %s %f",name,name,name,&v);
    if(num!=6) {
        printf("File format error: %s\n",s);
        printf("Usage: box shininess name v\n");
        exit(0);
    }
    j=find_box(name);
    if(j== -1) {
        printf("Box %s not known\n",name);
        exit(0);
    }
    box[j]->material[13]=v;
}

void read_box(s)
char *s;
{
	char ss[40];

	sscanf(s,"%s %s",ss,ss);
	if(!strcmp(ss,"create")) read_create_box(s);
	if(!strcmp(ss,"diffuse")) read_box_diffuse(s);
    if(!strcmp(ss,"specular")) read_box_specular(s);
    if(!strcmp(ss,"shininess")) read_box_shininess(s);
}

void read_plan_size(s)
char *s;
{
	int x,y,z,num;
	char ss[40];

	num=sscanf(s,"%s %s %d %d %d",ss,ss,&x,&y,&z);
	if(num!=5) {
		printf("File format error: %s\n",s);
		printf("Usage: plan size x y z\n");
		exit(0);
	}
	plan_nx=x;
	plan_ny=y;
	plan_nz=z;
}

void read_plan_arm(s)
char *s;
{
    int num;
	float x,y,z,length;
    char ss[40];

    num=sscanf(s,"%s %s %f %f %f %f",ss,ss,&length,&x,&y,&z);
    if(num!=6) {
        printf("File format error: %s\n",s);
        printf("Usage: plan arm length x y z\n");
        exit(0);
    }
    arm_x=x;
    arm_y=y;
    arm_z=z;
	arm_length=length;
}

void read_plan_expand(s)
char *s;
{
    int num;
    int pass;
    char ss[40];

    num=sscanf(s,"%s %s %d",ss,ss,&pass);
    if(num!=3) {
        printf("File format error: %s\n",s);
        printf("Usage: plan expand number\n");
        exit(0);
    }
	plan_passes=pass;
}

void read_plan(s)
char *s;
{
	char ss[40];

	sscanf(s,"%s %s",ss,ss);
	if(!strcmp(ss,"size")) read_plan_size(s);
	if(!strcmp(ss,"arm")) read_plan_arm(s);
	if(!strcmp(ss,"expand")) read_plan_expand(s);
}

void read_path(s)
char *s;
{
	char ss[40];
	float x,y,z;
	int num;

	num=sscanf(s,"%s %f %f %f\n",ss,&x,&y,&z);
	if(num!=4) {
		printf("File format error: %s\n",s);
		exit(0);
	}
	path_x[pathnum]=x;
	path_y[pathnum]=y;
	path_z[pathnum]=z;
	pathnum++;
}

void read_file(in)
FILE *in;
{
	char s[256];
	char ss[80];
	FILE *file[MAX_FILES],*cin;
	int done,numfiles;

	numvariables=0;
	numjoints=0;
	numentities=0;
	numboxes=0;
	numfiles=1;
	file[0]=in;

	cin=file[0];
	while(numfiles>0) {
		done=0;
		fgets(s,255,cin);
		while(!done) {
			if((strlen(s)>1)&&(!feof(cin))) {
				sscanf(s," %s",ss);
				if(!strcmp(ss,"angle")) define_angle(s);
				if(!strcmp(ss,"distance")) define_distance(s);
				if(!strcmp(ss,"bound")) bound_variable(s);
				if(!strcmp(ss,"set")) set_variable(s);
				if(!strcmp(ss,"joint")) define_joint(s);
				if(!strcmp(ss,"tensor")) set_tensor(s);
				if(!strcmp(ss,"friction")) set_friction(s);
				if(!strcmp(ss,"boundtime")) set_boundtime(s);
				if(!strcmp(ss,"numsteps")) set_numsteps(s);
				if(!strcmp(ss,"timestep")) set_timestep(s);
				if(!strcmp(ss,"graph")) read_graph(s);
				if(!strcmp(ss,"from")) read_from(s);
				if(!strcmp(ss,"to")) read_to(s);
				if(!strcmp(ss,"schematic")) read_schematic(s);
				if(!strcmp(ss,"include")) cin=read_include(s,file,&numfiles);
				if(!strcmp(ss,"runmode")) read_runmode(s);
				if(!strcmp(ss,"worldmode")) read_worldmode(s);
				if(!strcmp(ss,"visual")) read_visual(s);
				if(!strcmp(ss,"box")) read_box(s);
				if(!strcmp(ss,"plan")) read_plan(s);
				if(!strcmp(ss,"path")) read_path(s);
			}
			if(feof(cin)) {
				done=1;
			} else {
				fgets(s,255,cin);
			}
		}
		fclose(cin);
		numfiles--;
		if(numfiles>0) cin=file[numfiles-1];
	}
}
