/******************************************************************************\
********************************************************************************
********                                                                ********
******      Visual robot simulator                                        ******
****                                                                        ****
**          Author: Craig Dillon, Andrew Conway                               **
**                                                                            **
****        DYNAMICS.C                                                      ****
******                                                                    ******
********                                                                ********
********************************************************************************
\******************************************************************************/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#ifdef R_SGI
#include <gl/gl.h>
#endif
#include <sys/types.h>
#include "rob.h"

#define MAX(a,b) (((a)>(b))?(a):(b))
#define matmul(A,B,C) mul4(A,B,C)
#define matcopy(A,B) copy(A,B)

typedef struct {  /* Extra useful stuff for dynamics */
	double sinalpha,cosalpha;
	double sintheta,costheta;
	mat H;        /* Transform previous coords to this coords */
} DYNAJOINT;

#define VARTYPE_THETA 0
#define VARTYPE_ALPHA 1
#define VARTYPE_D     2
#define VARTYPE_A     3

typedef struct {
	int jointnum;   /* Index into main joint structure */
	int vartype;    /* variable type ... see above */
	mat Hdot;		/* partial derivative of H wrt. variable */
	mat Hdotdot;    /* double partial derivative of H wrt. variable */
} DYNAVAR;


double ***dyna_c;		  /* Christoffel Symbols */
DYNAJOINT *dynajoint;
mat **dyna_H;             /* H[a][b]=H_a^b - transforms from b to a */
DYNAVAR *dynavars;
int numvars;
int numaccels;   		  /* no. of variables for which we have accelerations */
mat **Hpartial;			  /* Hpartial[i][k]=S^i_k = dH_0^i / dq_k   */
mat ***Hdoublepartial;    /* Hdoublepartial[i][j][k] = dH_0^i / dq_j dq_k */
double **dyna_d;		  /* d coefficients for Kinetic energy */
double ***dyna_d_dot;     /* d[i][j][k]= d dyna_d[i][j] / d q_k */

#define NEWARRAY(typ,num) ((typ *)malloc(sizeof(typ)*(num)))

void dyna_init_mat(mat A,double sa,double ca,double st,double ct,double d,double a,double flaga,double flagb,double flagc)
{
    A[0][0]=flaga*ct;
    A[0][1]= -st*ca;
    A[0][2]=st*sa;
    A[0][3]=a*ct;
    A[1][0]=flaga*st;
    A[1][1]=ct*ca;
    A[1][2]= -ct*sa;
    A[1][3]=a*st;
    A[2][0]=0;
    A[2][1]=flagb*sa;
    A[2][2]=flagb*ca;
    A[2][3]=d;
    A[3][0]=0;
    A[3][1]=0;
    A[3][2]=0;
    A[3][3]=flagc;
}

void matzero(mat A)
{
	int i,j;
	for (i=0;i<4;i++) for (j=0;j<4;j++) A[i][j]=0;
}

void dyna_init1() /* Calculate some H matrices */
{
	int i,j;
	double sa,ca,st,ct,a,d;
	for (i=0;i<numjoints;i++) {  /* 1a */
		sa=dynajoint[i].sinalpha=sin(global_joint[i]->alpha);
		ca=dynajoint[i].cosalpha=cos(global_joint[i]->alpha);
		st=dynajoint[i].sintheta=sin(global_joint[i]->theta);
		ct=dynajoint[i].costheta=cos(global_joint[i]->theta);
		d=global_joint[i]->d;
		a=global_joint[i]->a;
		dyna_init_mat(dynajoint[i].H,sa,ca,st,ct,d,a,1.0,1.0,1.0);
	}
	for (i=0;i<=numjoints;i++) { /* 1b */
		identity(dyna_H[i][i]);
		for (j=i+1;j<=numjoints;j++) 
			mul4(dyna_H[i][j-1],dynajoint[j-1].H,dyna_H[i][j]);
	}
	for (i=0;i<numvars;i++) { /* 1 c&d */
		j=dynavars[i].jointnum;
		sa=dynajoint[j].sinalpha;
		ca=dynajoint[j].cosalpha;
		st=dynajoint[j].sintheta;
		ct=dynajoint[j].costheta;
		d=global_joint[j]->d;
		a=global_joint[j]->a;
		switch(dynavars[i].vartype) {
			case VARTYPE_A:
				dyna_init_mat(dynavars[i].Hdot,0.0,0.0,st,ct,0.0,1.0,0.0,0.0,0.0);
				matzero(dynavars[i].Hdotdot);
				break;
			case VARTYPE_D:
				dyna_init_mat(dynavars[i].Hdot,0.0,0.0,0.0,0.0,1,0.0,0.0,0.0,0.0);
				matzero(dynavars[i].Hdotdot);
				break;
			case VARTYPE_THETA:
				dyna_init_mat(dynavars[i].Hdot,sa,ca,ct,-st,0.0,a,1.0,0.0,0.0);
				dyna_init_mat(dynavars[i].Hdotdot,sa,ca,-st,-ct,0.0,a,1.0,0.0,0.0);
				break;
			case VARTYPE_ALPHA:
				dyna_init_mat(dynavars[i].Hdot,ca,-sa,st,ct,0.0,0.0,0.0,1.0,0.0);
				dyna_init_mat(dynavars[i].Hdotdot,-sa,-ca,st,ct,0,0,0.0,1.0,0.0);
				break;
		}
	}
}

void dyna_init2() /* Calculate some H matrices */
{
	int i,j,k;
	for (k=0;k<numvars;k++) {
		j=dynavars[k].jointnum+1;
		matmul(dyna_H[0][j-1],dynavars[k].Hdot,Hpartial[j][k]);
		for (i=j+1;i<=numjoints;i++) {
			matmul(Hpartial[i-1][k],dynajoint[i-1].H,Hpartial[i][k]);
		}
	}	
}

double mattrace(mat M)
{
	int i;
	double res=0;
	for (i=0;i<4;i++) res+=M[i][i];
	return res;
}

void mattranspose(mat A,mat B)
{
	int i,j;
	for (i=0;i<4;i++) for (j=0;j<4;j++) B[i][j]=A[j][i];
}

void dyna_init3() /* calculare d matrix */
{
	int i,j,k,jjoint,kjoint,maxjoint;
	for (j=0;j<numvars;j++) for (k=0;k<numvars;k++) {
		if (j>k) dyna_d[j][k]=dyna_d[k][j];
		else {
			jjoint=dynavars[j].jointnum;
			kjoint=dynavars[k].jointnum;
			maxjoint=MAX(jjoint,kjoint);
			dyna_d[j][k]=0;
			for (i=maxjoint+1;i<=numjoints;i++) {
				mat C,D,E;
				matmul (Hpartial[i][j],global_joint[i-1]->J,C);
				mattranspose(Hpartial[i][k],D);
				matmul (C,D,E);
				dyna_d[j][k]+=mattrace(E);
			}
		}
	}
}

void dyna_init4() /* calculate double derivatives of H matrices */
{
	int i,j,k,jjoint,kjoint;
	for (j=0;j<numvars;j++) for (k=0;k<numvars;k++) {
		if (j>k) for (i=0;i<=numjoints;i++) 
			matcopy(Hdoublepartial[i][k][j],Hdoublepartial[i][j][k]);
		else {
			int svar,lvar,sjoint,ljoint;
			jjoint=dynavars[j].jointnum;
			kjoint=dynavars[k].jointnum;
			if (jjoint<kjoint) {svar=j;lvar=k;sjoint=jjoint;ljoint=kjoint;}
			else {svar=k;lvar=j;sjoint=kjoint;ljoint=jjoint;}
			if (sjoint==ljoint) {
				if (j==k) {
					matmul(dyna_H[0][ljoint],dynavars[k].Hdotdot,
							Hdoublepartial[ljoint+1][j][k]);
				} else {
					mat A;
					int type1,type2;
					double sa=dynajoint[ljoint].sinalpha;
					double ca=dynajoint[ljoint].cosalpha;
					double st=dynajoint[ljoint].sintheta;
					double ct=dynajoint[ljoint].costheta;
					type1=dynavars[j].vartype;
					type2=dynavars[k].vartype;
					if (type1==VARTYPE_THETA) {
						type1=type2;
						type2=VARTYPE_THETA;
					}
					switch (type1) {
						case VARTYPE_D:
							matzero(Hdoublepartial[ljoint+1][j][k]);
							break;
						case VARTYPE_A:
							if (type2==VARTYPE_THETA) {
								dyna_init_mat(A,0.0,0.0,ct,st,0.0,1.0,0.0,0.0,0.0);
								matmul(dyna_H[0][ljoint],A,
									Hdoublepartial[ljoint+1][j][k]);
							} else matzero(Hdoublepartial[ljoint+1][j][k]);
							break;
						case VARTYPE_ALPHA:
							if (type2==VARTYPE_THETA) {
								dyna_init_mat(A,ca,-sa,ct,-st,0.0,0.0,0.0,0.0,0.0);
								matmul(dyna_H[0][ljoint],A,
									Hdoublepartial[ljoint+1][j][k]);
							} else matzero(Hdoublepartial[ljoint+1][j][k]);
							break;
					}
				}
			} else {
				matmul(Hpartial[ljoint][svar],dynavars[lvar].Hdot,
					Hdoublepartial[ljoint+1][j][k]);
			}
			for (i=ljoint+2;i<=numjoints;i++) {
				matmul(Hdoublepartial[i-1][j][k],dynajoint[i-1].H,
						Hdoublepartial[i][j][k]);
			}
		}
	}
}

void dyna_init5()
{
	int i,j,k,a,ijoint,jjoint,kjoint,maxjoint;
	for (i=0;i<numvars;i++) for (j=0;j<numvars;j++) for (k=0;k<numvars;k++) {
		if (j>k) dyna_d_dot[j][k][i]=dyna_d_dot[k][j][i];
		else {
			dyna_d_dot[j][k][i]=0;
			ijoint=dynavars[i].jointnum;
			jjoint=dynavars[j].jointnum;
			kjoint=dynavars[k].jointnum;
			maxjoint=MAX(jjoint,kjoint);
			maxjoint=MAX(ijoint,maxjoint);
			for (a=maxjoint+1;a<=numjoints;a++) {
				mat C,D,E;
				matmul (Hdoublepartial[a][k][i],global_joint[a-1]->J,C);
				mattranspose(Hpartial[a][j],D);
				matmul (C,D,E);
				dyna_d_dot[j][k][i]+=mattrace(E);
				matmul (Hdoublepartial[a][j][i],global_joint[a-1]->J,C);
				mattranspose(Hpartial[a][k],D);
				matmul (C,D,E);
				dyna_d_dot[j][k][i]+=mattrace(E);
			}
		}
	}
}

void dyna_init6()
{
	int i,j,k;
	for (i=0;i<numvars;i++) for (j=0;j<numvars;j++) for (k=0;k<numvars;k++) {
/*		printf("in init6, i=%d, j=%d, k=%d \n",i,j,k);   */
		dyna_c[i][j][k]=(dyna_d_dot[k][j][i]+dyna_d_dot[k][i][j]-
						 dyna_d_dot[i][j][k])/2.0;

	}
}

void dyna_debug()
{
	int i,j,k;
	print4(dynajoint[0].H);
	print4(dynavars[0].Hdot);
	print4(dynavars[0].Hdotdot);
	print4(Hpartial[1][0]);
	print4(Hdoublepartial[1][0][0]);
	print4(global_joint[0]->J);

	for (i=0;i<numvars;i++) for (j=0;j<numvars;j++)  {
		printf("d_%d %d = %lf\n",i,j,dyna_d[i][j]);
	}
	for (i=0;i<numvars;i++) for (j=0;j<numvars;j++) for (k=0;k<numvars;k++) {
		printf("c_%d %d %d = %lf\n",i,j,k,dyna_c[i][j][k]);
	}
}

void dyna_init_mem()
{
	int i,j,thisvar,mode;
	dynajoint=NEWARRAY(DYNAJOINT,numjoints);
	numvars=0;
	for (i=0;i<numjoints;i++) {
		if (global_joint[i]->ca==ROB_CON) numvars++;
		if (global_joint[i]->cd==ROB_CON) numvars++;
		if (global_joint[i]->calpha==ROB_CON) numvars++;
		if (global_joint[i]->ctheta==ROB_CON) numvars++;
	}
	if (numvars==0) {
		printf("No variables!!!!\n");
		exit(0);
	}
	dynavars=NEWARRAY(DYNAVAR,numvars);
	thisvar=0;
	for (mode=0;mode<2;mode++) {
		for (i=0;i<numjoints;i++) {
			if (mode^(global_joint[i]->sim_mode==SIM_F)) continue;
			/* The last line makes the simulated froces go last */
			if (global_joint[i]->ca==ROB_CON) {
				dynavars[thisvar].jointnum=i;
				dynavars[thisvar++].vartype=VARTYPE_A;
			}
			if (global_joint[i]->cd==ROB_CON) {
				dynavars[thisvar].jointnum=i;
				dynavars[thisvar++].vartype=VARTYPE_D;
			}
			if (global_joint[i]->ctheta==ROB_CON) {
				dynavars[thisvar].jointnum=i;
				dynavars[thisvar++].vartype=VARTYPE_THETA;
			}
			if (global_joint[i]->calpha==ROB_CON) {
				dynavars[thisvar].jointnum=i;
				dynavars[thisvar++].vartype=VARTYPE_ALPHA;
			}
		}
		if (!mode) numaccels=thisvar;
	}
	if (thisvar!=numvars) {
		printf("thisvar != numvars in dyna_init_mem\n");
		exit(0);
	}
	dyna_H=NEWARRAY(mat *,numjoints+1);
	for (i=0;i<numjoints+1;i++) dyna_H[i]=NEWARRAY(mat,numjoints+1);
	Hpartial=NEWARRAY(mat *,numjoints+1);
	for (i=0;i<=numjoints;i++) Hpartial[i]=NEWARRAY(mat,numvars); 
	Hdoublepartial=NEWARRAY(mat **,numjoints+1);
	for (i=0;i<=numjoints;i++) {
		Hdoublepartial[i]=NEWARRAY(mat *,numvars);
		for (j=0;j<numvars;j++) Hdoublepartial[i][j]=NEWARRAY(mat,numvars);
	}
	dyna_d=NEWARRAY(double *,numvars);
	for (i=0;i<numvars;i++) dyna_d[i]=NEWARRAY(double,numvars);
	dyna_d_dot=NEWARRAY(double **,numvars);
	for (i=0;i<numvars;i++) {
		dyna_d_dot[i]=NEWARRAY(double *,numvars);
		for (j=0;j<numvars;j++) dyna_d_dot[i][j]=NEWARRAY(double,numvars);
	}
	dyna_c=NEWARRAY(double **,numvars);
	for (i=0;i<numvars;i++) {
		dyna_c[i]=NEWARRAY(double *,numvars);
		for (j=0;j<numvars;j++) dyna_c[i][j]=NEWARRAY(double,numvars);
	}
}

void free_old()
{
	int i,j;
	if (dynajoint) {
		free(dynajoint);
		dynajoint=NULL;
	}
	if (dynavars) {
		free(dynavars);
		dynavars=NULL;
	}
	if (dyna_H) {
		for (i=0;i<numjoints+1;i++) free(dyna_H[i]); 
		free(dyna_H); 
		dyna_H=NULL;
	}
	if (Hpartial) {
		for (i=0;i<numjoints+1;i++) free(Hpartial[i]); 
		free(Hpartial); 
		Hpartial=NULL;
	}
	if (Hdoublepartial) {
		for (i=0;i<numjoints+1;i++) {
			for (j=0;j<numvars;j++) free(Hdoublepartial[i][j]);
			free(Hdoublepartial[i]);
		}
		free(Hdoublepartial);
		Hdoublepartial=NULL;
	}
	if (dyna_d) {
		for (i=0;i<numvars;i++) free(dyna_d[i]); 
		free(dyna_d); 
		dyna_d=NULL;
	}
	if (dyna_d_dot) {
		for (i=0;i<numvars;i++) {
			for (j=0;j<numvars;j++) free(dyna_d_dot[i][j]);
			free(dyna_d_dot[i]);
		}
		free(dyna_d_dot);
		dyna_d_dot=NULL;
	}
	if (dyna_c) {
		for (i=0;i<numvars;i++) {
			for (j=0;j<numvars;j++) free(dyna_c[i][j]);
			free(dyna_c[i]);
		}
		free(dyna_c);
		dyna_c=NULL;
	}
}

void dyna_init_lot()
{
	free_old();
	dyna_init_mem();
/*	printf("numvars = %d (init_mem) \n",numvars); */
	dyna_init1();
	dyna_init2();
	dyna_init3();
	dyna_init4();
	dyna_init5();
	dyna_init6();
/*	printf("numvars = %d (init6) \n",numvars); */
	/* dyna_debug(); */
}
	
void dyna_load_data(double *qdot,double *tau,double *qdotdot,double **storetau,
	double **storeqdotdot,double *friclin,double *fricquad,double *fricmin)
{
	int i,j;
	for	(i=0;i<numvars;i++) {
		j=dynavars[i].jointnum;
		switch(dynavars[i].vartype) {
			case VARTYPE_D:
				qdot[i]=global_joint[j]->dd;
				tau[i]=global_joint[j]->fd;
				qdotdot[i]=global_joint[j]->ddd;
				storetau[i]= &(global_joint[j]->fd);
				storeqdotdot[i]= &(global_joint[j]->ddd);
				friclin[i]=global_joint[j]->fric_dd;
				fricquad[i]=global_joint[j]->fric_dds;
				fricmin[i]=global_joint[j]->fric_dmin;
				break;
			case VARTYPE_A:
				qdot[i]=global_joint[j]->ad;
				tau[i]=global_joint[j]->fa;
				qdotdot[i]=global_joint[j]->add;
				storetau[i]= &(global_joint[j]->fa);
				storeqdotdot[i]= &(global_joint[j]->add);
				friclin[i]=global_joint[j]->fric_ad;
				fricquad[i]=global_joint[j]->fric_ads;
				fricmin[i]=global_joint[j]->fric_amin;
				break;
			case VARTYPE_ALPHA:
				qdot[i]=global_joint[j]->alphad;
				tau[i]=global_joint[j]->falpha;
				qdotdot[i]=global_joint[j]->alphadd;
				storetau[i]= &(global_joint[j]->falpha);
				storeqdotdot[i]= &(global_joint[j]->alphadd);
				friclin[i]=global_joint[j]->fric_alphad;
				fricquad[i]=global_joint[j]->fric_alphads;
				fricmin[i]=global_joint[j]->fric_alphamin;
				break;
			case VARTYPE_THETA:
				qdot[i]=global_joint[j]->thetad;
				tau[i]=global_joint[j]->ftheta;
				qdotdot[i]=global_joint[j]->thetadd;
				storetau[i]= &(global_joint[j]->ftheta);
				storeqdotdot[i]= &(global_joint[j]->thetadd);
				friclin[i]=global_joint[j]->fric_thetad;
				fricquad[i]=global_joint[j]->fric_thetads;
				fricmin[i]=global_joint[j]->fric_thetamin;
				break;
			default:
				printf("Help! VARTYPE is strange in dyna_load_data\n");
				exit(0);
		}
	}
}

void dyna_store_data(double *tau,double *qdotdot,double **storetau,double **storeqdotdot)
{
	int i;
	for	(i=0;i<numvars;i++) {
		*(storetau[i])=tau[i];
		*(storeqdotdot[i])=qdotdot[i];
	}
}

void dyna_calc_friction(double *friction,double *qdot,
		double *friclin,double *fricquad,double *fricmin)
{
	int i;
	for (i=0;i<numvars;i++) {
		double vel=fabs(qdot[i]);
		friction[i]=vel*(vel*fricquad[i]+friclin[i])+fricmin[i];
		if (qdot[i]<0) friction[i]= -friction[i];
	}
}

void dyna_calc_c(double *c,double *qdot)
{
	int i,j,k;
	for (k=0;k<numvars;k++) {
		c[k]=0;
		for (i=0;i<numvars;i++) for (j=0;j<numvars;j++) 
			c[k]+=dyna_c[i][j][k]*qdot[i]*qdot[j];
	}
}

void dyna_calc_phi(double *phi)
{
	int i,j,k,joint;
	double g_str=9.81;
	for (k=0;k<numvars;k++) {
		phi[k]=0;
		joint=dynavars[k].jointnum;
		for (i=joint+1;i<=numjoints;i++) {
			mat A;
			matmul(Hpartial[i][k],global_joint[i-1]->J,A);
			phi[k]+=g_str*global_joint[i-1]->J[3][3]*A[2][3];
		}
	}
}

void invert(double *A,double *inv,int n)
{
	int i,j,k;
#define from(x,y) (A[(x)+(y)*n])
#define to(x,y) (inv[(x)+(y)*n])
	for (i=0;i<n;i++) for (j=0;j<n;j++) to(i,j)=(i==j);
	for (i=0;i<n;i++) {
		double lc,maxval=0;
		int maxpos;
		for(j=i;j<n;j++) /* find pivot */ 
			if ((lc=fabs(from(i,j)))>maxval) {
				maxval=lc;
				maxpos=j;
			}
		if (!maxval) {
			printf("Help --- matrix doesn't invert\n");
			exit(10);
		}
		if (maxpos!=i) {
			for(j=0;j<n;j++) { /* swap rows i and maxpos */
				double temp=from(j,i);
				from(j,i)=from(j,maxpos);
				from(j,maxpos)=temp;
				temp=to(j,i);
				to(j,i)=to(j,maxpos);
				to(j,maxpos)=temp;
			}
		}
		maxval=from(i,i);    /* allow for negative */
		for (j=0;j<n;j++) {
			from(j,i)/=maxval;
			to(j,i)/=maxval;
		}
		for (j=0;j<n;j++) if (j!=i) {
			double mul=from(i,j);
			for (k=0;k<n;k++) {
				from(k,j)-=mul*from(k,i);
				to(k,j)-=mul*to(k,i);
			}
		}
	}
	for (i=0;i<n;i++) for (j=0;j<n;j++) {
		double err=from(i,j)-(i==j);
		if (fabs(err)>0.0001) printf("Bad inversion Routine i=%d j=%d err=%f\n",i,j,err);
	}
}

void forces_and_accelerations()  /* Super-function !!! */
{
	double *tau,*c,*qdot,*phi,*qdotdot,**storetau,**storeqdotdot;
	double *d12,*work1,*d3,*d4,*d4inv;
	double *friction,*friclin,*fricquad,*fricmin;
	int i,j,a,b;
	

	dyna_init_lot();
	a=numaccels;
	b=numvars-numaccels;

	tau=NEWARRAY(double,numvars);
	c=NEWARRAY(double,numvars);
	friction=NEWARRAY(double,numvars);
	friclin=NEWARRAY(double,numvars);
	fricquad=NEWARRAY(double,numvars);
	fricmin=NEWARRAY(double,numvars);
	qdot=NEWARRAY(double,numvars);
	phi=NEWARRAY(double,numvars);
	qdotdot=NEWARRAY(double,numvars);
	storetau=NEWARRAY(double *,numvars);
	storeqdotdot=NEWARRAY(double *,numvars);
	d12=NEWARRAY(double,a*(a+b));
	work1=NEWARRAY(double,numvars);
	d3=NEWARRAY(double,a*b+1);
	d4=NEWARRAY(double,b*b+1);
	d4inv=NEWARRAY(double,b*b+1);
	
	dyna_load_data(qdot,tau,qdotdot,storetau,storeqdotdot,
					friclin,fricquad,fricmin);
	dyna_calc_c(c,qdot);
	dyna_calc_phi(phi);
	dyna_calc_friction(friction,qdot,friclin,fricquad,fricmin);
	for (i=0;i<numvars;i++) for (j=0;j<numvars;j++) {
		double d=dyna_d[i][j];
		if (i<a) {
			d12[j+i*(a+b)]=d;
		} else {
			if (j<a) d3[j+(i-a)*a]=d;
			else d4[j-a+(i-a)*b]=d;
		}
	}
	if (b) {
		invert(d4,d4inv,b);
		domul(d3,qdotdot,work1,b,a,1);
		for(i=0;i<b;i++) 
			work1[i]=tau[a+i]-work1[i]-c[a+i]-phi[a+i]-friction[a+i];
		domul(d4inv,work1,qdotdot+a,b,b,1);
	}
	if (a) domul(d12,qdotdot,work1,a,a+b,1);
	for (i=0;i<a;i++) tau[i]=work1[i]+c[i]+phi[i]+friction[i];
	dyna_store_data(tau,qdotdot,storetau,storeqdotdot);
	free(tau);
	free(c);
	free(friction);
	free(friclin);
	free(fricmin);
	free(fricquad);
	free(phi);
	free(qdot);
	free(qdotdot);
	free(storetau);
	free(storeqdotdot);
	free(d12);
	free(work1);
	free(d3);
	free(d4);
	free(d4inv);
}


