/* 
 * points.c -- definition of global points
 * 
 */

#include <stdio.h>

#include "constants.h"
#include "structures.h"
#include "parameters.h"
#include "constructs.h"
#include "display.h"

#define MAXPOINTS 100

point *				/* list of global points */
pointlist[MAXPOINTS];

int				/* pointer to points list */
pointlist_p = 0;

float
point_phi(point *p)		/* returns the direction to the point */
{				/* between -PI and PI in radians */
  float x = p->x;
  float y = p->y;
  if (x == 0.0 && y == 0.0) return(0.0);
  return(norm2_angle((float)atan2(y,x)));
}

void
cc_point(point *from, point *to) /* carbon-copy one point to another */
{
  to->x = from->x;
  to->y = from->y;
  to->th = from->th;
}

void
seg_to_point(surf *s, point *p)	/* convert a segment to a point */
{
  float dx, dy;
  p->x = (s->x1 + s->x2) * 0.5;
  p->y = (s->y1 + s->y2) * 0.5;
  dx = s->x2 - s->x1;
  dy = s->y2 - s->y1;
  if (dx == 0.0 && dy == 0.0) p->th = 0.0;
  else p->th = atan2(dy,dx);
}

void
point_to_seg(point *p, float d, surf *s) /* convert from point to surface segment */
{
  float cs = cos(p->th);
  float sn = sin(p->th);
  s->d = d;
  d = d*0.5;
  s->x1 = d*cs + p->x;
  s->y1 = d*sn + p->y;
  s->x2 = -d*cs + p->x;
  s->y2 = -d*sn + p->y;
  s->cx = p->x;
  s->cy = p->y;
}

void
add_point(point *p)		/* adds P to pointlist */
{
  if (pointlist_p < MAXPOINTS)
    pointlist[pointlist_p++] = p;
}

void
rem_point(point *p)		/* remove it */
{
  int i;
  for (i=0; i<pointlist_p; i++)
    if (pointlist[i] == p)	/* found it, move everything up */
      {
	for (; i<pointlist_p; i++)
	  {
	    pointlist[i] = pointlist[i+1];
	  }
	pointlist_p--;
	break;
      }
}

void
update_point(point *p, float th, float cth, float sth, float dx, float dy)
{
  float tx, ty;
  tx = p->x - dx;
  ty = p->y - dy;
  p->x = tx * cth - ty * sth;
  p->y = ty * cth + tx * sth;
  p->th = add_ang(p->th,th);
}

void
update_pointlist(float x, float y, float th) /* updates all points on the list */
{
  int i;
  float cth, sth;
  th = -th;			/* historical reasons... */
  cth = cos(th);
  sth = sin(th);
  for (i=0; i<pointlist_p; i++)
    update_point(pointlist[i],th,cth,sth,x,y);
}


void
add_target(int x, int y)	/* adds a target point at screen x,y */
{
  float sx, sy;
  point *p;
  sx = (float)x - screen_matrix.a13;
  sy = screen_matrix.a23 - (float)y;
  p = (point *)malloc(sizeof(point));
  p->x = (- sy) / screen_matrix.a12;
  p->y = sx / screen_matrix.a21;
  p->th = 0.0;
  p->type = PT_POS;
  add_point(p);
}

point *
find_cp()			/* returns the first control point in pointlist */
{
  int i;
  for (i=0; i<pointlist_p; i++) {
    if (pointlist[i]->type == PT_POS) return(pointlist[i]);
  }
  return(NULL);
}

