/* 
 * motor.c -- functions for sending motor controls
 * 
 */

#include <stdio.h>

#include "constants.h"
#include "structures.h"
#include "parameters.h"

struct mctl
{
  float vel,			/* forward/reverse velocity */
        rot,			/* rotational velocity or */
        head;			/* heading */
} control,			/* current commands */
  prev_control;			/* previous ones */

float
target_vel()			/* returns targeted velocity */
{
  return(control.vel);
}

float
target_head()			/* returns targeted heading, in degrees */
{
  float ctl = flakey.control;
  if (ABS(ctl) > 100.0) ctl = 0.0;
  return(ctl*RAD_TO_DEG);
}


set_control(int type,float value) /* accumulates new controls */
{				/* currently use velocity and heading */
  switch(type)
    {
    case VEL:
      control.vel += value;
      break;
    case HEAD:
      control.head += value;
      break;
    }
}



float go_acc = 100;		/* speeding up acceleration */
float slow_acc = 100;		/* slowing down acceleration */
float brake_acc = 200;		/* braking acceleration */

set_flakey_accel(float vel) /* checks current acceleration */
{
  static float acc = 0.0;	/* current */

  if (vel < 10.0) vel = brake_acc;
  else if (vel < control.vel) vel = slow_acc;
  else vel = go_acc;

  if (vel != acc)
    {
      acc = vel;
      command_to_flakey_i(SETA_COM,(int)acc);
    }
}



#define VEL_REPEAT 10		/* number of times to wait on redundant vel commands */
#define VEL_THRESH 10.0		/* this much makes a difference */

send_vel_command(float vel)	/* send out a velocity command */
{
  static int count = VEL_REPEAT;
  float x;
 
  count--;
  x = ABS(prev_control.vel - vel);
  if (count < 0 || x >= VEL_THRESH)
    {
      set_flakey_accel(vel);
      command_to_flakey_i(VEL_COM,(int)vel);
      count = VEL_REPEAT;
      prev_control.vel = vel;
    }
}


#define HEAD_REPEAT 10		/* number of times to wait on redundant head commands */
#define HEAD_THRESH (1.0 * DEG_TO_RAD) /* this much makes a difference */
#define HEAD_MAX    (30.0 * DEG_TO_RAD)	/* max degrees we ever head */

send_head_command(float th)	/* send out a heading command */
{
  static int count = HEAD_REPEAT;
  
  count--;
  if (count < 0 || ABS(th) >= HEAD_THRESH)
    {
      if (flakey.control > 999.0 ||
	  (th > 0.0 && flakey.control < HEAD_MAX) ||
	  (th < 0.0 && flakey.control > (- HEAD_MAX)))
	{
	  command_to_flakey_i(DHEAD_COM,(int)(th*RAD_TO_DEG));
	}
      count = VEL_REPEAT;
      control.head = 0.0;
      prev_control.head = th;
    }
}


incr_velocity(float inc)	/* increment the velocity setpoint */
{
  control.vel +=  inc;
}

incr_head(float inc)		/* increment the heading setpoint */
{				/* use DEGREES here... */
  control.head +=  inc * DEG_TO_RAD;
}

robot_stop()			/* stop everything */
{
  control.vel = 0.0;
  control.head = 0.0;
  control.rot = 0.0;
  flakey.control = flakey.th;
}


motor_proc()			/* sends out motor control commands */
{
  send_vel_command(control.vel);
  send_head_command(control.head);
}


/*
 * Keyboard functions 
 */

extern int single_step_mode, step_wait;

process_key(x)			/* processes a keystroke command */
{
/*   printf("\n %d",x); */
  switch(x)
	{
	case 68:		/* I key: velocity */
	  incr_velocity(40.0);
	  break;
	case 113:		/* M key: reverse velocity */
	  incr_velocity(-40.0);
	  break;
	case 90:		/* J key: left turn */
	  incr_head(4.0);
	  break;
	case 92:		/* L key: right turn */
	  incr_head(-4.0);
	  break;
	case 128:		/* space: stop */
	  if (single_step_mode)
	    {
	      command_to_flakey_i(STEP_COM,STEP_STEP_ARG);
	      step_wait = 0;
	    }
	  else robot_stop();
	  break;
	}
}

