/* 
 * Routines for reading/writing packets
 */

#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sgtty.h>
#include "structures.h"
#include "constants.h"
#include "parameters.h"

/*
 * Conversion factors for input from the robot
 *
 */
float vel_conv_factor = 1.0;	/* convert to velocity in mm/sec */
float angle_conv_factor = RAW_TO_RAD; /* convert to angles in radians */
float dist_conv_factor = 1.0;	/* convert to distance in mm */
float diff_conv_factor = (1.0 / ANG_VEL_TO_WHEEL_DIFF);	
                  /* convert to angular velocity in radians/sec */
float range_conv_factor = 1.0;	/* range units to mm */

write_sint_to_client(int i)	/* write a short integer as two bytes, low-order first */
{
  write_byte_to_client(0xff & i);
  write_byte_to_client(0xff & (i>>8));
}

write_string_to_client(char *s)	/* write string s, with length as first byte */
{
  write_byte_to_client(strlen(s));
  while(*s)
    write_byte_to_client(*s++);
}

int
read_sint_from_client()		/* reads a short integer from input stream */
{
  int n;
  n = read_byte_from_client();
  n = n | (read_byte_from_client() << 8);
  if (n > 0x7fff) n -= 0x10000;
  return(n);
}

int
read_usint_from_client()	/* reads a short unsigned integer from input stream */
{
  int n;
  n = read_byte_from_client();
  n = n | (read_byte_from_client() << 8);
  return(n);
}


int 
read_word_from_client()		/* reads a long integer (4 bytes) */
{
  int n;
  n = (read_byte_from_client() << 24);
  n = n | (read_byte_from_client() << 16);
  n = n | (read_byte_from_client() << 8);
  n = n | read_byte_from_client();
  return(n);
}

char buf[20];

int
sync_client()			/* syncs to a new client by echoing end-chars */
{
  int i, c;
  int count = 0;
  for (i = 0; i < 6; i++)
    {
      sprintf(buf,"Synching %d",count);
      message(buf);
      clear_client_input();
      write_byte_to_client(count);
      output_packet();
      if (client_data_waiting_timeout(1000))
	{
	  ms_pause(100);
	  c = 0;
	  if (process_input()>=0)
	    c = read_byte_from_client();
	  if (c != count++) count = 0;
	  if (count > 2) return(0);
	}
    }
  return(-1);
}


/* sending commands to flakey */

write_command(c)
{
  write_byte_to_client(c);
}


int trace_commands = 0;
int trace_sonar = 0;

command_to_flakey(com)
{
  if (trace_commands)
    printf("\nCom: %d",com);
  write_command(com);
  output_packet();
}

command_to_flakey_i(com,arg)	/* integer argument */
{
  if (trace_commands)
    printf("\nCom: %d %d",com,arg);
  write_command(com);
  if (arg >= 0) write_byte_to_client(ARGchar);
  else write_byte_to_client(NARGchar);
  write_sint_to_client(ABS(arg));
  output_packet();
}

command_to_flakey_s(int com,char *str) /* string argument */
{
  if (trace_commands)
    printf("\nCom: %d %s",com,str);
  write_command(com);
  write_byte_to_client(SARGchar);
  write_string_to_client(str);
  output_packet();
}
 


open_motor_controller(high_power, speech) /* try opening motor controller */
{
  int i, arg;
/*  command_to_flakey(CLOSE_COM); */ 
  for (i=0; i<4; i++)
    {
      clear_client_input();
      if (!speech) arg = -1;
      else if (high_power) arg = HIGH_POWER_ARG;
      else arg = NORMAL_POWER_ARG;
      command_to_flakey_i(OPEN_COM,arg);
      command_to_flakey_i(PULSE_COM,MOTOR_ARG); /* send a packet requesting info */
      if (wait_client_packet(1000))
	{
	  process_one_client_packet(1);
	  return(0);
	}
    }
  return(-1);
}
      


/* Process all waiting incoming packets */


int
client_packet_waiting()		/* returns 1 if packets are waiting in the input queue */
{
  while(client_data_waiting())
    {
      if (process_input() >= 0)
	return(1);
      else 
	return(0);
    }
  return(0);

}

int
wait_client_packet(int wait)	/* waits until a client packet appears */
{
  int count = 0;
  while (process_input() < 0)
    { 
      ms_pause(100);
      count += 100;
      if (count >= wait) return(0);
    }
  return(1);
}


process_waiting_packets()	/* processes any input packets */
{
  while(client_packet_waiting())
    process_one_client_packet(0);
}


int trace_packets = 0;

sdata sbucket[MAX_SONARS];	/* holds sonar data */

process_sonar_packet(float x,float y,float th,int n,int s)
{
  sdata *r;
  if (n >= 0 && n < sonar_num)	/* check bounds */
    {
      r = &sbucket[n];
      r->fx = x; r->fy = y; r->fth = th; 
      r->range = s; r->new = True;
      sonar_coords(n);		/* calculate RW coordinates */
    }
  else
    {
      printf("\n*** Illegal sonar number: %d\n",n); 
      fflush(stdout);
    }
}


process_one_client_packet(int init) /* process one packet */
{
  int pac_type, arg, range;
  int status;
  char *s;
  int n, x,y,th, tx, ty, tth;	/* odometry readings */
  float dx, dy, dth;		/* incremental movement */

  /* Read the Packet */
  pac_type = read_byte_from_client();
  status = pac_type & 0x0f;	/* extract status */
  pac_type  = pac_type & 0xf0;	/* and pac type nibbles */
/*  read_usint_from_client();	/* for tracing... */
  if (trace_packets)
    { 
      switch(pac_type)
	{
	case MOTORpac:
	  s = "mot";
	  break;
	case DEPTHpac:
	  s = "dep";
	  break;
	case SONARpac:
	  s = "son";
	  break;
	default: 
	  s = "oth";
	  break;
	}
      printf(" %s",s);
      fflush(stdout);
    }
  x = read_usint_from_client();	/* odometer absolute values in mm */
  y = read_usint_from_client();
  if (init)
    { flakey.last_x = x; flakey.last_y = y; }
  tx = x - flakey.last_x;
  ty = y - flakey.last_y;
  if (tx > 0x1000) tx -= 0x8000; /* overflow check */
  if (tx < -0x1000) tx += 0x8000; /* overflow check */
  if (ty > 0x1000) ty -= 0x8000; /* overflow check */
  if (ty < -0x1000) ty += 0x8000; /* overflow check */
  dx = dist_conv_factor * (float)tx; 
  dy = dist_conv_factor * (float)ty;
  if (dx > 200.0) printf("\nLarge DX: %x %x",x,flakey.last_x);
  if (dx < -200.0) printf("\nLarge DX: %x %x",x,flakey.last_x);
  if (dy > 200.0) printf("\nLarge DY: %x %x",y,flakey.last_y);
  if (dy < -200.0) printf("\nLarge DY: %x %x",y,flakey.last_y);
                    /* rotate to current direction */
  prot(&dx,&dy,sub_ang((angle_conv_factor * flakey.last_th),flakey.th)); 
  th = read_sint_from_client();
  if (init)
    { flakey.last_th = th; }
  tth = th - flakey.last_th;		/* this is the incremental change in angle */
  dth = norm2_angle(angle_conv_factor * tth);

  switch(pac_type)		/* do differential packet processing */
    {
    case MOTORpac:
    case SMOTORpac:
      flakey.motor_packet_count++;
      flakey.status = status;
      flakey.last_th = th;		/* update previous readings */
      flakey.last_x  = x;
      flakey.last_y  = y;
      flakey.x += dx;
      flakey.y += dy;
      flakey.th = add_ang(dth,flakey.th);
      flakey.leftv = vel_conv_factor * (float)read_sint_from_client();
      flakey.rightv = vel_conv_factor * (float)read_sint_from_client();
      flakey.tv = (flakey.leftv + flakey.rightv) * 0.5;
      flakey.rv =  (flakey.leftv - flakey.rightv) * diff_conv_factor;
      flakey.battery = read_byte_from_client();
      flakey.bumpers = read_sint_from_client();
      th = read_sint_from_client(); /* heading control */
      if (th < 0) flakey.control = 1000.0;
      else flakey.control = norm2_angle(angle_conv_factor*(th - flakey.last_th));
      flakey.say = read_byte_from_client();
      if (pac_type == MOTORpac) break;
      n = read_byte_from_client(); /* number of sonar entries */
      if (n>6) n = 0;		/* just in case... */
      while(n--) 
	{ 
	  range = read_sint_from_client();
	  status = read_byte_from_client();
	  if (trace_sonar) printf(" %d,%d ",status,range);
	  range = range * range_conv_factor;
	  process_sonar_packet(flakey.x,flakey.y,flakey.th,status,range);
	  flakey.sonar_packet_count++;
	}
      break;
      
    case SONARpac:
      flakey.sonar_packet_count++;
      range = read_sint_from_client();
      if (trace_sonar) printf(" %d,%d ",status,range);
      range = range * range_conv_factor;
      process_sonar_packet(dx+flakey.x,dy+flakey.y,add_ang(dth,flakey.th),status,range);
      break;
    }
}




