/* 
 * behaviors.c -- some behavior definitions
 * 
 */

#include <stdio.h>

#include "constants.h"
#include "structures.h"
#include "constructs.h"
#include "parameters.h"

extern param *
beh_alloc(int n)
{
  return((param *)calloc(n,sizeof(param)));
}


float
f_or(float a, float b)
{
  return(MAX(a,b));
}

float
f_and(float a, float b)
{
  return(MIN(a,b));
}

float
f_not(float a)
{
  return(1.0 - a);
}

float
straight_down(float x, float a, float b)
{
  if (x < a) return(1.0);
  if (x > b) return(0.0);
  if (a == b) return(0.0);
  return((b - x) / (b - a));
}

float
up_straight(float x, float a, float b)
{
  if (x > a) return(1.0);
  if (x < b) return(0.0);
  if (a == b) return(0.0);
  return((x - a) / (b - a));
}

float
f_smaller(float a, float b,float delta)
{
  return(straight_down(a,(b - delta),b));
}

float
f_greater(float a, float b,float delta)
{
  return(f_smaller(b,a,delta));
}

float
f_eq(float a, float b,float delta)
{
  return(straight_down(ABS(a-b),0.0,delta));
}


float
f_very(float a)
{
  return(up_straight(a, 0.7, 1.0));
}

float
f_near(float a, float delta)
{
  return(straight_down(ABS(a), 0.0, delta));
}


/* 
 * Constant velocity -- achieve and maintain a constant velocity
 * 
 */

#define CV_PARAMS 6
#define CV_TOO_FAST_n 0
#define CV_TOO_FAST params[0].f
#define CV_TOO_SLOW_n 1
#define CV_TOO_SLOW params[1].f
#define CV_SPEED_n 2
#define CV_SPEED    params[2].f
#define CV_MAX_SPEED_n 3
#define CV_MAX_SPEED params[3].f
#define CV_SPEEDUP_n 4
#define CV_SPEEDUP params[4].f
#define CV_SLOWDOWN_n 5
#define CV_SLOWDOWN params[5].f

beh_params 
cv_setup(beh_params p)		/* setup function */
{
  beh_params params;
  params = beh_alloc(CV_PARAMS);
  CV_MAX_SPEED = 500.0;
  CV_SPEED = p[0].f;
  return(params);
}


void
cv_check_speed(beh_params params) /* update function */
{
  CV_TOO_SLOW = f_smaller(target_vel(),MIN(CV_SPEED,CV_MAX_SPEED),CV_SPEED*.2);
  CV_TOO_FAST = f_greater(target_vel(),CV_SPEED,CV_SPEED*.2);
  CV_SPEEDUP  = MIN(CV_SPEED,CV_MAX_SPEED) - target_vel();
  CV_SLOWDOWN = target_vel() - CV_SPEED;
  c_act_v = f_or(CV_TOO_FAST,CV_TOO_SLOW);
  c_act_t = 0.0;
  c_progress = 1.0;
  c_goal = 0.0;
}


behavior
constant_velocity = { "Constant Vel", cv_setup, cv_check_speed, 2,
			{ { "Speed-Up", CV_TOO_SLOW_n, ACCEL, CV_SPEEDUP_n },
			  { "Slow-Down", CV_TOO_FAST_n, DECEL, CV_SLOWDOWN_n }
			}
		    };




/* 
 * Avoid collision --- avoid obstacles close to the robot
 * 
 */

#define AC_PARAMS 10
#define AC_LEFT_n 0
#define AC_LEFT params[0].f
#define AC_RIGHT_n 1
#define AC_RIGHT params[1].f
#define AC_FRONT_n 2
#define AC_FRONT params[2].f
#define AC_REAR_n 3
#define AC_REAR params[3].f
#define AC_TURN_DEPTH_n 4
#define AC_TURN_DEPTH params[4].f
#define AC_SLOWDOWN_n 5
#define AC_SLOWDOWN params[5].f
#define AC_FRONT_SN_n 6
#define AC_FRONT_SN params[6].f
#define AC_SIDE_SN_n 7
#define AC_SIDE_SN params[7].f
#define AC_TURN_SN_n 8
#define AC_TURN_SN params[8].f
#define AC_BLOCKED_n 9
#define AC_BLOCKED params[9].f

beh_params			/* setup function, with sensitivities */
ac_setup(beh_params p)		/* p[0] is front, p[1] is side, p[2] is turn */
{
  beh_params params;
  params = beh_alloc(AC_PARAMS);
  AC_TURN_DEPTH = 4.0;
  AC_SIDE_SN = p[1].f;
  AC_FRONT_SN = p[0].f;
  AC_TURN_SN = p[2].f;
  return(params);
}

#define AC_SPEED -30.0

void
ac_collision_danger(beh_params params) /* update function */
{
  int fincd = 200, fincs = 70, fstart = 120, fedge1 = 100,
      fedge2 = 300, sincd = 100, sincs = 100, sstart = 200,
      sedge1 = 50, sedge2 = 300,
      fstand = 400, sstand = 340,
      fsense = 50;
  
  float cf, crs, crf, cls, clf, vel, fdist, sdist;
  float right, front_right, left, front_left, front;

  fedge2 += fsense*AC_FRONT_SN;

/*  vel = (int)MAX(flakey.tv,target_vel()); */
  vel = flakey.tv;
  clf = MIN(occupied2(fstart,fedge1,fedge2,1),
	    MIN(occupied2(fstart+fincd, fedge1, fedge2+fincs, 1),
		occupied2(fstart+fincd+fincd, fedge1, fedge2+fincs+fincs, 1)));
  cls = MIN(occupied2(sstart,sedge1,sedge2,2),
	    MIN(occupied2(sstart+sincd, sedge1, sedge2+sincs, 2),
		occupied2(sstart+sincd+sincd, sedge1, sedge2+sincs+sincs, 2)));
  fedge1 = -fedge1;
  fedge2 = -fedge2;
  fincs = -fincs;

  crf = MIN(occupied2(fstart,fedge1,fedge2,1),
	    MIN(occupied2(fstart+fincd, fedge1, fedge2+fincs, 1),
		occupied2(fstart+fincd+fincd, fedge1, fedge2+fincs+fincs, 1)));
  crs = MIN(occupied2(sstart,sedge1,sedge2,-2),
	    MIN(occupied2(sstart+sincd, sedge1, sedge2+sincs, -2),
		occupied2(sstart+sincd+sincd, sedge1, sedge2+sincs+sincs, -2)));
  
  cf = occupied2(100,-300,300,1);

  fdist = AC_FRONT_SN*MAX(150,vel); /* distance of front sensitivity, 900mm @ 300mm/sec */
  sdist = AC_SIDE_SN*MAX(80,vel); /* distance of side sensitivity, 300mm @ 300mm/sec */

  right = straight_down(crs,sstand,sstand+sdist);
  front_right = straight_down(crf,fstand,fstand+fdist);
  left = straight_down(cls,sstand,sstand+sdist);
  front_left = straight_down(clf,fstand,fstand+fdist);
  front = straight_down(cf,fstand,fstand+fdist);
  front = MAX(front,
	      MAX(front_right*.5,front_left*.5));

  AC_RIGHT = f_and(f_or(right,front_right),
		   f_and(f_not(front_left),
			 f_not(left)));

  AC_LEFT  = f_and(f_or(left,front_left),
		   f_and(f_not(front_right),
			 f_not(right)));

  AC_FRONT = f_or(front,
		  f_or(front_left,front_right));

  AC_BLOCKED = f_and(AC_FRONT, f_eq(AC_RIGHT,AC_LEFT,1.0));

  if (front_right > .2)
    draw_centered_rect(crf*.5,-30.0+crf*-.5,crf,crf);

  if (front_left > .2)
    draw_centered_rect(clf*.5,30.0+clf*.5,clf,clf);

  if (right > .2)
    draw_centered_rect(100.0,-30.0+crs*-.5,100.0,crs);

  if (left > .2)
    draw_centered_rect(100.0,30.0+cls*.5,100.0,cls);

  c_act_t = MAX(right,
		 MAX(left,
		     MAX(front_right,
			 MAX(front_left,front))));
  c_act_v = c_act_t;

  AC_TURN_DEPTH = AC_TURN_SN;

  AC_SLOWDOWN = 0.0;
  if (target_vel() > AC_SPEED)
    { AC_SLOWDOWN = target_vel() - AC_SPEED; }

  c_progress = 1.0;
  c_goal = 0.0;
}


behavior
avoid_collision = { "Avoid Collision", ac_setup, ac_collision_danger, 4,
			{ { "Collision-Right", AC_RIGHT_n, TURN_LEFT, AC_TURN_DEPTH_n },
			  { "Collision-Left",  AC_LEFT_n, TURN_RIGHT, AC_TURN_DEPTH_n },
			  { "Collision-Front", AC_FRONT_n, DECEL, AC_SLOWDOWN_n },
			  { "Collision-Blocked-Turn", AC_BLOCKED_n, TURN_LEFT, AC_TURN_DEPTH_n },
			}
		    };




/*
 *             GO TO POS
 *
 *  PURPOSE:  To go to an X,Y position
 *  PARAMETERS: speed, target-point, radius
 *
 */


#define GT_PARAMS 14
#define GT_TOO_FAST_n 0
#define GT_TOO_FAST params[0].f
#define GT_TOO_SLOW_n 1
#define GT_TOO_SLOW params[1].f
#define GT_SPEED_n 2
#define GT_SPEED    params[2].f
#define GT_MAX_SPEED_n 3
#define GT_MAX_SPEED params[3].f
#define GT_SPEEDUP_n 4
#define GT_SPEEDUP params[4].f
#define GT_SLOWDOWN_n 5
#define GT_SLOWDOWN params[5].f
#define GT_POS_n 6
#define GT_POS params[6].p	/* this is the control point */
#define GT_POS_ON_LEFT_n 7
#define GT_POS_ON_LEFT params[7].f
#define GT_POS_ON_RIGHT_n 8
#define GT_POS_ON_RIGHT params[8].f
#define GT_ANGLE_WAY_OFF_n 9
#define GT_ANGLE_WAY_OFF params[9].f
#define GT_POS_ACHIEVED_n 10
#define GT_POS_ACHIEVED params[10].f
#define GT_TURN_AMOUNT_n 11
#define GT_TURN_AMOUNT params[11].f
#define GT_RADIUS_n 12
#define GT_RADIUS params[12].f
#define GT_SLOWIT_n 13
#define GT_SLOWIT params[13].f


beh_params 
gt_setup(beh_params p)		/* setup function */
{
  beh_params params;
  params = beh_alloc(GT_PARAMS);
  GT_MAX_SPEED = 500.0;
  GT_SPEED = p[0].f;
  GT_POS = p[1].p;
  GT_RADIUS = p[2].f;
  return(params);
}


#define GP_TURN_FACTOR 6.0

void
gt_update(beh_params params) /* update function */
{
  float x, y, phi, delta;
  point *p;
  int lhs_clear, rhs_clear;
  float ach, left, right, bleft, bright, wayoff;
  float too_slow, too_fast;


  too_fast = f_greater(target_vel(),GT_SPEED,GT_SPEED*.2);
  too_slow = f_smaller(target_vel(),MIN(GT_SPEED,GT_MAX_SPEED),GT_SPEED*.2);

  p = (point *)GT_POS;		/* get target position */
  x = p->x; y = p->y; 
  phi = RAD_TO_DEG * point_phi(p);
  delta = target_head();
  if (phi < 0.0)
    {
      if (delta > phi) delta = delta - phi;
      else delta = 0.0;
    }
  else
    {
      if (delta < phi) delta = phi - delta;
      else delta = 0.0;
    }
  GT_TURN_AMOUNT = GP_TURN_FACTOR * f_not(f_near(delta,6.0)); 
  rhs_clear = occ(300,-600,400,600);
  lhs_clear = occ(300,600,400,600);

  ach = straight_down(MAX(ABS(x),ABS(y)),GT_RADIUS,GT_RADIUS * 0.5);
  left = up_straight(phi,10.0,30.0);
  right = up_straight(-phi,10.0,30.0);
  bleft = straight_down(lhs_clear,100.0,300.0);
  bright = straight_down(rhs_clear,100.0,300.0);
  wayoff = up_straight(ABS(phi),50.0,70.0);

  GT_POS_ON_LEFT = f_and(left,f_and(f_not(ach),
				    f_or(wayoff,f_not(bleft))));
  GT_POS_ON_RIGHT = f_and(right,f_and(f_not(ach),
				    f_or(wayoff,f_not(bright))));
  GT_POS_ACHIEVED = f_very(f_very(ach));
  GT_ANGLE_WAY_OFF = wayoff;
  
  GT_TOO_SLOW = f_and(too_slow, f_and(f_not(ach), f_not(wayoff)));
  GT_TOO_FAST = f_and(too_fast, f_not(wayoff));
  GT_SPEEDUP  = MIN(GT_SPEED,GT_MAX_SPEED) - target_vel();
  GT_SLOWDOWN = target_vel() - GT_SPEED;
  GT_SLOWIT = target_vel();

  c_act_v = c_act_t = 1.0;
  c_progress = 1.0;
  c_goal = ach;
}


behavior
go_to_pos = { "Go To Position", gt_setup, gt_update, 6,
		{ 
		  { "Speed-Up", GT_TOO_SLOW_n, ACCEL, GT_SPEEDUP_n },
		  { "Slow-Down", GT_TOO_FAST_n, DECEL, GT_SLOWDOWN_n },
		  { "Pos-on-left", GT_POS_ON_LEFT_n, TURN_LEFT, GT_TURN_AMOUNT_n },
		  { "Pos-on-right", GT_POS_ON_RIGHT_n, TURN_RIGHT, GT_TURN_AMOUNT_n },
		  { "Pos-here", GT_POS_ACHIEVED_n, DECEL, GT_SLOWIT_n },
		  { "Angle-way-off", GT_ANGLE_WAY_OFF_n, DECEL, GT_SLOWIT_n },
		}
	    };
