/*#########################################
 * flakey-seg.lisp
 *
 * Line-segment operations for the
 * Flakey simulator.
 *#########################################
 */

/* Basic definitions of segments, lines and points.
 * Uses homogenous (3-dim) coordinates for 2-dim geometry.
 * Distances are in mm
 * Angles are in rads * 1000, segment rads normalized 0 - pi
 * Uses single-floats
 */


#include <malloc.h>
#include "structures.h"
#include "constants.h"

/* simple types of operations: cross-product, dot-product, normalize */

float norm_angle(float a)	/* between 0 and 2PI */
{
  if (a > ANG360) a = a - ANG360;
  else if (a < 0.0) a = a + ANG360;
  return(a);
}

float norm2_angle(float a)	/* between -PI and PI */
{
  if (a > ANG180) a = a - ANG360;
  else if (a < (- ANG180)) a = a + ANG360;
  return(a);
}


float norm_seg_angle(float a)	/* between 0 and PI */
{
  if (a >= ANG180) a = a - ANG180;
  else if (a < 0) a = a + ANG180;
  return(a);
}

void normalize_pt(pt *p)	/* destructively normalize a point */
{
  p->x = p->x / p->w;
  p->y = p->y / p->w;
}

float dot_product(pt *p,line *l) /* dot product of a point and line */
{
  float r;
  r = p->x * l->a;
  r += p->y * l->b;
  r += p->w * l->c;
  return(r);
}

void cross_product_pt(p1,p2,l)	/* cross product of two points, as a line */
     pt *p1, *p2; line *l;
{
  l->a = (p2->w * p1->y) - (p2->y * p1->w);
  l->b = (p1->w * p2->x) - (p1->x * p2->w);
  l->c = (p2->y * p1->x) - (p1->y * p2->x);
}
  
void cppp(float x1,float y1,float x2, float y2,
	  float *a,float *b,float *c)
     /* returns line params of x1,y1 to x2,y2 in a,b,c */
{
  *a = y1 - y2;
  *b = x2 - x1;
  *c = (y2 * x1) - (x2 * y1);
}

void cpll(float a,float b,float c,float d,float e,float f,
	  float *x,float *y)	/* returns line intersection in x,y */
{
  float n;
  n = (e * a) - (d * b);
  if (n == 0.0) {*x = 0.0; *y = 0.0;}
  else {
    *x = ((f*b) - (e*c)) / n;
    *y = ((c*d) - (a*f)) / n;
  }
}


void intpps(float x1,float y1,float x2,float y2,
	    seg *s,float *x,float *y)
     /* returns point of intersection of s and endpts in x,y */
{
  float *a, *b, *c;
  cppp(x1,y1,x2,y2,a,b,c);
  cpll(*a,*b,*c,s->a,s->b,s->c,x,y);
}

int
between(float x1,float y1,float x2,float y2,
	float x,float y) /* returns 1 if x,y is between the endpoints, else 0 */
{
  return(((x1 == x2) || ((x >= x1) ? x < x2 : x >= x2)) &&
	 ((y1 == y2) || ((y >= y1) ? y < y2 : y >= y2)));
}

line *perp_line(float x,float y,seg *s)
     /* returns a new line object perp to seg s, thru point x,y */
{
  line *l;
  l = (line*)malloc(sizeof(line));
  if (l == NULL) perror("line allocation");
  l->a = s->b;
  l->b = - s->a;
  l->c = (s->a * y) - (s->b * x);
  return(l);
}


void set_seg_params(s)		/* sets up seg params from x,y */
     seg *s;
{
  line *l;
  cppp(s->x1,s->y1,s->x2,s->y2,&(s->a),&(s->b),&(s->c));
  l = perp_line(s->x1,s->y1,s);
  s->a1 = l->a;
  s->b1 = l->b;
  s->c1 = l->c;
  l = perp_line(s->x2,s->y2,s);
  s->a2 = l->a;
  s->b2 = l->b;
  s->c2 = l->c;
  s->norm = sqrt(s->a*s->a + s->b*s->b);
  s->th = norm_seg_angle(atan2(s->y1 - s->y2, s->x1 - s->x2));
}


seg *create_seg(float x1,float y1,float x2,float y2)	
     /* creates and returns a segment using endpoints */
{
  seg *s;
  s = (seg*)malloc(sizeof(seg));
  s->x1 = x1;
  s->x2 = x2;
  s->y1 = y1;
  s->y2 = y2;
  set_seg_params(s);
  return(s);
}

void move_seg(seg *s,float dx,float dy)	/* translate by adding dx,dy to each endpt */
{
  s->x1 += dx;
  s->x2 += dx;
  s->y1 += dy;
  s->y2 += dy;
  s->c = s->y2*s->x1 - s->x2*s->y1;
  s->c1 = s->a*s->y1 - s->b*s->x1;
  s->c2 = s->a*s->y2 - s->b*s->x2;
}

float dist_to_seg_line(p,s)	/* distance from normalized pt to segment */
     pt *p;  seg *s;
{
  return(abs((p->x*s->a + p->y*s->b + s->c) / s->norm));
}


float p_dist_to_seg_line(float x,float y,seg *s)
     /* distance from normalized pt to segment */
{
  return(abs((x*s->a + y*s->b + s->c) / s->norm));
}

int
side_of(float x,float y,seg *s)	/* which side of seg the point x,y is on */
{
  return((int)(x*s->a + y*s->b + s->c));
}


side_of_1(float x,float y,seg *s) /* which side of seg endpt #1 x,y is on */
				/* positive is towards other endpt */
{
  return((int)(x*s->a1 + y*s->b1 + s->c1));
}

side_of_2(float x,float y,seg *s) /* which side of seg endpt #1 x,y is on */
				/* negative is towards other endpt */
{
  return((int)(x*s->a2 + y*s->b2 + s->c2));
}


float 
sqdist(float x1,float y1,float x2,float y2) /* squared distance between points */
{
  return((x1-x2)*(x1-x2) + (y1-y2)*(y1-y2));
}


int 
within_dist_to_seg(float x,float y,seg *s,float d)
				/* point x,y is distance d to seg s */
{
  if (side_of_1(x,y,s) < 0)
    { if (sqdist(x,y,s->x1,s->y1) <= d*d) return(1); else return(0); }
  else if (side_of_2(x,y,s) > 0)
    { if (sqdist(x,y,s->x2,s->y2) <= d*d) return(1); else return(0); }
  else
    { if (p_dist_to_seg_line(x,y,s) <= d) return(1); else return(0); }
}

float 
angle_of_line(float a,float b,float c) /* returns angle of line, normalized */
{
  if (a == 0.0) return(0.0);
  else if (b == 0.0) { return(ANG90); }
  else return(norm_seg_angle(rad_to_ang(atan2((c / b),(- (c / a))))));
}
    

float
angle_of_pts(float x1,float y1,float x2,float y2)
				/* angle between two points, not normalized */
{
  return(rad_to_ang(atan2(y2-y1,x2-x1)));
}

float
angle_to_seg(float x,float y,seg *s)
				/* returns angle of closest pt on s to x,y */
{
  float a;
  if (side_of_1(x,y,s) < 0)
    { return(angle_of_pts(x,y,s->x1,s->y1)); }
  else if (side_of_2(x,y,s) > 0)
    { return(angle_of_pts(x,y,s->x2,s->y2)); }
  else
    { a = angle_of_line(s->a,s->b,s->c);
      if (side_of(x,y,s) >= 0)
	{ if (a > ANG90) return(a - ANG90); else return(a + ANG270); }
      else
	{ return(a + ANG90); }
    }
}
      

