/* 
 * Routines for reading/writing packets
 */

#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sgtty.h>
#include "structures.h"
#include "constants.h"
#include "parameters.h"

write_sint_to_client(int i)	/* write a short integer as two bytes, low-order first */
{
  write_byte_to_client(0xff & i);
  write_byte_to_client(0xff & (i>>8));
}

write_string_to_client(char *s)	/* write string s, with length as first byte */
{
  write_byte_to_client(strlen(s));
  while(*s)
    write_byte_to_client(*s++);
}

int
read_sint_from_client()		/* reads a short integer from input stream */
{
  int n;
  n = read_byte_from_client();
  n = n | (read_byte_from_client() << 8);
  if (n > 0x7fff) n -= 0x10000;
  return(n);
}

int 
read_word_from_client()		/* reads a long integer (4 bytes) */
{
  int n;
  n = (read_byte_from_client() << 24);
  n = n | (read_byte_from_client() << 16);
  n = n | (read_byte_from_client() << 8);
  n = n | read_byte_from_client();
  return(n);
}

write_end_char(int forcep)	/* writes an end-char to output stream */
{
  write_byte_to_client(ENDchar);
  if (forcep) force_client_output();
}

int
read_end_char()			/* returns 1 if next char is ENDchar */
{
  return(read_byte_from_client() == ENDchar);
}


int
sync_client()			/* syncs to a new client by echoing end-chars */
{
  int loop = 1;
  while (client_data_waiting_timeout(1000))
    {
      if (read_end_char()) write_end_char(1);
      else return(0);
    }
  return(-1);
}

/* Variables used in sending out info packets */

int motor_packet_count = 0;	/* packet sending variables */
int sonar_packet_count = 0;
int sonar_polling_flag = 0;

int sonar_num;			/* sonar parameters */
int current_sonar_num = 0;

float head_control = -1.0;	/* heading control variable */
int say_control = 0;		/* acknowledgment number for speech */

extern int single_step_mode;
extern int step_wait;
extern int nohalt;

/* Send out robot info packets to client */

send_client_packets()		/* send out info */
{
  if (motor_packet_count-- > 0)	/* motor packets */
    { send_motor_packet(); }
  if (sonar_polling_flag && sonar_packet_count-- > 0)
    { send_sonar_packet(); }	/* sonar packets */
}


write_packet_header(int n)	/* writes a packet header of type n */
{
  write_byte_to_client(SYNC1char);
  write_byte_to_client(SYNC2char);
  write_byte_to_client(n);
}


send_motor_packet()
{
  int d;
  write_packet_header(motor_header());
  write_sint_to_client((int)erratic.dx);
  write_sint_to_client((int)erratic.dy);
  write_sint_to_client((int)(RAD_TO_RAW * ang_to_rad(erratic.dth)));
  d = (int)(ANG_VEL_TO_WHEEL_DIFF * erratic.rv);
  write_sint_to_client((int)erratic.tv - d); /* left wheel velocity */
  write_sint_to_client((int)erratic.tv + d); /* right wheel velocity */
  write_byte_to_client(25);	/* battery voltage */
  write_sint_to_client(0);	/* bumpers */
  write_sint_to_client((head_control < 0) ? -1 : 
		       (int)(RAD_TO_RAW * head_control));
  write_byte_to_client(say_control); /* ack always */
  write_end_char(1);
}

int
motor_header()			/* motor header flag */
{
  return(MOTORpac | ((erratic.tv == 0.0 && erratic.rv == 0.0) ? 2 : 3));
}
 

int 
sonar_header()			/* sonar header flag */
{
  if (++current_sonar_num >= sonar_num)
    current_sonar_num = 0;
  return(SONARpac | current_sonar_num);
}


send_sonar_packet()		/* send out two sonar packets */
{
  write_packet_header(sonar_header());
  write_sint_to_client((int)erratic.dx);
  write_sint_to_client((int)erratic.dy);
  write_sint_to_client((int)(RAD_TO_RAW * ang_to_rad(erratic.dth)));
  write_sint_to_client(current_sonar_value());
  write_end_char(1);

  write_packet_header(sonar_header());
  write_sint_to_client((int)erratic.dx);
  write_sint_to_client((int)erratic.dy);
  write_sint_to_client((int)(RAD_TO_RAW * ang_to_rad(erratic.dth)));
  write_sint_to_client(current_sonar_value());
  write_end_char(1);
}

set_sonar_polling(int arg)	/* sets up sonar polling */
{
}


/* Process all waiting incoming packets */


int
client_packet_waiting()		/* returns 1 if packets are waiting in the input queue */
{
  while(client_data_waiting())
    {
      if (read_byte_from_client() == SYNC1char)
	return(read_byte_from_client() == SYNC2char);
    }
  return(0);

}

int
wait_client_packet()		/* waits until a client packet appears */
{
  int state = 0;
  int n;
  while(1)
    { 
      n = read_byte_from_client();
      if (state == 0)		/* we're waiting for SYNC1 */
	{ if (n == SYNC1char) state = 1; }
      else if (n == SYNC2char) return(1);
      else if (n == SYNC1char) state = 0;
    }
}


void
process_waiting_packets()	/* processes any input packets */
{
  while(client_packet_waiting())
    process_one_client_packet();
}


process_one_client_packet()	/* process one packet */
				/* we assume we have already seen a SYNC1 SYNC2 sequence */
{
  int pac_type, arg, count;
  char *str;
  char * form_arg_string();
				/* Read the Packet */
  arg = 0; count = 0;
  pac_type = read_byte_from_client();
  switch(arg = read_byte_from_client()) /* read args */
    {
    case ARGchar:
      arg = read_sint_from_client();
      break;
    case NARGchar:
      arg =  - read_sint_from_client();
      break;
    case SARGchar:
      str = form_arg_string();
      break;
    case ENDchar:
      arg = 0;
      break;
    default:
      printf("\n*** Warning: bad argument type %d in packet.",arg);
      break;
    }

  if (debug) printf("\nPac type: %d   Arg: %d",pac_type,arg);

  switch(pac_type)
      {
      case STEP_COM:		/* stepping mode */
	if (arg == 0)		/* turn it off */
	  { printf("\nReturning to continuous simulation");
	    single_step_mode = 0;
	  }
	else if (arg > 0)
	  { printf("\nStarting single step mode");
	    single_step_mode = 1;
	    step_wait = 1;
	  }
	else
	  { printf("\nstep...");
	    step_wait = 0;
	  }
	break;

      case PULSE_COM:		/* pulse motor or sonar */
	count = (arg & 0xff00) >> 8;
	if (count <= 0) count = 1;
	if (arg & MOTOR_ARG)
	  motor_packet_count = count;
	if (arg & SONAR_ARG)
	  sonar_packet_count = count;
	break;

      case OPEN_COM:
	printf("\nClient open request");
	break;

      case CLOSE_COM:		/* close client connection */
	printf("\nClient close request");
	if (arg > 0)
	  {
	    shutdown_client();
	    nohalt = 0;
	    step_wait = 0;
	  }
	break;

      case POLLING_COM:		/* start or stop sonar polling */
	if (arg)
	  {
	    sonar_polling_flag = 1;
	    set_sonar_polling(arg);
	  }
	else sonar_polling_flag = 0;
	break;

      case SETV_COM:		/* set max velocity, in mm/sec */
	erratic.mtv = (float)arg;
	break;
	
      case SETRV_COM:		/* set rot velocity increment, in deg/sec */
	arg = deg_to_ang((float)arg);
	erratic.mrv = arg;
	break;

      case SETA_COM:		/* set acceleration, in mm/sec/sec */
	erratic.mta = (float)arg;
	erratic.mra = (float)arg / 500.0;
	break;
	 
      case SETO_COM:		/* set origin */
	robot_set_origin();
	break;

      case MOVE_COM:		/* move forward, in mm */
	head_control = -1.0;
	c_move((float)arg);
	break;

      case ROTATE_COM: /* rotate to a heading, in degrees */
	head_control = -1.0;
	c_rotate_deg((float)arg);
	break;

      case DROTATE_COM:		/* rotate arg degrees */
	head_control = -1.0;
	c_rotate_deg((float)arg);
	break;

      case VEL_COM:		/* achieve a velocity and hold it, in mm/sec */
	c_trvel((float)arg);
	break;

      case RVEL_COM:		/* achieve a rot velocity and hold it, in ang/sec */
	head_control = -1.0;
	c_rotvel((float)arg);
	break;

      case HEAD_COM:		/* achieve a heading, in deg */
	head_control = norm_angle(deg_to_ang((float)arg));
	c_rotate(sub2_ang(head_control,erratic.dth));
	break;

      case DHEAD_COM:		/* change heading, in deg */
	if (arg)
	  {
	    if (head_control < 0) head_control = erratic.dth;
	    head_control = add_ang(head_control,deg_to_ang((float)arg));
	    c_rotate(sub2_ang(head_control,erratic.dth));
	  }
	else
	  {
	    head_control = -1;
	    c_rotvel(0.0);
	  }
	break;

      case DCHEAD_COM:		/* change heading from current, in deg */
	head_control = erratic.dth;
	head_control = add_ang(head_control,deg_to_ang((float)arg));
	c_rotate(sub2_ang(head_control,erratic.dth));
	break;

      case SAY_COM:
	printf("\nSAYING: %s\n",str);
	say_control++;
	break;
      default:
	printf("\n*** Warning: bad packet type %d.",pac_type);
	break;

      }
}


