;;; -*- Package: Toolset; Syntax: Common-Lisp; Mode: Lisp; Base: 10 -*-

(in-package 'toolset)

;;; Assorted utility functions.


;;; Some additions to the pcl interface, will probably be a real
;;; part of PCL when the metaobject protocol is finished.

(defmacro slot-empty-p (instance slot-name) 
  ;; called with (slot-empty-p obj 'slot)
  `(if (or (not (pcl::slot-boundp ,instance ,slot-name))
	  (null (slot-value ,instance ,slot-name)))
      t nil))

(defmacro slot-unbound (instance slot-name)
  ;; called with (slot-unbound obj 'slot)
  `(if (not (pcl::slot-boundp ,instance ,slot-name))
       t nil))


(defmacro subclassp (sub super)
  `(let ((subclass (if (symbolp ,sub) (pcl::class-named ,sub) ,sub))
	 (superclass (if (symbolp ,super) (pcl::class-named ,super) ,super)))
     (if (not (and (pcl::iwmc-class-p subclass)
		   (pcl::iwmc-class-p superclass)))
       (error 
	"One of ~S and ~S is not a recognized class or the name of a recognized class." 
	,sub ,super)
       (pcl::memq superclass (pcl::class-precedence-list subclass)))))


;;; An addition to be used with backquote - ',@(fn-returning-list) is
;;; illegal, but this performs the equivalent - 
;;; @,(quote-list (fn-returning-list)) inserts the values returned
;;; in a list by fn-returning-list, but quotes all of them first.

(defun quote-list (list)
  (let ((newlist nil))
    (dolist (item (if (eq (car list) 'quote)
		      (cadr list) list))
	    (if (or (and (listp item)
			 (not (eq (car item) 'quote)))
		    (and (symbolp item)
			 (not (member (char (string item) 0)
				      '(#\& #\:)))))
		(setf newlist (cons (list 'quote item) newlist))
	        (setf newlist (cons item newlist))))
    (setf newlist (reverse newlist))
    newlist))


;;; Methods to take an array and return its contents in a form
;;; suitable for inserting into a make-array form to recreate the
;;; array (used by SAVE-DRAGON).

(defmethod list-contents ((a t)) 
  (if (null a)
      (return-from list-contents nil)
    (error "ERROR: contents of slot should be an array or nil.")))


(defmethod list-contents ((a array)) 
  (let ((contents nil)
	(contents-list nil)
	(dimensions (array-dimensions a)))
    (dotimes (i (car dimensions))
	     (dotimes (j (cadr dimensions))
		      (setf contents (cons (aref a i j)
					   contents)))
	     (setf contents (nreverse contents))
	     (setf contents-list (cons contents contents-list))
	     (setf contents nil))
    (setf contents-list (nreverse contents-list))
    (return-from list-contents contents-list)))


(defmethod list-contents ((a vector))
  (let ((contents nil))
    (dotimes (i (length a))
	     (setf contents (cons (aref a i)
				  contents)))
    (setf contents (nreverse contents))
    (return-from list-contents contents)))


(defun make-slot-array (instance slot-name &optional (size nil size-given))
  (if (slot-empty-p instance slot-name)
      (return-from make-slot-array nil)
    (return-from make-slot-array 
		 (list 'make-array
		       (if size-given size
			 `',(array-dimensions (slot-value instance slot-name)))
		       ':initial-contents
		       `',(list-contents (slot-value instance slot-name))))))


;;; Method to cope with slots filled with nil vs. slots filled with
;;; #:slot-unbound, which can't be tested with #'eq.

(defmacro write-string-slot (obj slot) 
  `(if (or (slot-empty-p ,obj ,slot)
	   (not (stringp (slot-value ,obj ,slot))))
       ;; change all the #:slot-unbound's to nil's - shouldn't be
       ;; anything else
       nil
       (slot-value ,obj ,slot)))


