;-*- Mode:  LISP; Syntax: Common-lisp -*-

;;;; Example of dependency-directed search using JTRE
;;;; Version 8, Last edited 3/19/91

;;; Copyright (c) 1986, 1989, 1990, 1991 Kenneth D. Forbus, Northwestern University,
;;; Johan de Kleer and Xerox Corporation.
;;; All rights reserved.

;; This program uses dependency-directed search to solve the n-queens
;; problem.  The input is the size of the chessboard.  We follow the usual
;; trick and assume that exactly one queen will be placed in each column.
;; Then the problem reduces to finding a set of assignments for the
;; rows such that the queens do not capture each other.

(defvar *queen-choices* nil) ;; List of choice sets.
(defvar *queen-solutions* nil) ;; Answers

(defvar *n-assumptions* 0) ;; Statistics
(defvar *n-tests* 0)

(proclaim '(special *JTRE*))

(defvar *queen-rules-file*
  #+KDF-RT "/u/bps/code/jtms/jqrule.lisp"
  #+QRG-RT "/u/forbus/bps/code/jtms/jqrule.lisp"
  #+GCLISP "c:\\bps\\jtms\\"
  #+PARC "virgo:/virgo/dekleer/bps/code/jtms/jqrule.lisp")

(defun n-queens (n &optional (debugging? nil))
  (setup-queens-problem n debugging?)
  (solve-queens-problem *queen-choices*)
  (length *queen-solutions*))

(defun setup-queens-problem (n &optional (debugging? nil))
  (in-JTRE (create-jtre (format nil "~D-Queens JTRE" n) 
			:debugging debugging?))
  (setq *queen-solutions* nil)
  (setq *n-assumptions* 0 *n-tests* 0)
  (load *queen-rules-file*)
  (build-queen-choices n))

(defun build-queen-choices (n)
  (do ((column 1 (1+ column))
       (column-queens nil nil)
       (choices nil))
      ((> column n) (setq *queen-choices* (nreverse choices)))
    (dotimes (row n)
	     (push `(Queen ,column ,(1+ row)) column-queens))
    (push (nreverse column-queens) choices)))

(defun solve-queens-problem (choices)
  (cond ((null choices) (gather-queens-solution))
	(t (dolist (choice (car choices))
	    (unless (in? `(not ,choice) *jtre*)
	     ;respect nogood information
     (multiple-value-bind (nogood? asns)
      (try-in-context choice
		      `(solve-queens-problem ',(cdr choices))
		      *jtre*)
      (incf *n-assumptions*)
      (when nogood?
	    ;;This assumption lost, so justify the negation
	    ;; based on the other relevant assumptions.
	    (assert! `(not ,choice)
		     `(Nogood ,@ 
			      (remove choice asns))))))))))

(defun try-in-context (asn thunk jtre &aux try-marker result)
  (setq try-marker (cons 'TRY asn))
  (with-contradiction-handler (jtre-jtms jtre)
        #'(lambda (jtms contras)
	    (try-contradiction-handler
	     contras jtms asn try-marker jtre))
        (unwind-protect
	  (progn (unless (in? asn jtre)
		   (setq result (catch 'TRY-CONTRADICTION-FOUND
				  (assume! asn try-marker jtre)))
		   (when (and (listp result) (eq (car result) ':ASNS))
		     (return-from TRY-IN-CONTEXT
				  (values t (mapcar #'view-node (cdr result)))))
		   (setq result (catch 'TRY-CONTRADICTION-FOUND
				  (run-rules jtre)))
		   (when (and (listp result) (eq (car result) ':ASNS))
		     (return-from TRY-IN-CONTEXT
				  (values t (mapcar #'view-node (cdr result)))))
		   (eval thunk) ;; use the thunk
		   (progn (retract! asn try-marker t)
			  (return-from TRY-IN-CONTEXT
				       (values nil nil))))))))

(defun try-contradiction-handler (contras jtms asn marker *JTRE* &aux node)
  (unless (eq jtms (jtre-jtms *JTRE*))
    (error "~%High Contradiction Weirdness: ~A not jtms for ~A!"
	   jtms *JTRE*))
  (unless contras (return-from TRY-CONTRADICTION-HANDLER nil))
  (unless asn (return-from TRY-CONTRADICTION-HANDLER nil))
  (setq node (get-tms-node asn))
  (dolist (cnode contras)
    (let ((asns (assumptions-of-node cnode)))
      (when (member node asns)
	(retract! asn marker)
	(throw 'TRY-CONTRADICTION-FOUND (cons ':ASNS asns))))))

;;; Other helpers

(defun queens-so-far ()
  (remove-if #'(lambda (q) (out? q *jtre*)) (fetch `(Queen ?c ?r) *jtre*)))

(defun gather-queens-solution ()
  (push (queens-so-far)	*queen-solutions*))

(defun test-queens (from to)
  (do ((n from (1+ n)))
      ((> n to))
    (time (n-queens n))
    (format t "~%~D solutions, ~D assumptions." (length *queen-solutions*)
	    *n-assumptions*)))
