" (c) 1992 Copyright (c) University of Washington
  Written by Tony Barrett.

  All rights reserved. Use of this software is permitted for non-commercial
  research purposes, and it may be copied only for that use.  All copies must
  include this copyright message.  This software is made available AS IS, and
  neither the authors nor the University of Washington make any warranty about
  the software or its performance.

  When you first acquire this software please send mail to
  bug-ucpop@cs.washington.edu; the same address should be used for problems."

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; The following code is used to implement a graphic interface to snlp
;;; that uses the clim graphics package.

(in-package "ZENO")

(export '(play))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 1. Variables and structures

(defvar *enable-vcr* nil)
(defvar *tape* nil)			; A SNLP run trace
(defvar *display* nil)			; Where to show the movie
(defvar *window* nil)
(defvar *max-open* 1)
(defvar *max-unsafe* 1)
(defvar *max-links* 1)
(defvar *max-steps* 1)
(defvar *trees* nil)
(defvar *movie-frame* nil)
(defvar *tree-scale* 8)
    
(defstruct (plan-tree (:print-function print-plan-tree))
  entry
  reason
  parent
  (displayed nil)
  (next nil)
  (prev nil)
  (children nil)
  (xpos nil)
  (ypos nil))

(clim:define-presentation-type plan-picture ())

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 2. Defining the display interface

(clim::define-application-frame vcr
                          ()
  ()
  (:panes ((graph :application
                  :initial-cursor-visibility :off)
           (movie :application
                  :scroll-bars nil)
           (text :application)
           (menu :command-menu)
           (documentation :pointer-documentation)))
  (:layout ((main
             (:column 1
              (:column :rest
                (graph :rest)
                (menu :compute))
              (:column 3/10
                (:row :rest
                 (movie 1/4)
                 (text  3/4)))
              (documentation :compute)))
            )))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 3. Handy interface functions for using the program

(defun ENABLE-VCR ()
  (setf *enable-vcr* t
	*tape* nil))

(defun DISABLE-VCR ()
  (setf *enable-vcr* nil
	*tape* nil))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Record a single frame of the movie.
(defun VCR-ADD-FRAME (parent reason child)
  (when *enable-vcr*
    (push (list parent reason child) *tape*)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Load and display a tape using the clim graphics package.
(defun PLAY (message display)
  (setf *trees* (list (cons (compute-tree *tape*) message)))
  (setf *movie-frame* nil)
  (when (null *display*)
    (setq *display* (clim::open-root-window :clx :host display)))
  (when (null *window*)
      (multiple-value-bind (left top right bottom)
          (size-vcr-frame *display* 0 0 800 600)
        (setq *window* 
	  (clim:make-application-frame 'vcr
				       :parent *display*
				       :left left :top top
				       :right right :bottom bottom))))
  (clim:run-frame-top-level *window*))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 3. Commands

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  This macro is used to define command menu buttons
(defmacro DEFINE-VCR-BUTTON (name windows &body body)
  `(define-vcr-command 
       (,(intern (format nil "COM-~A-vcr" name)) :menu ,name) ()
     (let ,(mapcar #'(lambda (w)
		       `(,w (clim:get-frame-pane 
			     clim:*application-frame* (quote ,w))))
	    windows)
       ,@body)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; This command is used to make bugging a plan picture cause that
;;; plan to be displayed in the text window.
(define-vcr-command (com-display-plan) ((plan 'plan-picture))
  (display-a-plan plan)
  (when (or (plan-tree-next plan) (plan-tree-prev plan))
    (next-frame plan
		(clim:get-frame-pane clim:*application-frame* 'graph)
		(clim:get-frame-pane clim:*application-frame* 'movie))))
  
(clim:define-presentation-to-command-translator display-a-plan
    (plan-picture com-display-plan vcr)
  (object)
  `(,object))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Animate backward from the current movie position.
(define-vcr-button "<<" (movie graph)
  (loop
    (unless (and *movie-frame* (plan-tree-prev *movie-frame*) 
		 (vcr-sleep movie)) 
      (return nil))
    (next-frame (plan-tree-prev *movie-frame*) graph movie)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Move the movie one step backward.
(define-vcr-button "<" (movie graph)
  (when (and *movie-frame* (plan-tree-prev *movie-frame*))
    (next-frame (plan-tree-prev *movie-frame*) graph movie)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Move the movie one step forward.
(define-vcr-button ">" (movie graph)
  (when (and *movie-frame* (plan-tree-next *movie-frame*))
    (next-frame (plan-tree-next *movie-frame*) graph movie)))
    
;;;;;;;;;;;;;;;;;;;;;;;;
;;; Animate forward from the current movie position.
(define-vcr-button ">>" (movie graph)
  (loop
    (unless (and *movie-frame* (plan-tree-next *movie-frame*) 
		 (vcr-sleep movie)) 
      (return nil))
    (next-frame (plan-tree-next *movie-frame*) graph movie)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; blow up the tree diagram
(define-vcr-button "Enlarge" (graph movie)
  (when (car *trees*) 
    (setf *tree-scale* (/ *tree-scale* 2))
    (clim:window-clear graph)
    (clim:with-scaling (graph (/ 1 *tree-scale*))
      (draw-entry (caar *trees*) graph))
    (next-frame *movie-frame* graph movie)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; shrink the tree diagram
(define-vcr-button "Shrink" (graph movie)
  (when (car *trees*) 
    (setf *tree-scale* (* *tree-scale* 2))
    (clim:window-clear graph)
    (clim:with-scaling (graph (/ 1 *tree-scale*))
      (draw-entry (caar *trees*) graph))
    (next-frame *movie-frame* graph movie)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;
(define-vcr-button "Reset" (graph movie text)
  (when *trees* 
    (clim:window-clear graph)
    (clim:with-scaling (graph (/ 1 *tree-scale*))
      (draw-entry (caar *trees*) graph))
    (clim:window-clear text)
    (clim:with-text-style ('(:sans-serif :roman :small) text)
      (clim:write-string (format nil (cdar *trees*)) text))
    (next-frame (caar *trees*) graph movie)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; This command is used to exit the interface.
(define-vcr-button "Exit" (graph movie text)
  (clim:window-clear graph)
  (clim:window-clear movie)
  (clim:window-clear text)
  (setf *trees* nil
	*max-open* 1 *max-unsafe* 1 *max-links* 1 *max-steps* 1)
  (clim:frame-exit clim:*application-frame*))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Define  an interaction button
(clim:define-presentation-to-command-translator display-a-plan
    (plan-picture com-display-plan vcr)
  (object)
  `(,object))

(defun GRAPH-BUTTON (stream plan)
  (clim:with-output-as-presentation
      (:type 'plan-picture :object plan :stream stream :single-box t)
    (graph-plan stream plan :label (< *tree-scale* 16))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4. plan-tree setup

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Compute the search tree from a tape
(defun COMPUTE-TREE (tape &aux (nodes nil))
  (dolist (entry tape)
    (setf *max-open*   
      (max *max-open*   (length (plan-flaws (caddr entry))))
      *max-unsafe* 
      (max *max-unsafe* (length (plan-flaws (caddr entry))))
      *max-links*  
      (max *max-links*  (length (plan-links (caddr entry))))
      *max-steps*  
      (max *max-steps*  (length (plan-steps (caddr entry)))))
    (push (make-plan-tree :parent (car entry) 
			  :reason (cadr entry)
			  :entry (caddr entry))
	  nodes))
  (do* ((p nil (car n))
	(n nodes (cdr n)))
      ((null n) nil)
    (setf (plan-tree-next (car n)) (when (cdr n) (cadr n))
	  (plan-tree-prev (car n)) p))
  (do ((ns nodes (cdr ns)))
      ((null ns) nil)
    (dolist (n (cdr ns))
      (when (eq (plan-tree-parent n) (plan-tree-entry (car ns)))
	(push n (plan-tree-children (car ns))))))
  (order-tree (car nodes))
  (tree-node-pos (car nodes) 0 0)
  (car nodes))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Order a tree so that the winning path is left justified
(defun ORDER-TREE (node)
  (cond ((or (plan-flaws (plan-tree-entry node))
             (member (car (plan-tree-reason node)) '(:open :unsafe)))
         (dolist (c (plan-tree-children node))
           (when (order-tree c)
             (setf (plan-tree-children node)
               (cons c (remove c (plan-tree-children node)))
	       (plan-tree-displayed c) t)
             (return t))))
        (t t)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Compute search tree node display positions.
(defun TREE-NODE-POS (node x y)
  (setf (plan-tree-xpos node) x
	(plan-tree-ypos node) y)
  (if (plan-tree-children node)
      (dolist (c (plan-tree-children node) x)
	(setf x (tree-node-pos c x (+ y 200))))
    (+ x 200)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Draw the tree.
(defun DRAW-ENTRY (p stream)
  (clim:with-translation (stream (plan-tree-xpos p) (plan-tree-ypos p))
    (graph-button stream p))
  (when (and (plan-tree-displayed p) (plan-tree-children p))
    (clim:draw-line* stream
		     (+ (plan-tree-xpos p) 55) (+ (plan-tree-ypos p) 160)
		     (+ (plan-tree-xpos p) 55) (+ (plan-tree-ypos p) 180))
    (clim:draw-line* stream
     (+ (plan-tree-xpos p) 55) (+ (plan-tree-ypos p) 180)
     (+ 55 (plan-tree-xpos (car (last (plan-tree-children p)))))
     (+ (plan-tree-ypos p) 180))
    (dolist (c (plan-tree-children p)) 
      (clim:draw-line* stream
		       (+ (plan-tree-xpos c) 55) (- (plan-tree-ypos c) 20)
		       (+ (plan-tree-xpos c) 55) (plan-tree-ypos c))
      (draw-entry c stream))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Initialize the window when it is first created.
(defmethod initialize-instance :after ((frame vcr) &key)
  (let ((graph (clim:get-frame-pane frame 'graph))
	(movie (clim:get-frame-pane frame 'movie)))
    (clim:window-clear graph)
    (clim:window-clear movie)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4. Utility functions

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  This routine is used to sleep between frames of an animation
;;;  it stops the animation if any button is pressed
(defun VCR-SLEEP (ws)
  (multiple-value-bind (gesture type)
      (clim:read-gesture :stream ws :timeout (/ 1 10))
    (declare (ignore gesture))
    (eq type :timeout)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  This routine is used to resize the vcr window.
(defun SIZE-VCR-FRAME
    (root desired-left desired-top desired-width desired-height)
  (declare (values left top right bottom))
  (multiple-value-bind (left top right bottom)
      (clim:window-inside-edges root)
    (let ((desired-right (+ desired-left desired-width))
          (desired-bottom (+ desired-top desired-height)))
      (when (> desired-right right)
        (setf desired-right right
              desired-left (max left (- desired-right desired-width))))
      (when (> desired-bottom bottom)
        (setf desired-bottom bottom
              desired-top (max top (- desired-bottom desired-height))))
      (values desired-left desired-top desired-right desired-bottom))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 5. Graphics routines

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  This routine makes a histogram grapp of a plan.  The bars stand for
;;;  # open conditions, # links, # unsafe, # steps respectively.
(defun GRAPH-PLAN (stream node &key (fill nil) (label nil) &aux plan)
  (setf plan (plan-tree-entry node))
  (flet ((draw-histo (fraction color)
	   (clim:draw-rectangle* stream 0 (* 110 (- 1 fraction)) 20 110
				 :ink color)))
    (cond ((plan-tree-displayed node)
	   (when (not fill)
	     (clim:draw-rectangle* stream 0 0 110 110 :ink clim::+background+)
	     (draw-histo (/ (length (plan-flaws plan)) *max-open*) 
			 clim::+blue+)
	     (clim:with-translation (stream 30 0)
	       (draw-histo (/ (length (plan-links plan)) *max-links*) 
			   clim::+wheat+)
	       (clim:with-translation (stream 30 0)
		 (draw-histo (/ (length (plan-flaws plan)) *max-unsafe*) 
			     clim::+red+)
		 (clim:with-translation (stream 30 0)
		   (draw-histo (/ (length (plan-steps plan)) *max-steps*) 
			       clim::+green+)))))
	   (clim:draw-polygon* stream '(0 0 110 0 110 110 0 110)
			       :closed t :filled fill :line-thickness 1))
	  (t (clim:draw-polygon* stream '(0 0 110 0 110 110 0 110)
				 :closed t :filled nil :line-thickness 1)
	     (clim:draw-polygon* stream '(10 10 100 10 100 100 10 100)
				 :closed t :filled nil :line-thickness 1)))
    (when label
      (clim:draw-text stream (iconic-reason-label (plan-tree-entry node))
		      (clim:make-point 55 115) :align-x :center :align-y :top
		      :text-style '(:sans-serif :roman :tiny)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; display the next frame in the animation
(defun NEXT-FRAME (frame graph movie)
  (when (null (plan-tree-displayed frame))
    (setf (plan-tree-displayed frame) t) 
    (clim:with-scaling (graph (/ 1 *tree-scale*))
      (draw-entry frame graph)))
  (clim:with-scaling (graph (/ 1 *tree-scale*))
    (when *movie-frame*
      (clim:with-translation (graph (plan-tree-xpos *movie-frame*) 
				    (plan-tree-ypos *movie-frame*))
	(graph-plan graph *movie-frame* :label (< *tree-scale* 16))))
    (setf *movie-frame* frame)
    (clim:with-translation (graph (plan-tree-xpos *movie-frame*) 
				  (plan-tree-ypos *movie-frame*))
      (graph-plan graph *movie-frame* :fill t :label (< *tree-scale* 16))))
  (multiple-value-bind (w h) (clim:window-inside-size movie)
    (clim:window-clear movie)
    (clim:with-scaling (movie (/ (min w h) 200))
      (clim:with-translation (movie 45 60)
	(graph-plan movie *movie-frame* :label t)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; This function prints out a single plan in the text window
(defun DISPLAY-A-PLAN (plan)
  (let ((txt (clim:get-frame-pane clim:*application-frame* 'text)))
    (clim:window-clear txt)
    (clim:with-text-size (:small txt)
      (clim:write-string (format nil "~a" plan) txt))))
  
;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Print out na plan tree entry
(defun PRINT-PLAN-TREE (plan &optional (stream t) depth)
  (declare (ignore depth))
  (display-plan (plan-tree-entry plan) stream)
  (format stream "~%Reason: ~a" (reason (plan-tree-entry plan)) stream)
  (format stream "~%--------------------------------"))


(defun iconic-reason-label (plan)
  (let ((reason (cdr (assoc :reason (plan-other plan)))))
    (case (car reason)
     (:bogus "bogus")
     (:init "start")
     (:goal (format nil "G:~a" (caadr reason)))
     (:step
      (format nil "S:~a" 
	      (car (p-step-action 
		    (find (cadr reason) (plan-steps plan)
			  :key #'p-step-id)))))
     (:link
      (format nil "L:~a"
	      (car (p-step-action 
		    (find (cadr reason) (plan-steps plan)
			  :key #'p-step-id)))))
     (:cw-assumption "CWA")
     (:order
      (with-output-to-string (out)
	(print-interval (cadr reason) out)
	(format out "<")
	(print-interval (caddr reason) out))))))

(defun reason (plan)
  (let ((reason (cdr (assoc :reason (plan-other plan)))))
    (case (car reason)
      (:init "Initial plan")
      (:bogus
       (format nil "Remove bogus unsafe condition"))
      (:goal 
       (format nil "Add goal [~a] for step [~d]" (cadr reason) (caddr reason)))
      (:step
       (format nil "Add step ~a to provide ~a"
	       (p-step-action 
		(find (cadr reason) (plan-steps plan)
		      :key #'p-step-id))
	       (caddr reason)))
      (:forever "Fact")
      (:link
       (format nil "Link to step ~a for effect ~a" 
	       (cadr reason)
	       (caddr reason)))
      (:cw-assumption "Make a closed world assumption")
      (:order
       (with-output-to-string (out)
	 (format out "Order time ")
	 (print-interval (second reason) out)
	 (format out " before ")
	 (print-interval (third reason) out)))
      (otherwise (format nil "~s" reason)))))
