(in-package "ZENO")

" (c) 1990, 1991, 1992, 1993 Copyright (c) University of Washington
  Written by J. Scott Penberthy, Tony Barrett, Daniel Weld."

(use-package "VARIABLE")

(export '(define reset-domain record-vcr))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 1. Variables

(defvar *Templates* nil)		; list of dummy steps
(defvar *scr-rules* nil)		; list of scrs
(defvar *search-limit* 2000)		; max number of plans created

;;; Statistics related variables

(defvar *nodes-visited* 0)		; # plans visited in search
(defvar *plans-created* 0)		; # plans created in search
(defvar *branch* 0)			; compute avg branch factor

;;; Variables for Stuart Russell's IE search routine

(defvar *ie-limit*)
(defvar *ie-branches*)
(defvar *recording* nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 2. Interface functions

(defun gen-fact-function (name)
  (intern (format nil ".FACT-FN-~a." name) "ZENO"))

(defmacro DEFINE ((dtype name) &body body)
  (ecase dtype
    (operator `(apply #'defoper ',(cons name body)))
    (axiom (error "Generic axioms not implemented.  You must specify~%~
                   them as effects of the initial step."))
    (fact
     (let ((fn (gen-fact-function (car name))))
       `(progn
	  (defun ,fn ,(cdr name) ,@body)
	  (deffact ',(car name) ',fn))))
    (scr `(apply #'defscr ',(cons name body)))))

;;
;;; Fact stuff
;;

(defun get-fact-fn (predicate) (cdr (assoc predicate *facts*)))

(defun procedurally-defined? (theta)
  (and (not (eq '== (car theta)))
       (not (eq ':not (car theta)))
       (assoc (theta-pred theta) *facts*)))

(defun deffact (predicate fn)
  (let ((existing (assoc predicate *facts*)))
    (cond (existing
	   (rplacd existing fn))
	  (t
	   (push (cons predicate fn) *facts*)))
    (values predicate)))

(defun match-fact (form plan)
  (let ((fn (get-fact-fn (car form))))
    (if fn
      (apply fn (bind-variable (cdr form) (plan-bindings plan)))
      :undefined)))

(defun setb (&rest var-value-pairs)
  (cond ((null var-value-pairs)
	 nil)
	((atom var-value-pairs)
	 (error "Uneven arguments to SETB."))
	(t
	 (push (cons (car var-value-pairs)
		     (second var-value-pairs))
	       (setb (cddr var-value-pairs))))))
		     
;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Purge a previously defined domain theory.
(defvar *facts* nil "Alist of (predicate . function)")

(defun RESET-DOMAIN ()
  (setf *templates* nil)
  (setf *facts* nil)
  (setf *domain-functions* nil))

(defun add-domain-function (name)
  (push name *domain-functions*))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Enable the vcr
(defun RECORD-VCR ()
  (setf *recording* t))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 3. Defining operators

(defun DEFOPER (name &key parameters at-time context precondition
			  effects effect description resources)
  (declare (ignore description))
  (let ((start (or (first at-time) '?S))
	(end (or (second at-time) '?E)))
    (when context
      (error "Contexts are not implemented."))
    (when (member name *templates* 
            :key #'(lambda (a) (car (p-step-action a))))
      (error "Two actions with name ~a" name))
;    (setf *var-counter* 0)
    (let* ((vars (list* start end (parm-vars parameters)))
	   (pre (remove nil `(,precondition ,@(parm-types parameters))))
	   (step nil))
      (setf pre (if (> (length pre) 1) (cons :and pre) (car pre)))
      (when pre (test-wd pre vars))
      (setf step
	 (make-p-step :action (cons name vars)
		      :start start :end end
		      :parms (parm-vars parameters)
		      :precond (krsl-parse pre)
		      :res resources))
      (parse&install-effects step (nconc effects
					 (zenofy-effect effect nil)))
      (push step *templates*))))

(defun functions-only (form)
  (cond ((atom form) nil)
	((is-a-fn? form) (cons form nil))
	((or (eq (car form) :and) (eq (car form) :or))
	 (mapc 'functions-only (cdr form)))
	(t nil)))

(defun ZENOFY-EFFECT (e vlst &aux (ret nil))
  ;; transform prodigy-style effects into Zeno speak.
  (labels 
      ((eff* (e when forall)
	 (case (car e)
	   (:and 
	    (dolist (x (cdr e)) (eff* x when forall)))
	   (:forall
	    ;; if time appears, treat it as an interval effect.
	    (let ((univs (zenofy-forall (cadr e))))
	      (cond ((and (consp (car univs))
			  (eq 'time (caar univs)))
		     (push `(:cause (:forall ,univs ,(third e))
				    :when ,(cond ((null when) nil)
						 ((null (cdr when))
						  (car when))
						 (t (cons :and when)))
				    :forall ,forall)
			   ret))
		    (t
		     (setf univs (append forall univs))
		     (dolist (other (cddr e))
		       (eff* other when univs))))))
	   (:when 
	       (eff* (caddr e) (cons (cadr e) when) forall))
	   (otherwise
	    (push `(:cause ,e
			   :forall ,forall 
			   :when ,(cond ((null when) nil)
					((null (cdr when))
					 (car when))
					(t (cons :and when))))
		  ret)))))
    (when e (eff* e nil nil))
    ret))

(defvar *whitespace-chars* '(#\space #\tab #\return #\newline))

(defun read-interval (string)
  (flet ((barf ()
	   (error "Bad interval format: ~s" string)))
    (let ((times nil)
	  (start-pos nil)
	  (start-type nil)
	  (end-pos nil)
	  (end-type nil))
      (setf start-pos (position #\[ string)
	    start-type :closed)
      (unless start-pos
	(setf start-pos (position #\( string)
	      start-type :open))
      (unless start-pos (barf))
      (setf end-pos (position #\] string)
	    end-type :closed)
      (unless end-pos
	(setf end-pos (position #\) string)
	      end-type :open))
      (unless (and end-pos
		   (> end-pos start-pos))
	(barf))
      (let ((*whitespace-chars* '(#\,)))
	(setf times (mapcar #'read-from-string
			    (string-tokenize
			     (subseq string (1+ start-pos) end-pos) )))
	(unless (= 2 (length times)) (barf))
	(cons
	 (ecase start-type
	   (:open
	    (ecase end-type
	      (:closed :open-start)
	      (:open :open)))
	   (:closed
	    (ecase end-type
	      (:closed :closed)
	      (:open :open-end))))
	 times)))))

    
(defun grab-token (string &optional (start 0))
  "returns two values:  a substring for the next contiguous token, plus
                        POS, 1+ position of the end of this token"
  (let ((len (length string))
	tstart tend)
    (setq tstart
      (block a
	(do ((index start (1+ index)))
	    ((= index len) nil)
	  (when (not (member (elt string index) *whitespace-chars*
			     :test #'char=)) 
	    (return-from a index)))))
    (setq tend
      (and tstart
	   (block b
	     (do ((index tstart (1+ index))) 
		 ((= index len) len)
	       (when (member (elt string index) *whitespace-chars*
			     :test #'char=)
		 (return-from b index))))))
    (cond ((and tstart tend)
	   (values
	    (subseq string tstart tend)
	    tend))
	  (t
	   (values
	    nil
	    len)))))

(defun string-tokenize (line &optional (start 0))
  ;; This function separates a string into a list of substring tokens.  The
  ;; tokens are separated by characters in *whitespace-chars*.
  (multiple-value-bind (token new-start) (grab-token line start)
    (cond ((null token)
	   nil)
	  (t
	   (cons token
		 (string-tokenize line new-start))))))

(defun zenofy-forall (thing)
  (parse-universals
   (cond ((atom thing)
	  (cons thing nil))
	 ((and (symbolp (car thing))
	       (not (variable::variable? (car thing))))
	  (cons thing nil))
	 (t
	  thing))))

(defun parse-universals (forall)
  (let ((this-one (car forall)))
    (cond ((null this-one)
	   nil)
	  (t
	   (cons
	    (cond ((atom this-one)
		   this-one)
		  ((eq 'time (car this-one))
		   (if (stringp (third this-one))
		       `(time ,(second this-one)
			      ,@(read-interval (third this-one)))
		     this-one))
		  (t this-one))
	    (parse-universals (cdr forall)))))))

(defun PARSE&INSTALL-EFFECTS (step effects)
  ;;  (print (p-step-id step))
  (let ((ca nil)
	(efx nil)
	(efx-2 nil)
	(init? nil)
	(pre nil)
	(id (p-step-id step)))
    (setf init? (eq 0 id))
    (dolist (e effects)
      (setf efx-2 nil)
      (multiple-value-bind (efx-1 ca-1)
	  (krsl-parse-effect e)
	(unless init?
	  (setf efx-2 (remove-effects-with-time efx-1 (p-step-start step)))
	  (setf efx-2 (remove-effects-with-time efx-2 :forever)))
	(cond ((and (not init?) (null efx-2))
	       (let ((temp (functions-only
			    (cons :and (mapcar #'effect-post efx-1)))))
		 (setf efx (nconc efx efx-1))
		 (setf pre (nconc pre temp))
		 (setf ca (nconc ca ca-1))))
	      (t
	       (setf efx (nconc efx efx-1))
	       (setf ca (nconc ca ca-1))))))
    (setf ca
      (nconc ca (determine-equalities-among-vars efx)))
    (setf efx (update-forever-table efx))
    (unless init?
      (setf efx (remove-effects-with-time efx (p-step-start step))))
    (setf (p-step-ca step) ca)
    (if pre
	(if (p-step-precond step)
	    (setf (p-step-precond step)
	      (flatten-and-tree
	       `(:and ,@pre ,(p-step-precond step))))
	  (setf (p-step-precond step)
	    (flatten-and-tree `(:and ,@pre)))))
    (dolist (e efx)
      (setf (effect-id e) id)
      (add-effect step e))
    ))

(defun theta-syntactic-equal (t1 t2)
  (and (eq (car t1) (car t2))
       (eq (theta-time t1) (theta-time t2))
       (eq (theta-pred t1) (theta-pred t2))
       (equal (theta-args t1) (theta-args t2))))

(defun determine-equalities-among-vars (effects)
  (let ((equals nil)
	(table nil)
	(this-one nil))
    (dolist (e effects)
      (setf this-one (effect-post e))
      (let ((entry (find this-one table :test 'theta-syntactic-equal)))
	(cond ((null entry)
	       (push this-one table))
	      (t
	       (push `(= ,(theta-var entry) ,(theta-var this-one))
		     equals)))))
    (values equals)))

(defun add-effect (step effect)
  (let ((pred (theta-pred (effect-post effect))))
    (let ((entry (assoc pred (p-step-add step))))
      (cond ((null entry)
	     (push (list pred effect) (p-step-add step)))
	    (t
	     (push effect (cdr entry)))))))

(defun find-atom (the-atom tree)
  (cond ((null tree) nil)
        ((atom tree)
         (eq the-atom tree))
        (t
          (or (find-atom the-atom (car tree))
              (find-atom the-atom (cdr tree))))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Test to see if logical equation is syntacticly correct
(defun TEST-WD (wd vlst)
  (let ((header (car wd)))
    (cond ((or (eq header :exists) (eq header :forall))
	   (unless (null (cdddr wd)) (cerror "" "illegal expression: ~a" wd))
	   (or (atom (cadr wd))
	       (if (consp (caadr wd))
		   (dolist (v (cadr wd))
		     (test-typed-var v)
		     (push (cadr v) vlst))
		 (progn (test-typed-var (cadr wd))
			(push (cadr (cadr wd)) vlst))))
	   (test-wd (caddr wd) vlst))
	  ((eq header :fact)
	   (test-term (second wd) vlst))
	  ((or (eq header :eq) (eq header :neq))
	   (unless (null (cdddr wd)) (cerror "" "illegal expression: ~a" wd))
	   (test-arguments (cdr wd) vlst))
	  ((or (eq header :and) (eq header :or))
	   (dolist (e (cdr wd))
	     (test-wd e vlst)))
	  ((eq header :not)
	   (unless (null (cddr wd)) (cerror "" "illegal expression: ~a" wd))
	   (test-wd (cadr wd) vlst))
	  ((eq header :holds-at)
	   (unless (and (atom (second wd))
			(test-term (third wd) vlst)
			(null (fourth wd)))
	     (cerror "" "illegal :holds-at expression: ~a" wd)))
	  (t (test-term wd vlst)))))

(defun TEST-TERM (wd vlst)
  (unless (and (consp wd) (symbolp (car wd)))
    (cerror "" "illegal term: ~a" wd))
  (cond ((eq (car wd) :not)
	 (unless (null (cddr wd)) (cerror "" "illegal term: ~a" wd))
	 (test-term (cadr wd) vlst))
	((r-formula-p wd) t)
	(t (test-arguments (cdr wd) vlst) t)))

(defvar *ALL-METRIC-FUNCTIONS* '(+ - * / min max))

(defun valid-constraint-arg? (arg)
  (eq-member (car arg) *all-metric-functions*))
  
(defun TEST-ARGUMENTS (as vlst)
  (dolist (p as)
    (when (and (variable? p) (not (eq-member p vlst)))
      (format t "~&Warning: Unbound variable ~a~%" p))
    (when (and (consp p)
	       (not (valid-constraint-arg? p)))
      (cerror "" "illegal argument ~a" p))))

(defun TEST-TYPED-VAR (v)
  (unless (and (consp v)
	       (symbolp (car v)) (not (variable? (car v)))
	       (variable? (cadr v)) (null (cddr v)))
    (cond ((and (eq 'time (car v))
		(or (and (stringp (third v))
			 (= 3 (length v)))
		    (and
		     (member (third v) '(:open-end :open-start
					 :closed :open))
		     (= 5 (length v)))))
	   nil)
	  (t
	   (cerror "" "illegal typed variable ~a" v)))))

(defun PARM-VARS (parms)
  (mapcar #'(lambda (p) (if (listp p) (cadr p) p)) parms))

(defun PARM-TYPES (parms)
  (mapcar #'(lambda (p)
	      (if (atom p) (error "Bad parameter: ~s" p)
		`(,(car p) :forever ,@(cdr p))))
	  (remove-if #'symbolp parms)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4. Search routines

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  A simple best first search strategy.  Returns 3 values: the found state,
;;;  the average branching factor, and the number of generated but unexplored 
;;;  states.  The search will only generate up to LIMIT states.  
(defun BESTF-SEARCH (initial-state daughters-fn goal-p rank-fn limit)
  (let ((branches nil))                   ; compute average branch factor
    (do* ((current-entry nil (car search-queue))
          (current-state initial-state (cdr current-entry))
          (search-queue nil (cdr search-queue)))
         ((or (null current-state)
	      (funcall goal-p current-state)
              (> 0 limit))
          (values current-state
                  (if (null branches) 0
                      (div* (apply #'+ branches) (length branches)))
                  (length search-queue)))
      (count-stat .visited.)
      (let ((children (funcall daughters-fn current-state)))
        (decf limit (length children))
        (setf search-queue
              (merge
                  'list search-queue
                  (sort (mapcar #'(lambda (x) (cons (funcall rank-fn x) x))
                                children)
                        #'< :key #'car)
                  #'< :key #'car))
        (push (length children) branches)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  IE search function written by Stuart Russell 
;;;  (See "Efficient Memory-Bounded Search Methods" in ECAI-92)

(defun CALL-IE (node successors goalp rank-fn limit)
  (setf *ie-limit* limit)
  (setf *ie-branches* (cons 0 0))
  (let ((solution (ie (cons (funcall rank-fn node) node)
                      goalp successors rank-fn most-positive-single-float)))
    (values solution (if (zerop (car *ie-branches*)) 0 
                         (div* (cdr *ie-branches*) (car *ie-branches*))) 0)))

(defun IE (vnode goalp successors rank-fn bound &aux children)
  (cond ((or (funcall goalp (cdr vnode)) (> 0 *ie-limit*))
         (cdr vnode))
        ((null (setf children (mapcar #'(lambda (child)
                                          (cons (funcall rank-fn child) child))
                                      (funcall successors (cdr vnode)))))
         (setf (car vnode) most-positive-single-float)
         nil)
        (t (count-stat .visited.)
           (decf *ie-limit* (length children))
           (incf (car *ie-branches*)) (incf (cdr *ie-branches*) 
					    (length children))
           (dolist (vn children)    ;;; pathmax
             (setf (car vn) (max (car vn) (car vnode))))
           (do ()
               ((> (car vnode) bound))
             (setf children
                   (sort children #'< :key #'car))
             (let* ((best (car children))
                    (rest (cdr children))
		    ;; best sibling value
                    (new-bound (apply #'min (cons bound (mapcar #'car rest)))))
               (let ((v (ie best goalp successors rank-fn new-bound)))
                 (when v (return v)))
               (if (and rest (< (caar rest) (car best)))
                 (setf (car vnode) (caar rest))
                 (setf (car vnode) (car best))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 5. Miscellaneous other routines

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Record a single frame of the movie in the vcr.
(defun VCR-FRAME (parent reason child)
  (when *recording*
    (vcr-add-frame parent reason child))
  child)

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Add new high step number to each variable id for every term in 
;;; step from template.
(defun INSTANTIATE-STEP (step num)
  (let ((s (find num (p-step-cache step) :key #'p-step-id))
	(start (instantiate-term (p-step-start step) num))
	(end (instantiate-term (p-step-end step) num)))
    (if s s
      (car (push 
	    (make-p-step
	     :id num
	     :start start :end end
	     :res (instantiate-term (p-step-res step) num)
	     :ca (instantiate-term (p-step-ca step) num)
	     :action (instantiate-term (p-step-action step) num)
	     :precond (instantiate-term (p-step-precond step) num)
	     :add (instantiate-effect-table (p-step-add step) num))
	    (p-step-cache step))))))

(defun CONSTRAIN-STEP-TIMES (plan step)
  ;; Look for all time points mentioned in STEP other
  ;; than the start or end and constrain them appropriately.
  (let ((start (p-step-start step))
        (end (p-step-end step))
        (time nil)
        (ztm (plan-ordering plan)))
    (constrain-times (p-step-precond step) start end ztm)
    (dolist (alist (p-step-add step))
       (dolist (e (cdr alist))
      (setf time (effect-time e))
      (constrain-times (effect-pre e) start end ztm)
	 (cond ((eq :point (i-type time))
             (constrain-times (effect-post e) start end ztm))
            (t
  	      (constrain-one-time (i-start time) start end ztm)
              (constrain-one-time (i-end time) start end ztm)))))))
  
(defun GET-TIME-UNIVS (scope)
  (let ((result nil))
   (dolist (x scope result)
     (if (and (consp x) (eq 'time (car x)))
         (push x result)))))

(defun FIND-ALL-TIMES (form &optional (uni nil))
 (let ((result nil))
  (cond ((eq-member (car form) '(:and :or))
        (dolist (arg (cdr form))
          (setf result (nconc result (find-all-times arg uni)))))
       ((eq-member (car form) '(:forall :exists))
        (let ((new-uni (get-time-univs (second form))))
          (cond ((null new-uni)
                 (setf result (find-all-times (third form) uni)))
                (t
                 (setf result
                      (find-all-times (third form) (append new-uni uni)))))))
       ((eq-member (car form) *algebra*) nil)
       (form
	   (let ((time (theta-time form)))
              (unless (eq-member time uni)
                (push time result))))
    (values result))))

(defun CONSTRAIN-TIMES (form start end ztm &optional (uni nil))
 (cond ((eq-member (car form) '(:and :or))
        (dolist (arg (cdr form))
	     (constrain-times arg start end ztm uni)))
       ((eq-member (car form) '(:forall :exists))
        (let ((new-uni (get-time-univs (second form))))
          (cond ((null new-uni)
                 (constrain-times (third form) start end ztm uni))
                (t
                 (dolist (u new-uni)
		   ;; u is (:time arg type i-start i-end)
                   (constrain-one-time (fourth u) start end ztm)
                   (constrain-one-time (fifth u) start end ztm)
                   (push u uni))
                 (constrain-times (third form) start end ztm uni)))))
       ((eq-member (car form) *algebra*) nil)
       (form
	   (let ((time (theta-time form)))
          (unless (eq-member time uni)
             (constrain-one-time time start end ztm))))))

(defun CONSTRAIN-ONE-TIME (tau start end ztm)
  (unless (or (eq tau start) (eq tau end)
	      (eq tau :forever))
    (let ((s (find-ztime ztm start))
          (e (find-ztime ztm end))
          (ztau (find-ztime ztm tau)))
      (zset<= ztm s ztau)
      (zset<= ztm ztau e)
	 )))

(defun INSTANTIATE-EFFECT-TABLE (table num)
  (let ((result nil))
    (dolist (entry table)
      (push (instantiate-effects entry num) result))
    (values result)))

(defun INSTANTIATE-EFFECTS (effect-entry num)
  (let ((key (car effect-entry))
	(result nil))
    (dolist (e (cdr effect-entry))
      (push
       (make-effect
	:id num
	:influence-p (instantiate-term (effect-influence-p e) num)
	:time (instantiate-term (effect-time e) num)
	:pre (instantiate-term (effect-pre e) num)
	:post (instantiate-term (effect-post e) num)
	:forall (instantiate-term (effect-forall e) num)
	:exists (instantiate-term (effect-exists e) num)
	:ca (instantiate-term (effect-ca e) num))
       result))
    (values (cons key result))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Funarg related stuff
;;;

;;
;; "Theta" is the Greek symbol used to describe
;;  relations R(t1,x1,...,xn)  and function assignments
;;  f(t1,x1,...,xn) in my thesis.
;;

(defun is-a-fn? (form)
  (eq '== (car form)))

(defun theta-time (form)
  (cond ((or (eq (car form) '==)
	     (eq (car form) ':not))
	 (cadadr form))
	(t
	 (cadr form))))

(defun theta-pred (form)
  (cond ((or (eq (car form) '==)
	     (eq (car form) ':not))
	 (caadr form))
	(t
	 (car form))))

(defun theta-args (form)
  (cond ((or (eq (car form) '==)
	     (eq (car form) ':not))
	 (cddadr form))
	(t
	 (cddr form))))

(defun theta-var (form)
  (cond ((eq (car form) '==)
	 (third form))
	(t
	 nil)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Get today's date
(defun TODAY ()
  (let ((d (multiple-value-list (get-decoded-time))))
    (format nil "~a/~a ~a at ~a:~a:~a"
            (nth 4 d) (nth 3 d) (nth 5 d) (nth 2 d) (nth 1 d) (nth 0 d))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Reset the global variables used for collecting statistics on the
;;;  planner.
(defun RESET-STAT-VARS ()
  (reset-stat)
  (setf *rank-history* nil)
  (setf *forever* nil)
  (setf *nodes-visited* 0)
  (setf *unify-count* 0)
  (setf *t-counter* 0)			;temporary time points 
  (setf *compute-rank-unifies* 0)
  (setf *add-bind-count* 0)
  (setf *branch* 0)
  (setf *plans-created* 0))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  A divide routine that does not blow up on zero.
(defun div* (x y)
  (if (= y 0) (* 99999 99999 99999) (/ x y)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 6. Replace slow lisp utilities

(defun MY-MEMBER (elt lst &key (test #'eq))
  (dolist (l lst nil)
    (when (funcall test elt l) (return t))))

(defun REMOVE-1 (elt lst &key (test #'eq))
  (cond ((null lst) nil)
	((funcall test elt (car lst)) (cdr lst))
	(t (cons (car lst) (remove-1 elt (cdr lst) :test test)))))

(defun DELETE-1 (elt lst &key (test #'eq))
  (cond ((null lst) nil)
	((funcall test elt (car lst)) (cdr lst))
	(t (setf (cdr lst) (delete-1 elt (cdr lst) :test test))
	   lst)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 7. Print functions

;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Print function for Templates
(defun PRINT-TEMPLATES (&optional (templates *templates*))
  (format t "~&~%Templates:")
  (dolist (templ-n templates)
    (let ((action (p-step-action templ-n))
	  (pre-cond (p-step-precond templ-n))
	  (add (p-step-add templ-n)))
      (format t "~&~a~%  Pre  : ~a~%  Add  :~%"
	      action pre-cond)
      (dolist (a add)
	(format t "  <when: ~a  Add: ~a>~%" 
		(effect-precond a) (effect-add a))))))
		
;;
;;; Various display hacks
;; 

(defun display-history (plan)
  (let ((mom (cdr (assoc :parent (plan-other plan)))))
    (cond ((null mom)
	   ;; Now we can show it!
	   (print (assoc :reason (plan-other plan)))
	   (display-plan plan)
	   (when *debug*
	     (cond ((consp (plan-constraints plan))
		    (format t "~&Metric constraints unchanged.~%"))
		   (t
		    (show (plan-constraints plan))))))
	  (t
	   (display-history mom)
	   (print (assoc :reason (plan-other plan)))
	   (display-plan plan)))))
