;;
;;; Temporally quantified domains
;;

(in-package "ZENO")

(defparameter *tests* nil)

(defstruct problem name start end domain inits goal (rank-fun 'rank3) )

(defmacro defproblem (name (start end) &body body)
  (unless (evenp (length body))
    (error "Bad problem spec for ~s." name))
  `(progn
     (setf *tests* (remove ',name *tests* :key 'problem-name))
     (push (make-problem :name ',name 
             :start ',start 
             :end ',end ,@body) *tests*)
     ',name))

(defun bf-show (prob &optional (display "okanogan:0.0"))
  (when (symbolp prob)
    (setf prob (find prob *tests* :key #'problem-name)))
  (funcall (problem-domain prob))
  (record-vcr)
  (multiple-value-bind (plan stat)
      (plan (problem-inits prob) (problem-goal prob)
            :rank-fun (problem-rank-fun prob))            
    (when plan (display-plan plan))
    (display-stat stat))
  (load-vcr "bf-show")
  (play display))

(defun bf-control (prob)
  (when (symbolp prob) 
    (setf prob (find prob *tests* :key #'problem-name)))
  (funcall (problem-domain prob))
  (let ((plan (plan (problem-start prob) (problem-end prob)
                    (problem-inits prob) (problem-goal prob) 	
		    :rank-fun (problem-rank-fun prob))))
    (when plan (display-plan plan))
    (display-stats)
    (values plan)))

(defun ie-search (prob)
  (when (symbolp prob) 
    (setf prob (find prob *tests* :key #'problem-name)))
  (funcall (problem-domain prob))
  (let ((plan (plan (problem-start prob) (problem-end prob)
                    (problem-inits prob) (problem-goal prob)
		    :search-fun #'call-ie :rank-fun (problem-rank-fun prob))))
    (when plan (display-plan plan))
    (display-stats)
    (values plan)))

(defun set-clause-time (thing time)
 (cond ((eq (car thing) :not)
        `(:not ,(set-clause-time (second thing) time)))
        (t
         `(,(car thing) ,time ,@(cdr thing)))))

(defun initify (things)
  (cons :and
	(mapcar #'(lambda (clause) (set-clause-time clause 0)) things)))

(defun goalify (things)
  (cons :and
	(mapcar #'(lambda (clause) (set-clause-time clause :goal)) things)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  This file contains a set of test problems.  Some of them work,
;;;  and others are intractable for ucpop.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun blocks-world-domain ()
  ;; purge old domain prior to defining a new domain
  (reset-domain)

  ;; Define step for placing one block on another.
  (define (operator puton)
      :at-time (?s ?e)
      :parameters (?x ?y ?z)
      :precondition (:and (:forall (time ?t "[?s,?e)")
			   (:and (on ?t ?x ?z) (clear ?t ?y)))
			  (:forall (time ?t "[?s,?e]") (clear ?t ?x))
			  (:neq ?Y ?Z) (:neq ?X ?Z)
			  (:neq ?X ?Y) (:neq ?X Table))
      :effect
      (:and
       (on ?e ?X ?Y)
       (:not (on ?e ?X ?Z))
       (:when (:neq ?z table) (clear ?e ?z))
       (:when (:neq ?y table) (:not (clear ?e ?y))))))

(defun sipe-blocks-world ()
  ;; purge old domain prior to defining a new domain
  (reset-domain)
  (add-domain-function 'size)

  ;; Define step for placing one block on another.
  (define (operator fuel-puton)
      :at-time (?s ?e)
      :resources ((robot ?r))
      :parameters (?r ?X ?Y ?Z)
      :precondition (:and (:static (robot ?r))
			  (:forall (time ?t "[?s,?e)") 
			    (:and (on ?t ?x ?z) (clear ?t ?y)))
			  (:forall (time ?t "[?s,?e]") (clear ?t ?x))
			  (> (:value-at ?s (fuel-level ?r)) (size ?x))
			  (:neq ?Y ?Z) (:neq ?X ?Z)
			  (:neq ?X ?Y) (:neq ?X Table))
      :effect
      (:and
	(= (fuel-level ?e ?r) (- (fuel-level ?s ?r) (size ?x)))
        (on ?e ?x ?y)
        (:not (on ?e ?x ?z))
        (:when (:neq ?z table) (clear ?e ?z))
        (:when (:neq ?y table) (:not (clear ?e ?y)))))

  (define (operator government-puton)
      ;; consume twice the fuel of PUTON
      :at-time (?s ?e)
      :resources ((robot ?r))
      :parameters (?r ?X ?Y ?Z)
      :precondition (:and (:fact (robot ?r))
			  (:forall (time ?t "[?s,?e)") 
                            (:and (on ?t ?x ?z) (clear ?t ?y)))
			  (:forall (time ?t "[?s,?e]") (clear ?t ?x))
			  (> (fuel-level ?s ?r) (size ?x))
			  (:neq ?Y ?Z) (:neq ?X ?Z)
			  (:neq ?X ?Y) (:neq ?X Table))
      :effect
      (:and 
	(= (fuel-level ?e ?r) (- (fuel-level ?s ?r) (* 2 (size ?x))))
	(on ?e ?x ?y)
        (:not (on ?e ?x ?z))
        (:when (:neq ?z ?table) (clear ?e ?z))
        (:when (:neq ?y ?table) (:not (clear ?y)))))
  )

(defproblem sipe-1 (0 :goal)
 :domain 'sipe-blocks-world
 :inits '(:and
           (= (fuel-level 0 flakey) 40)
           (:fact (robot flakey))
	   (:fact (block A))
	   (:fact (block Table))
	   (:fact (block B))
	   (:fact (block D))
           (= 15 (size A))
           (= 5 (size B))
           (= 5 (size D))
           (on 0 A B)
           (clear 0 A)
           (clear 0 D)
           (clear 0 Table)
           (on 0 D table)
           (on 0 B table))
 :goal '(:and
         (<= 5 (fuel-level :goal flakey) 50)
	 (on :goal ?x B)
	 (:neq ?x A)))

(defproblem sussman-anomaly (0 :goal)
  :domain 'blocks-world-domain
  :inits '(:and (on 0 C A)
  	        (on 0 A Table)
                (on 0 B Table) 
                (clear 0 C)
                (clear 0 B)
                (clear 0 Table))
  :goal '(:and (on :goal b c)
               (on :goal a b)))

(defproblem tower-invert (0 :goal)
       :domain 'blocks-world-domain
       :inits (initify '((on a b) (on b c) (on c table)
			 (clear a) (clear table)))
       :goal (goalify '((on b c) (on c a))))

(defun briefcase-world ()
  ;; purge old domain prior to defining a new domain
  (reset-domain)

  (define (operator mov-b)
      :at-time (?s ?e)
      :parameters (?m ?l)
      :precondition (:and (:neq ?m ?l)
			  (:forall (time ?t "[?s,?e)") (at ?t b ?m)))
      :effect
      (:forall ?o 
        (:when (:or (:eq ?o b) (in ?s ?o))
          (:and (at ?e ?o ?l) (:not (at ?e ?o ?m))))))

  (define (operator take-out)
      :at-time (?s ?e)
      :parameters (?x)
      :precondition (:neq ?x B)
      :effect (:not (in ?e ?x)))
      
  (define (operator put-in)
      :at-time (?s ?e)
      :parameters (?x ?l)
      :precondition (:and (:neq ?x B)
			  (at ?e ?x ?l) 
                          (at ?e b ?l))
      :effect (in ?e ?x))
  )


(defproblem get-paid (0 :goal)
       :name 'get-paid
       :domain 'briefcase-world
       :inits (initify '((at B home) (at P home) (at D home) (in P)))
       :goal (goalify '((at B office) (at D office) (at P home))))

(defproblem get-paid2 (0 :goal)
       :domain 'briefcase-world
       :inits (initify '((at B home) (at P home) (at D home) (in P)))
       :goal (goalify '((at P home) (at D office) (at B home))))

(defproblem get-paid3 (0 :goal)
       :domain 'briefcase-world
       :inits (initify '((at B home) (at P home) (at D home) (in P)))
       :goal (goalify '((at P bank) (at D office) (at B home))))

(defproblem get-paid4 (0 :goal)
       :domain 'briefcase-world
       :inits (initify '((at B home) (at P home) (at D home) (in P)))
       :goal (goalify '((at B home) (at D office) (at P bank))))

(defun clown-world ()
  ;; purge old domain prior to defining a new domain
  (reset-domain)

  (define (operator rub-tummy)
     :at-time (?s ?e)
     :parameters (?x)
     :precondition (:and (:fact (hand ?x)) (free ?s ?x))
     :effect
     (:and (:forall (time ?t "(?s,?e)") 
            (:and (rubbing-tummy ?t ?x) (:not (free ?t ?x))))
           (:not (rubbing-tummy ?e ?x))
           (free ?e ?x)))

  (define (operator pat-head)
     :at-time (?s ?e)
     :parameters (?x)
     :precondition (:and (:fact (hand ?x)) (free ?s ?x))
     :effect
     (:and (:forall (time ?t "(?s,?e)")
              (:and (:not (free ?t ?x)) (patting-head ?t ?x)))
           (free ?e ?x)
           (:not (patting-head ?e ?x))))

  (define (operator slap-knee)
     :at-time (?s ?e)
     :parameters (?x)
     :precondition (:and (:fact (hand ?x)) (free ?s ?x))
     :effect
     (:and (:forall (time ?t "(?s,?e)")
              (:and (:not (free ?t ?x)) (slapping-knee ?t ?x)))
          (:not (slapping-knee ?e ?x))
          (free ?e ?x)))
  )


(defproblem clown1 (0 :end)
      :domain 'clown-world
       :inits '(:and (:fact (hand left))
		(:fact (hand right)) (free 0 left) (free 0 right))
       :goal (goalify '((patting-head ?a) (rubbing-tummy ?b))))

;; This next problem should NEVER return an answer.

(defproblem clown2 (0 :goal)
       :domain 'clown-world
       :inits (initify '((hand left) (hand right) (free left) (free right)))
       :goal (goalify '((hand ?a) (hand ?b) (hand ?c)
			(patting-head ?a) (rubbing-tummy ?b)
			(slapping-knee ?c))))

;;
;;; Now, some metric stuff
;;

(defun jug-world ()
  (reset-domain)
  (add-domain-function 'capacity)
  
  (define (operator fill)
      :at-time (?s ?e)
      :parameters (?jug)
      :precondition (:and (:fact (jug ?jug))
			  (>= (volume ?s ?jug) 0))
      :effect (= (volume ?e ?jug) (capacity ?jug)))

    (define (operator empty)
	:at-time (?s ?e)
	:parameters (?jug)
	:precondition (:and (:fact (jug ?jug))
			    (> (volume ?s ?jug) 0))
	:effect (= (volume ?e ?jug) 0))

    (define (operator pour)
	:at-time (?s ?e)
	:parameters (?jug1 ?jug2)
	:precondition (:and (:neq ?jug1 ?jug2)
			    (:fact (jug ?jug1))
			    (:fact (jug ?jug2)))
	:effect
	(:and
	 (= (volume ?e ?jug2)
	    (min (capacity ?jug2)
		 (+ (volume ?s ?jug1)
		    (volume ?s ?jug2))))
	 (= (volume ?e ?jug1)
	    (max 0
		 (+ (volume ?s ?jug1)
		    (volume ?s ?jug2)
		    (* -1 (capacity ?jug2)))))))
    )


(defproblem jug0 (0 :goal)
  :domain 'jug-world
  :inits '(:and (:fact (jug big))
	   (:fact (jug small))
	   (= 4 (capacity big))
	   (= 3 (capacity small))
	   (= 0 (volume 0 big))
	   (= 0 (volume 0 small)))
  :goal '(:and
	  (= 3 (volume :goal ?jug))))

(defproblem jug0-a (0 :goal)
  :domain 'jug-world
  :inits '(:and (:fact (jug big))
	   (:fact (jug small))
	   (= 4 (capacity big))
	   (= 3 (capacity small))
	   (= 0 (volume 0 big))
	   (= 0 (volume 0 small)))
  :goal '(:and
	  (:neq ?jug ?jugb)
	  (= 3 (volume :goal ?jug))
	  (= 4 (volume :goal ?jugb))))

(defproblem jug0-b (0 :goal)
       :domain 'jug-world
       :inits '(:and (:fact (jug big))
			      (:fact (jug small))
			      (= 4 (capacity big))
			      (= 3 (capacity small))
			      (= 0 (volume 0 big))
			      (= 0 (volume 0 small)))
       :goal '(:and
	       (= 3 (volume :goal big))
	       ))

(defproblem jug0-c (0 :goal)
       :domain 'jug-world
       :inits '(:and (:fact (jug big))
			      (:fact (jug small))
			      (= 4 (capacity big))
			      (= 3 (capacity small))
			      (= 0 (volume 0 big))
			      (= 0 (volume 0 small)))
       :goal '(:and
	       (= 2 (volume :goal small))
	       ))

(defproblem jug1 (0 :goal)
       :domain 'jug-world
       :inits '(:and (:fact (jug big))
			      (:fact (jug small))
			      (= 4 (capacity big))
			      (= 3 (capacity small))
			      (= 0 (volume 0 big))
			      (= 0 (volume 0 small)))
       :goal '(:and
	       (= 2 (volume :goal ?jug))))

(defproblem jug2 (0 :goal)
       :domain 'jug-world
       :inits '(:and (:fact (jug big))
		(:fact (jug small))
		(= 4 (capacity big))
		(= 3 (capacity small))
		(= 0 (volume 0 big))
		(= 0 (volume 0 small)))
       :goal '(:and
	       (= 2 (volume :goal big))))

(defproblem jug3 (0 :goal)
       :domain 'jug-world
       :inits '(:and (:fact (jug big))
		(:fact (jug small))
		(= 4 (capacity big))
		(= 3 (capacity small))
		(= 0 (volume 0 big))
		(= 0 (volume 0 small)))
       :goal '(:and
	       (> (volume :goal big) 1)
	       (> (volume :goal small) 1)
	       (> (volume :goal small)
		  (volume :goal big))))

(defun fly-domain ()
  (reset-domain)
  (add-domain-function 'dist)
  (add-domain-function 'mpg)

  (define (operator fly)
      :resources ((plane ?m))
      :parameters (?m ?l)
      :at-time (?s ?e)
      :precondition (:and (:neq ?m ?l)
			  (at ?s plane ?m)
			  (:forall (time ?t "[?s,?e]")
				   (> (fuel ?t plane) 0)))
      :effect
      (:and
       (= (- ?e ?s) (/ (dist ?m ?l) 650))
       (:influence ?s ?e (fuel plane) (- (/ 650 (mpg plane))))
       (:forall ?o
		(:when (:or (:eq ?o plane) 
			    (:forall (time ?t "[?s,?e]") (in ?t ?o)))
		  (:and
		   (at ?e ?o ?l)
		   (:forall (time ?t "(?s,?e]") (:not (at ?t ?o ?m))))))
       ))

  (define (operator deplane)
      :parameters (?x ?l)
      :at-time (?s ?e)
      :precondition (:and (:neq ?x plane)
			  (:forall (time ?t "[?s,?e]")
				   (:and (at ?t plane ?l)
					 (at ?t ?x ?l))))
      :effect (:and (= (- ?e ?s) 0.25) (:not (in ?e ?x))))

  (define (operator board)
      :parameters (?x ?l)
      :at-time (?s ?e)
      :precondition (:and (:neq ?x plane)
			  (:forall (time ?t "[?s,?e]")
				   (:and (at ?t plane ?l)
					 (at ?t ?x ?l))))
      :effect (:and (= (- ?e ?s) 0.50) (in ?e ?x)))
  )

(defproblem phd (7 13)
  :domain 'fly-domain
  :inits '(:and
	   (at 7 dan seatac)
	   (at 7 scott seatac)
	   (at 7 plane seatac)
	   (in 7 dan)
	   (:not (in 7 scott))
	   (= (fuel 7 plane) 1200)
	   (= (mpg plane) 3)
	   (= (dist seatac jfk) 3000))
  :goal '(:and
	  (at 12 dan seatac)
	  (at 13 scott jfk)
	  (at 13 plane jfk)))

(defun latch-world ()
  ;; From allen's Reasoning about Plans
  ;; This won't work unless we can model simultaneous action.
  (reset-domain)

  (define (operator turn-latch)
     :at-time (?s ?e)
     :effect
     (:and
       (:forall (time ?t "(?s,?e)") (latch ?t open))
       (:not (latch ?e open))))

  (define (operator pull-door)
     :at-time (?s ?e)
     :effect
     (:when (:forall (time ?t "[?s,?e]") (latch ?t open))
       (door ?e open))
     ))

(defproblem door-latch (0 :goal)
       :domain 'latch-world
       :inits '(:and (:not (latch 0 open)) (:not (door 0 open)))
       :goal '(:and (door :goal open)))
