;;
;;; Parse KRSL into Zeno's language
;;;
;;; Function assignments look like (== (f t x) v)
;;; Temporal relations look like (r t x)
;;;
;;
;; To do
;; - remove all conditional effects on :forever
;; - promote :forever effects to some global table
;; - remove all effects that modify a start time.
;;

(in-package "ZENO")

(defvar *algebra* '(< > <= >= = + - * / min max
		    :eq :neq))

(defvar *reserved* '(< > <= >= = + - * / min max
		    :eq :neq
		     :and :or :not :forall :exists :fact))

(defun r-formula-p (thing)
  (and (listp thing)
       (eq-member (car thing) '(= < > >= <=))))

(defun update-funarg-map (funargs funarg-map)
  (dolist (f funargs)
    (unless (find-funarg f funarg-map)
      (push (cons f (genvar f)) funarg-map)))
  (values funarg-map))

  
(defvar *DOMAIN-FUNCTIONS* nil
  "A list of the functions in the current domain.")

(defun domain-function-p (thing)
  (and (consp thing) 
       (eq-member (car thing) *domain-functions*)))

(defun funarg-p (thing)
  (and (listp thing)
       (atom (car thing))
       (not (eq-member (car thing) *reserved*))))

(defun find-funarg (funarg funarg-map)
  (dolist (entry funarg-map)
    (when (equal funarg (car entry))
      (return-from find-funarg (cdr entry)))))

(defun extract-funargs (sexp)
  "Return a list of all funargs within sexp."
  (cond ((atom sexp) nil)
	((funarg-p sexp)
	 (list sexp))
	(t
	 (append (extract-funargs (car sexp))
		 (mapcan #'extract-funargs (cdr sexp))))))


#||
(defstruct effect
  (id nil)				; id of step from whence this came
  (influence-p nil)			;dt variable, if an influence; nil, o/w
  (time nil)				;effect time, an interval
  (post nil)				;postcondition
  (pre  nil)				;precondition
  (forall nil)				;universal vars
  (exists nil)				;exists for the universals
  (ca nil)				;delayed Ca constraints
  )
||#
  
;(defvar *domain-functions* nil)

(defvar *var-counter* 0)
(defvar *long-names* nil)

#-aclpc(defun genvar (f)
  (if *long-names*
      (intern (make-symbol (format nil "?~a$~d" f
                           (incf *var-counter*))))
      (intern (make-symbol (format nil "?V~d"
                           (incf *var-counter*))))))

#+aclpc(defun genvar (f)
  (if *long-names*
      (intern (format nil "?~a$~d" f
                           (incf *var-counter*)))
      (intern (format nil "?V~d"
                           (incf *var-counter*)))))

(defun simple-rewrite (sexp funarg-map)
  "Replaces funargs with their constant values, and rewrite :facts
   into zeno logic."
  (cond ((atom sexp)
	 sexp)
	((find-funarg sexp funarg-map))
	((eq :fact (car sexp))
	 (let ((desc (second sexp)))
	   (cond ((eq :not (car desc))
		  (setf desc (second desc))
		  `(:not (,(car desc) :forever ,@(cdr desc))))
		 (t
		  `(,(car desc) :forever ,@(cdr desc))))))
	(t
	 (cons (simple-rewrite (car sexp) funarg-map)
		(simple-rewrite (cdr sexp) funarg-map)))))

(defun krsl-parse (form &optional (main-funarg-map nil))
  (multiple-value-bind (new-form funarg-map)
      (krsl-parse-internal form main-funarg-map)
    (flatten-and-tree
     `(:and ,@(mapcar #'funarg-entry->zeno funarg-map)
	    ,new-form
	    ))))

(defun krsl-parse-internal (form funarg-map)
  (cond ((atom form)
	 (values form funarg-map))
	((eq (car form) :forall) ;****
	 (parse-krsl-forall form funarg-map))
	((eq (car form) :and)
	 (multiple-value-bind (arglist new-funargs)
	     (krsl-parse-arglist (cdr form) funarg-map)
	   (values `(:and ,@arglist) new-funargs)))
	((eq (car form) :or)
	 (multiple-value-bind (arglist new-funargs)
	     (krsl-parse-arglist (cdr form) funarg-map)
	   (values `(:or ,@arglist) new-funargs)))
	((eq (car form) :exists)
	 (multiple-value-bind (new-form new-funargs)
	     (krsl-parse-internal (third form) funarg-map)
	   (values
	    `(:exists ,(zenofy-forall (second form)) ,new-form)
	    new-funargs)))
	((eq (car form) :implies)
	 (multiple-value-bind (arglist new-funargs)
	     (krsl-parse-arglist (cdr form) funarg-map)
	   (values `(:implies ,@arglist) new-funargs)))
	(t
	 (simple-krsl-parse form funarg-map))))

(defun krsl-parse-arglist (arglist funarg-map)
  (let ((new-args nil))
    (dolist (arg arglist)
      (multiple-value-bind (new-arg new-funargs)
	  (krsl-parse-internal arg funarg-map)
	(push new-arg new-args)
	(setf funarg-map new-funargs)))
    (values (nreverse new-args) funarg-map)))

(defun simple-krsl-parse (form funarg-map)
  ;;buggy... not.
  (when (and (consp form)
	     (eq-member (car form) *algebra*))
    (setf funarg-map
      (update-funarg-map (extract-funargs form) funarg-map)))
  (setf form (simple-rewrite form funarg-map))
  (values form funarg-map))

(defun extract-forall-vars (arglist)
  (cond ((null arglist)
	 nil)
	((listp (car arglist))
	 (cons (second (car arglist))
	       (extract-forall-vars (cdr arglist))))
	(t
	 (cons (car arglist)
	       (extract-forall-vars (cdr arglist))))))

(defun list-contains-one-of (list items)
  (dolist (i items)
    (if (member i list)
	(return-from list-contains-one-of t))))

(defun parse-krsl-forall (form funarg-map)
  (setf form `(:forall ,(zenofy-forall (second form))
		       ,@(cddr form)))
  (let ((uvars (extract-forall-vars (second form)))
	(arg nil)
	(name nil)
	(exists nil)
	(pass2 nil))
    (multiple-value-bind (pass1 new-funargs)
	(krsl-parse-internal (third form) funarg-map)
      (dolist (entry new-funargs)
	(setf arg (car entry))
	(setf name (cdr entry))
	(when (list-contains-one-of arg uvars)
	  (push entry exists)))
      (setf funarg-map
	(set-difference new-funargs exists))
      (cond (exists
	     (setf pass2
	       `(:forall ,(second form)
			 (:exists
			  ,(mapcar #'cdr exists)
			  (:and 
			   ,@(mapcar #'funarg-entry->zeno exists)
			   ,pass1)))))
	    (t
	     (setf pass2
	       `(:forall ,(second form) ,pass1))))
      (values pass2 funarg-map))))
		  
(defun funarg-entry->zeno (entry)
  ;; entry is (funarg . name)
  (let ((funarg (car entry))
	(name (cdr entry)))
    (cond ((domain-function-p funarg)
	   `(== (,(car funarg) :forever ,@(cdr funarg)) ,name))
	  (t
	   `(== ,funarg ,name)))))
	
(defun flatten-and-tree (tree)
  (cond ((eq :or (car tree))
	 `(:or ,@(mapcar #'flatten-and-tree (cdr tree))))
	((and (eq :and (car tree))
	      (cddr tree))
	 (let ((args nil))
	   (dolist (arg (cdr tree))
	     (cond ((eq (car arg) :and)
		    (dolist (other (cdr arg))
		      (push (flatten-and-tree other)
			    args)))
		   (t
		    (push (flatten-and-tree arg) args))))
	   `(:and ,@(nreverse args))))
	((eq :and (car tree))
	 (flatten-and-tree (second tree)))
	((eq :forall (car tree))
	 `(:forall ,(second tree)
		   ,(flatten-and-tree (third tree))))
	((eq :exists (car tree))
	 `(:exists ,(second tree)
		   ,(flatten-and-tree (third tree))))
	(t
	 tree)))

(defun fixup-interval-time-refs (forall)
  ;; Replace all existentials ?v relying on universally
  ;; quantified time points ?t with (:fn ?v), then eliminate
  ;; the existential.
  (let ((forms (if (eq (car forall) :and)
		   (cdr forall)
		 (list forall)))
	(result nil)
	(quants nil))
    (dolist (p forms)
      (cond ((eq :forall (car p))
	     (setf quants (first (second p)))
	     (when (not (eq 'time (car quants)))
	       (error "Forall forms may only have one TIME var"))
	     (let ((phi (third p))
		   (subst nil))
	       (when (eq :exists (car phi))
		 (setf quants (second phi))
		 (dolist (q quants)
		   (push (list q :fn q) subst))
		 (setf phi (sublis subst (third phi))))
	       (push `(:forall ,(second p) ,phi) result)))
	    (t
	     (push p result))))
    ;; return a list of elements of the FORALL
    (values (nreverse result))))

(defun parse-with-forall (forall form)
  ;; Assume that FORM is within a :forall form whobse
  ;; universally quantified variables are listed in 
  ;; the set FORALL.  Return a flattened conjunction
  ;; version of FORM, along with a list of existentials
  ;; associated with variables in the FORALL.
  (let ((raw (krsl-parse `(:forall ,forall ,form)))
	(conds nil)
	(exists nil))
    (when (eq (car raw) :and)
      (setf raw (cdr raw))
      ;; extract out all unquantified forms
      (loop until (eq (caar raw) :forall)
	  do
	    (push (pop raw) conds))
      (setf raw (car raw)))
    ;; raw is now (:forall () ...)
    (setf raw (third raw))
    (when (eq (car raw) :exists)
      ;; extract out the existentials that depend on the
      ;; original :forall
      (setf exists (second raw))
      (setf raw (third raw)))
    ;; if we have any unquantified postconditions,
    ;; include them now.
    (if conds
	(setq raw `(:and ,@conds ,raw)))
    ;; flatten what's left
    ;; return the raw postcondition and a list of existentials
    (values (flatten-and-tree raw) exists)))

(defun check-influence (form t1 t2 f dt-fn)
  ;; this should be more rigorous
  (unless (and (atom t1) (atom t2) (consp f) (consp dt-fn))
    (error "Bad influence format. I expected:~%~
            (:influence t1 t2 (f x1..xn) <constraint>) but you gave me~%~
        ~s" form)))

(defun parse-influence (form)
  ;; form should be (:influence t1 t2 (f x1..xn) <constraint>)
  ;; We return a list of one effect structure.
  (let ((t1 (second form))
        (t2 (third form))
        (f (fourth form))
        (dt-fn (fifth form))
        (dt nil)
        (fnvar nil)
        (ca nil)
        (pre nil))
    (check-influence form t1 t2 f dt-fn)
    (setf ca (krsl-parse dt-fn))
    (setf pre (extract-thetas ca))
    (setf ca (extract-constraints ca))
    (unless (null (cdr ca))
      (error "Malformed derivative in an Influence:~%~s" dt-fn))
    (setf dt (genvar `(dt ,(car f))))
    (setf ca `(= ,dt ,(car ca)))
    (setf fnvar `(:fn ,(genvar f)))
    (values
     (list
      (make-effect
       :pre (if pre `(:and ,@pre) nil)
       :post `(== (,(car f) ?t ,@(cdr f)) ,fnvar)
       :influence-p dt
       :ca ca
       :time `(:open-start ,t1 ,t2))))))

;(:influence ?t1 ?t2 (gas-vol ?plane)
 ;(- (/ (mpg ?plane) ?speed)))

(defun parse-postcond (forall form)
  ;; Return a list of postconditions in FORM for an effect
  ;; that has the given FORALL quantifiers.
  (multiple-value-bind (raw exists)
      (parse-with-forall forall form)
    (setf raw (fixup-interval-time-refs raw))
    (values raw exists)))

(defun parse-precond (forall form)
  ;; Return the preconditions for an effect that has
  ;; the given FORALL quantifiers.
  (parse-with-forall forall form))

(defun extract-constraints (form)
  (cond ((atom form) nil)
	((eq-member (car form) *algebra*)
	 (values (list form)))
	(t
	 (nconc (extract-constraints (car form))
		(extract-constraints (cdr form))))))

(defun extract-thetas (form &aux pred)
  (cond ((atom form) nil)
	((atom (car form))
	 (setf pred (car form))
	 (cond ((or (eq pred :exists)
		    (eq pred :forall))
		(extract-thetas (third form)))
	       ((or (eq pred :and)
		    (eq pred :or))
		(extract-thetas (cdr form)))
	       ((not (member pred *algebra*))
		(values (list form)))))
	(t
	 (nconc (extract-thetas (car form))
		(extract-thetas (cdr form))))))

(defun existential? (form exists)
  (cond ((atom form)
	 (not (null (member form exists))))
	(t
	 (or (existential? (car form) exists)
	     (existential? (cdr form) exists)))))

(defun krsl-parse-effect (form)
  (let ((post (second (member :cause form)))
	(efx nil)
	(ca nil))
    (cond ((eq (car post) :and)
	   (dolist (p (cdr post))
	     (multiple-value-bind (effects constraints)
		 (krsl-parse-effect `(:cause ,p
					     :forall ,(getf form :forall)
					     :when ,(getf form :when)))
	       (setf efx (nconc efx effects))
	       (setf ca (nconc ca constraints))))
	   (values efx ca))
	  ((eq (car post) :influence)
	   (unless (and (null (second (member :forall form)))
			(null (second (member :when form))))
	     (format t "Error -- I don't handle conditional or ~
                        quantified influences.  Sorry!"))
	   (parse-influence post))
	  ((null post)
	   (error "Missing :CAUSE keyword in an effect."))
	  (t
	   (krsl-parse-effect-1 post form)))))

(defun krsl-parse-effect-1 (post form)
  (let ((pre (getf form :when))
	(forall (getf form :forall))
	(ca nil)
	(ca-real nil)
	(ca-delayed nil)
	(exists nil)
	(efx nil))
    (multiple-value-bind (rho ex1)
	(parse-precond forall pre)
      (multiple-value-bind (posts ex2)
	  (parse-postcond forall post)
	(setf exists (nconc ex1 ex2))
	(dolist (post posts)
	  (cond ((member (car post) *algebra*)
		 (push post ca))
		(t
		 (setf efx
		   (nconc efx
			  (create-effect forall exists rho post))))))))
    (dolist (c ca)
      (if (existential? c exists)
	  (push c ca-delayed)
	(push c ca-real)))
    (when ca-delayed
      (dolist (e efx)
	(setf (effect-ca e)
	  (append ca-delayed (effect-ca e)))))
    (when (cdr efx)
      (setf efx (fixup-multi-way-effects efx)))
    (values (fixup-forall efx) ca-real)))
	  
(defun fixup-forall (efx)
  ;; assumes all effects in EFX have the same :FORALL
  ;; argument.  We take out all (type ?x) parameters
  ;; and make them (type :forever ?x) preconditios.
  (when efx
    (let ((quants (effect-forall (car efx)))
	  (pre nil)
	  (uvars nil))
      (when quants
	(dolist (q quants)
	  (cond ((listp q)
		 (push `(,(car q) :forever ,(cadr q)) pre)
		 (push (cadr q) uvars))
		(t
		 (push q uvars))))
	(if (cdr pre)
	    (setf pre (cons ':and pre))
	  (setf pre (car pre)))
	(dolist (e efx)
	  (setf (effect-forall e) uvars)
	  (when pre
	    (if (effect-pre e)
		(setf (effect-pre e)
		  (flatten-and-tree `(:and ,pre ,(effect-pre e))))
	      (setf (effect-pre e) pre)))))
      (values efx))))

(defun fixup-multi-way-effects (efx)
  ;; We have a set of effects that represent a multi-way constraint.
  ;; Thus, each effect is conditioned on the others postconditions...
  (let ((others nil)
	(fns (remove-if-not #'(lambda (e)
				(eq '== (car (effect-post e)))) efx)))
    (dolist (this-one fns)
      (setf others (mapcar #'effect-post (remove this-one fns)))
      (cond ((effect-pre this-one)
	     (setf (effect-pre this-one)
	       (flatten-and-tree `(:and ,@others ,(effect-pre this-one)))))
	    (t
	     (setf (effect-pre this-one)
	       	     (if (null (cdr others))
			 (car others)
		       `(:and ,@others))))))
    (values efx)))
	
;;
;; Funarg map stuff

(defun get-main-funarg-map (forms)
  (let ((main nil))
    (dolist (f forms)
      (multiple-value-bind (result new-map)
	  (krsl-parse-internal f main)
	(declare (ignore result))
	(setf main new-map)))))

(defun create-effect (forall exists rho post)
  ;; return a list of effect structures for the given
  ;; forall, exists, rho, post stuff.
  (let ((time nil)
	(ret nil))
    (cond ((eq (car post) :forall)
	   (setf time (car (second post)))
	   (when (not (eq 'time (car time)))
	     (error "Expected a TIME argument, but got ~s" time))
	   (setf time (cddr time))
	   (when (null time)
	     (error "You omitted the temporal classification in~%~
                     (:forall ~s ..." (second post)))
	   (let ((thetas (extract-thetas (third post)))
		 (eqns (extract-constraints (third post))))
	     ;; this is really lame -- a constant may also
	     ;; be any expression independent of the universal.
	     (when (and eqns (not (null (cdr thetas))))
	       (error "Sorry, this parser is lame.~%~
                       I can't handle multiple relations and constraints~%~
                       in a single interval effect.  I get confused as to~%~
                       whether you are following the restriction that all~%~
                       interval effects are of the form f(x)=c.  Please~%~
                       rewrite your effects into simpler forms:~%~%~s~%"
		      (third post)))
	     (dolist (th thetas)
	       (push
		(make-effect
		 :time time
		 :forall forall
		 :exists exists
		 :pre rho
		 :post th
		 :ca eqns)
		ret))
	     ret))
	  (t
	   (cons
	    (make-effect :time (theta-time post)
			 :forall forall
			 :exists exists
			 :pre rho
			 :post post
			 :ca nil)
	    nil)))))
	      
    
;;;;;;;;;;;;;;;;
;;;  Compute the peelings of an effect caused by binding a universal variable.
;;;  The last effect returned is the actual unpeeled effect.
;;;
;;;  Assumption: the universal variables are always the car of a binding

(defun PEEL-Ca (binds effect)
  (let ((exists (effect-exists effect))
	(new-exists nil)
	(subs nil)
        (ret (effect-ca effect)))
    (when exists
      (setf ret (copy-tree ret))
      (dolist (b binds)
	(when (member (car b) (effect-forall effect))
	  (unless new-exists
	    (setf new-exists (new-existentials exists (cdr b)))
	    (setf subs (pairlis exists new-exists)))
	(push b subs)))
      (when subs
	(setf ret (nsublis subs ret))))
    (values ret)))

(defun PEEL-goal (binds effect &aux (ret nil))
  (let ((exists (effect-exists effect))
	(new-exists nil)
	(subs nil))
    (setf ret (copy-tree (effect-pre effect)))
    (dolist (b binds)
      (when (member (car b) (effect-forall effect))
	(unless (or (null exists) new-exists))
	  (setf new-exists (new-existentials exists (cdr b)))
	  (setf subs (pairlis exists new-exists)))
	(push b subs))
      (when subs
	(setf ret (nsublis subs ret)))
    (values ret)))

(defun new-existentials (exists instance)
  (cond ((null exists) nil)
	(t
	 (mapcar #'(lambda (term)
		     (intern
		      (make-symbol (concatenate 'string (string term) "-"
						(string instance)))))
		 exists))))

(defun remove-effects-with-time (effect-list time)
  (remove-if #'(lambda (e)
		 (eq time (theta-time (effect-post e))))
	     effect-list))

(defun record-forever-theta (theta)
  (let ((p (theta-pred theta)))
    (let ((entry (assoc p *forever*)))
      (cond ((null entry)
	     (push (list p theta) *forever*))
	    (t
	     (push theta (cdr entry)))))
    (values)))

(defun update-forever-table (effect-list)
  (dolist (e effect-list)
    (when (and (eq :forever (theta-time (effect-post e)))
	       (null (effect-pre e)))
      (record-forever-theta (effect-post e))))
  (values (remove-effects-with-time effect-list :forever)))

(defun deref-fnvar (fnvar timepoint)
  ;;If FNVAR is of the form (:fn ?v), then return the unique symbol
  ;; ?v@timepoint.  Otherwise, return FNVAR.
  (if (atom fnvar)
      fnvar
    (intern
     (concatenate 'string
       (string (second fnvar))
       "@"
       (string timepoint)))))

(defun deref-constraint (con timepoint)
  ;; replace all fns (:fn ?v) with ?v@timepoint
  ;; in the constraints CON.
  (cond ((atom con)
	 con)
	((eq :fn (car con))
	 (deref-fnvar con timepoint))
	(t
	 (cons (deref-constraint (car con) timepoint)
	       (deref-constraint (cdr con) timepoint)))))

;;
;;; Simplifying things.
;;

(defun negate&simplify-precond (effect binds)
  (let ((b (mapcar #'(lambda (x) `(:eq ,(car x) ,(cdr x)))
		   (peel-binds binds effect)))
	(subgoal (peel-goal binds effect)))
    (cond ((and b subgoal)
	   (setf subgoal `(:and ,@b ,subgoal)))
	  (b
	    (setf subgoal `(:and ,@b))))
    (cond ((null subgoal) nil)
	  (t
	   (let ((result (simplify (canonical `(:not ,subgoal)))))
	     (cond ((eq result :true)
		    (error "Malformed action precondition -- always false!"))
		   ((eq result :false)
		    nil)
		   (t
		    result)))))))

(defun simplify (exp)
  (cond ((atom exp) exp)
	((eq (car exp) :and)
	 (simplify-and exp))
	((eq (car exp) :or)
	 (simplify-or exp))
	(t
	 exp)))

(defun simplify-and (exp)
  (let ((args nil))
    (dolist (arg (cdr exp))
      (setf arg (simplify arg))
      (if (eq arg :false)
	  (return-from simplify-and :false))
      (unless (eq arg :true)
	(push arg args)))
    (if (null args)
	(values :true)
      (if (null (cdr args))
	  (car args)
	`(:and ,@(nreverse args))))))

(defun simplify-or (exp)
  (let ((args nil)
	(good-args nil))
    (dolist (arg (cdr exp))
      (setf arg (simplify arg))
      (unless (eq arg :false)
	(push arg good-args)))
    (cond ((null good-args)
	   :false)
	  (t
	   (dolist (a good-args)
	     (unless (eq a :true)
	       (push a args)))
	   (if (null args)
	       :true
	     (if (null (cdr args))
		 (car args)
	       `(:or ,@args)))))))

