;;
;;; Nonlinear equation solver 
;;
;;  -- Delay everything until one of two arguments becomes ground.

(in-package "ZENO")
(proclaim '(optimize (speed 3) (safety 2) (space 0) (compilation-speed 0)))

(defclass NLIN-SOLVER ()
  ((table :initform NIL :accessor nlin-table)
   (values :initform NIL :accessor nlin-values)
   (csolver :accessor nlin-csolver :initarg :csolver)  
   (consistent? :initform T :accessor nlin-consistent?))
  )

(defmethod EQ-SOLVER ((self nlin-solver))
  (with-slots (csolver) self
    (cs-eqn csolver)))

(defmethod INEQ-SOLVER ((self nlin-solver))
  (with-slots (csolver) self
    (cs-ineq csolver)))

(defmethod SPAWN ((self nlin-solver) old-nlin-solver)
  ;; Makes this instance a duplicate of OLD-NLIN-SOLVER, minus
  ;; the link to the CSOLVER.
  ;;
  ;; THIS IS GOD AWFUL SLOW.. need to restructure it.
  ;;
  (with-slots (table consistent? values) self
    (cond ((nlin-consistent? old-nlin-solver)
	   (setf table (copy-nlin-table old-nlin-solver))
	   (setf values (nlin-values old-nlin-solver))
	   (setf consistent? t))
	  (t
	    (setf consistent? nil)))
    (values)))

(defmethod copy-nlin-table ((self nlin-solver))
  (let ((all nil)
	(table (nlin-table self))
	(new-table nil)
	(value nil)
	(args nil))
    (flet ((new-entry (name nlin)
	     (let ((entry (assoc name new-table)))
	       (cond ((null entry)
		      (push (list name nlin) new-table))
		     (t
		      (push nlin (cdr entry)))))))
      (dolist (entry table)
	(dolist (nlin (cdr entry))
	  (pushnew nlin all)))
      (setf all (if all (mapcar #'copy-nlin-eqn all)))
      (dolist (nlin all)
	(setf value (nlin-eqn-value nlin))
	(setf args (nlin-eqn-args nlin))
	(unless (numberp value)
	  (new-entry (cdr value) nlin))
	(dolist (arg args)
	  (unless (numberp arg)
	    (new-entry (cdr arg) nlin))))
      (values new-table))))
        

(defmethod SHOW-CONSTRAINTS ((self nlin-solver))
  (labels ((princ-num (num)
	     (format t "~f" (/ (round (* num 1000)) 1000)))
	   (princ-var (var)
	     (cond ((numberp var)
		    (if (< var 0)
			(format t "s~d" (abs var))
		      (format t "v~d" var)))
		   (t
		    (format t "~s" var))))
	   (princ-arg (thing)
	     (cond ((numberp thing) (princ-num thing))
		   ((== 1 (car thing)) (princ-var (cdr thing)))
		   (t
		    (princ-num (car thing))
		    (princ " ")
		    (princ-var (cdr thing))))))
    (let ((all (remove-duplicates (apply #'append (mapcar #'cdr (nlin-table self))))))
      (dolist (nlin all)
	(princ-arg (nlin-eqn-value nlin)) (princ " = ")
	(format t "(~@(~a~) " (nlin-eqn-pred nlin))
	(do ((args (nlin-eqn-args nlin) (cdr args)))
	    ((null args))
	  (princ-arg (car args))
	  (if (cdr args) (princ " ")))
	(princ ")")
	(terpri)))
    (values)))

(defmethod reset ((self nlin-solver))
  (setf (nlin-table self) nil)
  (setf (nlin-consistent? self) t)
  (setf (nlin-values self) nil))

(defstruct (nlin-eqn (:type vector))
  pred
  args
  value)

(defun GROUND-ARG (nlin name value)
  ;; ground the argument with NAME in NLIN, modifying NLIN with
  ;; a fresh argument list.
  (let ((new nil)
	(copy nil)
	(old (nlin-eqn-args nlin))
	(val (nlin-eqn-value nlin)))
    (dolist (arg old)
      (cond ((numberp arg)
	     (push arg new))
	    ((eq name (cdr arg))
	     (push (* value (car arg)) new))
	    (t
	     (push arg new))))
    (dolist (arg new)			;reverse the list
      (push arg copy))
    (if (and (consp val)
	     (eq name (cdr val)))
	(setf (nlin-eqn-value nlin)
	  (* (car val) value)))
    (setf (nlin-eqn-args nlin) copy)))

(defmethod ADD-RAW-EQUATION ((self nlin-solver) raw-form)
  ;; form will be (:nonlinear <predicate> :args <args> :valu <value>)
  (if (nlin-consistent? self)
      (let ((args (getf raw-form :args))
	    (value (getf raw-form :value))
	    (pred (getf raw-form :nonlinear))
	    (nlin nil))
	(setf nlin (make-nlin-eqn :pred pred :args args :value value))
	(cond ((wakeup-new-equation self nlin)
	       ;; Don't store equations that can be linearized.
	       (if (eq :inconsistent (wakeup-equation self nlin))
		   (setf (cs-consistent? (nlin-csolver self)) nil))
	       (values))
	      (t
;	       (when (every #'numberp (nlin-eqn-args nlin))
;		 (error "Yep."))
	       (dolist (arg (nlin-eqn-args nlin))
		 (when (consp arg)	;a variable (coeff . name)
		   (add-nlin-entry self (cdr arg) nlin)))
	       (when (consp (nlin-eqn-value nlin))
		 (add-nlin-entry self (cdr value) nlin))
	       (values))))))
	    
(defmethod WAKEUP-NEW-EQUATION ((self nlin-solver) nlin)
  ;; See if any vars in NLIN were previously used to wake up
  ;; constraints.  If so, use the wakeup procedure to figure out
  ;; what to do with NLIN before stuffing it in the table.  NLIN 
  ;; is modified by grounding all such vars.
  (let ((entry nil)
	(doit? nil)
	(vals (nlin-values self)))
    (dolist (arg (nlin-eqn-args nlin))
      (when (consp arg)
	(when (setf entry (assoc (cdr arg) vals))
	  (ground-arg nlin (car entry) (cdr entry))
	  (setf doit? t))))
    (values doit?)))

(defmethod ADD-NLIN-ENTRY ((self nlin-solver) name nlin)
  (let ((entry (assoc name (nlin-table self))))
    (cond ((null entry)
	   (push (list name nlin) (nlin-table self)))
	  (t
	   (push nlin (cdr entry))))
    (values)))

(defmethod KILL-NLIN-ENTRY ((self nlin-solver) name nlin)
  (let ((entry (assoc name (nlin-table self))))
    (if entry
	(rplacd entry (remove-1 nlin (cdr entry))))))

(defmethod KILL-NLIN ((self nlin-solver) nlin)
  (let ((val (nlin-eqn-value nlin))
	(args (nlin-eqn-args nlin)))
    (unless (numberp val)
      (kill-nlin-entry self (cdr val) nlin))
    (dolist (a args)
      (unless (numberp a)
	(kill-nlin-entry self (cdr a) nlin)))
    (values)))

(defmethod WAKEUP ((self nlin-solver) var value)
  (let ((entry (assoc var (nlin-table self))))
    (push (cons var value) (nlin-values self))
    (setf (nlin-table self)
      (remove-1 entry (nlin-table self)))
    (dolist (nlin (cdr entry))
      (count-stat .wakeup.)
      (ground-arg nlin var value)
      (if (eq :inconsistent (wakeup-equation self nlin))
	  (return :inconsistent)))))

(defmethod WAKEUP-EQUATION ((self nlin-solver) nlin)
  ;; Oh, how I'd love Symbolics' dispatch methods right now...
  (ecase (nlin-eqn-pred nlin)
    (max (wakeup-max self nlin))
    (min (wakeup-min self nlin))
    (/   (wakeup-div self nlin))
    (*   (wakeup-*   self nlin))))

(defmethod BAD-NONLINEAR ((self nlin-solver))
  (setf (ineq-consistent? (ineq-solver self)) nil)
  (setf (eqn-consistent? (eq-solver self)) nil)
  (setf (nlin-consistent? self) nil)
  (values :inconsistent))

(defmethod WAKEUP-MAX ((self nlin-solver) nlin)
  (let* ((output (nlin-eqn-value nlin))
	 (in (nlin-eqn-args nlin))
	 (in1 (car in))
	 (in2 (cadr in))
	 (in1? (numberp in1))
	 (in2? (numberp in2))
	 (out? (numberp output)))
    (cond (out?
	   (cond ((and in1? in2?)
		  (unless (== output (max in1 in2))
		    (bad-nonlinear self)))
		 (in1?
		  (when (approx< output in1)
		    (bad-nonlinear self)))
		 (in2? 
		  (when (approx< output in2)
		    (bad-nonlinear self)))))
	  ((and in1? in2?)
	   (kill-nlin self nlin)
	   (add-raw-equation (eq-solver self)
			     `(:equation ,(max in1 in2) ,output))))))

(defmethod WAKEUP-MIN ((self nlin-solver) nlin)
  (let* ((output (nlin-eqn-value nlin))
	 (in (nlin-eqn-args nlin))
	 (in1 (car in))
	 (in2 (cadr in))
	 (in1? (numberp in1))
	 (in2? (numberp in2))
	 (out? (numberp output)))
    (cond (out?
	   (cond ((and in1? in2?)
		  (unless (== output (min in1 in2))
		    (bad-nonlinear self)))
		 (in1?
		  (when (approx> output in1)
		    (bad-nonlinear self)))
		 (in2?
		  (when (approx> output in2)
		    (bad-nonlinear self)))))
	  ((and in1? in2?)
	   (kill-nlin self nlin)
	   (add-raw-equation (eq-solver self)
			     `(:equation ,(min in1 in2) ,output))))))

(defmethod WAKEUP-DIV ((self nlin-solver) nlin)
  ;; Output = in1/in2
  (let* ((output (nlin-eqn-value nlin))
	 (in (nlin-eqn-args nlin))
	 (in1 (car in))
	 (in2 (cadr in))
	 (in1? (numberp in1))
	 (in2? (numberp in2))
	 (out? (numberp output)))
    (cond (out?
	   (cond ((and in1? in2?)
		  (when (or (== 0 in2)
			    (not (== output (/ in1 in2))))
		    (bad-nonlinear self)))
		 (in1?
		  (kill-nlin self nlin)
		  (if (== 0 output)
		      (bad-nonlinear self)
		    (add-raw-equation (eq-solver self)
				      `(:equation ,(/ in1 output) ,in2))))
		 (in2?
		  (kill-nlin self nlin)
		  (if (== 0 in2)
		      (bad-nonlinear self)
		    (add-raw-equation (eq-solver self)
				      `(:equation ,(* in2 output) ,in1))))))
	  ((and in1? in2?)
	   (kill-nlin self nlin)
	   (cond ((== 0 in2)
		  (bad-nonlinear self))
		 (t
		  (add-raw-equation (eq-solver self)
				    `(:equation ,(/ in1 in2) ,output)))))
          (in2?
	   (kill-nlin self nlin)
           (if (== 0 in2)
               (bad-nonlinear self)
	     (add-raw-equation (eq-solver self)
			       `(:equation 0 ,output
                                           ,(scale-sym in1 (/ -1.0d0 in2))))))
	  )))

(defmethod wakeup-* ((self nlin-solver) nlin)
  ;; Output = in1*in2
  (let* ((output (nlin-eqn-value nlin))
	 (in (nlin-eqn-args nlin))
	 (in1 (car in))
	 (in2 (cadr in))
	 (in1? (numberp in1))
	 (in2? (numberp in2))
	 (out? (numberp output)))
    (cond (out?
	   (cond ((and in1? in2?)
		  (unless (== output (* in1 in2))
		    (bad-nonlinear self)))
		 (in1?
		  (cond ((and (== 0 output) (not (== 0 in1)))
			 (kill-nlin self nlin)
			 (add-raw-equation (eq-solver self)
					   `(:equation 0 ,in2)))
			((not (== 0 in1))
			 (kill-nlin self nlin)
			 (add-raw-equation (eq-solver self)
					   `(:equation ,(/ output in1) ,in2)))))
		 (in2?
		  (cond ((and (== 0 output) (not (== 0 in2)))
			 (kill-nlin self nlin)
			 (add-raw-equation (eq-solver self)
					   `(:equation 0 ,in1)))
			((not (== 0 in2))
			 (add-raw-equation (eq-solver self)
					   `(:equation ,(/ output in2) ,in1)))))))
	  ((and in1? in2?)
	   (kill-nlin self nlin)
	   (add-raw-equation (eq-solver self)
			     `(:equation ,(* in1 in2) ,output)))
          (in1?
	   (kill-nlin self nlin)
	   (add-raw-equation (eq-solver self)
			     `(:equation 0 ,output
					 ,(scale-sym in2 (* in1 -1)))))
          (in2?
	   (kill-nlin self nlin)
	   (add-raw-equation (eq-solver self)
			     `(:equation 0 ,output
					 ,(scale-sym in1 (* in2 -1)))))
	  )))
