;;
;;; Crude GANTT charts
;;

(in-package "ZENO")

(defstruct GANTT
  glines
  max-time
  title-width
  time-alist
  )

(defstruct GLINE			;a gantt "line"
  text
  start
  end
  (highlight nil)			;NIL or T for highlighting
  (start-char #\|)
  (end-char #\|)
  )

(defun link-t1 (link)
  (i-start (link-time link)))

(defun link-t2 (link)
  (i-end (link-time link)))

(defun GET-LINKS-FROM-STEP (plan step-id)
  (let ((set nil))
    (dolist (link (plan-links plan))
      (when (eq (link-Si link) step-id)
	(push link set)))
    (values set)))

(defun descriptor (theta)
  (if (eq (car theta) '==)
      (second theta)
    theta))

(defun GATHER-LINKS-BY-DESCRIPTOR (plan links)
  "Return a list of alists, where each alist is of the form (desc . sublinks)
   and SUBLINKS is a subset of LINKS whose descriptors are equivalent."
  (let ((alists nil)
	(bind (plan-bindings plan))
	(found-it? nil)
	(this-key nil))
    (dolist (link links)
      (setf found-it? nil)
      (setf this-key (descriptor (link-condition link)))
      (dolist (alist alists)
	(when (nunify this-key (car alist) bind)
	  (push link (cdr alist))
	  (setf found-it? t)
	  (return)))
      (unless found-it?
	(push (list this-key link) alists)))
    (values alists)))

(defun TOP-SORT (ztm)
  (let ((times (explode-ztm ztm))
	(pred-counts nil)
	(result nil)
	(entry nil))
    (dolist (val times)
      ;; val is (id befores afters)
      ;; predcounts will be (count id successors)
      (push (list (list-length (second val))
		  (car val)
		  (third val))
	    pred-counts))
    (do* ((next-0 (find 0 pred-counts :key #'car)
		  (find 0 pred-counts :key #'car))
	  (time (cdr next-0) (cdr next-0)))
        ((null next-0))
;      (print 'pred-counts-are)
;      (print pred-counts)
      (push (car time) result)
      (setf pred-counts (delete next-0 pred-counts))
      (dolist (suc (cadr time))
	(setf entry (find suc pred-counts :key #'second))
	(when entry
	  (decf (car entry)))))
    (values (nreverse result))
    ))

;;;;;;;;;;;;;;;;
;;; This version does a toplogical sort and then prints out the steps
;;; in the order in which they should be executed
(defun GET-STEP-TOTAL-ORDER (plan)
  (let ((ztm (plan-ordering plan)))
    (let ((times (top-sort ztm))
	  (order nil))
      (print times)
      (dolist (t1 times)
	;; may have more than one step that ends at this time!
	(dolist (s (plan-steps plan))
	  (when (eql (ztm-canonicalize ztm (p-step-end s))
		     t1)
	    (push (p-step-id s) order))))
    (nreverse order))))

(defun GENERATE-TIME-ALIST (plan)
  (let* ((ztm (plan-ordering plan))
	 (order (top-sort ztm))
	(alist nil)
	(count 0))
    (dolist (time order)
      (push (cons (ztm-canonicalize ztm time) count) alist)
      (incf count))
    (values alist)))

 (defun compute-link-time-span (plan link time-alist)
  "Return two values:  the earliest start and the latest end time for
   all of the LINKS."
    (let ((ztm (plan-ordering plan)))
    (values	
     (cdr (assoc (ztm-canonicalize ztm (link-t1 link))
			 time-alist))
     (cdr (assoc (ztm-canonicalize ztm (link-t2 link))
			 time-alist)))))

(defun LINK-THREATENED? (plan link)
  (dolist (f (plan-flaws plan))
    (when (and (unsafe-p f)
	       (eq link (unsafe-link f)))
      (return-from link-threatened? p))))

(defun ANY-LINKS-THREATENED? (plan links)
  (dolist (link links nil)
    (if (link-threatened? plan link)
	(return T))))

(defun STRINGIFY-PREDICATE (plan predicate &optional (indent 0))
  (let ((args (bind-variable (theta-args predicate) (plan-bindings plan))))
    (format nil "~@(~,,V@a~)(~(~{~#[~;~a~:;~a,~]~^~}~))"
	    indent
	    (format nil "~:[~;~~~]~s" 
		    (eq :not (car predicate))
		    (theta-pred predicate))
	    args)))

(defconstant *left-paren* #\( ) ;)
;;(
(defconstant *right-paren* #\) )

(defun LINK-START-CHAR (link)
  (ecase (i-type (link-time link))
   (:open       *left-paren*)
   (:open-start *left-paren*)
   (:closed     #\|)
   (:open-end   #\|)))

(defun LINK-END-CHAR (link)
  (ecase (i-type (link-time link))
   (:open       *right-paren*)
   (:open-start #\|)
   (:closed     #\|)
   (:open-end   *right-paren*)))

(defun TRANSLATE-STEP-INTO-GLINES (plan step-id time-alist)
  "Return an ordered list of GLINES, suitable for display on a
   GANTT chart"
  (let ((links (get-links-from-step plan step-id))
	(step (get-step-with-id plan step-id))
	(glines nil)
	(ztm (plan-ordering plan)))
    (flet ((lookup-time (x)
	     (or (cdr (assoc (ztm-canonicalize ztm x) time-alist)) 0)))
        (push
             (make-gline
                   :text (stringify-predicate plan
                               (cons (car (p-step-action step))
                                   ;; skip one of the time args
                                   (cddr (p-step-action step))))
                   :start (lookup-time (p-step-start step))
                   :end (lookup-time (p-step-end step)))
             glines)
       (dolist (link links)
            (multiple-value-bind (start end)
                   (compute-link-time-span plan link time-alist)
                  (push
                       (make-gline
                             :start-char (link-start-char link)
                             :end-char (link-end-char link)
                             :text (stringify-predicate plan (link-condition link) 2)
                             :start start :end (or end start)
                             :highlight (link-threatened? plan link))
                       glines))))
      (values (nreverse glines))))

(defun CREATE-GANTT-FOR-PLAN (plan)
  (let ((order (get-step-total-order plan))
	(time-alist (generate-time-alist plan))
	(glines nil))
    (setf glines (translate-step-into-glines plan 0 time-alist))
    (dolist (step-id order)
      (unless (zerop step-id)
	(setf glines
	  (nconc glines
		 (translate-step-into-glines plan step-id time-alist)))))
    (setf (plan-gantt plan)
      (make-gantt
       :time-alist (sort time-alist #'< :key #'cdr)
       :max-time (apply #'max (mapcar #'cdr time-alist))
       :title-width (apply #'max 
                       (mapcar #'length (mapcar #'gline-text glines)))
       :glines glines))))

(defconstant *NORMAL-GANTT-FORMAT* "~&~V,,,'.a~VT~V,,0,'-a~a~%") 
(defconstant *HIGHLIGHTED-GANTT-FORMAT* "~&~V,,,'.a~VT~V,,0,'*a~a~%") 

(defvar *max-gantt-width* 72 "Max width for plans")

(defun SHOW-GANTT (gantt &optional (scale 4) (stream *standard-output*))
  (let ((width1 (gantt-title-width gantt))
	(start nil)
	(interval-width nil)
	(too-wide? nil)
	(w0 (max *max-gantt-width* 3))
	(w1 (- *max-gantt-width* 3)))
    (terpri)
    (dolist (g (gantt-glines gantt))
      (setf interval-width
	(* (- (gline-end g) (gline-start g)) scale)
	start (+ width1 2 (* (gline-start g) scale)))
      (setf too-wide? (< w0 (+ interval-width 2 start)))
      (setf interval-width
	(min (max 2 (- w0 (+ 3 start))) interval-width))
      (setf start (min w1 start))
      (format stream (if (gline-highlight g)
			 *highlighted-gantt-format*
		         *normal-gantt-format*)
	      width1 (gline-text g)
	      start interval-width
              (if (and too-wide? (= start w1))
		  #\> (gline-start-char g))
	      (if too-wide? #\>
		(gline-end-char g))
	      ))
    (show-timeline gantt scale stream)))

(defvar *max-label-size* 3)

(defun SHOW-TIMELINE  (gantt &optional (scale 4) (stream *standard-output*))
  (let* ((new-alist (mapcar #'(lambda (entry)
				(cons (format nil "~@(~a~)"
                                  (variable::strip-prepended-? (car entry)))
			      (cdr entry)))
			    (gantt-time-alist gantt)))
	 (offset (+ 2 (gantt-title-width gantt)))
	 (label-width (min *max-label-size*
			   (apply #'max
                        (mapcar #'length (mapcar #'car new-alist))))))
    (cond ((>= scale label-width)
	   ;;; Horizontal version
	   (dolist (entry new-alist)
	     (let ((time (cdr entry))
		   (name (car entry)))
	       (unless (> (+ offset (* time scale)) *max-gantt-width*)
		 (let ((len (min *max-label-size* (length name))))
		   (format stream "~V,0T~a"
			   (ceiling
			    (- (+ offset (* time scale))
			       (/ len 2)))
			   (subseq name 0 len)))))))
	  (t
	   ;;; Vertical version
	   (dotimes (index (min label-width *max-label-size*))
	     (dolist (entry new-alist)
	       (let ((time (cdr entry))
		     (name (car entry)))
	       (unless (> (+ offset (* time scale)) *max-gantt-width*)
		 (format stream "~V,0T~a"
			 (+ offset (* time scale))
			 (if (>= index (length name))
			     ""
			   (elt name index))))))
	     (terpri))))))
  
(defun DISPLAY-PLAN (plan &optional (stream t) ignore)
  (declare (ignore ignore))
  (let ((gantt (or (plan-gantt plan)
		   (create-gantt-for-plan plan))))
     (format stream "~&~s of rank ~s~%" plan (rank3 plan))
    (show-gantt gantt 4 stream)
    (when (plan-flaws plan)
      (format stream "~%Remaining goals:~%")
      (dolist (o (plan-flaws plan))
	(when (openc-p o)
	  (let ((c (openc-condition o)))
	    (let ((*print-case* :downcase))
	      (pprint (ground-theta c plan) stream))))))
    (unless (plan-flaws plan)
      (format stream "~%Complete!")
      (when *debug*
	(setf cp plan)
	(format stream "~&Bindings: ~:[bad~;ok~].~%"
		(plan-bindings plan))
	(format stream "~&Links:")
	(dolist (l (plan-links plan))
	  (print-link l stream))
	(format stream "~&Names:")
	(dolist (l (plan-names plan)) (pprint l stream))
	(if (plan-constraints plan)
	    (show (if (consp (plan-constraints plan))
		      (car (plan-constraints plan))
		    (plan-constraints plan))))))
    (terpri)
;    (zshow (plan-ordering plan))
    ))
  
(defun ground-theta (theta plan)
  (let ((bind (plan-bindings plan)))
    (cond ((is-a-fn? theta)
	   `(== (,(theta-pred theta) ,(theta-time theta)
		 ,@(bind-variable (theta-args theta) bind))
	      ,(theta-var theta)))
	((eq ':not (car theta))
	 `(:not (,(theta-pred theta)
		 ,(theta-time theta)
		 ,@(bind-variable (theta-args theta) bind))))
	(t
	 `(,(theta-pred theta) ,(theta-time theta)
	   ,@(bind-variable (theta-args theta) bind))))))


(defun show-metrics (plan)
  (let* ((ztm (plan-ordering plan))
	 (order (top-sort ztm))
	 (names (sort (copy-list (plan-names plan))
		      #'(lambda (x y)
			  (< (or (position (ztm-canonicalize ztm (theta-time x)) order)
				 999)
			     (or (position (ztm-canonicalize ztm (theta-time y)) order)
				 999))))))
    (dolist (entry names)
      (format t "~&At time ~4a, ~a=~s~%"
	      (variable::strip-prepended-? (ztm-canonicalize ztm
					    (theta-time entry)))
	      (stringify-predicate plan (ground-theta entry plan))
	      (get-metric-value plan (theta-var entry))))))


;;
;; Dump lisp stuff to disk
;;

(defun dump ()
  (with-open-file (out "~/dump" :direction :output
		   :if-exists :new-version
		   :if-does-not-exist :create)
    (dolist (x (reverse *rank-history*))
      (format out "~s~%" x))))
