(in-package "VARIABLE")

" (c) 1992 Copyright (c) University of Washington
  Written by Tony Barrett.

  All rights reserved. Use of this software is permitted for non-commercial
  research purposes, and it may be copied only for that use.  All copies must
  include this copyright message.  This software is made available AS IS, and
  neither the authors nor the University of Washington make any warranty about
  the software or its performance.

  When you first acquire this software please send mail to
  bug-ucpop@cs.washington.edu; the same address should be used for problems."

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; The following code is used to implement variable codesignation constraints.
;;; This code was created after looking at Steve Hanks's use of varsets.
;;; 
;;;

(export '(unify nunify bind-variable instantiate-term
	  nec-mgu mgu grounded-p
	  eq-member eql-member
	  add-bind new-bindings variable? uniquify-var
	  *unify-count* *compute-rank-unifies* 
	  *computing-rank* *add-bind-count*))                   

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Outline of this file
;;;
;;; 1. Structures used for implementing constraints
;;; 2. A handy interface function to unify two lists using constraint sets
;;; 3. CONSTRAINT SETS (CS)
;;; 4. VARSETS:  Each VARSET represents a set of codesignating variables.
;;; 5. VARIABLES  -- 
;;; 6. Print functions

;;;***************************************************************************
;;; 1. Structures used for implementing constraints

(defstruct (varset (:print-function print-varset))
  const         ;; the unique constant that codesignates with this varset.
  cd-set        ;; the set of variables (and constant) that codesignate
  ncd-set)	;; Vars & consts that must not codesignate with this set

(defstruct (var (:print-function print-var))
  name)					; The variable's name

(defvar *unify-count* 0)		; # of variable unifications made
(defvar *compute-rank-unifies* nil)	; # of unifications made in rank fn
(defvar *computing-rank* nil)		; flag: are we computing a rank?
(defvar *add-bind-count* 0)		; # of times new bindings were added
(defvar *free-vsets* nil)		; free varsets generated by unify
(defvar *context-alst* nil)		; Association list of known variables

;;;***************************************************************************
;;; 2. A handy interface function to unify two lists using constraint sets

;;;;;;;;;;;;;;;;;;;;;;;;                                                  
;;; Test whether two conditions can be unified consistent with plan bindings
;;; Returns a list of the list of bindings needed to unify, i.e. has an extra
;;; set of parentheses.  Returns nil if they cannot be unified.  Returns (nil)
;;; if they unify with no additional bindings.

(defvar *syntax-check* t "Set to T to test syntax of clauses.")

(defun MGU (theta1 theta2 bindings)
  ;; Theta1 and Theta2 are either temporal relations
  ;; (R t x1...xn) or function assignments (== (f t x1..xn) v).
  ;;
  ;; Return the most general unifier for matching theta1 and theta2,
  ;; assuming that their times overlap.
  ;;
  (let ((a1 nil) (a2 nil))
   (when (and (eq (car theta1) (car theta2))
	     (eq (zeno::theta-pred theta1)
		 (zeno::theta-pred theta2)))
	(setf a1 (zeno::theta-args theta1)
	      a2 (zeno::theta-args theta2))
	(if (or (not *syntax-check*) (= (length a1) (length a2)))
	    (unify a1 a2 bindings)
	  (error "Argument count mismatch for ~:[relation~;function~] ~s.~%~
                  This is probably due to a typo in your action or problem~%~
                  descriptions.  Please check them carefully.~%~%~
                  This check can be disabled by typing the following:~%   ~
                  (setf variable::*syntax-check* nil)~%"
		 (eq '== (car theta1))
		 (zeno::theta-pred theta1))))))

(defconstant *nec-mgu* '(nil)
  "Return value for necessary unification")

(defun NEC-MGU (theta1 theta2 bindings)
  ;; "Necessarily" the same
  (and (eq (car theta1) (car theta2))
       (eq (zeno::theta-pred theta1)
	   (zeno::theta-pred theta2))
       (nunify (zeno::theta-args theta1) (zeno::theta-args theta2) bindings)))

(defun UNIFY (cond1 cond2 bindings)
  (when (and (eq (car cond1) :not) (eq (car cond2) :not))
    (setf cond1 (cadr cond1)
	  cond2 (cadr cond2)))
  (let ((result nil)
	(a1 nil)
	(a2 nil)
	(cs (list (car bindings))))
    (do ((b1 cond1 (cdr b1))
	 (b2 cond2 (cdr b2)))
	((null b1) (free-vsets (car cs) (car bindings)
			       (if (null b2)
				   (list result))))
      (zeno::count-stat zeno::.unify.)
      (setf a1 (get-vset (car b1) cs))
      (setf a2 (get-vset (car b2) cs))
      (when (not (eq a1 a2))
	(push (cons (car b1) (car b2)) result)
	(let ((c (combine-varset a1 a2)))
	  (if c (push c (car cs))
	    (return-from unify 
	      (free-vsets (car cs) (car bindings) nil)))))
      )))

(defun NUNIFY (cond1 cond2 bindings)
  ;; "Necessarily" Unify
  (zeno::count-stat zeno::.unify.)
  (let ((a1 nil)
	(a2 nil)
	(cs (list (car bindings))))
    (do ((b1 cond1 (cdr b1))
	 (b2 cond2 (cdr b2)))
	((null b1) (if (null b2) *nec-mgu* nil))
      (setf a1 (or (get-vset! (car b1) cs) (car b1)))
      (setf a2 (or (get-vset! (car b2) cs) (car b2)))
     (when (not (eq a1 a2))
       (return-from nunify nil)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Replace variable with associated constant from plan-bindings
;;; Leave as a variable if not bound to a constant
(defun BIND-VARIABLE (var bindings)
  (cond ((null var) nil)
        ((null bindings) var)
	((listp var) (mapcar #'(lambda (x) (bind-variable x bindings))
			     var))
	(t (value var bindings))))

(defun NEW-BINDINGS ()
  (new-cs))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Changes variable symbols in the list into variable structures in
;;; the context NUM.

(defun instantiate-term (term num &optional (alst nil))
  (cond ((consp term)
	 (cons (instantiate-term (car term) num alst)
	       (instantiate-term (cdr term) num alst)))
	((variable? term)
	 (let ((u (assoc term alst)))
	   (if u (cdr u) (make-variable term "." num))))
	(t term)))

;;;;;;;;;;;;;;;;;;;;;;;;                                                  
;;;  Add new entries to bindings
;;;  1st input is list of pairs of 2 terms, each pair possibly preceded by :not
(defun ADD-BIND (new-bind bindings)
  (when bindings
    (setf bindings (copy-cs bindings))
    (dolist (pair new-bind bindings)        ; 
      (zeno::count-stat zeno::.codesg.)
      (zeno::count-stat zeno::.constraints.)
      (if (not (if (eql :not (car pair))
		   (constrain-neq (cadr pair) (cddr pair) bindings)
		 (constrain-eq (car pair) (cdr pair) bindings)))
	  (return-from add-bind nil))))
  bindings)

;;;***************************************************************************
;;; 3. CONSTRAINT SETS (CS) --
;;;   A constraint set is a list of varsets
;;;

(defun new-cs () 
  "Create a new constraint set."
  (list nil))

(defun copy-cs (cs) 
  (list (car cs)))

(defun constrain-eq (sym1 sym2 cs)
  "Ensure that two symbols codesignate in a constraint set.
   Returns nil if symbols cannot codesignate."
  (let* ((v1 (get-vset sym1 cs))
	 (v2 (get-vset sym2 cs))
	 (vs (combine-varset v1 v2)))
    (when vs
      (if (not (eq v1 v2)) (push vs (car cs)))
      t)))

(defun constrain-neq (sym1 sym2 cs)
  "Make sure that two symbols never codesignate in a constraint set.
   Returns nil if symbols already codesignate."
  (let* ((r1 (get-vset sym1 cs))
	 (v1 (restrict-varset r1 sym2))
	 (r2 (get-vset sym2 cs))
	 (v2 (restrict-varset r2 sym1)))
    (when (and v1 v2)
      (push v1 (car cs))
      (push v2 (car cs))
      t)))

(defun codesignates (sym1 sym2 cs)
  "Test to see if two symbols explicitely codesignate."
  (eq-member sym2 (varset-cd-set (get-vset sym1 cs))))

(defun EQ-MEMBER (elt lst)
  (declare (optimize (speed 3) (safety 1))
	   (type list lst))
  (dolist (l lst nil)
    (if (eq elt l) (return t))))

(defun EQL-MEMBER (elt lst)
  (declare (optimize (speed 3) (safety 1))
	   (type list lst))
  (dolist (l lst nil)
    (if (eql elt l) (return t))))

(defun value (sym cs)
  "Get the value of a symbol.  It might be a constant or variable."
  (if (variable? sym)
      (let ((v (get-vset sym cs)))
	(if (varset-const v) (varset-const v)
	  (first (varset-cd-set v))))
    sym))

(defun get-vset (sym cs)
  "Get the varset associated with a symbol.  Might have to create it."
;  (declare (optimize (speed 3) (safety 1))
;	   (type list cs))
  (assert (atom sym))
  (dolist (l (car cs))
    (dolist (v (varset-cd-set l))
      (when (eql v sym) (return-from get-vset l))))
  (car (push (make-empty-varset sym) (car cs))))

(defun get-vset! (sym cs)
  "Get the varset associated with a symbol.  Will not create it."
;  (declare (optimize (speed 3) (safety 1))
;	   (type list cs))
  (assert (atom sym))
  (dolist (l (car cs))
    (dolist (v (varset-cd-set l))
      (when (eql v sym) (return-from get-vset! l))))
  (values nil))

;;;***************************************************************************
;;; 4. VARSETS:  Each VARSET represents a set of codesignating variables.
;;; Store for each 
;;;    (1) the constant (if any) they codesignate with, and 
;;;    (2) a set of codesignating variables and possibly a constant
;;;    (3) a set of constants and variables they CANNOT codesignate with.
;;;
;;; Some thoughts on VARSETs...
;;;   - The codesignation sets form equivalence classes.  This means that
;;;     the cd sets in the varsets are disjoint.
;;;   - Constants have varsets too!  This means that the only place that we
;;;     have to explicitely reason about a constant is to make sure that no
;;;     two unique constants codesignate.
;;;   - Since the cd sets in each varset contain all codesignating symbols,
;;;     there is no need to make sure that all symbols that must not 
;;;     codesignate are in the ncd lists.  Puting only one element of a 
;;;     codesignating varset in the ncd is enough to ensure that the
;;;     noncodesignating varsets can never get combined.
;;;

(defmacro var-union (a b) ; &aux ret
  `(union ,a ,b :test #'eql))

(defmacro var-intersect? (a b)
  `(intersection ,a ,b :test #'eql))

(defun make-empty-varset (symb)
  "Make a new varset for SYMB (if it has never been seen before).
   Constants appear in varsets too!"
  (new-varset :cd-set (list symb)
	      :const (if (variable? symb) nil symb)
	      :ncd-set nil))

(defun combine-varset (a b)
  "Combine two varsets.  (called for each codesignation constraint)"
  (if (eq a b) a
      (when (or (null (varset-const a))
		(null (varset-const b)))
	(let ((cd-set  (append (varset-cd-set a) (varset-cd-set b)))
	      (ncd-set (var-union (varset-ncd-set a) (varset-ncd-set b)))
	      (const   (if (varset-const a) (varset-const a)
			   (varset-const b))))
	  (when (and (not (eql-member const ncd-set))
		     (not (var-intersect? cd-set ncd-set)))
	    (new-varset :const const
			:cd-set cd-set
			:ncd-set ncd-set))))))

(defun restrict-varset (vset restricter)
  "Make sure that VSET never codesignats with a varset containing RESTRICTER."
  (when (not (eql-member restricter (varset-cd-set vset)))
    (new-varset :const (varset-const vset)
		:cd-set (varset-cd-set vset)
		:ncd-set (var-union (list restricter) 
				     (varset-ncd-set vset)))))

(defun grounded-p (form bind)
  (cond ((variable? form)
	 (not (null (varset-const (get-vset form bind)))))
	((atom form) t)
	(t
	 (and (grounded-p (car form) bind)
	      (grounded-p (cdr form) bind)))))
	   

(defun new-varset (&key const cd-set ncd-set)
  (if *free-vsets*
      (let ((v (pop *free-vsets*)))
	(setf (varset-const v) const
	      (varset-cd-set v) cd-set
	      (varset-ncd-set v) ncd-set)
	v)
    (make-varset :const const :cd-set cd-set :ncd-set ncd-set)))

(defun free-vsets (v1 v2 ret-val &aux a)
  (do ()
      ((eq v1 v2) ret-val)
    (setf a v1)
    (setf v1 (cdr v1))
    (setf (cdr a) *free-vsets*)
    (setf *free-vsets* a)))
	    
;;;**************************************************************
;;; 5. VARIABLES  -- This code was written by Steve Hanks 
;;;                            and modified by J Scott Penberthy
;;;

(defvar *variable-cache* (make-hash-table :test #'equal))

(defun print-var (thing &optional (stream *standard-output*) depth)
  (declare (ignore depth))
  (princ (var-name thing) stream)
  (princ #\* stream))

(defun variable? (thing)
  (or (and (symbolp thing)
	   (char= #\? (elt (symbol-name thing) 0)))
      (var-p thing)))

(defun uniquify-var (name)
  ;; careful -- what if this isn't unique?  may generate 
  ;; strange results!!!!!!!!!!!  CAUTION
  (make-var :name name))

;  (make-variable name (gensym)))

(defun strip-prepended-? (thing)
  (cond ((symbolp thing)
	 (let ((name (symbol-name thing)))
	   (if (char= #\? (elt (symbol-name thing) 0))
	       (subseq name 1)
	     name)))
	(t (princ-to-string thing))))

(defun make-variable (&rest id)
  (multiple-value-bind (val got-it) (gethash id *variable-cache*)
   (unless got-it
    (setf (gethash id *variable-cache*)
	  (setq val
		(intern
		 (apply #'concatenate 'string
			(mapcar #'princ-to-string id))
		 (symbol-package (first id))))))
   (values val)))

;;;***************************************************************************
;;; 6. Print functions
;;;

(defun print-varset (self stream depth)
  (declare (ignore depth))
  (format stream "<VARSET {~a} = ~a not~a>" 
          (varset-cd-set self) 
          (varset-const self) 
          (varset-ncd-set self)))

(defun print-variable (self stream depth)
  (declare (ignore depth))
  (format stream "~a" (var-name self)))
