;;; -*- Syntax: Common-lisp; Package: POS; Mode:Common-Lisp; Base:10 -*-
;;; Copyright 1988 David Throop at the University of Texas at Austin

(in-package :pos)

#+:ccl
(use-package :ccl :pos)


#+symbolics
(use-package :scl :pos)

#+symbolics
(import '(self send defmethod))

;;;  This file holds the functions and parameters that control the PostScript output.
;;;  None of this code should be machine dependent.

;;; The definitions of the following parameters are machine dependent, but may be
;;; referenced by functions that are not.




(proclaim '(special *black* *flip* *gray-one* *gray-two* *gray-three* 
		    *gray-four* *gray-five* *gray-six* *gray-seven* *gray-eight*
		    *gray-nine* *gray-ten* *initial-state* *plain-font* *dot-font*
		    *postscript-output-file*  *qplot-output* *symbol-x-offset*
		    *symbol-y-offset* *white* axis-font bmargin label-font xscreen yscreen))



;;;  This controls whether the image is sent to the screen, a PS file,
;;;  or both/neither. Start out with things sent to screen, but not hardcopied

(defparameter *image-disposal* :screen)


;;; Standard postscript style sets up the page and finishes with SHOW.  Images
;;; for inclusion in Text formatting programs should not have these commands.

(defparameter *postscript-style* :standard)


;;; Column Width in Inches

(defparameter *Text-Format-text-width* 6.0)


;;; Column Height in Inches

(defparameter *Text-Format-text-height* 7.5)

;;; Should either be NIL, or the list (llx lly urx ury) where elements are in screen 
;;; coordinates.  When non-null, the %%BoundingBox is computed and written to the PS file.

(defparameter *bounding-box* nil)


;;; Everytime a new font is introduced into a postscript output file,
;;; the font must be loaded and scaled and named.  Keep track of what
;;; fonts have already been used in *used-fonts*.   *Used-fonts* is an
;;; alist of the form (((fontface fontsize) name)((fontface fontsize)
;;; name)...).

(defparameter *used-fonts* nil)


;;; The fonts and linewidths and darkness stay the same in PostScript
;;; until they are changed.   Keep track of their current values in
;;; *current-linewidth*, *current-font* and *current-grayval*.  Realize: these are *not*
;;; default values, they are just the memory of what this parameter was
;;; set to in the last gesture.

(defparameter *current-linewidth* 1)
(defparameter *current-font* nil)
(defparameter *current-grayval* 0)


;;; The following parameters control the location of the the PostScript
;;; Image on the page.  Since terminals are usually wider than high
;;; (landscape) and printer starts out with y = 11, x = 8.5, we must
;;; rotate the coordinates and translate the origin in order to fit
;;; image onto paper.


(defparameter *x-translation* 8.5)		; in inches
(defparameter *y-translation* 0.5)		; in inches
(defparameter *rotation* 90)
(defparameter *x-scale* 0.7
  "Mapping between screen width in pixels and paper width")
(defparameter *y-scale* .8
  "Mapping between screen height in pixels and paper height")


(defun enforce-string (font familiar-font size gatom pair) 
  (cond ((not familiar-font)     (load-font gatom font size pair)
                                 (format *postscript-output-file* "~&~s f~&" gatom)
	                         (setf *current-font* pair))
	((not (equal pair *current-font*))
	                         (format *postscript-output-file* "~&~s f~&" gatom)
	                         (setf *current-font* pair))
	(t                       nil)))



(defun load-font (gatom font size pair)
  (let ((*print-case* :upcase))
    (format *postscript-output-file*
	    "~& /~s ~&   /~a findfont ~s scalefont def~&"
	    gatom font size)
    (push (list pair gatom) *used-fonts*)))
 
;;; In PostScript, the current-linewidth is a global parameter.  Every time we draw a line, we check
;;; to make sure the current-linewidth we want is the same as the last line we drew.  If not, we
;;; must change it.

(defun checkwidth (thickness)
       (unless (= *current-linewidth* thickness)
               (format *postscript-output-file* " ~d w " thickness)
               (setf *current-linewidth* thickness)))

;;; Anytime we draw anything, we must check to see if the current gray level is unchanged
;;; from last the last stroke.  If it has changed, we must "setgray" the new level.
;;; Remember, in postscript, "setgray 0" is black and "setgray 1" is white. 


;;; Anytime we draw anything, we must check to see if the current gray
;;; level is unchanged from last the last stroke.  If it has changed, we
;;; must "setgray" the new level.  Remember, in postscript, "setgray 0"
;;; is black and "setgray 1" is white, while in Symbolics, gray-value =
;;; 0 gives white and gray-value=1 gives black.

#|
(defun checkgray (&optional (alu *black*) (gray-level))
  (let ((grayval (cond ((equal alu *white*)    1)	; White
		       (gray-level             (- 1 gray-level))
		       ((equal alu *black*)    0)	; Black 
		       (t                      0))))	; Black
    (unless (= *current-grayval* grayval)
            (format *postscript-output-file* " ~d g " grayval)
            (setf *current-grayval* grayval))))
|#


(defun checkgray (&optional (alu *black*))
  (let ((grayval (cond ((equal alu *white*)        1)
                       ((equal alu *gray-one*)    .1)
                       ((equal alu *gray-two*)    .2)
                       ((equal alu *gray-three*)  .3)
                       ((equal alu *gray-four*)   .4)
                       ((equal alu *gray-five*)   .5)
                       ((equal alu *gray-six*)    .6)
                       ((equal alu *gray-seven*)  .7)
                       ((equal alu *gray-eight*)  .8)
                       ((equal alu *gray-nine*)   .9)
                       ((equal alu *gray-ten*)    .9)
                       (t                          0))))
    (unless (= *current-grayval* grayval)
            (format *postscript-output-file* " ~d g " grayval)
            (setf *current-grayval* grayval))))


(defparameter *gray-level* 1)





;;; Whether the image is sent to the screen, or to a PS file, or both/neither is actually
;;; controlled by *IMAGE-DISPOSAL*.  Since it is anticipated that this functionality may
;;; change, we supply the following functions to test or bind this.  Normally, application
;;; code should not reference *image-disposal* directly.

(defun image-to-postscript-p ()
  (member *image-disposal* '(:ps :both)))

(defun image-to-screen-p ()
  (member *image-disposal* '(:screen :both)))

(defun set-image-disposal (image-disposal-value)
       (setq *image-disposal*		image-disposal-value))


(defmacro with-plotting-to-screen-inhibited (&rest args)
  `(let ((*image-disposal* (case *image-disposal*
			     (:both :ps)
			     (:screen nil)
			     (t *image-disposal*))))
     ,@args))

(defmacro with-plotting-to-postscript-inhibited (&rest args)
  `(let ((*image-disposal* (case *image-disposal*
			     (:both :screen)
			     (:ps nil)
			     (t *image-disposal*))))
     (when *image-disposal*
       ,@args)))

(defmacro with-plotting-to-screen-forced (&rest args)
  `(let ((*image-disposal* (case *image-disposal*
			     (nil :screen)
			     (:ps :both)
			     (t *image-disposal*))))
     ,@args))

(defmacro with-plotting-to-postscript-forced (&rest args)
  `(let ((*image-disposal* (case *image-disposal*
			     ((nil :ps) :ps)
			     (t :both))))
     ,@args))



;;; Turn on the use-lv-call slot for *qplot-output* so that drawing
;;; commands will actually draw.  See XPOS:XPOS;lispview-extensions.lisp
;;; for more details.
;;;
#+:pos-lispview
(defmacro with-plotting-to-lispview-forced (&rest args)
  `(progn
    ;; First we wait until the window is done with any drawing.
    (loop until (eq (status *qplot-output*) :idle))

    ;; Then we turn on the use-lv-call flag and plot.  This should
    ;; always be NIL, but maybe not
    (let ((old-lv-call (pos::use-lv-call *qplot-output*)))
      (unwind-protect
	   (progn
	     (with-plotting-to-postscript-inhibited
		 (setf (pos::use-lv-call *qplot-output*) t)
	         ,@args))
	(setf (pos::use-lv-call *qplot-output*) old-lv-call)))))


#-:pos-lispview
(defmacro with-plotting-to-lispview-forced (&rest args)
  `(with-plotting-to-postscript-inhibited
      ,@args))


 
;;; *Postscript-Bound* is a flag to test whether we already have opened
;;; a postscript output file.  Many of the higher level plotting
;;; routines call each other recursively - the *postscript-bound* flag
;;; stops us from dumping to multiple files simultaneously.

(defvar *postscript-bound* nil)

;;; *Allow-Plotting* allows us to turn off everything within the scope
;;; of the PostScript plotting.

(defparameter *allow-plotting* t)

;;; DEVICE-INTERFACE is a macro that handles directing the output of the
;;; simulation to the screen, a postscript file, or both.

(defmacro device-interface (&rest plotting-exp)
     `(when *allow-plotting*
          #+l-windows (setf xscreen (slb-cl::window-stream-inner-width *qplot-output*)
                            yscreen (slb-cl::window-stream-inner-height *qplot-output*))
       (let ((*print-case*   :upcase)
	     (*image-disposal* *image-disposal*))
	 (unwind-protect
	      (progn
		(when (image-to-postscript-p)
		  (get-and-open-postscript-file))
		,@plotting-exp)
	   (close-postscript-file-if-needed)))))


(defun get-and-check-postscript-file ()
  "This function will prompt the user for a postscript output file
name and will check to make sure that it can be created.  The file
name will be returned."
  (loop for file = (get-postscript-file)
	for exists = (probe-file file)
	for exit = (if exists
		       #+(or ti symbolics) t
		       #-(or ti symbolics) (y-or-n-p
					    "File already exists. Should I overwrite it?")
		       t)
	do (when exit (return file))))


(defun get-and-open-postscript-file ()
  "This function will get the name of a postscript file from the user and
it will open it.  If the user does not specify a file, this function return
nil.  Otherwise, it returns the name of the file.  IT will open the file and
start the postscript output."
  (let ((file (get-and-check-postscript-file)))
    (when file
      (setf *postscript-bound* t)
      #-:lispm (setf *postscript-output-file* (open file :direction :output
						    :if-exists :overwrite
						    :if-does-not-exist :create))
      #+:lispm (setf *postscript-output-file* (open file :direction :output))
      (start-postscript-file)
      file)))

(defun close-postscript-file-if-needed (&key (abort nil))
  "This function will close a postscript file if there is one open."
  (when (and *postscript-bound*
	     *postscript-output-file*
	     (streamp *postscript-output-file*))
    (setf *postscript-bound* nil)
    (end-postscript-file)
    (close *postscript-output-file* :abort abort)
    (setf *postscript-output-file* nil)))


;;; Clear the states of some global variables.  Write out a header and a function definition
;;; for INCH to the Postscript file.

(defun start-postscript-file ()
       (initialize-postscript-variables)
       (format *postscript-output-file*
	       "%! -*- Mode: Text; Default-character-style: (:FIX :BOLD :NORMAL) -*-")
       (when (eq *postscript-style* :ms-word)
             (format *postscript-output-file* "~&.pic.~&"))
       (when *bounding-box* (bounding-box))
       (format *postscript-output-file* "~2&/inch {72 mul} def~&")
       (case *postscript-style*
             ((:standard :Text-Format-insert :bounded :ms-word)
                   (compress-commands))
             (t    (error "Unknown *postscript-style*: ~s" *postscript-style*))))


(defun end-postscript-file ()
       (case *postscript-style*
             ((:standard :Text-Format-insert :bounded :ms-word)
                    (restore-environment))
             (t     (error "Unknown environment: ~s" *postscript-style*))))



(defun initialize-postscript-variables ()
  #+l-windows
  (setf *x-scale* (/ 720.0 xscreen)
	*y-scale* (/ (* 8.5 72) yscreen))
  (setq  *current-linewidth* 1
	 *current-font* nil
	 *used-fonts* nil))

(defun bounding-box ()
  (let ((morex (* *x-translation* 72))
	(morey (* *y-translation* 72)))
    (format *postscript-output-file* "~&%%BoundingBox: ~d ~d ~d ~d~&"
	    (round (+ morex (* (first *bounding-box*) *x-scale*)))
	    (round (+ morey (* (- yscreen (second *bounding-box*)) *y-scale*)))
	    (round (+ morex (* (third *bounding-box*) *x-scale*)))
	    (round (+ morey (* (- yscreen (fourth *bounding-box*)) *y-scale*))))
    *bounding-box*))

