;;; -*- Mode: LISP; Syntax: Common-Lisp; Base: 10; Package: QSIM -*-
;;;  $Id: props.lisp,v 1.1 1991/03/26 21:38:04 clancy Exp $

;;;----------------------------------------------------------------------
;;; A propagator for the = constraint.
;;
;; Written by Adam Farquhar with some consulting from D. Throop.
;; Written on: 14 Feb 1990
;;
;; 24.Oct.90 Adam Farquhar.  Fixed the mysterious deleteion of a paren and "al" in the var cvals.
	      
(defun =-propagator (con cvals state)
  ;; Args: con, a constraint, cvals, always nil, as = does not take
  ;; corresponding values, and state.
  ;;
  ;; Return: a list of vars effected by propagation.
  ;; Side Effect: set the qvalue of x or y.
  ;;
  (when cvals
    (error "The = constraint does not accept corresponding values: ~a ~a ~a"
	   con cvals state))
  (let* ((template (constraint-name con))	; (= x y)
	 (vars     (constraint-variables con))	; (<x> <y>)
;	 (type (first template))    ;used only in tracing  - drt
	 (x-name    (second template))
	 (x         (first  vars))
	 (y-name    (third template))
	 (y         (second vars))
	 (changed nil))
;   Following is already reported elsewhere:	-- RM
;   (and trace-propagation (format t "~&*** Propagating (~a ~a ~a) ***~%" type x-name y-name))
    (cond ((and (consp y-name)
		(eq 'CONSTANT (car y-name)))
	   ;; Handle the special case where y-name = (constant mag).
	   (if
	     (assert-eq x (find (second y-name) (variable--qspace x)
				:key #'lmark-name)
			state)
	     (setq changed (list x)))
	   (if  (set-qdir x 'std)
		(pushnew x changed))
	   changed)
	  (t ;; The general case where they are both vars.
	   (union (glistify (prop-= x x-name y y-name state))
		  (glistify (prop-= y y-name x x-name state))
		  (glistify (propagate-qdir con cvals state)))))))



;;;-----------------------------------------------------------------------------
;;;  (M s1 ... sn) Constraints                                   (BJK:  10-26-90)
;;;
;;;  The M constraint represents a multivariable function relation y = f(x1 ... xn).
;;;  It takes any number of variables and a tuple of signs representing the
;;;  partial derivatives, si = dy/dxi (partial derivs).
;;;  The product of corresponding signs and partial are tested like SUM-ZERO:
;;;  Either all signs are zero, or there are both + and - values.
;;;-----------------------------------------------------------------------------

(declare-qsim-constraint
  (make-contype :name            'M
		:nargs           99
		:propagator      'M-propagator	; doesn't exist yet
		:checkfcn        'check-M-constraint
		:qmag-relation   nil
		:qdir-relation   nil
		:cvals-allowed-p t))

; When the constraint structure is build for an instance of the M constraint
; (M s1 ... sn), the sign tuple (s1 ... sn -) must be put into constraint.partials.


; Things to do:
;  - Write M-propagator.

(defun check-M-constraint (tuple constraint)
  (let ((partials (constraint-partials constraint)))
    (and (check-M-qdirs tuple partials)
	 (check-M-infinite-values tuple partials)
	 (check-M-cvals tuple constraint partials))))

; Test whether a tuple of qdirs, multiplied by a tuple of partials, is consistent.

(defun check-M-qdirs (tuple partials)
  (do ((pluses 0)
       (minuses 0)
       (L tuple (cdr L))
       (P partials (cdr P)))
      ((null P) (eql (= pluses 0)
                     (= minuses 0)))
    (cond ((eql (car P) 0))
	  ((eql (qval-qdir (car L)) 'ign) (return-from check-M-qdirs t))
	  ((eql (qval-qdir (car L)) 'std))
          ((sign-qdir-match (car P) (qval-qdir (car L))) (incf pluses))
          (t (incf minuses)))))


(defun plus-minus (orders partials)
  (let ((a-nil nil)
	(unknown nil)
	(pluses 0)
	(minuses 0)
	(i 0))
    (mapc #'(lambda (o p)
	      (cond ((eql P 0))
		    ((eql O  0))
		    ((eql O nil)
		     (setq unknown (list i P))
		     (if a-nil
			 (return-from plus-minus nil)
			 (setq a-nil t)))
		    ((eq P O) (incf pluses))
		    (t (incf minuses)))
	      (incf i))
	  orders partials)
    (values pluses minuses unknown)))

(defun qdir-order (qdir)
  (ecase qdir
    (inc '+)
    (std 0)
    (dec '-)
    ((nil) nil)
    (ign '*)))

(defun M-propagator (con cvs state)
  (union (propagate-M-qdir con)
	 (propagate-M-cvs con cvs state)))

(defun propagate-M-qdir (con &aux orders)
  (setq orders (mapcar #'(lambda (var)
			   (qdir-order (var-qdir var)))
		       (constraint-variables con)))
  (unless (member '* orders)
    (multiple-value-bind (plus minus unknown)
	(plus-minus orders (constraint-partials con))
      (when plus
	(let* ((qdir-if-plus
		 (if (zerop plus)
		     (if (zerop minus) 'std 'inc)
		     (if (zerop minus) 'dec nil)))
	       (new-qdir
		 (if (eq (second unknown) '+)
		     qdir-if-plus
		     (qdir-inverse qdir-if-plus)))
	       (var (nth (first unknown) (constraint-variables con))))
	  (when new-qdir
	    (set-qdir var new-qdir)
	    (list var)))))))

(defun propagate-M-cvs (con cvs state &aux var)
  (dolist (cv cvs)
    (setq var
	  (or (propagate-m-corr-values con cv state)
	      var)))
  (list var))
  
(defun PROPAGATE-M-CORR-VALUES (con corr-values state)
  (let ((orders (mapcar #'(lambda (var val)
			    (qmag-order (variable-qmag var)
					val
					(variable--qspace var)))
			(constraint-variables con)
			corr-values)))
    (multiple-value-bind (plus minus unknown)
	(plus-minus orders (constraint-partials con))
      (when plus
	(let* ((order-if-plus
		 (if (zerop plus)
		     (if (zerop minus) 0 '+)
		     (if (zerop minus) '- nil)))
	       (order
		 (if (eq (second unknown) '+)
		     order-if-plus
		     (opp-sign order-if-plus))))
	  (when order
	    (let ((var (nth (first unknown) (constraint-variables con)))
		  (mag (nth (first unknown) corr-values)))
	      (ecase order
		(+ (assert-lb var mag state))
		(- (assert-ub var mag state))
		(0 (assert-eq var mag state)))
	      var)))))))

(defun qdir-inverse (qdir)
  (ecase qdir
    (std 'std)
    (inc 'dec)
    (dec 'inc)
    (ign 'ign)
    (nil 'nil)))
			;
(defun sign-qdir-match (sign qdir)
  (ecase sign
    (+ (eql qdir 'inc))
    ;;(0 (eql qdir 'std))
    (- (eql qdir 'dec))))

; Infinity check depends on [x]_inf = +,0,- for infinity,finite,neg-infinity, respectively.

(defun check-M-infinite-values (tuple partials)
  (do ((pluses 0)
       (minuses 0)
       (L tuple (cdr L))
       (P partials (cdr P)))
      ((null L) (eql (= pluses 0)
                     (= minuses 0)))
    (cond ((eql (car P) 0))
	  (t (let ((s (sign-wrt-infinity (qval-qmag (car L)))))
	       (cond ((eql s 0))
		     ((eql s (car P)) (incf pluses))
		     (t (incf minuses))))))))

(defun sign-wrt-infinity (qmag)
  (cond ((listp qmag) 0)
	((eql qmag *inf-lmark*) '+)
	((eql qmag *minf-lmark*) '-)
	(t 0)))

; Check wrt corresponding values computes a sign tuple wrt each cval tuple.
;   This can be speeded up by replacing the (M-check (mapcar ... )) with a DO.

(defun check-M-cvals (tuple constraint partials)
  (let ((qspaces (mapcar #'variable--qspace (constraint-variables constraint)))
	(cv-tuples (constraint--cvals constraint)))
    (dolist (cv-tuple cv-tuples t)
      (or (M-check (mapcar #'(lambda (qval cval qspace)
			       (qmag-order (qmag qval) cval qspace))
			   tuple
			   cv-tuple
			   qspaces)
		   partials)
	  (return-from check-M-cvals nil)))))

(defun M-check (tuple partials)
  (do ((pluses 0)
       (minuses 0)
       (L tuple (cdr L))
       (P partials (cdr P)))
      ((null P) (eql (= pluses 0)
                     (= minuses 0)))
    (cond ((eql (car P) 0))
	  ((eql (car L) 0))
          ((eql (car L) (car P)) (incf pluses))
          (t (incf minuses)))))

