;;; -*- Mode: LISP; Syntax: Common-lisp; Package: QSIM; Base: 10 -*-
;;;  $Id: hod-simplify.lisp,v 1.1 1991/03/26 21:37:56 clancy Exp $

;;; Author: Adam Farquhar, October 1989.
;;;
;;; The HOD derivation requires some algebraic transformation and
;;; simplification.  This is implemented in this file through the
;;; functions QA-SIMPLIFY and QA-TRANSFORM.  It is achieved by applying
;;; a set of simplification or transformation rules to an expression.
;;; 
;;; I use qrules to translate a list of rules into a compiled lisp
;;; function which allows for fast pattern matching with minimal
;;; consing.  These rulesets are optimal for this approach as they are
;;; fairly small and applied often.  The rule application functions are
;;; defined by the macro DEF-HOD-RULES, and are created at compile time.
;;; They may be redefined after changing the rule lists by recompiling
;;; the form (def-hod-rules).
;;;
;;;======================================================================

;; (setq  *trace-successful-rules* t)
;; (setq  *trace-successful-rules* nil)

;;; *transforms* is a list of transformations of fullexp.
(defun QA-TRANSFORM (fullexp part)
  (declare (special *transforms*))
  (apply-tr-rules part fullexp)
  (if (listp part)
      (dolist (piece part)
	(qa-transform fullexp piece)))
  (length *transforms*))

(defun QA-SIMPLIFY (exp)
  (when (listp exp)
    ;; Try to simplify the whole expression.
    (do ((result (apply-identity-rule exp)
		 (apply-identity-rule exp)))
	((eq result 'failed))
      (setq exp result))
    ;; Now try to simplify its parts
    (when (listp exp)
      (setq exp (mapcar #'qa-simplify exp))
      ;; and do the whole thing again. I think that this should
      ;; probably be recursive.
      (do ((result (apply-identity-rule exp)
		   (apply-identity-rule exp)))
	  ((eq result 'failed))
	(setq exp result))))
  exp)

;;;======================================================================
; These are the transformation rules for deriving Higher-Order Derivative constraints
; (sometimes called Curvature-at-Steady Constraints).
;   - The first clause in the rule is matched against the sd2 expression.
;   - Additional clauses before "->" are matched against QDE constraints, after substitutions.
;   - The clause after the "->" has bindings substituted, and is returned.

; => extend to make these state-dependent, and handle S+/S-.

(defun parse-rule (rule)
  (declare (values (head condition result)))
  (values (car rule)
	  (if (eql (second rule) '->)
	      nil
	      (second rule))
	  (car (last rule))))

(defparameter *transformation-rules*
	      '(((sd2 ?x) (M+ ?x ?y) -> (sd2 ?y))
		((sd2 ?y) (M+ ?x ?y) -> (sd2 ?x))
		;
		((sd2 ?x) (M- ?x ?y) -> (- 0 (sd2 ?y)))
		((sd2 ?y) (M- ?x ?y) -> (- 0 (sd2 ?x)))
		;
		((sd2 ?z) (add ?x ?y ?z) -> (+ (sd2 ?x) (sd2 ?y)))
		((sd2 ?x) (add ?x ?y ?z) -> (- (sd2 ?z) (sd2 ?y)))
		((sd2 ?y) (add ?x ?y ?z) -> (- (sd2 ?z) (sd2 ?x)))
		;
		((sd2 ?z) (mult ?x ?y ?z) -> (+ (* ?y (sd2 ?x))
						(+ (* ?x (sd2 ?y))
						   (* 2 (* (sd1 ?x) (sd1 ?y))))))
		((sd2 ?x) (mult ?x ?y ?z) -> (- (/ (sd2 ?z) ?y)
						(- (* 2 (* (sd1 ?z)
							   (/ (sd1 ?y) (^ ?y 2))))
						   (- (* 2 (* ?z (/ (^ (sd1 ?y) 2)
								    (^ ?y 3))))
						      (* ?z (/ (sd2 ?z) (^ ?y 2)))))))
		((sd2 ?y) (mult ?x ?y ?z) -> (- (/ (sd2 ?z) ?x)
						(- (* 2 (* (sd1 ?z)
							   (/ (sd1 ?x) (^ ?x 2))))
						   (- (* 2 (* ?z (/ (^ (sd1 ?x) 2)
								    (^ ?x 3))))
						      (* ?z (/ (sd2 ?z) (^ ?x 2)))))))
		;
		((sd2 ?x) (minus ?x ?y) -> (- 0 (sd2 ?y)))
		((sd2 ?y) (minus ?x ?y) -> (- 0 (sd2 ?x)))
		;
		((sd2 ?x) (d/dt ?x ?y) -> (sd1 ?y))
		))

(defparameter *identity-rules*
	      '(((+ 0 ?x) -> ?x)		; additive and multiplicative identities
		((+ ?x 0) -> ?x)
		((- ?x 0) -> ?x)
		((- ?x ?x) -> 0)
		((* 0 ?x) -> 0)
		((* ?x 0) -> 0)
		((/ 0 ?x) -> 0)
		((* 1 ?x) -> ?x)
		((* ?x 1) -> ?x)
		((/ ?x 1) -> ?x)
		((^ ?x 0) -> 1)
		((^ 0 ?x) -> 0)
		((^ ?x 1) -> ?x)

		((- 0 (- 0 ?x)) -> ?x)		; strict simplifications
		((- 0 (- ?x ?y)) -> (- ?y ?x))
		((- 0 (+ (- 0 ?x) ?y)) -> (- ?x ?y))
		((- ?x (- 0 ?y)) -> (+ ?x ?y))
		((- ?x (- ?x ?y)) -> ?y)
		((- ?x (+ ?x ?y)) -> (- 0 ?y))
		((- ?x (+ ?y ?x)) -> (- 0 ?y))
		((- (+ ?x ?y) ?x) -> ?y)
		((- (+ ?x ?y) ?y) -> ?x)
		((+ ?x (- ?y ?x)) -> ?y)
		((+ (- ?x ?y) ?y) -> ?x)
		((- (- ?x ?y) ?x) -> (- 0 ?y))

		((+ ?x ?x) -> (* 2 ?x))		; preferable form
		((+ (+ ?x ?y) ?z) -> (+ ?x (+ ?y ?z)))
		((- (- ?x ?y) ?z) -> (- ?x (+ ?y ?z)))
		((* (* ?x ?y) ?z) -> (* ?x (* ?y ?z)))
		;
		((sd2 ?x) (constant ?x) -> 0)
		((sd2 ?x) (independent ?x) -> 0)	; final simplifications
		((sd1 ?x) (chatter ?x) -> 0)
		))



;;; When using the qrules function translate-net, we need to provide the
;;; code to be executed when a rule's head successfully matches a datum.
;;; This is the FINISH-CODE.  All of our rules have a common part -- the
;;; condition must be checked, perhaps resulting in additional variable
;;; bindings, and the bindings must be substituted into a result.  Then
;;; we either return the result for the identity-rules, or we save it
;;; for the transformation rules.  

(defun identity-finish-code (link pattern-vars)
  (finish-code '(return-from RULES result)
	       link pattern-vars))

(defun transform-finish-code (link pattern-vars)
  (finish-code '(pushnew (qa-simplify
			   (subst (qa-simplify result)
				  datum
				  fullexp))
			 *transforms* :test #'equalp)
	       link pattern-vars))


;; FINISH is a piece of code to be executed at the very finish of the
;; finish.  
;; We do two optimizations here.
;;  1. we only build up binding lists for the variables which are USED.
;;  2. we avoid calls to substitute-vars unless they are really
;;  necessary.
;;
(defun finish-code (finish link pattern-vars)
  ;; This code is meant to be executed in an environment alist and all
  ;; of the patter-vars are bound.
  `(progn
     ,@(mapcar
	 #'(lambda (rule)
	     (multiple-value-bind (ignore condition result)
		 (parse-rule rule)
	       (let ((condition-code
		       (when condition
			 `((setq alist
				 (check-condition datum
						  ',condition
						  ,(binding-list
						     (intersection
						       pattern-vars
						       (union (expression-vars result)
							      (expression-vars condition)))))))))
		     (substitute-code
		       (cond ((variable? result)
			      (cond ((member result pattern-vars)
				     `(,result))
				    (condition
				     `((cdr (assoc ',result alist))))
				    (T (error "Variable ~a not bound"))))
			     ((atom result) `(',result))
			     ((expression-vars result)
			      `((substitute-vars ',result
						 ,(if condition
						      'alist
						      (binding-list (expression-vars result))))))
							    
			     (T `(',result)))))
				    
		 `(progn ,@condition-code
			 ;;(format t "~&Checked ~a: ~a ~%" ',rule alist )
			 (unless ,(if condition
				      '(eq alist 'failed)
				      nil)
			   (let ((result ,@substitute-code))
			     (trace-successful-rule
			       ',rule datum result)
			     ,FINISH))))))
	 (link-contents link))))

(defun substitute-vars (x alist)
  (if (and (consp x) (consp alist))
      (sublis alist x)
      (or (cdr (assoc x alist))
	  x)))

;;; As the dnet building tends to reverse the order of the rules on the
;;; lists, we reverse them first.  We also rename the vars in the
;;; transformation rules. It was easier than rewriting them.
(defmacro DEF-HOD-RULES ()
  (let ((id-rules (reverse *identity-rules*))
	(tr-rules (nreverse (rename-vars-in-rules *transformation-rules*))))
    `(progn
       (defun apply-identity-rule (DATUM)
	 (block RULES
	   (let (alist)
	     (if (null ,(translate-net  (dnet id-rules)
					'DATUM  'identity-finish-code))
		 'failed))))
       (defun apply-tr-rules (DATUM fullexp)
	 (declare (special *transforms*))
	 (let (alist)
	   ,(translate-net (dnet tr-rules)
			   'DATUM 'transform-finish-code :conjunction 'progn))))))

;;; >>>>>>>>>>>>>>>>>>>>>>>>>><<<<<<<<<<<<<<<<<<<<<<<<<<<<<
;;; >>>>>>> DEFINE THE FUNCTION APPLY-IDENTITY-RULE <<<<<<<

			    (DEF-HOD-RULES)

;;; >>>>>>>>>>>>>>>>>>>>>>>>>><<<<<<<<<<<<<<<<<<<<<<<<<<<<<
;;;----------------------------------------------------------------------
;;; QSIM RULE CONDITION

;; Return new alist or 'failed
;;
(defun CHECK-CONDITION (datum condition alist)
  (declare (special *qde* *constraints* *independents* *chatter-var*))
  (declare (ignore datum))
  (ecase (car condition)
    ((NIL) alist)
    ((INDEPENDENT)
     (if  (member (cdr (assoc (second condition) alist))
		  *independents*)
	  alist
	  'failed))
    ((CHATTER)
     (if  (eql (cdr (assoc (second condition) alist))
		  *chatter-var*)
	  alist
	  'failed))
    ((M+ M- ADD MULT MINUS D/DT CONSTANT)
     (dolist (con (qde-constraints *qde*) 'FAILED)
       (if (eql (car condition)
		(contype-name (constraint-type con)))
	   (let ((result (match-data (rest condition)
				(constraint-variables con)
				alist)))
	     (if (not (eql result 'FAILED))
		 (RETURN result))))))))

;; pat may contatin pattern variables,
;; dat is pure data.  
;; Return: new alist or 'failed
;;
(defun match-data (pat data alist)
  (cond ((eql alist 'failed) alist)
	((null pat)
	 (if (null data)  alist 'failed))
	((variable? pat)
	 (let ((binding (assoc pat alist)))
	   (if (null binding)
	       (acons pat data alist)
	       (if (eql (cdr binding) data)
		   alist
		   'failed))))
	((atom pat)
	 (if (eql pat data) alist 'failed))
	((atom data) 'failed)
	(T (match-data (cdr pat) (cdr data)
		       (match-data (car pat) (car data) alist)))))

;;; A test function, get some timing results.
;;;
(defun ^ (x y)
  (expt x y))
(defun test ()
  (declare (special t1 t2 t3 t3 t4 t5))
  (setq t1 '(+ 0 (+ 0 (+ (* 2 3) (^ (* 88 23) 0)))))
  (setq t2 '(+ (- 0 (- 4 (- 4 (^ 1 1)))) 1))
  (setq t3 '(+
	      (+ (^
		   (- 9 (+ 9 0))
		   (* 1 (* 1 (* 1 90))))
		 (- 0 1))
	      (+ 0 (+ (* 2 3) (* (- 4 (- 4 (^ 99 1))) (^ (* 88 23) 0) )))))
  (setq t4 `(+ ,t3 (* ,t1 ,t2)))
  (setq t5 `(+ (- 0 (+ ,t4 ,t1)) (+ ,t1 ,t4)))
  (dolist (exp (list t1 t2 t3 t4 t5))
    (let ((e1  (time (qa-simplify exp)))
	  (v   (eval exp)))
      (if (= (eval e1) v)
	  (format t "~&SIMPLIFY CORRECT:~%   ~a = ~a~%" v e1)
	(format t "~%****SIMPLIFICATION WRONG~@
                       ****Actual value = ~a, ~@
                       ****Simpified value = ~a,~@
                       EXPRESSION: ~a,
                       SIMPLIFIED TO: ~a~2%")))))
