;;; -*- Mode:Common-Lisp; Package:QSIM; Base:10 -*-

(in-package :qsim)

;;; If qmag is an atom it will return the qmag, if it is an
;;; interval it will return the lower-lmark of the range.

(defmacro lower-lmark (qmag)
  `(if (atom ,qmag)
       ,qmag
       (car ,qmag)))

;;; If qmag is an atom it will return the qmag, if it is an
;;; interval it will return the upper-lmark of the range.

(defmacro upper-lmark (qmag)
  `(if (atom ,qmag)
       ,qmag
       (cadr ,qmag)))

(defparameter *trace-envision-guide* nil)
(defparameter *envision-guide-filter* nil)
(defparameter *envision-menu*
	      (list (list 'S  "Specify New Subgraph")
		    (list 'B  "Perform Envisionment Guided Simulation")
		    (list 'SL "Enter new state limit.")
		    (list 'DB "Display one of the Envision Guided Simulations")
		    (list 'DE "Display Envisionment")))

(defother sim envision-guided-sims 
  :description 
  "A list of the SIMs for behavior tree simulations which were 
   executed using this envisionment as a guide.")   
(defother sim envision-paths
  :description "Used on the SIM of an envisionment guided simulation.  It is
  a list of the paths within the envisionment which are used as the path.")
(defother sim envision-roots
  :description "Used on the SIM of an envisioment guided simulation.  It is a
  list of the roots of the various paths through the envisionment graph
  which has been selected.")

(defother state envision-abstraction
  :description "Points to the state in the envisionment which is an abstraction of this state")
(defother state envision-path
  :description "identifies if this state is on the envisionment path.  It contains a list of
  the predecessors of this state along the selected path.")

(pushnew 'envision-guide-filter *global-state-filters*)



;;;
;;;  ENVISION GUIDE
;;; 
;;;  Input Parameters:  ENV-SIM  a sim from which the envisionment
;;;                              should be executed.  It should include
;;;                              an intial state.
;;;
;;;  This function provides a nice user interface to perform
;;;  envisionment guided simulations.  It will receive a sim and will
;;;  perform an envisionment on for that sim.  It modifies ENV-SIM to
;;;  contain the appropriate speical variable settings for an
;;;  envisionment.
;;;
;;;  It then provides the user with a menu to specify subgraphs within
;;;  the envisionment, perform and envisionment guided simulation and to
;;;  display the results of some of these simulations.  Multiple
;;;  envisionment guided simulations can be performed.  For each one
;;;  performed, a new sim is created along with an initial state.  The
;;;  slot ENVISIONMENT-GUIDED-SIMS on ENV-SIM will contain a list of
;;;  these sims.  The new sim created for the behavior tree simulation
;;;  will keep track of the subgraphs selected for that simulation and
;;;  the roots of that subgraph.

(defun envision-guide (sim-or-state)
  (let* ((options *envision-menu*)
	 (env-sim (if (sim-p sim-or-state)
		      sim-or-state
		      (state-sim sim-or-state)))
	 (init-state (if (state-p sim-or-state)
			 sim-or-state
			 (sim-state sim-or-state)))
	 (state-limit (sim-state-limit env-sim))
	 subgraph)
    (unless (> (sim-state-count env-sim) 1)
      (let ((*simulation-type* :depth-first))
	(format *Qsim-Report* "~&PERFORMING THE ENVISIONMENT ")
	(qsim (sim-state env-sim))
	(qsim-display env-sim)))
    (loop for cmd = (general-menu options :skip-lines t)
	  when (null cmd) return nil
	  do (let ((cmd-str (car cmd)))
	       (cond ((equal cmd-str 'S)
		      (setf subgraph (specify-subgraph init-state)))
		     ((equal cmd-str 'A)
		      (setf subgraph (append subgraph
					     (specify-subgraph init-state))))
		     ((equal cmd-str 'SL)
		      (setf state-limit (cond ((get-state-limit state-limit))
					      (t state-limit))))
		     ((equal cmd-str 'B)
		      (if (null subgraph)
			  (format *Qsim-Report* "~&No subgraph specified.  Cannot perform ~
                                         envisionment guided simulation")
			  (let ((new-sim (make-beh-sim-from-env-sim env-sim))
				(env-roots (remove-duplicates (mapcar #'car subgraph) :test #'equal))
				(*envision-guide-filter* t)
				(*simulation-type* :breadth-first))
			    (setf (sim-envision-paths new-sim)
				  subgraph
				  (sim-envision-roots new-sim)
				  env-roots)
			    (setf (sim-state-limit new-sim) state-limit)
			    (clear-env-path (sim-state env-sim))
			    (mark-env-paths subgraph)
			    (pushnew new-sim
				     (sim-envision-guided-sims env-sim))
			    (setf subgraph nil)
			    (qsim (sim-state new-sim))
			    (qsim-display new-sim))))
		     ((equal cmd-str 'DE)
		      (with-envisioning (qsim-display env-sim)))
		     ((equal cmd-str 'DB)
		      (clear-env-path (sim-state env-sim))
		      (mark-env-paths subgraph)
		      (let ((selected-sim (select-env-guided-sim
					    (sim-envision-guided-sims env-sim))))
			(when selected-sim (qsim-display
					     (second selected-sim))))))))
    env-sim))

(defun get-state-limit (current-limit)
  (format *Qsim-Report* "~&Current state limit it ~a. Enter new limit (Nil to stay the same): "
	  current-limit)
  (loop for cmd = (read)
	when (null cmd) return it
	when (integerp cmd) return cmd
	do (format *Qsim-Report* "~&Invalid Entry.  Try Again. ")))


(defun select-env-guided-sim (sims)
  "This function will allow the user to select from a list of SIMs"
  (let* ((count 0)
	 (options (mapcar #'(lambda (sim)
			      (incf count)
			      (list count sim))
			  (reverse sims))))
    (general-menu options :skip-lines t)))


(defun make-beh-sim-from-env-sim (sim)
  "This function creates a copy of SIM.  It assumes that SIM is an envisionment
SIM and it will create a SIM with the appropriate settings for a behavior tree 
simulation.  It creates a new initial-state and sets various flags apprpriately."
  (let ((new-sim (make-sim-from-sim sim)))
    (setf (sim-state new-sim) (state-copy (sim-state sim))
	  (sim-qde new-sim) (sim-qde sim)
	  (sim-display-block new-sim) (sim-display-block sim)
	  (sim-enable-landmark-creation new-sim) t
	  (sim-cross-edge-envisionment new-sim) nil
	  (sim-states new-sim) nil
	  (sim-envision-guided-sims new-sim) nil)
    (setf (state-sim (sim-state new-sim))
	  new-sim)
    new-sim))




;;;  SPECIFY-SUBGRAPH
;;;
;;;  This function provides the user with a number of different ways
;;;  to specify a subgraph within the envisionment.  All of these
;;;  mechanism require that the subgraph be Cyclic or end in a
;;;  terminal state.

(defun specify-subgraph (&optional (initial-state *initial-state*))
  (let* (;(cmd (get-subgraph-spec-mech))
	 (options (list (list 'F "Specify Initial and Final States" #'specify-init-and-final-states)
			(list 'B "Specify Behaviors" #'specify-and-complete-behaviors)))
	 (selection (general-menu options)))
    (when selection
      (funcall (third selection) initial-state))))

(defun specify-and-complete-behaviors (&optional (initial-state *initial-state*))
  "Prompts the user for a set of behaviors numbers.  It will complete these behaviors from 
the envisionment and return the behaviors."
  (let* ((behs (get-behaviors initial-state))
	 (btotal (length behs))
	 (bnums (make-list-of-bnums (specify-behs-by-bnum btotal)))
	 (behs-to-allow (mapcar #'(lambda (bnum)
				    (nth (1- bnum) behs))
				bnums)))
    (complete-behs behs-to-allow)))


(defun specify-init-and-final-states (&optional (initial-state *initial-state*))
  (declare (ignore initial-state))
  (format *Qsim-Report* "~%This cpability is not implemented yet."))


(defun mark-env-paths (behs)
  "This function will mark the states along the envisionment path.  This is
done by maintaining a list of predecessor states for each marked state.  This
means that this state is along the path extending from the predecessor
states in the list."
  (mapcar #'(lambda (beh)
	      (let ((pred (car beh)))
		(mapcar #'(lambda (state)
			    (pushnew pred (state-envision-path state))
			    (setf pred state))
			(cdr beh))))
	  behs))


(defun get-marked-paths (states)
  "This function will receive a list of initial states for the portion of the
envisionment graph which has been marked.  It will return a set of the behaviors
which have been marked which extend from these initial-states."
  (let ((*traverse-xedges* t))
    (if (listp states)
	(mapcan #'get-marked-paths-rec states)
	(get-marked-paths-rec states))))


(defun get-marked-paths-rec (state)
  (when state
    (let ((marked-succ (filter-for-env-path (successor-states state) state)))
      (if marked-succ
	  (mapcar #'(lambda (beh)
		      (cons state beh))
		  (mapcan #'get-marked-paths-rec marked-succ))
	  (list (list state))))))


		    
(defun complete-behs (behs)
  "This function will receive a list of behaviors.  It will complete each 
behavior by traversing any cross edges at the end of each behavior."
  (let ((*traverse-xedges* t)
	(*develop-graph* t))
    (mapcan #'(lambda (beh)
		  (mapcar #'(lambda (completion)
			      (append beh (cdr completion)))
			  (get-behaviors (car (last beh))
					 :visited-nodes (butlast beh))))
	    behs)))


(defun clear-env-path (&optional (state *initial-state*))
  "This function will set envision-path to nil for the tree
extending from *initial-state*."
  (when state
    (setf (state-envision-path state) nil)
    (mapcan #'clear-env-path (successor-states state))))


;;;
;;;  ENVISION-GUIDE-FILTER
;;;
;;;  This filter will eliminate states which are not on the selected
;;;  path in the envisionment graph.  If this filter is being used, then
;;;  links will be maintained between states in the behavior tree and
;;;  their abstract representations in the envisionment graph.  
;;;  
;;;  A link is maintained between each state and its abstraction in the
;;;  envisionment graph.  This link will be traveresed to find the
;;;  matching state in the envisionment graph of the predecessor of
;;;  STATE.  If this state is a time-point, then STATE must match
;;;  against one of its successors which are along the selected path.
;;;  If this state is a time-interval, then STATE must either match
;;;  against it or against one of its successors which are on the
;;;  selected path.   If STATE is the first state in a simulation,
;;;  then it must match agaisnt one of the roots for the paths selected
;;; which are stored on the SIM. 
;;;  

(defun envision-guide-filter (state)
  (if *envision-guide-filter*
      (let* ((sim (state-sim state))
	     (envision-roots (sim-envision-roots sim))
	     (pred (real-state-predecessor state))
	     env-state)
	(cond ((null pred)     ; if state is a root of the tree
	       (setf env-state (car (member state envision-roots :test #'abstraction?))))
	      (t (let* ((env-pred (state-envision-abstraction pred))
			(*traverse-xedges* t)
			(env-pred-successors (filter-for-env-path (successor-states env-pred)
						       env-pred)))
		   (if (and (i-state-p env-pred)
			    (abstraction? state env-pred))
		       (setf env-state env-pred)
		       (setf  env-state (car (member state env-pred-successors :test
						     #'abstraction?)))))))
	(cond (env-state
	       (setf (state-envision-abstraction state) env-state)
	       (when *trace-envision-guide*
		 (format *Qsim-Report* "~&State ~a matches against ~a on the envision path." 
			 state env-state))
	       state)
	      (t (pushnew 'inconsistent (state-status state))
		 (when *trace-envision-guide*
		   (format *Qsim-Report* "~&State ~a is not on the envision path." state))
		 nil)))
      state))

(defun real-state-predecessor (state)
  "Returns predecessor of this state"
  (let ((car-just (car (state-justification state)))
	(pred-just (cadr (state-justification state))))
    (cond ((member car-just '(initialized-with copy-of))
	   nil)
	  ((equal car-just 'one-of-several-completions-of)
	   (when (equal 'transition-from 
			(car (state-justification pred-just)))
	     (second (state-justification pred-just))))
	  (t pred-just))))


(defun filter-for-env-path (states env-pred)
  "This function will remove those states in the STATES list which have
not been selected along the envisionment path with ENV-PRED as their
predecessor."
  (remove-if-not #'(lambda (state)
		     (member env-pred (state-envision-path state)))
		 states))


;;;
;;;  ABSTRACTION?
;;;
;;;  THis function will determine if a state in the envisionment graph
;;;  is an abstraction of a state in the envisionment guided behavior
;;;  tree simulation.
;;;

(defun abstraction? (other-state abs-state)
  "This function determines if ABS-STATE is an abstraction of
OTHER_STATE.  ABS-STATE should be a state from the envisionment graph
while OTHER_STATE will be a state from the envisionment guided
simulation."
  (when *trace-envision-guide*
    (format t "~&   ABS-STATE: ~a  OTHER-STATE: ~a" abs-state other-state))
  (cond ((not (eq (state-qde abs-state) (state-qde other-state)))
	 (when *trace-envision-guide*
	   (format t "  FAILED.  DIfferent QDEs.")))
        ((and (p-state-p abs-state)		; if ABS-STATE is a time point then
	      (i-state-p other-state))          ; OTHER-STATE must be one too.
	 (when *trace-envision-guide*
	   (format t "  FAILED. ABS-STATE is a time point and OTHER-STATE is an interval.")))
	((not (every #'(lambda (abs-qvalue other-state-qvalue)
			 (or (qval-equal (cdr abs-qvalue) (cdr other-state-qvalue))
			     (let* ((abs-qmag (qmag (cdr abs-qvalue)))
				    (other-state-qmag (value-in-orig (qmag (cdr other-state-qvalue))))
				    (qmag-equal? (qmag-equal abs-qmag other-state-qmag)))
			       (if qmag-equal?
				   qmag-equal?
				   (when *trace-envision-guide*
				     (format t "    QMAGS not EQUAL: ~a  ~a" abs-qmag other-state-qmag)))))) 
		     (remove-time (state-qvalues abs-state))
		     (remove-time (state-qvalues other-state))))
	 (when *trace-envision-guide*
	   (format t "   FAILED.")))
	(t (if *trace-envision-guide*
	       (not (format t "     SUCCEEDED."))
	       t))))


;;; This function will receive a qmag and it will return the value that
;;; this qmag would have in the orignal qspace.

(defun value-in-orig (qmag)
  (if (point-p qmag)
      (let ((where-defined (lmark-where-defined qmag)))
	(if where-defined
	    (let ((lower-bound (value-in-orig (lower-lmark where-defined)))
		  (upper-bound (value-in-orig (upper-lmark where-defined))))
	      (list (lower-lmark lower-bound)
		    (upper-lmark upper-bound)))
	    qmag))
      (let ((lower-where-defined (lmark-where-defined (lower-lmark qmag)))
	    (upper-where-defined (lmark-where-defined (upper-lmark qmag))))
	(list (if lower-where-defined
		  (lower-lmark (value-in-orig (lower-lmark lower-where-defined)))
		  (lower-lmark qmag))
	      (if upper-where-defined 
		  (upper-lmark (value-in-orig (upper-lmark upper-where-defined)))
		  (upper-lmark qmag))))))


;;;  FROM NEW-QUAL-EQUIV



(defun qmag-equal (qmag1 qmag2)
  (cond ((and (point-p qmag1)
	      (point-p qmag2))
	  (robust-lmark-equal qmag1 qmag2))
	 ((and (interval-p qmag1)
	       (interval-p qmag2))
	  (and (robust-lmark-equal (car qmag1)
			    (car qmag2))
	       (robust-lmark-equal (cadr qmag1)
			    (cadr qmag2))))
	 (t nil)))

;;;  FROM MOD-OCC-BRANCH code

(defun state-copy (state)
 (let* ((nstate 
	   (make-state :qde           (state-qde state)
		       :qvalues       (copy-qvalues (state-qvalues state))
		       :qspaces       (copy-alist (state-qspaces state))
		       :cvalues       (copy-alist (state-cvalues state))
		       :other         (copy-tree (state-other state))
		       :text          (format nil "Copy of ~a" state)
		       :justification `(initialized-with ,(state-qvalues state))
		       :name          (genname 'S)
		       )))
    (setf (state-sim nstate) *current-sim*)     ; added DJC 07/15/91
    ;; The statement to set time (below) is not actually necessary because
    ;; time is included in the qvalues list.
    ;; (setf (state-time nstate) (state-time state))
    (set (state-name nstate) nstate)
    ;(setf (state-abstractions nstate) nil)
    ;(setf (state-abstraction-of nstate) nil)
    nstate))





;;; GENERAL-MENU
;;;
;;;  This is a general function which takes a list of OPTIONS and creates
;;;  a menu for these options.  Each element of the list of options should
;;;  be of the form
;;;
;;;      (<char-string> <text> [<function>])
;;;
;;;  where the <char-string> is the string to be entered to make the
;;;  selection, <text> describes the selection, and <function> is an
;;;  optional argument which may or may not be used by the calling
;;;  function in determining the action to take in response to the
;;;  selection.
 
(defun general-menu (options &key (skip-lines nil))
  (when (member 'Q options :test #'(lambda (cmd option)
						(equal cmd (car option))))
    (error "Q cannot be an option on the menu.  It is used to quit."))
  (loop for cmd = (display-prompt options skip-lines)
	when (or (null cmd)
		 (equal cmd 'Q)) return nil
	when (car (member cmd options :test #'(lambda (cmd option)
						(equal cmd (car option)))))
	return it
	do (format *Qsim-Report* "~&Incorrect Entry.  Try again.")))

(defun display-prompt (options &optional (skip-lines t))
  (format *Qsim-Report* "~&Select Option: (Q to Quit) ")
  (loop for option on options 
	do (format *Qsim-Report* " ~a=~a" (caar option) (second (car option)))
	do (if (cdr option)
	       (if skip-lines
		   (format *Qsim-Report* "~%                           ")
		   (format *Qsim-Report* ","))
	       (format *Qsim-Report* ": ")))
  (clear-input)
  (read))
