;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
(in-package 'QSIM)

; Copyright (c) 1986, 1987, 1988, 1989, 1990, Benjamin Kuipers.

; The qualitative plotter.
;     This tries to be device-independent, with the device-dependent code
;     segregated into (e.g.) plot-on-symbolics-screen.lisp.  
;     However, the coordinates are currently in symbolics screen dots!

; Global variables:
;     controlling the layout of the screen as a whole.

;;; >>>>>>>>>>>vvvvvvvvvvvvvvvvvvvvvvvvvvvvv<<<<<<<<<<< 
;;; >>>>>>>>    LOAD THE FILE ACOMMAND-LOOP   <<<<<<<<<
;;; >>>>>>>>>>^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^<<<<<<<<<<<

;;; An attempt at a slightly more rational command interface for
;;; displaying and examining qsim states and behaviors, etc.
;;;
;;; The structure QD (for qsim display?) packages together most of the
;;; information needed to guide the displaying.  It has slots for the
;;; root of the behavior, its initial states, the list of behaviors, the
;;; number, the length of the longest one, the qde, the current plotting
;;; mode, various types of layouts, etc.
;;;
;;; The global *dis* (display object?) is bound to one of these.  Many
;;; of the commands make use of it.
;;;
;;; I am using the acommand-loop code (see the file acommand-loop).
;;; This is a simple command interpreter which provides matching of
;;; partial input strings to commands.  The command definitions are
;;; somewhat similar to those which the symbolics uses.  The interpreter
;;; code is written in pure CL, and has been tested under a variety of
;;; CLs.
;;;
;;;----------------------------------------------------------------------
;;;
;;; ptb 9-90: I've picked up the ball on this code and will be adding
;;; comments as I go. Hopefully, most of the routines will be commented
;;; in time.
;;;

(use-package :acommand)

(defstruct (qd
	     (:type list))
  root
  initial-states
  all-behaviors
  current
  behavior-count
  longest-behavior-size
  qde
  plot-mode					;one of time-plot, variable-plot, phase-plot
  phase-plot-layout
  time-plot-layout
  variable-plot-layout
  (plot-tree-p t))
  
(defvar *dis* (make-qd)
  "A discriptor for the current display setup.")

(defparameter *qd-commands* nil
  "The command table for qsim display.")

(defun qdisplay-prompt ()
  (fresh-line)
  (if (integerp (qd-current *dis*))
      (format t "~&qdisplay ~a/~a> " (qd-current *dis*) (qd-behavior-count *dis*))
      (format t "~&qdisplay> ")))


;;;----------------------------------------------------------------------
;;; This is the "main" routine for running Tom's commad completion code
;;; for qsim displays. As I understand it, it is meant as a replacemet
;;; of the "qsim-display" function.
;;;
;;;
;;; NOTE:  IF THIS IS USED, CHANGES MUST BE MADE TO HANDLE THE DISPLAY BLOCK
;;;        AS IN THE CURRENT QSIM-DISPLAY.


(defun qdisplay (&optional (initial *initial-state*))
  "A simple command interpreter for displaying and examing qsim states
  and behaviors."
  (unless (consp *dis*) (setq *dis* (make-qd)))	;make real structure!
  (init-display-descriptor (return-a-state initial) *dis*)
  (catch :exit
    (device-interface 
      (command-top-level 'qdisplay-prompt *qd-commands*))))


(defun init-display-descriptor (initial-state dis)
  "Given an initial state, set up the display descriptor.  This includes
   setting up some sort of default layout too."
  (setf (qd-root dis) initial-state)
  (setf (qd-initial-states dis)
	(get-list-of-initial-states initial-state))
  (setf (qd-qde dis) (state-qde (first (qd-initial-states dis))))
  (setf (qd-all-behaviors dis) (delete nil (get-behaviors initial-state)))
  (setf (qd-behavior-count dis) (length (qd-all-behaviors dis)))
  (setf (qd-longest-behavior-size dis)
	(apply #'max (mapcar #'length (qd-all-behaviors dis))))
  (setf (qd-time-plot-layout dis)
	(find-some-layout (qd-all-behaviors dis) *reference-states*))
  (setf (qd-plot-mode dis) 'time-plot)
  (setf (qd-current *dis*) 1)
  (compute-layout  (qd-time-plot-layout dis)))

(defun ensure-valid-descriptor ()
  t)


;;;======================================================================
;;;
;;; COMMANDS
;;;
;;;

(def-com "QContinue Simulation" *qd-commands*
  (format t "~&Current State Limit is ~a.~%" *state-limit*)
  (q-continue :new-state-limit (read-in :prompt "New State Limit" :type 'integer))
  (init-display-descriptor (qd-root *dis*) *dis*))

(def-com "Help" *qd-commands*
  (format t "~&Valid commands are:~%")
  (show-matching-commands () *qd-commands*))

(def-com "?" *qd-commands* com-help)

(def-com "Clear Descriptor" *qd-commands*
  "Kill any pointers we might have to qsim objects.  Good hygene."
  (fill *dis* nil))

(def-com "Exit" *qd-commands*
  (throw :exit (values)))

(def-com "Next" *qd-commands*
  "Plot the next behavior."
  (setf (qd-current *dis*)
	(1+ (mod  (qd-current *dis*) (qd-behavior-count *dis*))))
  (com-plot-behavior))

(def-com "Goto Behavior" *qd-commands*
  "Goto a new behavior and plot it."
  (setf  (qd-current *dis*)
	 (read-integer-between 1 (qd-behavior-count *dis*) "Goto Behavior"))
  (com-plot-behavior))

(def-com "Plot Behavior" *qd-commands*
  "Plot the current behavior under the current mode.  I.e. a time-plot
   or a phase plot."
  (ecase (qd-plot-mode *dis*)
    (time-plot (com-time-plot-behavior))
    (phase-plot (com-phase-plot-behavior))))

(def-com "Show Variables" *qd-commands*
  "Show the variables referenced in the behaviors rooted at the initial state."
  (format t "~{~&    ~a~}~%" (get-reachable-vars (qd-initial-states *dis*))))

(def-com "Show-Initial-State" *qd-commands*
  (format t "Initial State: ~a" *initial-state*))


(def-com "Set-Initial-State" *qd-commands*
  (com-show-initial-state)
  (setq *initial-state*
	(return-a-state (read-with-prompt "Select an initial state"))))

(def-com "Show-State" *qd-commands*
  "Prompt for and display a state."
  (ps (return-a-state (read-with-prompt "Select a state"))))

(def-com "Show-Behaviors" *qd-commands*
  "Print the lists of states in each of the behaviors."
  (loop for i from 1
	for b in (qd-all-behaviors *dis*)
	do
    (format t "~&~3d ~a" i b)))

(def-com "Show-Layout" *qd-commands*
  (format t "~&The current time-plot layout:")
  (pprint (qd-time-plot-layout *dis*))
  (format t "~%The current phase-plot layout:")
  (pprint (first (qd-phase-plot-layout *dis*))))

(def-com "Reset QDE Layout from Time Plot Layout" *qd-commands*
  (setf (qde-layout (qd-qde *dis*)) (qd-time-plot-layout *dis*)))

(def-com "Reset Time Plot Layout from QDE" *qd-commands*
  (setf (qd-time-plot-layout *dis*) (qde-layout (qd-qde *dis*))))

(def-com "Set Time Plot Layout" *qd-commands*
  (setf (qd-time-plot-layout *dis*)
	(get-time-plot-layout (qd-root *dis*))))

(def-com "Time Plot Behavior" *qd-commands*
  "Produce a qualitative plot of the current behavior."
  (ensure-valid-descriptor)
  (unless (qd-time-plot-layout *dis*)
    (com-set-time-plot-layout))
  (let ((layout (qd-time-plot-layout *dis*)))
    (unless (eql 'time-plot (qd-plot-mode *dis*))
      (compute-layout layout))
    (setf (qd-plot-mode *dis*) 'time-plot)
    (time-plot-behavior
      (qd-initial-states *dis*)
      *reference-states*
      (qd-all-behaviors *dis*)
      layout
      nil ;; Purportedly not used: (qd-qspaces *dis*)
      (qd-longest-behavior-size *dis*)
      (qd-behavior-count *dis*)
      (qd-current *dis*)
      (qd-plot-tree-p *dis*))))

(def-com "Set Phase Plot Layout" *qd-commands*
  (setf (qd-phase-plot-layout *dis*)
	(get-phase-plot-layout (qd-root *dis*) t)))

(def-com "Phase Plot Behavior" *qd-commands*
  "Plot variables against each other."
  (ensure-valid-descriptor)
  (unless (qd-phase-plot-layout *dis*)
    (com-set-phase-plot-layout))
  (let ((layout (qd-phase-plot-layout *dis*)))
    (unless (eql 'phase-plot (qd-plot-mode *dis*))
      (compute-rectangular-layout (second layout) (third layout)))
    (setf (qd-plot-mode *dis*) 'phase-plot)
    (phase-plot-behavior
      (qd-initial-states *dis*)
      (qd-all-behaviors *dis*)
      (first layout)
      (third layout)
      (qd-longest-behavior-size *dis*)
      (qd-behavior-count *dis*)
      (qd-current *dis*))))

(def-com "Toggle Time Points" *qd-commands*
  (setq *time-points-only* (not *time-points-only*))
  (format t "~&Time intervals are now ~a plotted in phase plots."
	  (if *time-points-only* "not" "")))

(def-com "Plot Tree" *qd-commands*
  "Plot the behavior tree."
  (ensure-valid-descriptor)
  (qplot-behavior-tree
    (qd-initial-states *dis*)
    (qd-longest-behavior-size *dis*)
    (qd-behavior-count *dis*)
    0))
    
(def-com "Plot Variables Across Behaviors" *qd-commands*
  "Plot the variable values across behaviors."
  (var-slice-viewer))

(def-com "Plot Phases Across Behaviors" *qd-commands*
  "Plot a phase portrait across a set of behaviors."
  (phase-slice-viewer))


(def-com "Display Initial States" *qd-commands*
  (SHOW-QVAL-DIFFS (qd-initial-states *dis*) T T 'EQUAL-ORIGINAL-LMARKS))

(def-com "Display Final States" *qd-commands*
  (SHOW-QVAL-DIFFS (qd-root *dis*) T T 'EQUAL-ORIGINAL-LMARKS))

(def-com "Display Representative Final States" *qd-commands*
  (SHOW-QVAL-DIFFS (representative-final-states (qd-root *dis*)) T T 'EQUAL-ORIGINAL-LMARKS))

(defun display-constraints (constraints qde)
  (loop for constraint in constraints
	for cvalue = (cdr (assoc constraint (qde-cvalues qde)))
	do (format t "~&  ~a" constraint)
	   (when cvalue
	     (format t "    ~a" cvalue)))
  (terpri))

(def-com "Display Constraints" *qd-commands*
  (display-constraints (qde-constraints (qd-qde *dis*)) (qd-qde *dis*)))

(def-com "Display Constraints On Variable" *qd-commands*
  "Display the constraints on a particular variable."
  (display-constraints
    (cdr
      (assoc
	(qde-name (qd-qde *dis*))
	(variable-constraints
	  (find (read-in :prompt "A Variable"
			 :type `(member ,@(mapcar #'variable-name
						  (qde-variables (qd-qde *dis*)))))
		(qde-variables (qd-qde *dis*)) :key #'variable-name))))
    (qd-qde *dis*)))

(def-com "Display Quantity Spaces" *qd-commands*
  "Display the intitial quantity spaces."
  (format t "~%~{~&  ~a~}~%" (qde-qspaces (qd-qde *dis*))))

(def-com "Trim Layout" *qd-commands*
  (let ((redundant (redundant-variables (qd-all-behaviors *dis*) (qd-qde *dis*))))
    (when  redundant
      ;; flatten the layout
      ;; remove the redundant vars
      ;; put in rowxcol form.
      (let ((layout (apply #'append (qd-time-plot-layout *dis*))))
	(setq layout (delete 'time layout))
	(setq layout (delete-if #'(lambda (v)
				    (member v redundant :key #'car))
				layout))
	(setf (qd-time-plot-layout *dis*)
	      (loop for row from 0 upto (1- (length (qd-time-plot-layout *dis*)))
		    collect
		      (loop for col from 0 upto (1- (apply #'max
							   (mapcar #'length
								   (qd-time-plot-layout *dis*))))
			    collect (pop layout)))))
      (compute-layout (qd-time-plot-layout *dis*)))))

(defun redundant-variables (behaviors qde &aux commons)
  ;; This is a DUMB implementation:
  (setq commons (user::state-known-values (first (first behaviors))))
  (dolist (b behaviors)
    (dolist (s (cdr b))
      (setq commons (intersection (user::state-known-values s)
				  commons
				  :test #'equal))
      (if (null commons)
	  (return-from redundant-variables (qde-layout qde)))))
  commons)




(defun phase-plot-behavior (initials behaviors pplanes cols bmax btotal bnum)
  (qplot-new-behavior)
  (qplot-label (nth (1- bnum) behaviors) bnum btotal)
  (when (<= btotal 20)
    (plot-state-tree initials bmax btotal (- xscreen rmargin xbsize) ybloc xbsize ybsize bnum))
  (do ((ypos tmargin (+ ypos *ysize* ysep))
       (plist pplanes (nthcdr cols plist)))
      ((null plist))
    (do* ((ct cols (1- ct))
	  (ps plist (cdr ps))
	  (parms (car ps) (car ps))	    
	  (xpos lmargin (+ xpos *xsize* xsep)))
	 ((zerop ct))
      (when (car ps)
	(let* ((label (format nil "~a vs ~a" (car parms) (second parms))))
	  (plot-one-phase-diagram
	    parms (nth (1- bnum) behaviors)
	    xpos ypos *xsize* *ysize* label)))))
  (qplot-end-display))



(defun get-time-plot-layout (initial)
  (do ((layout (prompt-for-layout) (prompt-for-layout)))
      ((layout-okay-p layout initial) layout)))

(defun prompt-for-layout ()
  (format *qsim-report*
	  "~&Enter new layout as list of lists of variables.  ~
           Each sublist will be a row in the display.  ~
           Use nils for spacing.~%Layout: ")
  (read))

(defun layout-okay-p (layout state)
  (cond ((not (listp layout))
	 (format *qsim-report* "~&Input layout ~a is not a list." layout) nil)
	(t
	 (let ((var-list (get-reachable-vars state)))
	   (do ((L layout (cdr L)))
	       ((null L) t)
	     (when (not (listp (car L)))
	       (format *qsim-report* "~&In layout, ~a is not a proper sublist." (car L))
	       (return nil))
	     (when (do ((S (car L) (cdr S)))
		       ((null S) nil)
		     (cond ((null (car S)))
			   ((member (car S) var-list))
			   (t (format *qsim-report* "~&~a is an invalid variable.  ~
                                           The valid variables are: ~a" (car S) var-list)
			      (return t))))
	       (return nil)))))))

(defun get-phase-plot-layout (initial prompt-only-p)
  (let* ((pplanes (or (and (null prompt-only-p) (get-phase-planes-from-qde initial))
		      (get-phase-planes-from-prompt initial)))
	 (n (length pplanes))
	 rows
	 cols)
    (cond ((< n 9) (setq rows 2 cols 4))
	  ((< n 19) (setq rows 3 cols 6))
	  (t (setq rows 4 cols 8)))
    (list pplanes rows cols)))

(defun get-phase-planes-from-qde (state)
  (let ((pplanes (cdr (assoc 'phase-planes (qde-other (state-qde state))))))
    (if (pplanes-okay-p pplanes state)
	pplanes
	(get-phase-planes-from-prompt state))))

(defun get-phase-planes-from-prompt (state)
  (do* ((pplanes (prompt-for-pplanes) (prompt-for-pplanes))
	(check (pplanes-okay-p pplanes state)
	       (pplanes-okay-p pplanes state)))
       (check
	 (when (y-or-n-p "Do you want to use this as the QDE phase-planes clause? ")
	   (if (assoc 'phase-planes (qde-other (state-qde state)))
	       (rplacd (assoc 'phase-planes (qde-other (state-qde state)))
		       pplanes)
	       (if (qde-other (state-qde state))
		   (nconc (qde-other (state-qde state))
			  (list (cons 'phase-planes pplanes)))
		   (setf (qde-other (state-qde state))
			 (list (cons 'phase-planes pplanes))))))
	 pplanes)))

(defun prompt-for-pplanes ()
  (format *qsim-report* "~&Enter list of phase planes, each being a list of two phase variables ~
               (nils may be used for spacing out phase plots).~%Phase planes: ")
  (read))

(defun pplanes-okay-p (pplanes state)
  (cond ((not (listp pplanes)) (format *qsim-report* "~&Input is not a list.") nil)
	((> (length pplanes) 32) (format *qsim-report* "~&Too many phase planes (> 32).") nil)
	(t
	 (let ((var-list
		 (get-reachable-vars state)))
	   
	   (do ((plist pplanes (cdr plist)))
	       ((null plist) t)
	     (cond ((null (car plist)))
		   ((or (not (listp (car plist))) (not (eql (length (car plist)) 2)))
		    (format *qsim-report* "~&~a is not a sublist of length 2." (car plist))
		    (return nil))
		   ((or (not (member (caar plist) var-list))
			(not (member (cadar plist) var-list)))
		    (format *qsim-report* "~&Invalid phase variable in ~a.  ~
                               The valid phase variables are: ~a" (car plist) var-list)
		    (return nil))))))))

(defun get-var-list-from-state (state)
   (mapcar #'(lambda (pair) (car pair))
	   (state-qspaces state)))

(defun find-some-layout (behaviors refstates)
  (or (qde-layout (state-qde (caar behaviors)))
      (allvars-layout (qdes-of-behs behaviors refstates))))

(defun allvars-layout (qdes)
  (let* ((vars (remove-duplicates
		 (loop for qde in qdes
		       nconc (mapcar #'car (qde-var-alist qde)))))
	 (avars (sort vars #'string<)))
    (unless *include-time-in-layout*
      (setf avars (remove 'time avars)))
    (loop for avartail on avars by #'(lambda (list)
				       (nthcdr *columns-for-layout* list))
	  collect (loop for n from 0 below *columns-for-layout*
			collect (nth n avartail)))))




; Plot the behavior of a single parameter.

(defun perturb-link (x1 y1 x2 y2)
  (flet ((i (a b)
	   (+ a (* .15 (- b a)))))
    (qplot-vector (i x1 x2)(i y1 y2)
		  (i x2 x1)(i y2 y1) :shaftthick *perturb-arrow-thickness*)
    (qplot-string "p" (+ (/ (+ x1 x2) 2)
			 (if (< y1 y2) 1 -4))
		  (- (/ (+ y1 y2) 2) 4)
		  :font #+symbolics '(:fix :roman :very-small)
		  #-symbolics axis-font)))

; QSPACE-FROM-BEHAVIOR is designed to produce a suitable qspace for labeling
; the vertical axis, even if it has to dig through the behavior looking for
; an appropriate state just before a region transition.

(defun qspace-from-behavior (varname behavior)
  (let* ((last-state (car (last behavior)))
	 (qspace     (qspace varname last-state)))
    (dolist (state behavior)
      (if (eql (car (state-successors state)) 'transition-identity)
	  (setq qspace (qspace-union (qspace varname state) qspace))))
    qspace))

(defun retrieve-lmark (lmark alist) ;1/25/90 by D.B.
  (second (assoc (lmark-name lmark) alist :key #'lmark-name)))
                                                        ; Modified 1/25/90 by D.B.
(defun must-retrieve-lmark (lmark alist)		; lookup with error on failure.
  (cond ((retrieve-lmark lmark alist))
	(t (error "No value for ~a in ~a." lmark alist))))


;;; If a parameter is in an interval, GUESS-VLOC will guess at the best
;;; location to plot it.  This can be tricky - if the parameter is in an
;;; interval which a later state will split with a new landmark, we must aviod
;;; the appearance of plotting the current value at that lmark.

(defun guess-vloc (qval yalloc param n behavior)
  (let ((qmag (qval-qmag qval)))
    (cond
      ((eql (qdir qval) 'inc)
       (- (must-retrieve-lmark (car qmag) yalloc)
	  (/ *VshareSize* 1.0)))
      ((eql (qdir qval) 'dec)
       (+ (must-retrieve-lmark (second qmag) yalloc)
	  (/ *VshareSize* 1.0)))
      ((and (eql (qdir qval) 'ign) param)
       (let (lmark-qmag)
	 (if (setq lmark-qmag (or (get-prev-lmark-qmag param n behavior)
				  (get-next-lmark-qmag param n behavior)))
	     (if (lmark-equal lmark-qmag (car qmag))
		 (- (must-retrieve-lmark (car qmag) yalloc)
		    (/ *VshareSize* 1.0))
		 (+ (must-retrieve-lmark (second qmag) yalloc)
		    (/ *VshareSize* 1.0)))
	     (/ (+ (must-retrieve-lmark (car qmag) yalloc)
		   (must-retrieve-lmark (second qmag) yalloc))
		2.))))
      (t  (/ (+ (must-retrieve-lmark (car qmag) yalloc)
		(must-retrieve-lmark (second qmag) yalloc))
	     2.)))))

;;; The (AND ...) check in the next two functions is necessary because a
;;; parameter may not even exist in the quantity space of a previous /
;;; succeeding state, if there is a region transition.

(defun get-prev-lmark-qmag (param n behavior)
  (do ((pos (1- n) (1- pos)))
      ((< pos 0) nil)
    (when (and (alookup param (state-qvalues (nth pos behavior)))
	       (atom (qmag (alookup param (state-qvalues (nth pos behavior))))))
      (return (qmag (alookup param (state-qvalues (nth pos behavior))))))))

(defun get-next-lmark-qmag (param n behavior)
  (do ((pos (1+ n) (1+ pos)))
      ((>= pos (length behavior)) nil)
    (when (and (alookup param (state-qvalues (nth pos behavior)))
	       (atom (qmag (alookup param (state-qvalues (nth pos behavior))))))
      (return (qmag (alookup param (state-qvalues (nth pos behavior))))))))

;;;  QPLOT-STATE  - Plotting of states within tree diagram.
;;;  Modified by DLD, per Ben's suggestion, to display time points as
;;;  filled circles (discs) and time intervals as empty circles.
;;;  Unfinished states are followed by a short dashed line to indicate
;;;  that the state's successors have not been generated.

(defun qplot-state (state x y)
  (let ((size 3)
	(status (car (state-status state)))
	(fill   (time-point-p (state-time state))))
    (case status
      ;; Unfinished state displayed with trailing dashed line.
      ((NIL) (qplot-circle x y size :filled fill)
       (qplot-dashed-line (+ x size 1) y (+ x 19) y :dash-pattern '(3 3)))
      ;; Final state displayed as a small disc inside a larger circle.
      ((QUIESCENT FINAL-STATE)
       (qplot-circle x y (+ size 1))
       (qplot-circle x y (- size 1) :alu *black* :filled t))
      ;; Inconsistent state is marked with "X".
      ((INCONSISTENT SELF-INTERSECTION)
       (qplot-circle x y size :alu *black* :filled fill)
       (qplot-line (- x size -10) (- y size) (+ x size 10) (+ y size))
       (qplot-line (- x size -10) (+ y size) (+ x size 10) (- y size)))
      ;; Normal intermediate state.
      (OK (qplot-circle x y size :alu *black* :filled fill))
      ;; Cycle displayed as small circle inside large circle.
      (CYCLE (qplot-circle x y (+ size 1))
	     (qplot-circle x y (- size 1)))
      ;; Transition state displayed as large circle with vertical line through it.
      (TRANSITION (qplot-circle x y (+ size 1))
		  (qplot-line x (- y size) x (+ y size)))
      ;; This branch only used when  plotting an unknown status.
      (T (qplot-circle x y (+ size 5))		
	 (qplot-symbol x y '?)))))


; draw a dotted line connecting two plotted points.

(defun connect-with-dots (x1 y1 x2 y2)
  (let ((dx (/ (- x2 x1) (+ ndots 1.0)))
	(dy (/ (- y2 y1) (+ ndots 1.0))))
    (do ((N  ndots  (- N 1))
	 (x  (+ x1 dx)  (+ x dx))
	 (y  (+ y1 dy)  (+ y dy)))
	((= N 0))
      (qplot-dot x y))))



; Draw the behavior tree.
;   initial-state is a list of initial-states
;   behaviors is a list of lists of behaviors.

(defun plot-state-tree (initial-state depth width xloc yloc xsize ysize bnum)
  (let* ((xstep (min  (/ xsize depth) tree-xstep-max))
	 (ystep (min  (/ ysize width) tree-ystep-max)))
    (plot-trees initial-state xloc yloc xstep ystep nil nil)
    (cond ((> bnum 0)
	   (qplot-symbol (+ xloc xsize) (+ yloc (* (- bnum 1) ystep)) 'left)))
    ))

; plot-tree must return the y-level for the next call.
;   x,y  =  coordinates for this node
;   dx,dy  =  step amounts to later nodes
;   ox,oy  =  previous node to draw connections from.

(defun plot-trees (states x y dx dy ox oy)
  (cond ((null states) y)
	(t (qplot-state (car states) x y)
	   (if (numberp ox) (draw-tree-limb ox oy x y))
	   (let ((successors (successor-states (car states)))
		 (ny nil))
	     (cond ((null successors)
		    (setq ny (+ y dy))
		    (plot-trees (cdr states) x ny dx dy ox oy))
		   (t (setq ny (plot-trees successors (+ x dx) y dx dy x y))
		      (plot-trees (cdr states) x ny dx dy ox oy)))))))

(defun draw-tree-limb (x1 y1 x2 y2)      ; => L-shaped, not diagonal.
  (let ((size 3))				; radius of standard circle
    (if (= y1 y2)
	(qplot-hline (+ x1 size) y1 (- x2 x1 size size))
	(progn
	  (qplot-vline x1 (+ y1 size) (- y2 y1 size))
	  (qplot-hline x1 y2 (- x2 x1 size))))))



(defun transition-follows (state)
  (and (eql (car (state-successors state)) 'transition-identity)
       (second (state-successors state))))

(defun transition-precedes (state) ;;10/23/89 D.B.  
 (let ((pred (predecessor-of-state state)))
   (and pred					; Modified so it also finds completions of incomplete 
	(transition-follows pred))))		; states after region transitions - DRT.

;;; The remaining pages contain code for the new version of the QSIM display.

(proclaim '(special *time-points-only*))

;; New version of Displays.



(defun layout-from-state (initial)
  (qde-layout (state-qde (if (atom initial)	; ugly!
			     initial (car initial)))))

;;;======================================================================
;;;
;;; SOME UTILITES
;;;


(defun read-integer-between (low high &optional (prompt "Input an integer"))
  (format t "~&~a (~a-~a): " prompt low high)
  (unless (<= low high)
    (error "~a must be less than or equal to ~a, or we'll never return!" low high))
  (let ((n (read)))
    (cond ((and (integerp n)
		(<= low n high))
	   n)
	  (T
	   (format t "~&It must be an integer between ~a and ~a." low high)
	   (read-integer-between  low high prompt)))))

(defun map-behaviors (function root)
  "Apply Function to each state in each behavior reachable from
   Root, which is either a state or a list of states."
  #+symbolics (declare (sys:downward-funarg function))
  (dolist (initial (get-list-of-initial-states root))
    (map-behaviors-1 function initial)))

(defun map-behaviors-1 (function state)
  "Apply Function to each state in each behavior reachable from State."
  #+symbolics (declare (sys:downward-funarg function))
  (funcall function state)
  (mapc #'(lambda (s)
	    (map-behaviors-1 function s))
	(successor-states state)))

(defun get-reachable-qdes (root-state &aux qdes)
  "Return a list of the qdes which are used in behaviors reachable
   from Root-state."
  (map-behaviors #'(lambda (state)
		     (pushnew (state-qde state) qdes))
		 root-state)
  qdes)

(defun get-reachable-vars (root)
  "Return a list of the variables used in the qdes which appear
   attached to states reachabel from Root."
  (reduce #'(lambda (sofar qde)
	      (nconc (loop for pair in (qde-var-alist qde)
			   unless (member (car pair) sofar)
			     collect (car pair))
		     sofar))
	  (get-reachable-qdes root)
	  :initial-value nil))


(defun GET-VAR-LIST-FROM-QDES ()
  (apply #'append (mapcar #'(lambda (qde) (mapcar #'(lambda (pair) (car pair))
						  (qde-var-alist qde)))
			  (display-block-qdes-in-beh-tree *current-display-block*))))


(defun return-a-state (state)
  "If state is one, return it, otherwise make sure that the user provides one."
  (cond ((find-state state))
	(T
	 (cerror "Supply a state"
		 "The argument, ~a, is not a valid qsim state."
		 state)
	 (format t "~&Enter the name of a state to view: ")
	 (return-a-state (read)))))
	 
(defun find-state (x)
  "If X is a state, return it.  If it is a symbol, then try to find a state
   in X's package or the qsim package which has the same name as the symbol does."
  (cond ((state-p x) x)
	((symbolp x)
	 (cond ((and (boundp x)
		     (state-p (symbol-value x)))
		(symbol-value x))
	       (T
		(let ((qsymbol (find-symbol (string x) :qsim)))
		  (and qsymbol
		       (boundp qsymbol)
		       (state-p (symbol-value qsymbol))
		       (symbol-value qsymbol))))))))

;;;From Mallory's Customize file:

(defparameter *PRINT-QVAL-SYMBOLS* t
  "If non-nil and *detailed-printing* is nil, qvals are printed with symbols
  by qval-printer.")
;; (setq *detailed-printing* nil) t)

(defun QVAL-PRINTER (qval stream ignore)
  
  "Print-function for structure qval."
  (declare (special *detailed-printing* *print-qval-symbols*))
  (let ((qmag (qval-qmag qval))
	(qdir (qval-qdir qval)))
    (cond (*detailed-printing*
	   (format stream "#<Qval ~(~A~): (~A ~A)>"
		   (variable-name (qval-variable qval)) qmag qdir))
	  ((and (boundp *print-qval-symbols*) *print-qval-symbols*) 
	   (format stream "~A,~A"
		   (cond ((null qmag) "?")
			 ((listp qmag) (substitute "?" nil qmag))
			 (t qmag))
		   (case qdir
		     (std "=")
		     (inc #+(or TI Symbolics) "" #-(or TI Symbolics) "^")
		     (dec #+(or TI Symbolics) "" #-(or TI Symbolics) "v")
		     (ign "*")
		     (nil "?")
		     (t (format nil "~(~A~)" qdir)))))
	  (t ; Ordinary printing
	   (format stream "(~A ~(~A~))" qmag qdir)))))


(defparameter *PRINT-ORIGINAL-LMARKS* nil
  "If non-nil and *detailed-printing* is nil, lmarks are printed by qval-printer with
  pairs of original lmarks (e.g., (0 inf)) rather than with new lmarks (e.g., x-21)
  created during simulation.")

;;(setq *PRINT-ORIGINAL-LMARKS* t)
(defun LMARK-PRINTER (lmark stream ignore)

  "Print-function for structure lmark"
  (declare (special *detailed-printing*))
  (cond (*detailed-printing*
	 (format stream "#<Lm ~(~a~)>" (lmark-name lmark)))
	((and (boundp *print-original-lmarks*)
	      *print-original-lmarks*
	      (lmark-where-defined lmark))	; a two-list, if non-nil
	 (format stream "~a"			; ordinary printing does each component
		 (get-original-lmarks lmark)))
      (t ; Ordinary printing
       (format stream "~(~a~)" (lmark-name lmark)))))


(defun EQUAL-ORIGINAL-LMARKS (qval1 qval2)

  "Determine whether 2 qvalues are equal.  Added lmarks are considered equal if they were
  added between the same original landmarks."
  (and (eq (qval-qdir qval1) (qval-qdir qval2))
       (equal (get-original-lmarks (qval-qmag qval1))
	      (get-original-lmarks (qval-qmag qval2)))))


(defun GET-ORIGINAL-LMARKS (lmark)
  
  "Lmark should be an lmark or a 2-list of lmarks.  If it is an original landmark in its
  qspace, return it.  If lmark is an added landmark, return a 2-list of lmarks in the
  original qspace between which lmark was added.  If lmark is a 2-list of lmarks, return
  them or the original lmarks they were added between."

  ;; Lmark-where-defined is either nil, for lmarks in the original qspace, or a pair of
  ;; lmarks, between which the instant lmark was added during simulation.  In the latter
  ;; case, the two lmarks may or may not be original.

  (let ((lmark-list
	  (if (listp lmark)
	      lmark				; pair of lmarks
	      (lmark-where-defined lmark))))	; original or added lmark
    (if lmark-list				; added lmark or pair of lmarks
	 (list (get-original-lmark (first  lmark-list) #'first)
	       (get-original-lmark (second lmark-list) #'second))
	 lmark)))				; original lmark


(defun GET-ORIGINAL-LMARK (lmark which)

  "Return which (first or second) where-defined landmark of lmark."
  (let ((wd (lmark-where-defined lmark)))
    (if (null wd)
	lmark
	(get-original-lmark (funcall which wd) which))))

(defun SHOW-QVAL-DIFFS (&optional (states *initial-state*)
			(check-uniform-rows t) (check-duplicate-cols t) (test #'equal))
  
  "Print a table showing all qvals in (1) all states in states (if states is a list of
  states or state names) or (2) all final states in all behaviors (finished or not)
  starting in states (if states is a single state).  The handling of uniform rows or
  repeated columns of values is controlled by the parameters check-uniform-rows and
  check-duplicate-cols.  See documentation for function Print-Table."

  (cond ((atom states)
	 (setq states (get-final-states states)))
	((not (state-p (first states)))		; Assume list of state names
	 (setq states (mapcar #'eval states)))
	(t nil))
  (let ((var-names (mapcar #'first (state-qvalues (first states))))
	;(*detailed-printing* nil)
	;(*print-qval-symbols* t)
	;(*print-original-lmarks* nil)
	)
    (declare (special *print-original-lmarks* *print-qval-symbols*))
    (print-table
      (transpose (mapcar #'(lambda (state)
			     (mapcar #'rest (state-qvalues state)))
			 states))
      var-names
      (mapcar #'state-name states)
      :check-uniform-rows check-uniform-rows
      :check-duplicate-cols check-duplicate-cols
      :test test)))

(defun convert-to-user-qvalues (alist)
 "Alist is a list (var . qvalue)."
 (loop for (var . qvalue) in alist
       collect (list var (user-qvalue qvalue))))

(defun user-qvalue (qvalue)
  (unless (qval-p qvalue)
    (error "~a is not a qval." qvalue))
  (list (user-lmark (qval-qmag qvalue))
	 (qval-qdir qvalue)))

(defun user-lmark (qmag)
  (cond ((null qmag) nil)
	((consp qmag)
	 (list (user-lmark (first qmag))
	       (user-lmark (second qmag))))
	((lmark-p qmag)
	 (lmark-name qmag))
	(T (error "~a is not a qmag."))))

(defun make-similar-state (state)
  (make-new-state :from-state state
		  :assert-values
		  (convert-to-user-qvalues (state-qvalues state))
		  :completions nil))


(defun final-states (x)
  (cond ((consp x)
	 (loop for b in x
	       collect (car (last b))))
	((state-p x)
	 (get-final-states x))
	(T (error "~a must be a state or a list of behaviors"))))

;;;----------------------------------------------------------------------
;;; A useful exploration scheme:  
;;;
;;; The final states of a set of behaviors may be broken into
;;;equivalence classes which are determined by the the state qvalues.
;;;Two qvalues are taken to be equal if they are generated between the
;;;same lmarks.  E.g. (0 b-19 b*) (0 b-29 b*), the lmarks b-19 and b-29
;;;are considered to be equivalent.
;;;
;;; The function REPRESENTATIVE-FINAL-STATES does this for the final
;;; states rooted in an initial state, It returns two values: a list
;;; containing a representative member of each of these eqv classes, and
;;; the eqivalence classes them selves.  The eqc are an alist of
;;; (qvalues . states).
 
(defun representative-final-states (initial)
  (let* ((eqc (form-state-eqv-classes (final-states initial)))
	 (reps (mapcar #'second eqc)))
    (values reps eqc)))

(defun form-state-eqv-classes (states &optional eqcs)
  (if (null states)
      eqcs
      (let ((eqc (assoc (cdr (state-qvalues (car states)))
			eqcs
			:test 'equal-qvalue-lists)))
	(if eqc
	    (push (car states) (cdr eqc))
	    (setq eqcs (cons (list (cdr (state-qvalues (car states)))
				   (car states))
			     eqcs)))
	(form-state-eqv-classes (cdr states) eqcs))))

(defun equal-qvalue-lists (l1 l2)
  (loop for e1 in l1
	for e2 in l2
	finally 
	  (return t)
	when (not (EQUAL-ORIGINAL-LMARKS (cdr e1)(cdr e2)))
	  do (return-from equal-qvalue-lists nil)))