;;;  -*- Mode:Common-Lisp; Package: QSIM; Syntax: COMMON-LISP; Base:10 -*-
;;;  Copyright (c) 1989 by Daniel Dvorak.

(in-package 'QSIM)


;;;-----------------------------------------------------------------------------
;;;        S A V E   &   R E S T O R E    E N V I S I O N M E N T S
;;;
;;;  This file contains functions for saving the results of a QSIM simulation
;;;  (i.e., the attainable envisionment) to a file and also functions for
;;;  reading such a file and restoring the envisionment into memory.
;;;
;;;  Saving and restoring envisionments is useful in circumstances where:
;;;  -- an envisionment takes a long time to create, so you don't want to
;;;     re-simulate each time that you want to use/examine it; or
;;;  -- a bunch of envisionments are being created and collected, and it is
;;;     impractical to do them all at the same time.
;;;
;;;  Saving an envisionment is easy -- we just output a printed representation
;;;  of what is in each slot of each state.  Restoring an envisionment is a bit
;;;  harder since STATEs, LMARKs, and QVALs need to be created and pointers
;;;  among these structures need to be restored.
;;;-----------------------------------------------------------------------------


;;; Modifications by Bert Kay :
;;;    - The explorers complain sometimes when with-open-file is called with
;;;      :if-exists is set to NIL.  This is changed in qsim-save-states so that 
;;;      :if-exists defaults to :error.
;;;    - *initial-state* can be a list of states rather than just a singleton.


(defvar *state-filename* "qsim-envisionments.lisp")

;;;-----------------------------------------------------------------------------
;;;  Function:  qsim-save-states
;;;
;;;  Purpose:   This function saves all of the states in the current behavior
;;;             tree (the tree whose root is in *initial-state*) to a file.
;;;             Each state is saved as a list with keyword arguments and
;;;             printed values.
;;;-----------------------------------------------------------------------------

(defun qsim-save-states ()
  (format t "~%~%Please enter file name (default ~A) ---> "
	  *state-filename*)
  (let ((filename (read-line *query-io* nil nil nil))
	(truename  nil)
	(if-exists :error))
    
    (when (or (null filename) (string= "" filename))
      (setq filename *state-filename*))
    (when (string= "q" filename)
      (return-from qsim-save-states (values)))

    (when (setq truename (probe-file filename))
      (if (y-or-n-p "~%File ~a already exists.~
                     ~%Do you want to create a new version (Y) or append (N)? "
		    truename)
	  (setq if-exists :new-version)
	  (setq if-exists :append)))

    ;; Open the file to save the knowledge base into.
    (with-open-file (stream filename
			    :direction :output
			    :if-exists if-exists
			    :if-does-not-exist :create)

      (format t "~%Saving QSIM states to ~a " (truename stream))
      (multiple-value-bind (second minute hour date month year) (get-decoded-time)
        (format stream ";;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; -*-~
                     ~2%;;; ~d/~d/~d  ~2d:~2,'0d:~2,'0d~
                     ~2%;;; Envisionment for ~a"
		month date (mod year 100) hour minute second
		(qde-name (state-qde 
			    (if (listp *initial-state*) 
				(car *initial-state*) 
				*initial-state*)))))

      ;; ### Save each state if *initial-state* has more than one state in it
      (dolist (state (get-list-of-initial-states *initial-state*))
	(save-states state stream)
	(format stream "~2%(initial-state ~s)"
		(state-name state)))
      (terpri stream))))


(defun save-states (root stream)
  (write-char #\. t)
  (format stream "~2%(state :name          ~s~
                   ~%       :qde           ~:(~s~)~
                   ~%       :text          ~s~
                   ~%       :status        ~s~
                   ~%       :time-label    ~s"
	  (state-name root)
	  (qde-name (state-qde root))
	  (state-text root)
	  (state-status root)
	  (state-time-label root))
  
  ;; Output justification.
  (format stream "~%       :justification (~(~s~) ~s)"
	  (car (state-justification root))
	  (let ((x (cadr (state-justification root))))
	    (if (state-p x) (state-name x) nil)))

  ;; Output successors (and save them for recursive call).
  ;; PROBLEM : STATE-SUCCESSORS = (? NIL) vs. NIL
  (let ((successor-states (if (null (state-successors root))
			      NIL
			      (cdr (state-successors root)))))
    (if (null successor-states)
;	(format stream "~%       :successors    (~(~s~))"
;		(car (state-successors root)))
	(format stream "~%       :successors    NIL"
		(car (state-successors root)))
	(format stream "~%       :successors    (~(~s~) ~{~s ~})"
		(car (state-successors root))
		(mapcar #'(lambda (state)
			    (when state (state-name state)))
			successor-states)))

    ;; Output qspaces.
    (format stream "~%       :qspaces       (")
    (let ((first-line t))
      (dolist (entry (state-qspaces root))
	(let ((varname (car entry))
	      (qspace  (cdr entry)))
	  (if first-line
	      (format stream "(~(~a~) ~36T(~{~(~s~) ~}))"
		      varname (mapcar #'lmark-name qspace))
	      (format stream "~%~23T(~(~a~) ~36T(~{~(~s~) ~}))"
		      varname (mapcar #'lmark-name qspace)))
	  (setq first-line nil))))
    (princ ")" stream)

    ;; Output qvalues.
    (format stream "~%       :qvalues       (")
    (let ((first-line t))
      (dolist (entry (state-qvalues root))
	(let* ((varname (car entry))
	       (qval    (cdr entry))
	       (qmag    (qmag qval)))
	  (format stream "~:[~%~23T~;~](~(~a~) ~36T(~(~s ~s~)))"
		  first-line
		  varname
		  (if (lmark-p qmag)
		      (lmark-name qmag)
		      (list (lmark-name (first qmag)) (lmark-name (second qmag))))
		  (qdir qval))
	  (setq first-line nil))))
    (princ ")" stream)

    ;; Output cvalues.
    (format stream "~%       :cvalues       (")
    (let ((first-line t))
      (dolist (entry (state-cvalues root))
	(let ((con   (car entry))
	      (cvals (cdr entry)))
	  (format stream "~:[~%~23T~;~](~(~s~) ~36T~{~(~s~) ~})"
		  first-line
		  (constraint-name con)
		  (mapcar #'(lambda (cv-tuple) (mapcar #'lmark-name cv-tuple)) cvals))
	  (setq first-line nil))))
    (princ ")" stream)

    ;; Output other.
    (format stream "~%       :other         ~s" (state-other root))

    ;; All done.
    (princ ")" stream)

    ;; Do recursive call on successor states.
    (dolist (state successor-states)
      (when state
	(save-states state stream)))
    )
  
  )


;;;-----------------------------------------------------------------------------
;;;  Function:  qsim-read-states
;;; 
;;;  Purpose:   This function reads all of the envisionments from a specified
;;;             file and restores them to memory.  Note that a file may contain
;;;             many envisionments, not just one.  This function assumes that
;;;             all of the QDEs referenced in the envisionment are defined in
;;;             memory when this function is called, otherwise it will complain.
;;;
;;;  Returns:   a list of the initial states of each envisionment.
;;;
;;;  Design:    An envisionment is an interconnected set of STATEs, LMARKs,
;;;             QVALS, and QDEs, and it is not possible to restore all the
;;;             connections in a single pass.  Qsim-read-states does it in
;;;             three passes:
;;;
;;;             1.  Read each symbolic state form, create an instance of the
;;;                 STATE structure for it, and fill in its slots with the
;;;                 symbolic values.
;;;             2.  Convert the successors and justification slots of each STATE
;;;                 from symbolic state names to STATEs.  Now, the states are
;;;                 connected together.
;;;             3.  Convert qspaces, qvalues, and cvalues.
;;;-----------------------------------------------------------------------------

(defun qsim-read-states ()
  (format t "~%~%Please enter file name (default ~A) ---> "
	  *state-filename*)
  (let ((filename (read-line *query-io* nil nil nil)))
    (when (or (null filename) (string= "" filename))
      (setq filename *state-filename*))
    (when (string= "q" filename)
      (return-from qsim-read-states (values)))

    ;; Open the file to save the knowledge base into.
    (with-open-file (stream filename
			    :direction :input
			    :if-does-not-exist :error)

      (format t "Reading QSIM states from ~a " (namestring (truename stream)))

      (let ((initial-states      nil)
	    (*state-name-alist*  nil)
	    form)
	(declare (special *state-name-alist*))

	;; PASS 1: Read in symbolic state forms.
	(loop
	  (setq form (read stream nil 'eof nil))
	  (if (eql 'eof form)
	      (return))
	  (case (first form)
	    (state          (load-state form))
	    (initial-state  (push (alookup (second form) *state-name-alist*) initial-states)
			    (setq *state-name-alist* (nreverse *state-name-alist*))
			    (restore-states)
			    (setq *state-name-alist* nil))
	    (otherwise      (format t "~%Ignoring unknown form: ~a" form))))

	;; All done.
	(terpri)
	(princ "All done -- envisionment has been restored.")
	(reverse initial-states)
	))))
	

(defun restore-states ()
  (declare (special *state-name-alist*))

  ;; PASS 2: Convert state-justification and state-successors
  (dolist (name-and-state *state-name-alist*)
    (mapl #'(lambda (state-names)
	      (rplaca state-names (alookup (car state-names) *state-name-alist*)))
	  (cdr (state-justification (cdr name-and-state))))
    (mapl #'(lambda (state-names)
	      (rplaca state-names (alookup (car state-names) *state-name-alist*)))
	  (cdr (state-successors (cdr name-and-state)))))

  ;; PASS 3: Convert qspaces, qvalues, and cvalues.
  (dolist (name-and-state *state-name-alist*)
    (let ((state (cdr name-and-state)))
      (setf (state-qspaces state) (restore-qspaces state)
	    (state-qvalues state) (restore-qvalues state)
	    (state-cvalues state) (restore-cvalues state)))))


(defun load-state (form)
  (declare (special *state-name-alist*))
  (let ((state nil))
    (do* ((rform   (cdr form)   (cddr rform))
	  (keyword (car rform)  (car rform))
	  (value   (cadr rform) (cadr rform)))
	 ((endp rform))
    
      ;;;(format t "~%Keyword: ~S   Value: ~A" keyword value)

      (case keyword
	(:name          (setq state (make-state :name value))
	                (push (cons value state) *state-name-alist*))
	(:qde           (setf (state-qde state)
			      (if (boundp value)
				  (eval value)
				  (error "QDE ~a is not defined; please load it first." value))))
	(:text          (setf (state-text state)          value))
	(:status        (setf (state-status state)        value))
	(:time-label    (setf (state-time-label state)    value))
	(:justification (setf (state-justification state) value))
	(:successors    (setf (state-successors state)    value))  ; ### mistake : state-qvalues
	(:qspaces       (setf (state-qspaces state)       value))
	(:qvalues       (setf (state-qvalues state)       value))
	(:cvalues       (setf (state-cvalues state)       value))
	(:other         (setf (state-other state)         value))
	(otherwise      (format t "~%Ignoring unknown form: (~a ~a)"
				keyword value)))))
  (write-char #\. t))


(defun dump-state (state)
  (format t "~%name          : ~a" (state-name state))
  (format t "~%qde           : ~a" (state-qde state))
  (format t "~%text          : ~a" (state-text state))
  (format t "~%status        : ~a" (state-status state))
  (format t "~%time-label    : ~a" (state-time-label state))
  (format t "~%justification : ~a" (state-justification state))
  (format t "~%successors    : ~a" (state-successors state))
  (format t "~%qspaces       : ~a" (state-qspaces state))
  (format t "~%qvalues       : ~a" (state-qvalues state))
  (format t "~%cvalues       : ~a" (state-cvalues state))
  (format t "~%other         : ~a" (state-other state)))

(defun restore-qspaces (state)
  (let* ((qde  (state-qde state))
	 (pred-state (get-state-predecessor state qde)))
    (mapcar #'(lambda (entry)
		(let* ((varname         (car entry))
		       (symbolic-qspace (cadr entry))
		       (initial-qspace  (alookup varname (qde-qspaces qde))))
		  (cons varname (mapcar #'(lambda (lname)
					    (cond ((find lname initial-qspace :key #'lmark-name))
						  ((when pred-state
						     (find lname (alookup varname (state-qspaces pred-state)) :key #'lmark-name)))
						  (t (make-lmark :name lname))))
					symbolic-qspace))))
	    (state-qspaces state))))


(defun get-state-predecessor (state qde)
  "Returns closest predecessor state of the given qde."
  (let ((pred (state-predecessor state)))
    (when pred
      (if (eq qde (state-qde pred))
	  pred
	  (get-state-predecessor pred qde)))))


(defun restore-qvalues (state)
  (convert-user-values (state-qvalues state) state))


(defun restore-cvalues (state)
  (mapcar #'(lambda (entry)
	      (let* ((con-name       (car entry))
		     (cv-tuple-list  (cdr entry))
		     (con    (find con-name (qde-constraints (state-qde state))
				   :key #'constraint-name  :test #'equal))
		     (cvars  (constraint-variables con)))
		(cons con (convert-cvals cv-tuple-list cvars))))
	  (state-cvalues state)))