;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-

(in-package 'QSIM)


; The two cascaded tanks is a classic example of the "chattering" branch.
; This structure produces it both by filling the tanks to equilibrium from
; empty, and by perturbing an initial equilibrium state.
;   These tanks don't overflow.

(define-QDE Cascaded-tanks-ch
  (text "Two cascaded tanks")
  (quantity-spaces
    (inflowa    (0 ifa* ifa+ inf))
    (amounta    (0 amta* inf))
    (outflowa   (0 ofa* inf))
    (netflowa   (minf 0 inf))
    (amountb    (0 amtb* inf))
    (inflowb    (0 ifb* inf))
    (outflowb   (0 ofb* inf))
    (netflowb   (minf 0 inf))
    )
  (constraints
    ((M+ amounta outflowa)            (0 0) (inf inf)   (amta* ofa*))
    ((ADD outflowa netflowa inflowa)  (0 0 0)           (ofa* 0 ifa*))
    ((d//dt amounta netflowa))
    ((m+ outflowa inflowb)            (0 0) (inf inf)   (ofa* ifb*))
    ((m+ amountb outflowb)            (0 0) (inf inf)   (amtb* ofb*))
    ((add outflowb netflowb inflowb)  (0 0 0)           (ofb* 0 ifb*))
    ((d//dt amountb netflowb)))
  (independent inflowa)
  (dependent amounta outflowa netflowa amountb inflowb outflowb netflowb)
  (history amounta amountb)
  (layout
     (nil amounta amountb nil)
     (nil inflowa inflowb nil)
     (nil outflowa outflowb nil)
     (nil netflowa netflowb nil))
  (print-names
    (inflowa   "flow(out->A)"  ifa)
    (amounta   "amount(A)"     amta)
    (outflowa  "flow(A->b)"    ofa)
    (netflowa  "d amount(A)"   nfa)
    (inflowb   "flow(a->B)"    ifb)
    (amountb   "amount(B)"     amtb)
    (outflowb  "flow(B->out)"  ofb)
    (netflowb  "d amount(B)"   nfb))
)


(defun fig-3 ()
  (let ((init (make-initial-state cascaded-tanks-ch
				   '((inflowa (ifa* std))
				     (amounta (0 nil))
				     (amountb (0 nil))))))
    (qsim init)
    (qsim-display init nil)
    ))


; This version of the structure is the same as the above, but says
; to ignore qdir distinctions on the problematical parameter NETFLOWB.
; The chatter is eliminated.

(define-QDE Cascaded-tanks-ign-test-ch
  (text "Two cascaded tanks (ignore qdirs on netflowb)")
  (quantity-spaces
    (inflowa    (0 ifa* ifa+ inf))
    (amounta    (0 amta* inf))
    (outflowa   (0 ofa* inf))
    (netflowa   (minf 0 inf))
    (amountb    (0 amtb* inf))
    (inflowb    (0 ifb* inf))
    (outflowb   (0 ofb* inf))
    (netflowb   (minf 0 inf))
    )
  (constraints
    ((M+ amounta outflowa)            (0 0) (inf inf)   (amta* ofa*))
    ((ADD outflowa netflowa inflowa)  (0 0 0)           (ofa* 0 ifa*))
    ((d//dt amounta netflowa))
    ((m+ outflowa inflowb)            (0 0) (inf inf)   (ofa* ifb*))
    ((m+ amountb outflowb)            (0 0) (inf inf)   (amtb* ofb*))
    ((add outflowb netflowb inflowb)  (0 0 0)           (ofb* 0 ifb*))
    ((d//dt amountb netflowb)))
  (independent inflowa)
  (dependent amounta outflowa netflowa amountb inflowb outflowb netflowb)
  (history amounta amountb)
  (layout
     (nil amounta amountb nil)
     (nil inflowa inflowb nil)
     (nil outflowa outflowb nil)
     (nil netflowa netflowb nil))
  (print-names
    (inflowa   "flow(out->A)"  ifa)
    (amounta   "amount(A)"     amta)
    (outflowa  "flow(A->b)"    ofa)
    (netflowa  "d amount(A)"   nfa)
    (inflowb   "flow(a->B)"    ifb)
    (amountb   "amount(B)"     amtb)
    (outflowb  "flow(B->out)"  ofb)
    (netflowb  "d amount(B)"   nfb))
  (other
    (ignore-qdirs netflowb))
)

(defun fig-5 ()
  (let ((init (make-initial-state cascaded-tanks-ign-test-ch
				   '((inflowa (ifa* std))
				     (amounta (0 nil))
				     (amountb (0 nil))))))
    (qsim init State-Limit)
    (qsim-display init nil)
    ))



; The Two-Tank System, or U-Tube.

(define-QDE Two-Tank-System-ch
  (text "Two tank system")
  (quantity-spaces
    (inflowa    (0 ifa* ifa+ inf))
    (amounta    (0 amta* inf))
    (pressurea  (0 pa* inf))
    (outflowa   (0 ofa* inf))
    (netflowa   (minf 0 inf))
    (amountb    (0 amtb* inf))
    (pressureb  (0 pb* inf))
    (delta-ab   (minf 0 dab* inf))
    (inflowb    (0 ifb* inf))
    (outflowb   (0 ofb* inf))
    (netflowb   (minf 0 inf))
    )
  (constraints
    ((M+ amounta pressurea)           (0 0)   (amta* pa*))
    ((ADD delta-ab pressureb pressurea)  (0 0 0)  (dab* pb* pa*))
    ((M+ delta-ab  outflowa)          (0 0)   (dab* ofa*))
    ((ADD outflowa netflowa inflowa)  (0 0 0) (ofa* 0 ifa*))
    ((d//dt amounta netflowa))
    ((m+ outflowa inflowb)            (0 0)   (ofa* ifb*))
    ((m+ amountb pressureb)           (0 0)   (amtb* pb*))
    ((m+ pressureb outflowb)          (0 0)   (pb* ofb*))
    ((add outflowb netflowb inflowb)  (0 0 0) (ofb* 0 ifb*))
    ((d//dt amountb netflowb)))
  (independent inflowa)
  (dependent amounta outflowa netflowa amountb inflowb outflowb netflowb
	     pressurea pressureb delta-ab)
  (history amounta amountb)
  (layout
     (nil amounta amountb nil)
     (nil inflowa inflowb nil)
     (nil outflowa outflowb nil)
     (nil netflowa netflowb nil))
  (print-names
    (inflowa   "flow(out->A)"  ifa)
    (amounta   "amount(A)"     amta)
    (pressurea "pressure(A)"   pa)
    (pressureb "pressure(B)"   pb)
    (delta-ab  "d press(A->B)" dab)
    (outflowa  "flow(A->b)"    ofa)
    (netflowa  "d amount(A)"   nfa)
    (inflowb   "flow(a->B)"    ifb)
    (amountb   "amount(B)"     amtb)
    (outflowb  "flow(B->out)"  ofb)
    (netflowb  "d amount(B)"   nfb))
)


(defun fig-3alt ()
  (let ((init (make-initial-state two-tank-system-ch
				   '((inflowa (ifa* std))
				     (amounta (0 nil))
				     (amountb (0 nil))))))
    (qsim init)
    (qsim-display init nil)
    ))


; The two tank model
; Testing the transfer from Explore-Qsim to Q
; 1. two-tank: two-tank-pdpi, fill-two-tank-from-empty
; 2. two-tank-with-curvature-constraint:
;            two-tank-pdpi-with-curvature-constraint, fill-two-tank-from-empty-with-curvature-constraint
; 3. two-tank-with-curvature-constraint-init-landmarks:
;            two-tank-pdpi-with-curvature-constraint-init-landmarks.

; 1. The two tank system. No curvature constraint, minimal set of landmarks.
(define-QDE two-tank
	    (text "Two-tank system")
  (quantity-spaces
    (amounta (0 amta* inf))
    (amountb (0 amtb* inf))
    (pressurea (0 pa* inf))
    (pressureb (0 pb* inf))
    (delta-ab (minf 0 dab* inf))
    (crossflowab (minf 0 crf* inf))
    (inflowa (0 ifa* inf))
    (outflowb (0 ofb* inf))
    (netflowa (minf 0 inf))
    (netflowb (minf 0 inf)))
  (constraints
    ((M+ pressurea amounta)(0 0)(pa* amta*))
    ((M+ pressureb amountb)(0 0)(pb* amtb*))
    ((add delta-ab pressureb pressurea)(0 0 0)(dab* pb* pa*))
    ((M+ delta-ab crossflowab)(0 0)(dab* crf*)(inf inf))
    ((m+ pressureb outflowb)(0 0)(pb* ofb*)(inf inf))
    ((add netflowa crossflowab inflowa)(0 0 0)(0 crf* ifa*))
    ((add netflowb outflowb crossflowab)(0 0 0)(0 ofb* crf*))
    ((d//dt amounta netflowa))
    ((d//dt amountb netflowb)))
  (independent inflowa)
  (dependent amounta amountb pressurea pressureb delta-ab crossflowab outflowb netflowa netflowb)
  (history amounta amountb)
  (layout
    (amounta amountb nil)
    (netflowa netflowb nil)
    (inflowa crossflowab outflowb))
  (print-names
    (amounta "amount (A)" amta)
    (amountb "amount (B)" amtb)
    (pressurea "pressure (A)" pa)
    (pressureb "pressure (B)" pb)
    (delta-ab "d press(Pa-Pb)" dab)
    (crossflowab "Crossflow (A->B)" crf)
    (inflowa "flow ( -> A)" ifa)
    (outflowb "flow (B->out)" ofb)    
    (netflowa "d amount (A)" nfa)
    (netflowb "d amount (B)" nfb)))

; 1a. A standard example to illustrate the difference between the inclusion of landmark
; and absence of initial landmark is for the initial state: (referred to as in the pipd-mode)
; amta=amta0 >amta* (positive), decreasing,
; amtb=amtb0 >amtb* (positive), increasing.
; 
(defun two-tank-pdpi()
  (let*((normal (make-initial-state two-tank
				    '((amounta (amta* nil))
				      (amountb (amtb* nil))
				      (crossflowab (crf* std))
				      (netflowa (0 std))
				      (netflowb (0 std))
				      (outflowb (ofb* std))
				      (inflowa (ifa* std)))))
	(init (make-modified-state normal
				   '((amounta ((amta* inf) nil))
				     (amountb ((amtb* inf) nil))
				     (crossflowab ((crf* inf) nil)) 				 
				     (netflowa ((minf 0) nil))
				     (netflowb ((0 inf) nil)) 				 
				     (inflowa (ifa* std))))))
    (qsim init state-limit)
    (qsim-display init `((normal ,normal)))))

; 1b. Fill the two tank system, i.e. fill-two-tanks-from empty.
(defun fill-two-tanks-from-empty()
  (let*((init (make-initial-state two-tank
				    '((amounta (0 nil))
				      (amountb (0 nil))
				      (inflowa (ifa* std))))))

    (qsim init state-limit)
    (qsim-display init nil)))



 
; 2. The two tank system with the inclusion of curvature constraints. Minimal set of landmarks.

(define-QDE two-tank-with-curvature-constraint
  (text "Two-tank system with curvature constraint")
  (quantity-spaces
    (amounta (0 amta* inf))
    (amountb (0 amtb* inf))
    (pressurea (0 pa* inf))
    (pressureb (0 pb* inf))
    (delta-ab (minf 0 dab* inf))
    (crossflowab (minf 0 crf* inf))
    (inflowa (0 ifa* inf))
    (outflowb (0 ofb* inf))
    (netflowa (minf 0 inf))
    (netflowb (minf 0 inf)))
  (constraints
    ((M+ pressurea amounta)(0 0)(pa* amta*))
    ((M+ pressureb amountb)(0 0)(pb* amtb*))
    ((add delta-ab pressureb pressurea)(0 0 0)(dab* pb* pa*))
    ((M+ delta-ab crossflowab)(0 0)(dab* crf*)(inf inf))
    ((m+ pressureb outflowb)(0 0)(pb* ofb*)(inf inf))
    ((add netflowa crossflowab inflowa)(0 0 0)(0 crf* ifa*))
    ((add netflowb outflowb crossflowab)(0 0 0)(0 ofb* crf*))
    ((d//dt amounta netflowa))
    ((d//dt amountb netflowb)))
  (independent inflowa)
  (dependent amounta amountb pressurea pressureb delta-ab crossflowab outflowb netflowa netflowb)
  (history amounta amountb)
  (layout
     (nil amounta amountb nil)
     (nil inflowa crossflowab nil)
     (nil crossflowab outflowb nil)		
     (nil netflowa netflowb nil))
  (print-names
    (amounta "amount (A)" amta)
    (amountb "amount (B)" amtb)
    (pressurea "pressure (A)" pa)
    (pressureb "pressure (B)" pb)
    (delta-ab "d press(Pa-Pb)" dab)
    (crossflowab "Crossflow (A->B)" crf)
    (inflowa "flow ( -> A)" ifa)
    (outflowb "flow (B->out)" ofb)    
    (netflowa "d amount (A)" nfa)
    (netflowb "d amount (B)" nfb))
  (other (curvature-at-steady(netflowa (deriv netflowb))(netflowb (deriv netflowa))))
  )



; 2b. Fill the two tank system, i.e. fill-two-tanks-from empty.
(defun fig-7 ()
  (let*((init (make-initial-state two-tank-with-curvature-constraint
				    '((amounta (0 nil))
				      (amountb (0 nil))
				      (inflowa (ifa* std))))))

    (qsim init)
    (qsim-display init nil)))
