;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
(in-package :qsim)

; The Two-Tank System, or U-Tube.
;  - no special constraint on simulation (intractable chattering)
;  - no new landmarks for netflowb (doesn't help)
;  - ignore qdirs (solves the problem)
;  - higher-order derivative (HOD) constraints (solves the problem)

(define-QDE Two-Tank-System
  (text "Two tank system")
  (quantity-spaces  
    (inflowa    (0 inf)       "flow(out->A)")
    (amounta    (0 inf)       "amount(A)")
    (pressurea  (0 inf)       "pressure(A)")
    (outflowa   (0 inf)       "flow(A->b)")
    (netflowa   (minf 0 inf)  "d amount(A)")
    (amountb    (0 inf)       "amount(B)")
    (pressureb  (0 inf)       "pressure(B)")
    (delta-ab   (minf 0 inf)  "d press(A->B)")
    (inflowb    (0 inf)       "flow(a->B)")
    (outflowb   (0 inf)       "flow(B->out)")
    (netflowb   (minf 0 inf)  "d amount(B)"))
  (constraints
    ((M+ amounta pressurea)           (0 0) (inf inf))
    ((ADD delta-ab pressureb pressurea))
    ((M+ delta-ab  outflowa)          (0 0) (inf inf))
    ((ADD outflowa netflowa inflowa))
    ((d/dt amounta netflowa))
    ((m+ outflowa inflowb)            (0 0) (inf inf))
    ((m+ amountb pressureb)           (0 0) (inf inf))
    ((m+ pressureb outflowb)          (0 0) (inf inf))
    ((add outflowb netflowb inflowb))
    ((d/dt amountb netflowb))
    ((constant inflowa)))
  (layout
    (amounta   nil      amountb)
    (pressurea delta-ab pressureb)
    (inflowa   outflowa outflowb)
    (netflowa  inflowb  netflowb))
)

(defun increase-two-tanks-inflow-rate ()
  (let* ((sim (make-sim :HOD-constraints nil
			:SD3-constraints nil))
	 (normal (make-new-state :from-qde two-tank-system
				 :assert-values '((inflowa ((0 inf) std))
						  (amounta ((0 inf) std))
						  (amountb ((0 inf) std)))
				 :sim sim))
	 (init (make-new-state :from-state normal
			       :perturb '((inflowa +))
			       :inherit '(amounta amountb))))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))

(defun fill-two-tanks-from-empty ()
  (let* ((sim  (make-sim :HOD-constraints nil
			:SD3-constraints nil))
	 (init (make-new-state :from-qde two-tank-system
			       :assert-values '((inflowa ((0 inf) std))
						(amounta (0 nil))
						(amountb (0 nil)))
			       :sim sim)))
    (qsim init)
    (qsim-display init)
    ))

;  pdmi designates that initially amounta=plus and increasing, amountb=minus and decreasing.

(defun two-tank-pdmi ()
  (setf (qde-derived-sd2-expressions two-tank-system) nil)
  (let* ((sim (make-sim :HOD-constraints nil
			:SD3-constraints nil))
	 (normal (make-new-state :from-qde two-tank-system
				 :assert-values '((inflowa ((0 inf) std))
						  (amounta ((0 inf) std))
						  (amountb ((0 inf) std)))
				 :sim sim))
	 (init (make-new-state :from-state normal
			       :perturb '((amounta +)
					  (amountb -))
			       :inherit '(inflowa))))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))


; The Two-Tank System, or U-Tube, ignoring some qdirs.

(defun increase-two-tanks-inflow-rate-ign-NFB ()
  (let* ((sim (make-sim :ignore-qdirs '(netflowb delta-ab outflowa netflowa inflowb)
			:HOD-constraints nil
			:SD3-constraints nil))
	 (normal (make-new-state :from-qde two-tank-system
				 :assert-values '((inflowa ((0 inf) std))
						  (amounta ((0 inf) std))
						  (amountb ((0 inf) std)))
				 :sim sim))
	 (init (make-new-state :from-state normal
			       :perturb '((inflowa +))
			       :inherit '(amounta amountb)
			       :text "Increase inflow from equilibrium")))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))

(defun fill-two-tanks-from-empty-ign-NFB ()
  (let* ((sim  (make-sim :ignore-qdirs '(netflowb delta-ab outflowa netflowa inflowb)
			:HOD-constraints nil
			:SD3-constraints nil))
	 (init (make-new-state :from-qde two-tank-system
			       :assert-values '((inflowa ((0 inf) std))
						(amounta (0 nil))
						(amountb (0 nil)))
			       :sim sim)))
    (qsim init)
    (qsim-display init)
    ))

;  pdmi designates that initially amounta=plus and increasing, amountb=minus and decreasing.

(defun two-tank-ign-pdmi ()
  (setf (qde-derived-sd2-expressions two-tank-system) nil)
  (let* ((sim (make-sim :ignore-qdirs '(netflowb delta-ab outflowa netflowa inflowb)
			:HOD-constraints nil
			:SD3-constraints nil))
	 (normal (make-new-state :from-qde two-tank-system
				 :assert-values '((inflowa ((0 inf) std))
						  (amounta ((0 inf) std))
						  (amountb ((0 inf) std)))
				 :sim sim))
	 (init (make-new-state :from-state normal
			       :perturb '((amounta +)
					  (amountb -))
			       :inherit '(inflowa))))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))


; normal inflow, high A.

(defun two-tanks-inflow-normal-A-high-ign-NFB ()
  (let* ((sim (make-sim :ignore-qdirs '(netflowb delta-ab outflowa netflowa inflowb)
			:HOD-constraints nil
			:SD3-constraints nil))
	 (normal (make-new-state :from-qde two-tank-system
				 :assert-values '((inflowa ((0 inf) std))
						  (amounta ((0 inf) std))
						  (amountb ((0 inf) std)))
				 :sim sim))
	 (init (make-new-state :from-state normal
			       :perturb '((amounta +))
			       :inherit '(inflowa amountb)
			       :text "Normal inflow, tank A initially high")))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))

(defun two-tanks-inflow-normal-AB-high-ign-NFB ()
  (let* ((sim (make-sim :ignore-qdirs '(netflowb delta-ab outflowa netflowa inflowb)
			:HOD-constraints nil
			:SD3-constraints nil))
	 (normal (make-new-state :from-qde two-tank-system
				 :assert-values '((inflowa ((0 inf) std))
						  (amounta ((0 inf) std))
						  (amountb ((0 inf) std)))
				 :sim sim))
	 (init (make-new-state :from-state normal
			       :perturb '((amounta +)
					  (amountb +))
			       :inherit '(inflowa)
			       :assert-values '((netflowa ((0 inf) nil))
						(delta-ab ((0 inf) nil)))
			       :text "Normal inflow, both tanks initially high")))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))

; Apply HOD constraints to the two-tank system.

;  pdmi designates that initially amounta=plus and increasing, amountb=minus and decreasing.

(defun two-tank-pdmi-acc ()
  (setf (qde-derived-sd2-expressions two-tank-system) nil)
  (let* ((sim (make-sim :HOD-constraints t
			:SD3-constraints nil))
	 (normal (make-new-state :from-qde two-tank-system
				 :assert-values '((inflowa ((0 inf) std))
						  (amounta ((0 inf) std))
						  (amountb ((0 inf) std)))
				 :sim sim))
	 (init (make-new-state :from-state normal
			       :perturb '((amounta +)
					  (amountb -))
			       :inherit '(inflowa))))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))

; 3b. Fill the two tank system, i.e. fill-two-tanks-from empty.

(defun fill-two-tanks-from-empty-acc ()
  (setf (qde-derived-sd2-expressions two-tank-system) nil)
  (let* ((sim  (make-sim :HOD-constraints t
			:SD3-constraints nil))
	 (init (make-new-state :from-qde two-tank-system
			       :assert-values '((inflowa ((0 inf) std))
						(amounta (0 nil))
						(amountb (0 nil)))

			       :sim sim)))
    (qsim init)
    (qsim-display init)
    ))