;;; -*- Mode:Common-Lisp; Package:QSIM; Base:10 -*-

(in-package :qsim) ;changed DJC

;;;  This file contains examples of Occurrence Branch Elimination.  Currenlty, it
;;;  contains two examples:
;;;
;;;         -  TRIPLE-TUBE which models three tubes which are being
;;;            filled simultaneously.  This is a simple example
;;;
;;;         -  4-REACTIONS which models a sequence of 4 chemical reactions.
;;;            This model is discussed in David Dalle Molle's PhD Dissertation.
;;;            This is a more complicated example.


(defun describe-occ-branch ()
  (format *Qsim-Report*
"~2%Event Occurrence Branching is a phenomenon which occurs when there are two or more variables
approaching landmarks within their respective qspaces.  If there is not constraint which restricts
the temporal ordering of these events (i.e. the reaching of a landmark by one of these variables),
then QSIM will define a behaviors for each possible temporal ordering.  Often the temporal ordering
of these events is irrelevant to the modeler.  This temporal ordering can significantly increase the
exponential bracnhing of the behavior tree.

The elimination of occurrence branching will abstract portions of the behavior tree into
Aggregate Intervals.  The aggregate intervals eliminate the temporal correlation of the
variables within this interval.  Thus, the branching in the tree is eliminated.  When viewing
a behavior which contains an aggregate interval, notice that the time points within a portion
of the time plot have been eliminated.  THis portion of the behavior corresponds to the
aggregate interval.

A model can exhibit multiple instances of occurrence branching.  Thus, there are various levels
of detail at which the model can be viewed.  Once occurrence branching has been eliminated, the modeler
can use the Occurrence Branchy menu to view the behavior tree at various levels of detail.
These levels correspond to the elimination of various instances of occurrence branching.
"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;             TRIPLE-TUBE
;;;
;;;

(define-QDE tri-tube
  (text "Triple tube")
  (quantity-spaces
    (inflowA     (0 ifa* inf))
    (inflowB     (0 ifb* inf))
    (inflowC     (0 ifc* inf))
    (amtA       (0 Amax inf))
    (amtB       (0 Bmax inf))
    (amtC       (0 Cmax inf))
    )
  (constraints
    ((constant inflowA))
    ((constant inflowB))
    ((constant inflowC))
    ((d/dt amtA inflowA))
    ((d/dt amtB inflowB))
    ((d/dt amtC inflowC))
    )
  (layout (amtA      amtB amtC)
	  (inflowA inflowB inflowC)
	  )
  )

(defun fill-trip-tube ()
  (setf *state-limit* 300)
  (let* ((init (make-initial-state tri-tube
				   `((amtA (0 nil))
				     (amtB (0 nil))
				     (amtC (0 nil))
				     (inflowC (ifc* nil))
				     (inflowB (ifb* nil))
				     (inflowA (ifa* nil))))))
    (qsim init)
    (format *Qsim-Report* 
"~%
To eliminate the occurrence branching, select the O (others) menu option and then select E.
The occurrence branching will be eliminated from the tree.  Use the Event Occurrence Branch
menu to expand the aggregate intervals for descriptions of the behavior at varions levels of
detail.") 
    (qsim-display init)
    ))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;            4-REACTIONS
;;;
;;;
;This qde models an isothermal batch reactor with 4 1st-order irreversible
;reactions in series A -> B -> C -> D -> E occurring.  The modeling equations 
;are:
;
; dCa/dt = -k1Ca
;
; dCb/dt = k1Ca - k2Cb
;
; dCc/dt = k2Cb - k3Cc
;
; dCd/dt = k3Cc - k4Cd
;
; dCe/dt = k4Ci
;
;where ki = KiCao*A/Fa (C = concentration, F = Flow rate, A = cross sectional area), 
;for a flow system.  (The Ki's are the first-order reaction rate constants).
;
;(11/15/88).
;
;This QDE should be run with *accept-only-analytical-solutions*, *perform-acc-analysis*,
;and *auto-sd3-evaluation* set to T to avoid impossible behaviors.  Also, *state-limit* 
;should be at least 1500.
;
;The QDE generates 39 behaviors after simulating 1435 states.  This is equivalent to the 
;behavior of a plug-flow reactor of infinite length with the same kinetics.
;
(define-qde batch-a->b->c->d->e
  (text "Isothermal batch CSTR reactor for the reactions A -> B -> C -> D -> E")
  (quantity-spaces
    (-cao (minf cao 0))
    (k1 (0 k1 inf))
    (k2 (0 k2 inf))
    (k3 (0 k3 inf))
    (k4 (0 k4 inf))
    (ca (0 cao inf))
    (cb (0 cao inf))
    (cc (0 cao inf))
    (cd (0 cao inf))
    (ce (0 cao inf))
    (ra (0 inf))
    (rb (0 inf))
    (rc (0 inf))
    (dca/dt (minf 0 inf))
    (dcb/dt (minf 0 inf))
    (dcc/dt (minf 0 inf))
    (dcd/dt (minf 0 inf))
    (dce/dt (minf 0 inf)))
  (constraints
    ((mult k1 ca ra))
    ((mult k2 cb rb))
    ((mult k3 cc rc))
    ((mult k4 cd dce/dt))
    ((minus dca/dt ra))
    ((add dcb/dt rb ra))			
    ((add dcc/dt rc rb))
    ((add dcd/dt dce/dt rc))
    ((sum-zero ca cb cc cd ce -cao)
     (cao 0 0 0 0 cao) (0 cao 0 0 0 cao) (0 0 cao 0 0 cao) (0 0 0 cao 0 cao) (0 0 0 0 cao cao))
    ((sum-zero dca/dt dcb/dt dcc/dt dcd/dt dce/dt))
    ((d/dt ca dca/dt))
    ((d/dt cb dcb/dt))				
    ((d/dt cc dcc/dt))				
    ((d/dt cd dcd/dt))				
    ((d/dt ce dce/dt)))				
  (independent k1 k2 k3 k4 -cao)
  (history ca cb cc cd ce dca/dt dcb/dt dcc/dt dcd/dt dce/dt ra rb rc)
;  4 behavior with this and no acc
;  (other (ignore-qdirs dcb/dt dcc/dt dcd/dt dce/dt))	
  (print-names
    (ca "Concentration of A" ca)
    (cb "Concentration of B" cb)
    (cc "Concentration of C" cc)
    (cd "Concentration of D" cd)
    (ce "Concentration of E" ce)
    (-cao "Initial Concentration of A" cao)
    (dca/dt "dcA/dt" dca)
    (dcb/dt "dcB/dt" dcb)
    (dcc/dt "dcC/dt" dcc)
    (dcd/dt "dcD/dt" dcd)
    (dce/dt "dcE/dt" dce))
  (layout
    (ca dca/dt ra)
    (cb dcb/dt rb)
    (cc dcc/dt rc)
    (cd dcd/dt)
    (ce dce/dt nil))
  )

(defun batch-a->b->c->d->e-short (limit)
  (setf *state-limit* limit
	*perform-acc-analysis* t
	*auto-sd3-evaluation* t
	*analytic-functions-only* t)
  (let ((*state-limit* limit) (*perform-acc-analysis* t) (*auto-sd3-evaluation* t)
	(*analytic-functions-only* t))
  
    (declare (special my-initial))
    (batch-a->b->c->d->e-text)
    (format t "~2%******************************************************************************")
    (format t "~2%   WARNING:  This simulation generates over 700 states and could take")
    (format t "~%   over an hour to run (in XQ, 10 minutes in NQ).  Be Patient")
    (format t "~2%******************************************************************************")
    (setq my-initial
	  (make-initial-state
	    batch-a->b->c->d->e
	    '((k1 (k1 std))
	      (k2 (k2 std))
	      (k3 (k3 std))
	      (k4 (k4 std))
	      (-cao (cao std))
	      (ca (cao nil))
	      (cb (0 nil))
	      (cc (0 nil))
	      (cd (0 nil))
	      (ce (0 nil)))
	    "Initial concentrations [A] = CAO, [B] = [C] = [D] = [E] = 0 at t = 0.")
	  )
    (qsim my-initial)
    (format *Qsim-Report* 
"~%
To eliminate the occurrence branching, select the O (others) menu option and then select E.
The occurrence branching will be eliminated from the tree.  Use the Event Occurrence Branch
menu to expand the aggregate intervals for descriptions of the behavior at varions levels of
detail.") 
         
    (qsim-display my-initial)
    ))

(defun batch-a->b->c->d->e-text ()
  (format t "~2%This qde models an isothermal batch reactor with 4 1st-order irreversible")
  (format t "~%reactions in series A -> B -> C -> D -> E occurring.  The modeling equations")
  (format t "~%are:")
  (format t "~2%  dCa/dt = -k1Ca")
  (format t "~2%  dCb/dt = k1Ca - k2Cb")
  (format t "~2%  dCc/dt = k2Cb - k3Cc")
  (format t "~2%  dCd/dt = k3Cc - k4Cd")
  (format t "~2%  dCe/dt = k4Ci")
  (format t "~2%where the ki's are the first-order reaction rate constants."))



