;;; -*- Mode: LISP; Syntax: Common-lisp; Package: QSIM; Base: 10 -*-
(in-package :qsim)

;;;=============================================================================
;;;
;;;        M A I N    D A T A    S T R U C T U R E S    O F    C C
;;;   -------------------------------------------------------------------------
;;;
;;;  Contents:	  This file contains the definitions for the main structures
;;;               and definitions of "CC", the component-connection modeling
;;;               package.
;;;
;;;  Comment:	  CC is a model-building system that takes as input a
;;;               component-connection description of a device and produces
;;;               as output a QDE (qualitative differential equation) structure
;;;               suitable for input to QSIM, a qualitative simulation system.
;;;               CC was conceived and originally implemented by David Franke;
;;;               Dan Dvorak and Adam Farquhar have made changes and extensions.
;;;
;;;  Overview:	  The following key points may be helpful in getting the 
;;;		  "big picture":
;;;
;;;               Component Interface:
;;;               A component interface defines a component type and an interface
;;;               for that type.  An interface description contains terminals
;;;               (or ports) and parameters.
;;;
;;;               Component Implementation:
;;;               A component implementation describes a decomposition of the
;;;               component into simpler (sub)components, possibly QSIM primitives.
;;;               A decomposition into simpler (sub)components is described in terms
;;;               of component instances and connections among the terminals
;;;               of the (sub)components and the terminals of the component being 
;;;               defined.
;;;
;;;               Configuration:
;;;               A configuration defines which component implementations are to be 
;;;               used in constructing a model (i.e. a completely specified component).
;;;
;;;               Model:
;;;               A model is a completely specified component, for which a QDE can
;;;               be constructed.
;;;
;;;=============================================================================


;;;-----------------------------------------------------------------------------
;;;  Variable:   *Component-Definitions*
;;;              
;;;  Purpose:    The list of currently defined component interfaces. New
;;;              definitions are added via the macro Define-Component-Interface.
;;;-----------------------------------------------------------------------------

(defvar *COMPONENT-DEFINITIONS* nil "List of component interface definitions")


;;;-----------------------------------------------------------------------------
;;;  Variable:   *Configurations*
;;;              
;;;  Purpose:    The list of currently defined configurations.  Configurations
;;;              are placed on this list as well as the list of configurations
;;;              of the associated component, as they may be referenced
;;;              independent of the component (i.e. as a model name).  New
;;;              definitions are added via the macro Define-Configuration.
;;;-----------------------------------------------------------------------------

(defvar *CONFIGURATIONS* nil "List of configurations")


;;;-----------------------------------------------------------------------------
;;;  Variable:   *Global-Configurations*
;;;              
;;;  Purpose:    List of configuration information specified for component types.
;;;              (i.e. specified via component type name).
;;;-----------------------------------------------------------------------------

(defvar *GLOBAL-CONFIGURATIONS* nil "List of component type configuration info")

;;;-----------------------------------------------------------------------------
;;;  Variable:   *Local-Configurations*
;;;              
;;;  Purpose:    List of configuration information specified for specific
;;;              instances (i.e. specified via instance name).
;;;-----------------------------------------------------------------------------

(defvar *LOCAL-CONFIGURATIONS* nil "List of instance specific configuration info")



;;;-----------------------------------------------------------------------------
;;;  Variable:   *CC-Transition-New-Mode-Values*
;;;              
;;;  Purpose:    Mode values determined at a mode transition.
;;;-----------------------------------------------------------------------------

(defvar *CC-TRANSITION-NEW-MODE-VALUES* nil "List of mode variable/value pairs.")


;;;-----------------------------------------------------------------------------
;;;  Variable:   *Constraints*
;;;              
;;;  Purpose:    Association list of constraints created while building a model.
;;;              List is indexed with modes, with common constraints indexed
;;;              with NIL.
;;;-----------------------------------------------------------------------------

(defvar *CONSTRAINTS* nil "A-list of constraints for the current QDE, indexed by mode.")


;;;-----------------------------------------------------------------------------
;;;  Variable:   *Model-Variables*
;;;              
;;;  Purpose:    List of model variables created while building a model.
;;;-----------------------------------------------------------------------------

(defvar *MODEL-VARIABLES* nil "List of model variables generated for the current QDE.")


;;;-----------------------------------------------------------------------------
;;;  Variable:   *Mode-Assumptions*
;;;              
;;;  Purpose:    List of mode variable/value lists representing mode values 
;;;              assumed in building this model.
;;;-----------------------------------------------------------------------------

(defvar *MODE-ASSUMPTIONS* nil "List of mode/value lists assumed for the current QDE.")


;;;-----------------------------------------------------------------------------
;;;  Variable:   *Mode-Variables*
;;;              
;;;  Purpose:    List of mode variables created while building a model.
;;;-----------------------------------------------------------------------------

(defvar *MODE-VARIABLES* nil "List of mode variables generated for the current QDE.")


;;;-----------------------------------------------------------------------------
;;;  Variable:   *NAME-STACK*
;;;              
;;;  Purpose:    List of component names to the current point in the name tree.
;;;              This list is used to create the display text for variables.
;;;-----------------------------------------------------------------------------

(defvar *NAME-STACK* nil "List of component names for current point in model.")


;;;-----------------------------------------------------------------------------
;;;  Variable:   *NAME-TREE*
;;;              
;;;  Purpose:    Tree of names used in the model used to convert a user specified
;;;              variable name (of the form (<comp> <comp> <comp> <var>)) with
;;;              the CC generated name.  This tree is placed on the OTHER slot
;;;              of the QDE.
;;;-----------------------------------------------------------------------------

(defvar *NAME-TREE* nil "Tree of component and variable names.")



;;;-----------------------------------------------------------------------------
;;; Trace variables
;;;-----------------------------------------------------------------------------

(defvar *CC-TRACE* nil)			; Ben's tutorial overview trace

(defvar *TRACE-CONNECTION-PROCESSING* nil)

(defvar *TRACE-KCL-CONSTRAINT-APPLICATION* nil)

(defvar *MODEL-MACRO-TRACE-FLAG* nil)

(defvar *TRACE-MODE-PROCESSING* nil)


;;;-----------------------------------------------------------------------------
;;;  Constant:   *GENERIC-VARIABLE-TYPES*
;;;              
;;;  Purpose:    Variable types independent of any domain, used when no domain
;;;              (NIL) or an unknown domain name is used.
;;;-----------------------------------------------------------------------------

(defconstant *generic-variable-types*
             '((effort       effort       "e")
	       (flow         flow         "f")
	       (displacement displacement "q")
	       (momentum     momentum     "p")
	       (resistance   resistance   "r")
	       (capacitance  capacitance  "c")
	       (inductance   inductance   "l")
	       (power        power        "w")
	       (energy       energy       "e")))


;;;-----------------------------------------------------------------------------
;;;  Variable:   *DOMAIN-SPECIFIC-VARIABLE-TYPES*
;;;              
;;;  Purpose:    An alist containing mappings between the standard symbols for 
;;;              effort, flow, resistance, and capacitance and the domain
;;;              specific symbols for the component currently being considered.
;;;              This variable is dynamically bound each time a component is
;;;              included in a model.
;;;-----------------------------------------------------------------------------

(defvar *DOMAIN-SPECIFIC-VARIABLE-TYPES*
  *generic-variable-types*
  "A-List of standard symbols to domain specific symbols.")


;;;-----------------------------------------------------------------------------
;;;  Variable:   *DOMAIN-CONNECTION-JUNCTION-TYPES*
;;;              
;;;  Purpose:    An alist indexed by domain giving the bond graph junction type
;;;              for the particular domain.  Junction types are 0 (shared effort,
;;;              flows sum to zero) and 1 (shared flow, efforts sum to zero).
;;;-----------------------------------------------------------------------------

(defvar *DOMAIN-CONNECTION-JUNCTION-TYPES*
  '((electrical . 0)
    (mechanical . 1)
    (mechanical-rotation . 1)
    (mechanical-translation . 1)
    (hydraulic . 0)
    (acoustic . 0)
    (thermal . 0)
    (magnetic . 0))
  "A-List of junction types for connections in different domains.")


;;;-----------------------------------------------------------------------------
;;;  Variable:   *COMPONENT-DOMAIN*
;;;              
;;;  Purpose:    The symbol representing the default domain of the component
;;;              currently being considered.  This variable is dynamically
;;;              bound each time a component is included in a model.
;;;-----------------------------------------------------------------------------

(defvar *COMPONENT-DOMAIN* nil "Domain of current component.")


;;;  The following lists for variables in different domains probably has
;;;  some mistakes and/or questionable choices for variable names.
;;;  Feel free to modify this.

(defconstant *mechanical*
	     '((effort        force         "F")
	       (flow          velocity      "V")
	       (momentum      momentum      "p")
	       (displacement  displacement  "X")
	       (power         power         "W")
	       (resistance    resistance    "R")
	       (capacitance   capacitance   "C")))

(defconstant *electrical*
	     '((effort        voltage       "V")
	       (flow          current       "I")
	       (resistance    resistance    "R")
	       (capacitance   capacitance   "C")
	       (displacement  charge        "Q")
	       (power         power         "P")
	       (inductance    inductance    "L")
	       (momentum      flux-linkage  "Fl")))

(defconstant *hydraulic*
	     '((effort        pressure      "P")
	       (flow          flow          "F")
	       (resistance    resistance    "R")
	       (capacitance   volume        "V")
	       (momentum      momentum      "mv")
	       (displacement  amount        "A")
	       (power         power         "W")))

(defconstant *thermal*
	     '((effort        temperature   "T")
	       (flow          heat-flow     "F")
	       (resistance    resistance    "R")
	       (capacitance   capacitance   "C")
	       (displacement  entropy       "Q")
	       (power         power         "W")))

(defconstant *magnetic*
	     '((effort        force         "F")))

(defvar *cc-domains*
	     `((mechanical             . ,*mechanical*)
	       (mechanical-rotation    . ,*mechanical*)
	       (mechanical-translation . ,*mechanical*)
	       (electrical             . ,*electrical*)
	       (hydraulic              . ,*hydraulic*)
	       (acoustic               . ,*hydraulic*)   ; acoustical and hydraulic are same.
	       (thermal                . ,*thermal*)
	       (magnetic               . ,*magnetic*)))



;;;-----------------------------------------------------------------------------
;;;  Structure:  Component-Info
;;;
;;;  Purpose:    A base structure inherited by Component-Interface and
;;;              Component-Implementation.  Several slots are common and a
;;;              uniform access function allows other code to be simpler.
;;;              Note:  Oh to be using a object-oriented language!!
;;;-----------------------------------------------------------------------------

(defstruct component-info
  "CC Base Component Information"
  (name nil)                ; Name of this component type
  (text nil)                ; Descriptive text
  (configurations nil)      ; Configurations for this component implementation
  (previous-definition nil) ; Previous definition of this component type (allow pop definition)
  (quantity-space-info nil) ; Information on defaults qspaces, qspace hierarchy
  (other nil))              ; Dynamic slots


;;;-----------------------------------------------------------------------------
;;;  Structure:  Component-Interface
;;;
;;;  Purpose:    Represents a component type definition.  The possible
;;;              implementations associated with this type are listed here.
;;;              The interface and parameters for this component type are
;;;              defined here.
;;;
;;;  Note:       The previous-definition field exists to allow redefinitions
;;;              which can be subsequently revoked (POPed).  This can be useful
;;;              for temporarily redefining a component type.
;;;-----------------------------------------------------------------------------

(defstruct (component-interface (:include component-info)
				(:print-function component-interface-printer))
  "CC Component Interface"
  (domain nil)              ; One of electrical, mechanical, hydraulic, thermal, ...
  (terminals nil)           ; Terminals defined (names) for this component type
  (parameters nil)          ; Parameters defined for this component type
  (implementations nil))    ; Alist of implementations

(defun component-interface-printer (obj stream ignore)
  (declare (special *detailed-printing*)
	   (ignore ignore))
  (if *detailed-printing*
      (format stream "#<Component I/F ~A ~A>" (component-interface-name obj)
	      (component-interface-text obj))
      (format stream "~A" (component-interface-name obj))))

;;;-----------------------------------------------------------------------------
;;;  Macro:      Define-Component-Interface
;;;
;;;  Purpose:    Initializes an instance of Component-Interface with the
;;;              information specified in the macro body.  Subsequent component 
;;;              implementations are placed on a list of implementations for the
;;;              appropriate interface structure.
;;;-----------------------------------------------------------------------------

(defmacro define-component-interface (component-type-name text domain &rest interface-specs)
  `(let ((new-definition 
	   (make-component-interface :name ',component-type-name
				     :text ,text
				     :domain ',domain))
	 (interface-specs ',interface-specs)
	 (old-definition (find ',component-type-name *component-definitions*
			       :key 'component-interface-name)))
     (setf (component-interface-terminals new-definition)
	   (cdr (assoc 'terminals interface-specs)))
     (setf (component-interface-parameters new-definition)
	   (cdr (assoc 'parameters interface-specs)))
     (setf (component-interface-quantity-space-info new-definition)
	   (digest-quantity-space-defaults new-definition
					   (cdr (assoc 'quantity-spaces interface-specs))
					   ',domain))
     ;; Error report unknown clause names
     (dolist (clause interface-specs)
       (unless (find (car clause) '(parameters quantity-spaces terminals))
	 (format *qsim-report* "~&~% Unknown clause ~S in interface definition ~S."
		 (car clause) ',component-type-name)))
     (when old-definition
       (format *qsim-report* "~&~% Redefining component type ~S.  Implementations not copied."
	       ',component-type-name)
       (setf (component-interface-previous-definition new-definition) old-definition)
       (setf *component-definitions* (delete old-definition *component-definitions*)))
     (push new-definition *component-definitions*)
     new-definition))

;;;-----------------------------------------------------------------------------
;;;  Macro:      Define-Component
;;;
;;;  Purpose:    Simplified call to both define-component-interface and
;;;              define-component-implementation, to provide a simpler 
;;;              syntax compatible with the QR book chapter on CC.
;;;              (BJK:  6-2-92)
;;;-----------------------------------------------------------------------------

(defmacro define-component (name type doc-string &rest body)
  (let ((terminal-clause (assoc 'terminal-variables body)))
    `(progn

      (define-component-interface ,name ,doc-string ,type
	(terminals . ,(mapcar #'car (cdr terminal-clause))))

      (define-component-implementation unique ,name ""
				       . ,body)
      )))



(defun DISPLAY-LOCAL-QUANTITY-SPACE-SPEC (qspace-spec stream)
  (format stream "(~S ~S)" (car qspace-spec) (cdr qspace-spec)))

(defun DISPLAY-COMPONENT-QUANTITY-SPACE-INFO (qspace-specs stream)
  (format stream "~%  (quantity-spaces")
  (dolist (spec qspace-specs)
    (case (car spec)
      (defaults 
	(format stream "~%    (defaults ")
	(display-local-quantity-space-spec (second spec) stream)
	(dolist (item (cddr spec)) 
	  (format stream "~%              ")
	  (display-local-quantity-space-spec item stream))
	(princ ")" stream))
      (hierarchical-parents
	(format stream "~%    (hierarchical-parents ")
	(display-local-quantity-space-spec (second spec) stream)
	(dolist (item (cddr spec)) 
	  (format stream "~%                          ")
	  (display-local-quantity-space-spec item stream))
	(princ ")" stream))
	  ))
  (princ ")" stream))

;;;-----------------------------------------------------------------------------
;;;  Function:   Display-Component-Interface
;;;
;;;  Purpose:    Fancy display of a component-interface structure instance.
;;;-----------------------------------------------------------------------------

(defun DISPLAY-COMPONENT-INTERFACE (ci stream)
  ;; Print the contents of the component-interface instance CI in a form that can
  ;; be easily read and output to a file in text form, suitable for reading in as
  ;; an interface definition.
  (format stream "~&~%(define-component-interface ~S~
                    ~%  ~S"
	  (component-interface-name ci) (component-interface-text ci))
  (when (component-interface-parameters ci)
    (format stream "~%  (parameters")
    (dolist (parm (component-interface-parameters ci)) (format stream " ~S" parm))
    (princ ")" stream))
  (when (component-interface-terminals ci)
    (format stream "~%  (terminals")
    (dolist (term (component-interface-terminals ci)) (format stream " ~S" term))
    (princ ")" stream))
  (when (component-interface-quantity-space-info ci)
    (display-component-quantity-space-info (component-interface-quantity-space-info ci) 
					   stream))
  (princ ")" stream)
  ci)


;;;-----------------------------------------------------------------------------
;;;  Structure:  Component-Implementation
;;;
;;;  Purpose:    Represents a component type implementation, and is recorded on
;;;              the implementations list of theassociated component type 
;;;              definition.
;;;-----------------------------------------------------------------------------

(defstruct (component-implementation (:include component-info)
				     (:print-function component-implementation-printer))
  "CC Component Implementation"
  (interface nil)           ; The defining component type structure
  ; type - virtual slot, either 'primitive (QSIM constraints) or 'composed (components)
  (components nil)          ; List of component instances of this definition
  (connections nil)         ; List of connections of this definition
  (terminal-variables nil)  ; List of terminal variable declarations of this definition
  (component-variables nil) ; List of component variables of this definition
  (mode-variables nil)      ; List of mode variables of this definition
  (constraints nil)         ; List of the constraints of this definition
  (body nil))               ; All specifications per defining macro


(defun component-implementation-printer (obj stream ignore)
  (declare (special *detailed-printing*)
	   (ignore ignore))
  (if *detailed-printing*
      (format stream "#<Component Impl ~A (~A) of ~A>" (component-implementation-name obj)
	      (component-implementation-text obj) 
	      (component-interface-name (component-implementation-interface obj)))
      (format stream "#<Component Impl ~A of ~A>" (component-implementation-name obj)
	      (component-interface-name (component-implementation-interface obj)))))


(defun component-implementation-type (impl)
  (if (component-implementation-components impl) 'composed 'primitive))


(defun component-implementation-composed-p (impl)
  (component-implementation-components impl))


(defun component-implementation-primitive-p (impl)
  (null (component-implementation-components impl)))


(defmacro component-implementation-terminals (impl)
  `(component-interface-terminals (component-implementation-interface ,impl)))


(defmacro componnet-implementation-type (impl)
  `(list (component-interface-name (component-implementation-interface ,impl))
	 (list 'mode (component-implementation-name ,impl))))

;;;-----------------------------------------------------------------------------
;;;  Macro:      Define-Component-Implementation
;;;
;;;  Purpose:    Initializes an instance of Component-Implementation and
;;;              determines whether this is a primitive or composed definition.
;;;-----------------------------------------------------------------------------

(defmacro define-component-implementation (implementation-name component-type-name text
					   &rest specs)
  `(let ((new-definition 
	   (make-component-implementation :name ',implementation-name
					  :text ,text))
	 (specs ',specs)
	 (interface-definition (find ',component-type-name *component-definitions*
				     :key 'component-interface-name))
	 (old-definition nil))
     (if (null interface-definition)
	 (format *qsim-report* "~% ** Unknown component type ~A, implementation ~S not recorded."
		 ',component-type-name ',implementation-name)
	 (progn (setf (component-implementation-interface new-definition) interface-definition)
		(setf (component-implementation-quantity-space-info new-definition)
		      (digest-quantity-space-defaults new-definition
						      (cdr (assoc 'quantity-spaces specs))
						      (component-interface-domain interface-definition)))
		(setf (component-implementation-components new-definition)
		      (cdr (assoc 'components specs)))
		(setf (component-implementation-connections new-definition)
		      (cdr (assoc 'connections specs)))
		(setf (component-implementation-terminal-variables new-definition)
		      (cdr (assoc 'terminal-variables specs)))
		(setf (component-implementation-component-variables new-definition)
		      (cdr (assoc 'component-variables specs)))
		(setf (component-implementation-mode-variables new-definition)
		      (cdr (assoc 'mode-variables specs)))
		(setf (component-implementation-constraints new-definition)
		      (cdr (assoc 'constraints specs)))
		(setf (component-implementation-body new-definition) specs)

		;; Error report unknown clauses
		(dolist (clause specs)
		  (unless (find (car clause) '(components component-variables connections
					       constraints mode-variables quantity-spaces
					       terminal-variables))
		    (format *qsim-report* 
			    "~&~% Unknown clause ~S in implementation ~S of component type ~S."
			    (car clause) ',implementation-name ',component-type-name)))

		;; Check connection specs
		(let ((terminal-names (component-interface-terminals interface-definition))
		      (component-names (mapcar #'car 
					       (component-implementation-components new-definition))))
		  (dolist (connection-spec (component-implementation-connections new-definition))
		    (dolist (terminal-spec (cdr connection-spec))
		      (if (symbolp terminal-spec)
			(unless (member terminal-spec terminal-names)
			  (format *qsim-report* "~&~% Unknown terminal name ~S in connection ~S."
				  terminal-spec connection-spec))
			(unless (member (car terminal-spec) component-names)
			  (format *qsim-report* "~&~% Unknown component name ~S in connection ~S."
				  (car terminal-spec) connection-spec))))))

		;; Stack previous implementation definition
		(when (setf old-definition 
			    (find ',implementation-name 
				  (component-interface-implementations interface-definition)
				  :key 'component-implementation-name))
		  (format *qsim-report* 
			  "~&~% Redefining implementation ~S of component interface ~S."
			  ',implementation-name ',component-type-name)
		  (setf (component-implementation-previous-definition new-definition) old-definition)
		  (setf (component-interface-implementations interface-definition)
			(delete old-definition 
				(component-interface-implementations interface-definition))))
		(push new-definition (component-interface-implementations interface-definition))
		new-definition))))

;;;-----------------------------------------------------------------------------
;;; These accessors are defined so that the representation of a component type
;;; reference can change (if needed) in the future.
;;; In list form, a component type reference is either a symbol denoting the
;;; type name, or a list whose first element is the type name and optional 
;;; elements denoting the implementation and mode.
;;;-----------------------------------------------------------------------------

(defmacro COMPONENT-TYPE-REF-TYPE-NAME (component-type-ref)
  `(if (listp ,component-type-ref) (first ,component-type-ref) ,component-type-ref))


(defmacro COMPONENT-TYPE-REF-IMPL-NAME (component-type-ref)
  `(if (listp ,component-type-ref) (cadr (assoc 'impl (cdr ,component-type-ref)))))


(defmacro COMPONENT-TYPE-REF-MODES (component-type-ref)
  `(if (listp ,component-type-ref) (cdr (assoc 'mode (cdr ,component-type-ref)))))


(defun DISPLAY-TERMINAL-VARIABLE-SPEC (terminal-spec stream)
  (format stream "~%    (~S ~S" (first terminal-spec) (second terminal-spec))
  (dolist (var-spec (cddr terminal-spec))
    (format stream "~%       ~S" var-spec))
  (princ ")" stream))

(defun DISPLAY-COMPONENT-CLAUSE (clause-name clause-specs stream)
  (when clause-specs
    (format stream "~%  (~A" clause-name)
    (dolist (spec clause-specs)
      (format stream "~%     ~S" spec))
    (princ ")" stream)))

;;;-----------------------------------------------------------------------------
;;;  Function:   Display-Component-Implementation
;;;
;;;  Purpose:    Fancy display of a component-implementation structure instance.
;;;-----------------------------------------------------------------------------

(defun DISPLAY-COMPONENT-IMPLEMENTATION (ci stream)
  ;; Print the contents of the component-implementation instance CI in a form that
  ;; can be easily read and output to a file in text form, suitable for reading in
  ;; as an interface definition.
  (format stream "~&~%(define-component-implementation ~S ~S~
                    ~%  ~S"
	  (component-interface-name (component-implementation-interface ci))
	  (component-implementation-name ci) (component-implementation-text ci))
  (when (component-implementation-quantity-space-info ci)
    (display-component-quantity-space-info (component-implementation-quantity-space-info ci)
					   stream))
  (display-component-clause "terminal-variables"
			    (component-implementation-terminal-variables ci)
			    stream)
  (display-component-clause "component-variables"
			    (component-implementation-component-variables ci)
			    stream)
  (display-component-clause "components" (component-implementation-components ci)
			    stream)
  (display-component-clause "constraints" (component-implementation-constraints ci)
			    stream)
  (display-component-clause "connections" (component-implementation-connections ci)
			    stream)
  (princ ")" stream)
  ci)


;;;-----------------------------------------------------------------------------
;;;  Structure:  Configuration
;;;
;;;  Purpose:    A configuration associates component instances with specific
;;;              implementations.  This allows such decisions to be delayed
;;;              from component definition time to model specification time.
;;;              Hence, changes in the model can be made quickly and easily.
;;;-----------------------------------------------------------------------------

(defstruct configuration
  "CC Configuration"
  (name nil)                ; Name of this configuration
  (text nil)                ; Descriptive text
  (component nil)           ; Component interface or implementation
  (mode nil)                ; Mode declaration
  (for-clauses nil)         ; (for <comp ref> [<comp ref> | <config>]) ...
  (previous-definition nil) ; Previous definition of this component type (allow pop definition)
  (other nil))              ; Dynamic slots



(defmacro define-configuration (name component text &rest specs)
  `(let ((new-definition (make-configuration :name ',name
					     :text ,text
					     :component ',component
					     :mode (component-type-ref-modes ',component)))
	 (old-definition (find ',name *configurations* :key 'configuration-name))
	 (component (find (component-type-ref-type-name ',component)
			  *component-definitions* :key 'component-interface-name)))
     (cond ((null component)
	    (format *qsim-report* "~% ** Unknown component type ~A, configuration ~A not recorded."
		    ',(if (listp component) (first component) component) ',name)
	    nil)
	   ((and (listp ',component)
		 (null (setf component (find (component-type-ref-impl-name ',component)
					     (component-interface-implementations component)
					     :key 'component-implementation-name))))
	    (format *qsim-report* "~% ** Unknown component implementation ~A for component type ~A, configuration ~A not recorded."
		    (component-type-ref-impl-name ',component)
		    (component-type-ref-type-name ',component) ',name)
	    nil)
	   (t (setf (configuration-for-clauses new-definition) ',specs)
	      (setf (configuration-component new-definition) component)
	      (when old-definition
		(setf (configuration-previous-definition new-definition) old-definition)
		(setf *configurations* (delete old-definition *configurations*))
		(setf (component-info-configurations component)
		      (delete old-definition (component-info-configurations component))))
	      (push new-definition (component-info-configurations component))
	      (push new-definition *configurations*)
	      new-definition))))



(defun CONFIGURATION-COMPONENT-INTERFACE (config)
  (if (typep (configuration-component config) 'component-interface)
      (configuration-component config)
      (component-implementation-interface (configuration-component config))))



(defun CONFIGURATION-COMPONENT-IMPLEMENTATION (config)
  (when (typep (configuration-component config) 'component-implementation)
    (configuration-component config)))



(defun CONFIGURATION-COMPONENT-MODE (config)
  (configuration-mode config))


(defun display-cc-model (m)
  ;; If m is of type component-interface, select the first implementation from
  ;; the list of implementations, if one exists.  If m is of type configuration,
  ;; display the corresponding component, with default configuration, if necessary.
  (cond ((typep m 'component-interface)
	 (if (null (setf m (car (component-interface-implementations m))))
	     (format *qsim-report* "~% ** Component type ~A has no implementations."
		     (component-interface-name m))
	     (format *qsim-report* "~% Component implementation ~A used by default."
		     (component-implementation-name m))))
	((typep m 'configuration)
	 nil))
  (when m
    (format t "~%(make-model~
             ~%  :name~17T~A-~A~
             ~%  :text~17T\"~A\"~
             ~%  :domain~17T~A"
	    (component-implementation-name m)
	    (component-interface-name (component-implementation-interface m))
	    (component-implementation-text m)
	    (component-interface-domain (component-implementation-interface m)))
    (when (component-implementation-composed-p m)
      (display-multilist ":components"  (component-implementation-components m))
      (display-multilist ":connections" (component-implementation-connections m))
;     (if (model-transitions m)
;         (display-multilist ":transitions" (model-transitions m)))
      )
    (when (component-implementation-primitive-p m)
      (display-multilist ":terminal-variables" (component-implementation-terminal-variables m))
      (display-multilist ":component-variables" (component-implementation-component-variables m))
      (display-multilist ":constraints" (component-implementation-constraints m)))
    (format *qsim-report* "~%)")))
  
(defun display-multilist (title multilist)
  (format *qsim-report* "~&  ~A~17T("  title)
  (let ((sublist (car multilist)))
    (if sublist
	(format *qsim-report* "~S"  sublist)))
  (dolist (sublist (cdr multilist))
    (format *qsim-report* "~%~18T~S"  sublist))
  (format *qsim-report* ")"))

(defun display-qde (qde &optional (show-corresponding-values nil))
  (format *qsim-report* "~%(make-QDE~
             ~%  :name~17T~A" (qde-name qde))
  (display-multilist ":qspaces" (qde-qspaces qde))
  (display-multilist ":constraints" (qde-constraints qde))
  (when show-corresponding-values
	(display-multilist ":cvalues" (qde-cvalues qde)))
  (format *qsim-report* "~&  :independent~17T~A
             ~%  :history~17T~A
             ~%  :transitions~17T~A"
	  (qde-independent qde) (qde-history qde) (qde-transitions qde))
  (display-multilist ":print-names" 
		     (mapcar #'(lambda (var)
				 (list (variable-name var) (variable-title var) (variable-prefix var)))
			     (qde-variables qde)))
  (format *qsim-report* "~&  :text~17T~A"  (qde-text qde))
  (display-multilist ":layout" (qde-layout qde))
  (display-multilist ":other" (qde-other qde))
  (format *qsim-report* "~%)"))

;;;-----------------------------------------------------------------------------
;;;  Structure:  Component-Variable
;;;
;;;  Purpose:    A component variable structure is created for each variable 
;;;              defined for a particular component.  For example, for a Battery,
;;;              component variables are generated for the voltage of each
;;;              terminal (of which there are two), and the voltage measured
;;;              across the two terminals 
;;;-----------------------------------------------------------------------------

(defstruct (COMPONENT-VARIABLE (:PRINT-FUNCTION PRINT-COMPONENT-VARIABLE))
  (component-name nil)				; Name of the corresponding component
  (name           nil)				; Name of the component variable
  (name-stack     nil)                          ; List containing component name hierarchy
  (domain         nil)                          ; Default domain for this component
  (type           nil)				; Type of the component variable
  (lm-symbol      nil)				; New landmark symbol
  (independent    nil)				; Component variable independent?
  (display        nil)				; Display corresponding model variable?
  (quantity-space nil)				; Quantity space of the component variable
  (initable       nil)				; Can initial values be assigned?
  (ignore-qdir    nil)                          ; Ignore direction of change?
  (new-landmarks  t)                            ; Create new landmarks for this variable?
  (model-variable nil))				; Corresponding model variable



(defun PRINT-COMPONENT-VARIABLE (cv stream ignore)
  (declare (ignore ignore))
  (format stream "(~S ~S)" (component-variable-component-name cv) (component-variable-name cv))
;  (format stream "<Component-Variable ~S of ~S, type ~S, mv ~S>"
;          (component-variable-name cv) (component-variable-component-name cv)
;          (component-variable-type cv)
;          (when (component-variable-model-variable cv)
;            (if (eql 'mode (component-variable-type cv))
;                (mode-variable-name (component-variable-model-variable cv))
;                (model-variable-name (component-variable-model-variable cv)))))
  )


;;;-----------------------------------------------------------------------------
;;;  Structure:  Model-Variable
;;;
;;;  Purpose:    A model variable is created for each unique variable in the
;;;              model.  One model variable may correspond to one or more
;;;              component variables.  
;;;-----------------------------------------------------------------------------

(defstruct (MODEL-VARIABLE (:PRINT-FUNCTION PRINT-MODEL-VARIABLE))
  (name           nil)				; Name of the model variable
  (display-text   nil)                          ; Display name for the model variable
  (domain         nil)                          ; Domain (electrical, thermal, etc.)
  (type           nil)				; Type of the model variable
  (independent    nil)				; Model variable independent?
  (display        nil)				; Display this model variable?
  (quantity-space nil)				; Quantity space of the model variable
  (initable       nil)				; Can initial values be assigned?
  (ignore-qdir    nil)                          ; Ignore direction of change?
  (new-landmarks  t)                            ; Create new landmarks for this variable?
  (component-variables nil))			; List of associated component variables


(defun PRINT-MODEL-VARIABLE (mv stream ignore)
  (declare (ignore ignore))
  (format stream "~S" (model-variable-name mv))
;  (format stream "<Model-Variable ~S, type ~S, from ~S>"
;          (model-variable-name mv) (model-variable-type mv)
;          (mapcar #'(lambda (cv) (list (component-variable-component-name cv)
;                                       (component-variable-name cv)))
;                  (model-variable-component-variables mv)))
  )

;;;-----------------------------------------------------------------------------
;;;  Structure:  Mode-Variable
;;;
;;;  Purpose:    A mode variable is created for each unique mode variable in the
;;;              model.
;;;-----------------------------------------------------------------------------

(defstruct (MODE-VARIABLE (:PRINT-FUNCTION PRINT-MODE-VARIABLE))
  (name           nil)				; Name of the mode variable
  (display-text   nil)                          ; Display name for the mode variable
  (display        nil)				; Display this mode variable?
  (static         nil)                          ; Static mode variable?
  (quantity-space nil)				; Quantity space of the mode variable
  (condition-alist nil)                         ; Mode conditions indexed by mode value
  (transition-expressions nil)                  ; Expressions for the transitions clause
  )


(defun PRINT-MODE-VARIABLE (mv stream ignore)
  (declare (ignore ignore))
  (format stream "~S" (mode-variable-name mv))
;  (format stream "<Mode-Variable ~S, values ~S>"
;          (mode-variable-name mv) (mode-variable-quantity-space mv))
  )

;;;-----------------------------------------------------------------------------
;;;  Structure:  Quantity-Space
;;;
;;;  Purpose:    A quantity-space structure is created for each quantity space
;;;              defined in a define-quantity-space macro, each quantity-space
;;;              clause in a define-component-interface or define-component-
;;;              implementation macro, and for each quantity-space clause in a
;;;              variable specification.  Note that these structures are built
;;;              at macro processing time, but may be incomplete (e.g. a default
;;;              hierarchical parent may be inherited from the model structure
;;;              hierarchy).  The structures created for variables will be
;;;              completed during model building.  This structure identifies the
;;;              hierarchical parent of the quantity-space, the source of the
;;;              quantity space definition, and any conservation correspondences.
;;;-----------------------------------------------------------------------------

(defstruct (CC-QUANTITY-SPACE (:PRINT-FUNCTION PRINT-CC-QUANTITY-SPACE))
  (name            nil)				; Name of the qspace, or pointer to source
  (landmark-list   nil)				; Quantity space of the component variable
  (parent-qspace   nil)				; Quantity space parent (hierarchical)
  (correspondences nil))			; Conservartion correspondences



(defun PRINT-CC-QUANTITY-SPACE (qs stream ignore)
  (declare (ignore ignore))
  (format stream "<Quantity-Space ~S>" 
	  (if (and (cc-quantity-space-name qs)
		   (symbolp (cc-quantity-space-name qs)))
	      (cc-quantity-space-name qs)
	    (cc-quantity-space-landmark-list qs))))


(defmacro define-quantity-space (qspace-name &rest body)
  `(let ((new-qspace (digest-quantity-space-spec ',qspace-name ',body)))
     (complete-quantity-space-spec new-qspace nil)
     (push new-qspace *quantity-spaces*)
     new-qspace))


;;;-----------------------------------------------------------------------------
;;;  Variable:   *Base-Quantity-Space*
;;;              
;;;  Purpose:    Basic quantity space, (minf 0 inf)
;;;-----------------------------------------------------------------------------

(defvar *BASE-QUANTITY-SPACE* 
  (make-cc-quantity-space :name 'base-quantity-space
			  :landmark-list '(minf 0 inf)
			  :correspondences '((minf inf) (0 0) (inf minf)))
  "Basic quantity space (minf 0 inf).")


;;;-----------------------------------------------------------------------------
;;;  Variable:   *Quantity-Spaces*
;;;              
;;;  Purpose:    List of quantity spaces defined.
;;;-----------------------------------------------------------------------------

(defvar *QUANTITY-SPACES*
  (list *base-quantity-space*)
  "List of quantity spaces defined.")


;;;-----------------------------------------------------------------------------
;;;  Variable:   *Quantity-Space-Defulats*
;;;              
;;;  Purpose:    List of quantity space defulats, dynamically bound during
;;;              flattening.
;;;-----------------------------------------------------------------------------

(defvar *QUANTITY-SPACE-DEFAULTS* nil "List of quantity space defaults.")



;;;-----------------------------------------------------------------------------
;;;  Structure:  Instantiated-CC-Model
;;;
;;;  Purpose:    An instantiated CC model structure is created for each top level
;;;              CC model created via build-qde.
;;;-----------------------------------------------------------------------------

(defstruct (INSTANTIATED-CC-MODEL (:PRINT-FUNCTION PRINT-INSTANTIATED-CC-MODEL))
  (name                nil)             ; Name for this model (prefix on moded models)
  (text                "")              ; Text description for the model
  (configuration-name  nil)             ; Name of instantiated configuration
  (interface-name      nil)		; Name of instantiated component interface
  (implementation-name nil)             ; Name of instantiated component implementation
  (mode                nil)             ; Initial mode name, presumably static
  (model-variables     nil)		; List of model variables
  (mode-variables      nil)		; List of mode variables
  (constraints         nil)		; List of constraints
  (name-tree           nil)		; Hierarchical name tree
  (qdes                nil)             ; QDEs generated for this model
  )

(defun PRINT-INSTANTIATED-CC-MODEL (im stream ignore)
  (declare (ignore ignore))
  (if (instantiated-cc-model-configuration-name im)
    (format stream "~S" (instantiated-cc-model-configuration-name im))
    (format stream "(~S (impl ~S))" 
	    (instantiated-cc-model-interface-name im)
	    (instantiated-cc-model-implementation-name im))))

;;;-----------------------------------------------------------------------------
;;; Function:  Make-CC-state
;;;
;;; Purpose:   Calls Make-new-state, adjusting syntax from CC names to QSIM names.
;;;
;;;-----------------------------------------------------------------------------

(defun make-cc-state  (&key (from-qde nil)	; specify either QDE or state
			    (from-state nil)
			    (inherit nil)	; variables whose values to inherit
			    (perturb nil)	; alist of (var sign) for perturbations.
			    (assert-values nil)	; assert qualitative values
			    (assert-ranges nil)	; assert quantitative values
			    (completions T)	; compute completions from partial state?
			    (text nil)		; comment text
			    (sim nil))          ; SIM structure to record information
  (let ((qde (or from-qde
		 (state-qde from-state)
		 (error "Must have a QDE from somewhere"))))
    (make-new-state :from-qde from-qde
		    :from-state from-state
		    :inherit (if inherit (CC-name qde :list inherit))
		    :perturb (if perturb (CC-name qde :alist perturb))
		    :assert-values (if assert-values
				       (CC-name qde :alist assert-values))
		    :assert-ranges (if assert-ranges
				       (mapcar #'(lambda (item)
						   (list (list (if (eql (caar item) 'time)
								   'time
								   (CC-name qde :name (caar item)))
							       (cadar item))
							 (cadr item)))
					       assert-ranges))
		    :completions completions
		    :text text
		    :sim sim)))

;;;-----------------------------------------------------------------------------
;;; Function:  CC-display
;;;
;;; Purpose:   Calls QSIM-display, renaming variables in layout from
;;;            CC names to QSIM variables.
;;;
;;;-----------------------------------------------------------------------------

(defun CC-display (&optional (initial-state-or-SIM *initial-state*)
		     &key 
		     (reference-states nil)
		     (layout))
  (let* ((qde (cond ((listp initial-state-or-SIM) (state-qde (car initial-state-or-SIM)))
		    ((state-p initial-state-or-SIM) (state-qde initial-state-or-SIM))
		    ((sim-p initial-state-or-SIM) (sim-qde initial-state-or-SIM))
		    (t (error "CC-display arg ~a of wrong type." initial-state-or-SIM))))
	 (qsim-layout (mapcar #'(lambda (l) (CC-name qde :list l))
			      layout)))
    (qsim-display initial-state-or-SIM
		  :reference-states reference-states
		  :layout qsim-layout)))
