/*
 *  'xform' transforms an ASCII QDE model input file into the internal
 *  representation used for all later processing.  Basically, 'xform'
 *  reads the input file and fills in the structures for parameters,
 *  quantity spaces, constraints, and corresponding values.  These
 *  structures were carefully designed with cross-reference pointers
 *  among themselves so that the simulator can almost always directly
 *  access the data it needs without searching through a list for it.
 *
 *  The parsing done by 'xform' is unavoidably messy, but I've tried to
 *  make it as straightfoward as possible.  I have not attempted to
 *  make these functions fast since they are only used once for each
 *  QDE input file.
 */

#define   EXTERN  extern
#include  <stdio.h>
#include  <ctype.h>
#include  "defs.h"
#include  "global.h"
#include  "io.h"


#define  SYNTAXERR	(-1)
#define  OK		1


#define  QSPACE		0
#define  INDEPENDENT	1
#define  CONSTRAINTS	2
#define  PRINTNAMES	3
#define  INITIALSTATE	4
#define  MODIFIEDSTATE	5


static struct
{
    int   section;
    char  name[20];
} part[] =
{
  { QSPACE,		"quantity-spaces"      },
  { INDEPENDENT,	"independent"	       },
  { CONSTRAINTS,	"constraints"	       },
  { PRINTNAMES,		"print-names"	       },
  { INITIALSTATE,	"make-initial-state"   },
  { MODIFIEDSTATE,	"make-modified-state"  },
  { -1,			""                     }
};


/*
 *  'strcci' is like 'strcmp' except that it ignores upper/lower case
 *  differences ("cci" = compare, case-independent).
 */
strcci( cp1, cp2 )
register char  *cp1, *cp2;
{
    register int  c1, c2, diff;

    while (1)
    {
	c1 = (int) *cp1++;
	c2 = (int) *cp2++;
	if (c1 == 0)
	{
	    if (c2 == 0)  return(0);
	    else return(-1);
	}
	if (c2 == 0)  return(1);

	if (islower(c1))  c1 = toupper(c1);
	if (islower(c2))  c2 = toupper(c2);
	diff = c1 - c2;
	if (diff != 0)  return(diff);

    }
}


/*
 *  'xform' is the top-level function called from the main program to
 *  read in the QDE input file.  'xform' simply reads the next section
 *  heading and hands control to the appropriate parsing routine.
 */
xform()
{
    register int   i, x, ret;
    char  c;

    /* Poised to expect left parenthesis of next major section */
    while ((x = token()) != MEOF)
    {
	if (x != LPAREN)  goto syntaxerr;
	if ((x = token()) < 0)  goto syntaxerr;

	for (i = 0;  part[i].section >= 0;  i++)
	    if (strcci(symbol[x],part[i].name) == 0)
		goto found;

	printf("Invalid section name: %s\n", symbol[x]);
	goto syntaxerr;

found:	switch(part[i].section)
	{
	case QSPACE:	    ret = doqspace();         break;
	case INDEPENDENT:   ret = doindependent();    break;
	case CONSTRAINTS:   ret = doconstraints();    break;
	case PRINTNAMES:    ret = doprintnames();     break;
	case INITIALSTATE:  ret = doinitialstate();   break;
	case MODIFIEDSTATE: ret = doinitialstate();   modified = TRUE;  break;
	}

	if (ret == SYNTAXERR)  goto syntaxerr;
	if (part[i].section == INITIALSTATE)   return;
	if (part[i].section == MODIFIEDSTATE)  return;
    }
    return;

syntaxerr:
    printf("Error in input file found just before what follows:\n");
    while ((c = getc(mstream)) != EOF)
		putchar(c);
    exit(1);
}


/*
 *  Quantity space input.
 */
doqspace()
{
    register struct param *p;
    register int  x;

    while((x = token()) != RPAREN)
    {
	if (x != LPAREN)        return(SYNTAXERR);
	if ((x = token()) < 0)  return(SYNTAXERR);

	if (nparams >= MAXPARAMS)
	{
	    printf("Exceeded limit of %d parameters\n", MAXPARAMS);
	    exit(1);
	}

	/*  Add new parameter name  */
	p = &param[nparams++];
	p->x = x;
	p->namep = symbol[x];
	p->independent = FALSE;
	p->assigned    = FALSE;
	p->qmagx = NOTSPECIFIED;
	p->qdir  = NOTSPECIFIED;
	p->ncons = 0;
	p->suffix = 1;
	p->titlep = "";
	p->prnamep = "X";
	p->range  = STD;

	/*  Expect start of quantity space here  */
	if (token() != LPAREN)        return(SYNTAXERR);

	/* Note:  global 'qsp' is pointer to next available qspace;
	 * and global 'lmp' is pointer to next landmark position.
	 */
	if (qsp >= qspmax)
	{
	    printf("Exceeded limit of %d qspaces\n", MAXQSPACES);
	    exit(1);
	}
	p->qspacep   = qsp;

	if (lmp >= lmpmax)
	{
	    printf("Exceeded limit of %d landmarks\n", MAXLMARKS);
	    exit(1);
	}
	qsp->lmarkp  = lmp;
	qsp->nlmarks = 0;
	qsp->zerox   = -1;
	qsp->minf    = qsp->inf = FALSE;

	/*  Loop to get each symbol of the quantity space  */
	while ((x = token()) != RPAREN)
	{
	    if (x < 0)  return(SYNTAXERR);
	    /* test for exceeding landmark space here */
	    if (qsp->nlmarks >= MAXQSIZE)  return(SYNTAXERR);
	    *lmp++ = x;
	    /* If landmark == "0", save its index for later use  */
	    if (strcmp(symbol[x],"0") == 0)
		qsp->zerox = qsp->nlmarks<<1;
	    else if (strcci(symbol[x],"minf") == 0)
		qsp->minf = TRUE;
	    else if (strcci(symbol[x],"inf") == 0)
		qsp->inf = TRUE;
	    qsp->nlmarks++;
	}
	if (qsp->nlmarks == 0)  return(SYNTAXERR);
	if (token() != RPAREN)  return(SYNTAXERR);
	qsp++;
    }
    return(OK);
}


/*
 *  Independent parameters input.
 */
doindependent()
{
    register int  x, i;

    while((x = token()) != RPAREN)
    {
	for (i = 0;  i < nparams;  i++)
	{
	    if (x == param[i].x)
	    {	param[i].independent = TRUE;
		break;
	    }
	}
	if (i == nparams)  return(SYNTAXERR);
    }
    return(OK);
}


int  mplus(), mminus(), add(), deriv(), mult();

struct
{
	char  name[10];
	int   type;
	int   nargs;
	int   (*funcp)();
}
  cnames[] =
	{ { "M+",		MPLUS,   2,  mplus },
	  { "M-",		MMINUS,  2,  mminus },
	  { "MINUS",		MMINUS,  2,  mminus },
	  { "ADD",		ADD,     3,  add },
	  { "D//DT",		DERIV,   2,  deriv },
	  { "MULT",		MULT,    3,  mult },
	  { "",			-1,     -1,       }
	};


/*
 *  Constraints input.
 */
doconstraints()
{
    register int  i, x;
    register struct constraint  *conp;

    int  j, k, px, argx, ncons;
    struct param  *p;

    conp = &constraint[0];
    nconstraints = 0;

    /* Loop over each constraint/corresponding-values */
    while ((x = token()) != RPAREN)
    {
	if (x != LPAREN)  	return(SYNTAXERR);
	if (token() != LPAREN)	return(SYNTAXERR);

	/* now get the constraint name (M+ or ADD or d//dt or MULT) */
	if ((x = token()) < 0)  return(SYNTAXERR);

	if (nconstraints >= MAXCONSTRAINTS)
	{
	    printf("Exceeded limit of %d constraints\n", MAXCONSTRAINTS);
	    exit(1);
	}

	/* Find matching constraint name in table */
	for (i = 0;  cnames[i].type >= 0;  i++)
	    if (strcci(symbol[x],cnames[i].name) == 0)
		goto found1;
	printf("Unknown constraint name: %s\n", symbol[x]);
	return(SYNTAXERR);

found1:	/*  Name of constraint found in cnames[] table  */
	conp->type  = cnames[i].type;
	conp->funcp = cnames[i].funcp;
	conp->nargs = cnames[i].nargs;
	conp->namep = symbol[x];
	px = 0;

	/* Loop over the arguments of the constraint */
	while ((x = token()) != RPAREN)
	{
	    if (x < 0)  return(SYNTAXERR);
	    for (j = 0;  j < nparams;  j++)
		if (x == param[j].x)  goto found2;
	    printf("Unknown param: %s\n", symbol[x]);
	    return(SYNTAXERR);

found2:	    /*  The px'th argument of this constraint has been identified as
	     *  parameter j.  Save this fact and increment the count of the
	     *  number of constraints sharing parameter j and add this 
	     *  constraint to parameter j's list of constraints.
	     */
	    ncons = param[j].ncons++;
	    param[j].conp[ncons] = conp;
	    conp->paramp[px++] = &param[j];

	    /*  If type MULT, then landmark 0 must appear in the qspace
	     *  of all its arguments.  If type DERIV, landmark 0 must appear
	     *  in the qspace of the second argument.
	     */
	    if (((conp->type == MULT) && (param[j].qspacep->zerox == -1)) ||
		((conp->type == DERIV) && (j == 1) &&
			(param[j].qspacep->zerox == -1)))
	    {
		puts("landmark 0 must be defined for all params of MULT");
		return(SYNTAXERR);
	    }
	}

	if (px != cnames[i].nargs)
	{   printf("Incorrect number of args to %s\n", cnames[i].name);
	    return(SYNTAXERR);
	}

	conp->ncorrs = 0;
	conp->corrp  = crp;

	/* Loop over the sets of corresponding values */
	while ((x = token()) != RPAREN)
	{
	    if (x != LPAREN)  return(SYNTAXERR);
	    /* Loop over the values in each set */
	    argx = 0;
	    while ((x = token()) != RPAREN)
	    {
		/* Look for symbol in param's qspace */
		p = conp->paramp[argx];
		k = findlmark( x, p->qspacep );
		if (k < 0)
		{
		    printf("cval not found in qspace: %s\n", symbol[x]);
		    return(SYNTAXERR);
		}

		/* store correspondence as a point/interval (not index) */
		crp->cvalx[argx++] = k<<1;
	    }
	    if (argx != cnames[i].nargs)
	    {   printf("Incorrect number of corrs to %s\n", cnames[i].name);
		return(SYNTAXERR);
	    }
	    if (crp >= crpmax)
	    {
		printf("Exceeded limit of %d correspondences\n", MAXCORR);
		exit(1);
	    }
	    crp++;
	    conp->ncorrs++;
	}
	if (nconstraints >= MAXCONSTRAINTS)
	{
	    printf("Exceeded limit of %d constraints\n", MAXCONSTRAINTS);
	    exit(1);
	}
	conp++;
	nconstraints++;
    }
    return(OK);
}
/*  above: should test for limits on crp and conp  */


/*
 *  Print-names input.
 */
doprintnames()
{
    register int  i, x, y;

    /* Loop over each constraint/corresponding-values */
    while ((x = token()) != RPAREN)
    {
	if (x != LPAREN)  	return(SYNTAXERR);

	/* now get the parameter name */
	x = token();
	if (x < 0)  return(SYNTAXERR);

	/* See if this is a known parameter name  */
	for (i = 0;  i < nparams;  i++)
	    if (x == param[i].x)  goto found;
	printf("Unknown param: %s\n", symbol[x]);
	return(SYNTAXERR);

found:	/*  Name of parameter has been found  */
	if ((y = token()) < 0)
	    return(SYNTAXERR);
	param[i].titlep = symbol[y];

	if ((y = token()) < 0)
	{
	    /*  If no prefix name present, use the parameter name */
	    param[i].prnamep = symbol[x];
	    if (y != RPAREN)  return(SYNTAXERR);
	}
	else
	{
	    /*  Save prefix name for later use in new landmark names  */
	    param[i].prnamep = symbol[y];
	    if ((x = token()) != RPAREN)  return(SYNTAXERR);
	}
    }
    return(OK);
}


/*
 *  "make-initial-state" and "make-modified-state" input.
 */
doinitialstate()
{
    register int  i, x;
    int   lmark1, lmark2;

    while((x = token()) != RPAREN)
    {
	if (x != LPAREN)  return(SYNTAXERR);
	if ((x = token()) < 0)  return(SYNTAXERR);
	for (i = 0;  i < nparams;  i++)
	    if (x == param[i].x) break;
	if (i == nparams)  return(SYNTAXERR);

	/* default is that qmag is exact, rather than a range */
	param[i].range = STD;

	/* have identified parameter ; now get its value */
	if ((x = token()) != LPAREN)  return(SYNTAXERR);
	x = token();
	if (x == LPAREN)
	{   /*
	     *  Magnitude is an interval.
	     */
	    if ((x = token()) < 0)  return(SYNTAXERR);
	    if (strcci(symbol[x],"nil") == 0)
	    {
		/*  if qmag = (nil ?) then flag as range from -end to X. */
		param[i].range = DEC;
	    }
	    else
	    {
		/*  qmag = (X ?), so see if X is valid.  */
		lmark1 = findlmark( x, param[i].qspacep );
		if (lmark1 < 0)  return(SYNTAXERR);
		param[i].qmagx = (lmark1 << 1) + 1;
	    }
	    if ((x = token()) < 0)  return(SYNTAXERR);
	    if (strcci(symbol[x],"nil") == 0)
	    {
		/*  qmag = (_ nil)  */
		if (param[i].range == DEC)
		    /*  qmag = (nil nil)  */
		    param[i].qmagx = (param[i].qspacep->nlmarks - 1)<<1;
		else
		    /*  qmag = (X nil)  */
		    param[i].range = INC;
	    }
	    else
	    {
		/*  qmag = (_ Y), so see if Y is valid.  */
		lmark2 = findlmark( x, param[i].qspacep );
		if (lmark2 < 0)  return(SYNTAXERR);
		if ((param[i].range == STD) && (lmark2 != (lmark1 + 1)))
		    /*  qmag = (X Y) but X not adjacent to Y in qspace  */
		    return(SYNTAXERR);
		if (param[i].range == DEC)
		    /*  qmag = (nil Y), so set upper bound.  */
		    param[i].qmagx = (lmark2 << 1) - 1;
	    }
	    if ((x = token()) != RPAREN) return(SYNTAXERR);
	}
	else
	{   /*
	     *  Magnitude is a point.
	     */
	    if (strcci(symbol[x],"nil") == 0)
		param[i].qmagx = NOTSPECIFIED;
	    else
	    {
		lmark1 = findlmark( x, param[i].qspacep );
		if (lmark1 < 0)  return(SYNTAXERR);
		param[i].qmagx = lmark1 << 1;
	    }
	}

	/* have the qmag; now get the qdir */
	if ((x = token()) < 0)  return(SYNTAXERR);
	if      (strcci(symbol[x],"std") == 0)  param[i].qdir = STD;
	else if (strcci(symbol[x],"inc") == 0)  param[i].qdir = INC;
	else if (strcci(symbol[x],"dec") == 0)  param[i].qdir = DEC;
	else if (strcci(symbol[x],"nil") == 0)  param[i].qdir = NOTSPECIFIED;
	else return(SYNTAXERR);
	if ((x = token()) != RPAREN)  return(SYNTAXERR);
	if ((x = token()) != RPAREN)  return(SYNTAXERR);
    }
    return(OK);
}


/*
 *  'findlmark' returns the index of a value in its quantity space.
 */
findlmark( x, qspacep )
register int  x;
struct qspace  *qspacep;
{
    register int    i;
    register short *lmarkp;

    lmarkp = qspacep->lmarkp;
    for (i = 0;  i < qspacep->nlmarks;  i++)
	if ( x == (int) *lmarkp++ )  return(i);
    return(-1);
}
