/*
 *  'print' contains routines for formatted output of states.
 */

#define   EXTERN  extern
#include  <stdio.h>

#ifdef ATTUNIX
#include  <string.h>
#else
#include  <strings.h>
#endif

#include  "defs.h"
#include  "global.h"

/* These column settings may be changed as desired */
#define  COLPARAM	5
#define  COLVALUE	17
#define  COLQSPACE	34

static int   firsttime = TRUE;	/* "first time" flag for title initialization*/
static char  title[80];		/* non-volatile title line */
static char  string[40];	/* non-volatile string return area */
static char  str2[40];		/* alternate string return area */


/*
 *  'timetostring' converts a time value (which is a point/interval value)
 *  to its printable representation.  For example, 2 becomes "1" and
 *  3 becomes "(1 2)".
 */
char *
timetostring( tim )
register int  tim;
{
	register int   x;

	x = tim >> 1;
	if (interval(tim))
		sprintf( string, "(%d %d)", x, x+1 );
	else
		sprintf( string, "%d", x );
	return(string);
}


/*
 *  'qmagtostring' converts an point/interval value to its printable
 *  representation.
 */
char *
qmagtostring( qmag, qspacep )
int  qmag;
register struct qspace *qspacep;
{
	register short  *lmarkp;
	register int    x;

	lmarkp = qspacep->lmarkp;
	if (qmag > ((qspacep->nlmarks - 1) << 1))
	{
		strcpy( string, "!!" );
 printf("[qmag=%d,nlmarks=%d]", qmag, qspacep->nlmarks);
		return(string);
	}
	x = qmag >> 1;
	if (interval(qmag))
		/*  qmag is an interval  */
		sprintf( string, "(%s %s)",
			symbol[*(lmarkp+x)], symbol[*(lmarkp+x+1)]);
	else
		/*  qmag is a point  */
		strcpy( string, symbol[*(lmarkp+x)] );
	return(string);
}

/*
 *  'qdirtostring' converts a qualitative direction to its printable
 *  representation.
 */
char *
qdirtostring( qdir )
register int  qdir;
{
	register char  *cp;

	switch( qdir )
	{
	case STD:  cp = "std";  break;
	case INC:  cp = "inc";  break;
	case DEC:  cp = "dec";  break;
	case ANY:  cp = "any";  break;
	default:   cp = "?";    break;
	}
	strcpy( str2, cp);
	return(str2);
}


/*
 *  'prstate' prints a formatted version of a state's number, time,
 *  predecessors, successors, parameter values and qspaces.
 */
prstate( statep )
struct state  *statep;
{
	register int  j, k;
	register char  *cp;

	int    i, statex;
	short  *lmarkp;
	char   buf[80];
	struct qval   *qvalp;
	struct qspace *qspacep;


	/* First time through, create title line with aligned headings */
	if (firsttime)
	{
		firsttime = FALSE;
		for (i = 0;  i < 79; )
			title[i++] = ' ';
		title[79] = '\0';
		strncpy(&title[COLPARAM],  "Parameter",      9);
		strncpy(&title[COLVALUE],  "Value",          5);
		strcpy(&title[COLQSPACE],  "Quantity Space");
	}

	qvalp  = statep->qvalp;

	/*  Print state header: state #, time, predecessors, successors  */
	statex = statep - &state[0];
	printf("State %d     Time = %s\n", statex, timetostring(statep->time));
	fputs("     Predecessor states:", stdout);
	prlist( statep->npreds, statep->predp);
	fputs("     Successor states:  ", stdout);
	prlist( statep->nsuccs, statep->succp);
	puts(title);

	/*  Loop over each parameter and print its name, value, and qspace  */
	for (i = 0, qvalp = statep->qvalp;  i < nparams;  i++, qvalp++)
	{
		/*  Clear print buffer  */
		for (j = 0;  j < 79; )
			buf[j++] = ' ';
		buf[79] = '\0';

		/*  Put parameter name in buffer  */
		j = COLPARAM;
		cp = param[i].namep;
		while( *cp )  buf[j++] = *cp++;

		/*  Put "(qmag qdir)" in buffer  */
		j = COLVALUE;
		buf[j++] = '(';
		cp = qmagtostring( qvalp->qmagx, qvalp->qspacep );
		while( *cp )  buf[j++] = *cp++;
		buf[j++] = ' ';
		cp = qdirtostring( qvalp->qdir );
		while( *cp )  buf[j++] = *cp++;
		buf[j++] = ')';

		/*  Put quantity space in buffer  */
		j = COLQSPACE;
		buf[j++] = '(';
		qspacep = qvalp->qspacep;
		lmarkp  = qspacep->lmarkp;
		for (k = 0;  k < qspacep->nlmarks;  k++)
		{
			cp = symbol[*lmarkp++];
			while ( *cp )
			{	/*  guard against exceeding buf length  */
				if (j > 77)  goto out;
				buf[j++] = *cp++;
			}
			buf[j++] = ' ';
		}
		buf[j-1] = ')';
out:		buf[j] = '\0';

		/*  Print buffer  */
		puts(buf);
	}
}


/*
 *  'prlist' prints the IDs of a list of states.  It is used to print out
 *  a state's predecessor and successor lists.
 */
prlist( n, listp )
register int  n;
struct state *listp[];
{
    register int  i, statex;

    if (n == 0)
	fputs(" none\n", stdout);
    else
    {
	for (i = 0;  i < n;  i++)
	{
	    statex = listp[i] - &state[0];
	    printf(" %d", statex);
	}
	putchar('\n');
    }
}

/*  'prpvals' prints the pvals of all parameters.  This is used as a
 *  tracing aid.
 */
prpvals()
{
    struct pval  *pvalp;
    int  i, j;

    /*  Loop over each parameter  */
    for (i = 0;  i < nparams;  i++)
    {
	printf("%s:  %d pvals:\n", param[i].namep, param[i].npvals);
	for (j = 0, pvalp=param[i].pvalp;  j < param[i].npvals;  j++, pvalp++)
	{
	    printf("        (%s  %s)",
		qmagtostring(pvalp->qmagx, param[i].qspacep),
		qdirtostring(pvalp->qdir));
	    if (pvalp->bad)  puts("  bad");
	    else  putchar('\n');
	}
    }
}

static int  argcol[5] = { 8, 24, 40, 56, 72 };

prcon( conp )
struct constraint  *conp;
{
    char buf[80], *cp, *sp;
    int  i;

    for (i = 0;  i < 80;  i++)
	buf[i] = ' ';
    /*  put constraint name in buffer  */
    cp = &buf[0];
    sp = conp->namep;
    while (*sp)  *cp++ = *sp++;
    /*  put argument names in buffer  */
    for (i = 0;  i < conp->nargs;  i++)
    {
	cp = &buf[argcol[i]];
	sp = conp->paramp[i]->namep;
	while (*sp)  *cp++ = *sp++;
    }
    *cp = '\0';
    puts(buf);
}
	
	

prtuple( conp, tupp )
struct constraint  *conp;
struct tuple       *tupp;
{
    char buf[80], *cp, *sp;
    int  i;
    struct pval  *pvalp;

    for (i = 0;  i < 80;  i++)
	buf[i] = ' ';
    /*  put constraint name in buffer  */
    cp = &buf[0];
    sp = conp->namep;
    while (*sp)  *cp++ = *sp++;
    /*  put argument qmag/qdir in buffer  */
    for (i = 0;  i < conp->nargs;  i++)
    {
	pvalp = tupp->pvalp[i];
	cp = &buf[argcol[i]];
	*cp++ = '(';
	sp = qmagtostring(pvalp->qmagx, conp->paramp[i]->qspacep);
	while (*sp)  *cp++ = *sp++;
	*cp++ = ' ';
	sp = qdirtostring(pvalp->qdir);
	while (*sp)  *cp++ = *sp++;
	*cp++ = ')';
/*
 printf("  <%d  %d>", pvalp->qmagx, pvalp->qdir);
*/
    }
/*
 putchar('\n');
*/
    *cp = '\0';
    fputs(buf,stdout);
}
	

prcandidate( conp, val1p, val2p, val3p )
struct constraint  *conp;
struct pval        *val1p, *val2p, *val3p;
{
    char buf[80], *cp, *sp;
    int  i;
    struct pval  *pvalp;
    struct pval  *pvalps[MAXARGS];

    pvalps[0] = val1p;
    pvalps[1] = val2p;
    pvalps[2] = val3p;

    for (i = 0;  i < 80;  i++)
	buf[i] = ' ';
    /*  put constraint name in buffer  */
    cp = &buf[0];
    sp = conp->namep;
    while (*sp)  *cp++ = *sp++;
    /*  put argument qmag/qdir in buffer  */
    for (i = 0;  i < conp->nargs;  i++)
    {
	pvalp = pvalps[i];
	cp = &buf[argcol[i]];
	*cp++ = '(';
	sp = qmagtostring(pvalp->qmagx, conp->paramp[i]->qspacep);
	while (*sp)  *cp++ = *sp++;
	*cp++ = ' ';
	sp = qdirtostring(pvalp->qdir);
	while (*sp)  *cp++ = *sp++;
	*cp++ = ')';
    }
    *cp = '\0';
    fputs(buf,stdout);
}


static char qbuf[80];

/*  Return pointer to formatted quantity space  */
char *
prqspace( qspacep )
struct qspace  *qspacep;
{
	int  j, k;
	char *cp;
	short *lmarkp;

	/*  Put quantity space in qbuffer  */
	j = 0;
	qbuf[j++] = '(';
	lmarkp  = qspacep->lmarkp;
	for (k = 0;  k < qspacep->nlmarks;  k++)
	{
		cp = symbol[*lmarkp++];
		while ( *cp )
		{	/*  guard against exceeding qbuf length  */
			if (j > 77)  goto out;
			qbuf[j++] = *cp++;
		}
		qbuf[j++] = ' ';
	}
	qbuf[j-1] = ')';
out:	qbuf[j] = '\0';

	return(qbuf);
}


prcorr( conp, corrp )
struct constraint  *conp;
struct corr        *corrp;
{
    char   buf[80], *cp, *sp;
    int    i, x, y;
    short  *lmarkp;

    for (i = 0;  i < 80;  i++)
	buf[i] = ' ';
    /*  put constraint name in buffer  */
    cp = &buf[0];
    sp = conp->namep;
    while (*sp)  *cp++ = *sp++;
    /*  put argument names in buffer  */
    for (i = 0;  i < conp->nargs;  i++)
    {
	cp = &buf[argcol[i]];
	if (interval(corrp->cvalx[i]))
	    printf("Bug!! cvalx[%d] = %d (odd)\n", i, corrp->cvalx[i]);
	x = corrp->cvalx[i] >> 1;
	lmarkp = conp->paramp[i]->qspacep->lmarkp;
	y = *(lmarkp + x);
	sp = symbol[y];
	while (*sp)  *cp++ = *sp++;
    }
    *cp = '\0';
    puts(buf);
}

/*  Print all correspondences of a given constraint  */
pracorr( conp )
struct constraint  *conp;
{
    int  i;
    struct corr  *corrp;

    prcon( conp );
    corrp = conp->corrp;
    for (i = 0;  i < conp->ncorrs;  i++)
	prcorr( conp, corrp++);
}
