/*
 *  'formstates' is applied after tuple generation, constraint checking and
 *  Waltz filtering.  From the remaining tuples it forms all possible
 *  consistent states.
 *
 *  This function is initially called for constraint 0; it selects a tuple
 *  and then calls itself recursively for constraint 1, which then selects
 *  a tuple consistent with the first one, and calls itself recursively for
 *  constraint 2, etc.  Through backtracking, every consistent combination
 *  of tuples will be generated.  Early pruning in the earliest recursive
 *  calls reduces the potentially large search space.
 *
 *  There is one improvement that could be made to this routine that would
 *  eliminate a bad worst-case behavior.  The worst-case would occur if the
 *  lowest-numbered constraints shared no parameters (thus, there would be
 *  no early pruning).  To eliminate this problem, the recursive calls
 *  should use a "connected ordering" of the constraints.  This "connected
 *  ordering" could be generated once by analyzing the constraints, and
 *  then saved for later use here.  Is this improvement worth the bother?
 *  Probably not, since constraint models are usually given in "connected
 *  order" anyway.
 */


#include  <stdio.h>
#include  "defs.h"
#define   EXTERN  extern
#include  "global.h"


formstates( n )
int   n;		/*  # of constraint to find a tuple for  */
{
    register struct param  *p;
    register struct tuple  *tupp;
    register int  k;
    register int  stackx;

    struct param  *stack[MAXARGS];
    struct tuple  *tuppstop;
    struct constraint  *conp;


    conp = &constraint[n];
    tupp = conp->tuplep;
    tuppstop = tupp + conp->ntuples;

    /*  Loop over all tuples for this constraint  */
    for ( ; tupp < tuppstop ; tupp++)
    {
	/*  If this tuple known to be invalid, skip it.  */
	if (tupp->bad)  continue;
	stackx = 0;

	/*  Loop over the 2 or 3 parameters of this tuple  */
	for (k = 0;  k < conp->nargs;  k++)
	{
	    p = conp->paramp[k];

	    /*  Does this parameter already have an assigned value?  */
	    if (p->assigned)
	    {
		/*  Is parameter's assigned pval same as tuple's pval?  */
		if (p->mypvalp == tupp->pvalp[k])
		    continue;
		else
		    goto trynexttuple;
	    }
	    else
	    {
		/*  Assign parameter the value in the tuple  */
		p->mypvalp = tupp->pvalp[k];
		/*  Remember which parameters I have assigned values to.  */
		p->assigned = TRUE;
		stack[stackx++] = p;
	    }
	}

	/*  Current tuple is consistent with our previous tuples.  */
	if (n+1 < nconstraints)
	{
	    /*  Not all constraints have been covered yet, so do recursive
	     *  call on the next constraint.
	     */
	    formstates(n+1);
	}
	else
	{
	    /*  We have now covered all constraints and therefore assigned
	     *  a value to each parameter.  Thus, this is a candidate state
	     *  to be filtered and, if it passes, added as a new state.
	     */
	    formnewstate();
	}

trynexttuple:
	/*  "Unassign" the parameters I previously assigned (necessary for
	 *  the next tuple consideration to work properly.
	 */
	while (stackx > 0)
	    stack[--stackx]->assigned = FALSE;
    }
}
