/*
 *  CQ -- definitions file.
 *
 *  Notational conventions used throughout CQ:
 *
 *  -- Variable names ending in "p" are usually pointers;
 *     variable names ending in "x" are usually array indices;
 *     variable names beginning with "n" and ending in "s" are usually
 *     counters.
 *
 *  -- External names (globals, function names) are unique in the first
 *     six characters to permit porting CQ to the widest range of machines.
 */

/*
 *  Trace flags:
 *    If the symbol TRACE is defined then optional trace code is compiled in
 *    to various routines (the code is bracketed by #ifdef TRACE, #endif).
 *    For minimum object code size and slightly faster running speed, remove
 *    the "#define TRACE" statement.
 *
 *    The remaining trace flags (that begin with "TR_") define bit positions
 *    in the global integer 'trace'.  Any of these bit positions may be set
 *    at run-time via analysis of an optional command-level argument.  Each
 *    piece of trace code checks to see if the appropriate bit is set.
 */
#define  TRACE
#define  TR_IVALS	0x001	/* trace creation of initial values */
#define  TR_PVALS	0x002	/* trace creation of possible values */
#define  TR_TUPLES	0x004	/* trace creation of tuples */
#define  TR_WALTZ	0x008	/* trace Waltz filtering of tuples */
#define  TR_STATES	0x010	/* trace creation of new states */
#define  TR_LANDMARKS	0x020	/* trace creation of new landmarks */
#define  TR_CVALS	0x040	/* trace creation of new correspondences */
#define  TR_DIVERGENT	0x100	/* allow prediction from divergent states */
#define  TR_QUIESCENT	0x200	/* allow prediction from quiescent states */
#define  TR_EXEC	0x400	/* print execution statistics summary */


#define  TRUE  1
#define  FALSE 0

#define  BIGNUM  32767		/* largest value for a short */


/*
 *  The following are possible values for a qualitative direction (qdir).
 *  Warning:  the associations INC=0, STD=1, DEC=2 are used for indexing
 *  into some pre-defined arrays (for transitions and qdir testing).
 *  So, don't change these values unless you also change those arrays.
 *
 *  (NOTSPECIFIED is also used with qualitative magnitude (qmag)).
 */
#define  INC  0
#define  STD  1
#define  DEC  2
#define  ANY  3
#define  NOTSPECIFIED  (-1)



/*
 *  The 'compare' macro compares two numbers and returns GT, EQ or LT
 *  corresponding to less-than, equal, or greater-than.  Warning: the
 *  associations GT=0, EQ=1, LT=2 are used not only for indexing into
 *  some pre-defined arrays, but the equivalences GT=INC, EQ=STD, and
 *  LT=DEC are also used in some constraint checking routines (deriv).
 */
#define  GT  0
#define  EQ  1
#define  LT  2
#define  compare(A,B)   ((A) < (B) ? LT : ((A) == (B) ? EQ : GT))


/*  The 'point' macro returns TRUE if QMAG is a point value.
 *  The 'interval' macro returns TRUE if QMAG is an interval value.
 */
#define  interval(QMAG)    (QMAG & 01)
#define  point(QMAG)     (!(QMAG & 01))



typedef  short  Boolean;


/*
 *  Constraint types (these values are currently not actually used anywhere).
 */
#define  MPLUS   0
#define  MMINUS  1
#define  ADD     2
#define  DERIV   3
#define  MULT    4



/*============================================================================
 *  The following definitions affect the size of problem that CQ can handle.
 *  The role of each definition is explained below.  If you need to change
 *  any values (such as expanding for larger problems), follow the guidelines
 *  suggested below.
 *============================================================================
 */

/*
 *  MAXCONSTRAINTS and MAXPARAMS are the maximum number of constraints and
 *  parameters, respectively, in the model.  MAXPARAMS should be roughly
 *  2-3 times larger than MAXCONSTRAINTS, reflecting the fact that all
 *  constraints are either 2-valued or 3-valued.
 */
#define  MAXCONSTRAINTS 40
#define  MAXPARAMS	100

/*
 *  MAXSYMBOLS is the maximum number of unique names of parameters, landmarks,
 *  and print-names.  To set it, first estimate the total number of landmarks
 *  that will exist when simulation is finished; call it TL.
 *  Set MAXSYMBOLS = TL + 3*MAXPARAMS.
 */
#define  MAXSYMBOLS	400

/*
 *  MAXSYMCHARS is the amount of character storage allocated to store all
 *  symbols.  If the average length of a parameter-name is 8, and the average
 *  length of a print-name is 24, and the average length of a landmark-name 
 *  is 4, then set MAXSYMCHARS to (8+24)*MAXPARAMS + 4*TL (where TL is the
 *  same as described for MAXSYMBOLS).
 */
#define  MAXSYMCHARS	3600

/*
 *  MAXPVALS is the maximum number of qval candidates allowed during any
 *  simulation cycle.  If you are doing an attainable envisionment with
 *  the initial state values completely specified, then MAXPVALS does not
 *  need to be any larger than 4*MAXPARAMS since no transition can generate
 *  more than 4 candidate values.  If you are doing a total envisionment and
 *  N is the average number of landmarks in the initial qspace of a parameter,
 *  then set MAXPVALS = (2*N - 1)*3*MAXPARAMS, approximately.  The factor 3
 *  represents the 3 qdirs and the factor 2*N-1 represents the number of
 *  possible magnitudes in a qspace of N landmarks.
 */
#define  MAXPVALS	400

/*
 *  MAXQVALS establishes how many 'qval' structure elements are allocated.
 *  A 'qval' element is needed for each parameter of each state.  Thus,
 *  ideally, set MAXQVALS = MAXSTATES * nparams, where 'nparams' is the
 *  actual number of parameters in the model.
 */
#define  MAXQVALS	2000

/*
 *  Each time a new landmark is created it requires a new qspace to hold it.
 *  MAXQSPACES is the maximum number of qspaces allowed.  If NL is the number
 *  of new landmarks estimated to be created, set MAXQSPACES = MAXPARAMS + NL.
 */
#define  MAXQSPACES	300	/* # distinct qspaces generated */

/*
 *  MAXLMARKS is the maximum for the sum of lengths of all distinct qspaces.
 *  If we assume that the average length of a qspace is initally 3, and that
 *  the average length of newly-generated qspaces is 5, then set
 *  MAXLMARKS = 3*MAXPARAMS + 5*NL, approximately.
 */
#define  MAXLMARKS	1300

/*
 *  MAXQSIZE is an arbitrary restriction on the maximum number of landmarks
 *  that any quantity space can hold.  This is used as a check against
 *  runaway landmark creation (it has no role in storage allocation).
 */
#define  MAXQSIZE	20

/*
 *  MAXCORR is the maximum number of sets of corresponding values.
 *  If EC is the number of existing correspondences in the model and
 *  NC is an estimate of the number of new correspondences generated
 *  (as a result of new landmarks), then set MAXCORR = EC + NC.
 *  As a first guess, try MAXCORR = 6*MAXPARAMS.
 */
#define  MAXCORR	1000

/*
 *  MAXSCORR is the maximum number of sets of corresponding values that can
 *  be saved for all states.  This should equal MAXSTATES * MAXCONSTRAINTS.
 */
#define  MAXSCORR	3200

/*
 *  MAXSTATES is the maximum number of states that can be saved.
 *  The number needed is highly model-dependent.  As a first guess,
 *  try MAXSTATES = 2*MAXCONSTRAINTS.
 */
#define  MAXSTATES	80

/*
 *  MAXPCONS is the maximum number of constraints that can share the same
 *  parameter.  Just make this as big as needed for your model.
 */
#define  MAXPCONS	4

/*
 *  MAXARGS is the maximum number of arguments per constraint.  Currently,
 *  ADD and MULT are the biggest, having 3 arguments.
 */
#define  MAXARGS	3	/* max. no. arguments per constraint */

/*
 *  MAXTUPLES is the maximum number of tuples that can be stored on any
 *  simulation cycle.  As a first guess, set MAXTUPLES = 10*MAXPARAMS.
 */
#define  MAXTUPLES	1000

/*
 *  MAXNEIGHBORS is the maximum number of predecessor or successor states
 *  that any state can have.  I think 3 or 4 is the most ever needed.
 */
#define  MAXNEIGHBORS	10

/*
 *  MAXAGENDA is the maximum length of the agenda, i.e., the maximum number
 *  of states awaiting prediction.  This value need never be larger than
 *  MAXSTATES.  Since each agenda item only occupies the space of a pointer,
 *  it's reasonable to set MAXAGENDA = MAXSTATES.
 */
#define  MAXAGENDA	80



/*
 *  'param' is used as the temporary storage for each parameter during
 *  simulation (permanent values are stored in 'qval's associated with
 *  each generated state).
 */
struct param
{
    short	  x;		/* index into symbol[] */
    char	  *namep;	/* pointer to param name */
    char	  *prnamep;	/* pointer to new landmark prefix name */
    char	  *titlep;	/* pointer to printable title string */
    short	  suffix;	/* suffix integer for new landmark names */
    short	  qmagx, qdir;	/* initial qualitative value */
    short	  range;	/* =STD if qmagx is exact (not a range),
				 * =INC if qmag is a range from qmagx to +end,
				 * =DEC if qmag is a range from -end to qmagx.
				 */
    short	  newx;		/* qmagx of new landmark, else BIGNUM */
    struct qspace *qspacep;	/* pointer to initial qspace */
    Boolean       independent;	/* independent param? TRUE or FALSE */
    Boolean       assigned;	/* TRUE if qmagx/qdir assigned by formstates */
    struct pval   *mypvalp;	/* set by 'formstates' */
    short         npvals;	/* number of possible values */
    struct pval   *pvalp;	/* pointer to N consecutive pval's (N=npvals)*/
    short   	  ncons;	/* number of constraints sharing this param */
    struct constraint *conp[MAXPCONS]; /* pointer to consecutive constraints*/
};



struct qspace
{
    short  nlmarks;	/* number of landmarks in this qspace */
    short  zerox;	/* index of zero landmark, else -1 */
    short  *lmarkp;	/* pointer to variable length array of landmarks,
			   each of which is an index into symbol[] */
    Boolean minf, inf;	/* TRUE if qspace contains minf,inf, respectively */
};



struct constraint
{
    char   *namep;		/* pointer to constraint name */
    int    (*funcp)();		/* pointer to constraint check function */
    short  type;		/* type: MPLUS, DERIV, ADD, etc. */
    short  nargs;		/* actual number of arguments */
    struct param *paramp[MAXARGS];   /* pointers to arguments (max 3 args) */
    short  ncorrs;		/* number of sets of corresponding values */
    struct corr *corrp;		/* pointer to start of correspondences */
    short  ntuples;		/* current number of valid tuples */
    struct tuple *tuplep;	/* pointer to start of tuples */
    Boolean covered;		/* TRUE if covered in 'formstates' */
};



/*
 *  'pval' holds a "possible value" for a given parameter.  If a given
 *  parameter has N possible values then there will be N consecutive 'pvals's
 *  pointed to by param[i].pvalp, with param[i].npvals = N.
 *  The 'nusers' counter is used during constraint checking for early
 *  elimination of a pval, when possible.
 */
struct pval
{
    short  qmagx;		/* magnitude: index into qspace */
    short  qdir;		/* direction: INC, DEC, STD */
    short  ncons;		/* number of constraints accepting this pval */
    Boolean used;		/* Is this pval used at all: TRUE or FALSE */
    Boolean bad;		/* Initially FALSE, but set TRUE if any
				   constraint can't use this value. */
};



/*
 *  'qval' holds the minimum information about a parameter needed in
 *  distinguishing different states.  A state is stored as N consecutive
 *  'qval's, where N is the number of parameters in the model.
 */
struct qval
{
    struct qspace *qspacep;	/* pointer to quantity space */
    short  qmagx;		/* magnitude: index into qspace */
    short  qdir;		/* direction: INC, DEC, STD */
};



/*
 *  Tuple -- this holds a 2-tuple or 3-tuple for a given constraint.
 */
struct tuple
{
    Boolean  bad;			/* TRUE if tuple not useable */
    struct pval  *pvalp[MAXARGS];	/* pointer to 2 or 3 pval's  */
};



/*
 *  Corresponding values -- this holds one set of corresponding values.
 */
struct corr
{
    short  cvalx[MAXARGS];	/* each element is a qmagx value  */
};



/*
 *  "state" contains (or points to) all data that form a state.
 */
struct state
{
    unsigned int checksum;		/* for fast state equality check */
    short  time;			/* point/interval value of time */
    short  npreds;			/* number of predecessor states */
    struct state *predp[MAXNEIGHBORS];	/* indices of predecessor states */
    short  nsuccs;			/* number of successor states */
    struct state *succp[MAXNEIGHBORS];	/* indices of successor states */
    struct qval *qvalp;			/* pointer to N successive qval's,
					 * one for each parameter */
    struct scorr *scorrp;		/* pointer to M successive scorr's,
					 * one for each constraint */
};



/*
 *  State correspondences.  For each state there is one of these structures
 *  for every constraint.
 */
struct scorr
{
    short  ncorrs;		/* # correspondences for this constraint */
    struct corr  *corrp;	/* pointer to successive corr's */
};
