;;; -*- Mode:Common-Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-

;;; Copyright (c) 1990 by James Crawford.
;;;  $Id: atrace.lisp,v 1.1 92/04/16 09:30:24 clancy Exp $

;;;                        ****** ATRACE ******

; Trace utilities for Algernon.

; All trace utilities return their FIRST argument except for trace functions
; for alogic and arules which always return a list of bindings or a stream,
; and trace functions for aframes which return their value argument.

; Could be optimized by making all these routines macros.

; These routines are proclaimed to be inline so all of Algernon should be recompiled
; if they are changed:

(proclaim '(inline TRACE-CONTRA))
(proclaim '(inline TRACE-INTEREST))
(proclaim '(inline TRACE-LOGIC))
(proclaim '(inline TRACE-OFF))
(proclaim '(inline TRACE-ALL))
(proclaim '(inline DEFAULT-TRACE))
(proclaim '(inline TAB-OVER))
(proclaim '(inline SET-TAB-OUT))
(proclaim '(inline SET-TAB-IN))
(proclaim '(inline RESET-TAB))
(proclaim '(inline TRACE-PREP-PATH))
(proclaim '(inline TRACE-PARTITION-UPDATES-END))
(proclaim '(inline TRACE-PARTITION-UPDATES))
(proclaim '(inline TRACE-PARTITION-RETRIEVAL))
(proclaim '(inline TRACE-PROPAGATION-END))
(proclaim '(inline TRACE-PROPAGATION-BEGIN))
(proclaim '(inline TRACE-SETS-IARS-COMPLETION-WRT-FRAME))
(proclaim '(inline TRACE-RULE-COMPLETION))
(proclaim '(inline TRACE-RULE-ADDITION))
(proclaim '(inline TRACE-RULE-END))
(proclaim '(inline TRACE-RULE-START))
(proclaim '(inline TRACE-ITERATIONS-END))
(proclaim '(inline PRED-LIST-2-LIST))
(proclaim '(inline TRACE-ITERATION))
(proclaim '(inline TRACE-ITERATIONS-START))
(proclaim '(inline TRACE-CONTRADICTION-RESOLUTION))
(proclaim '(inline TRACE-ASSUMPTION-DROP))
(proclaim '(inline TRACE-DROP-AN-ASSUMP))
(proclaim '(inline TRACE-CONTRADICTION))
(proclaim '(inline TRACE-VALUE-GET))
(proclaim '(inline TRACE-VALUE-DELETION))
(proclaim '(inline TRACE-NEW-ASSUMPTION-INSERTION-END))
(proclaim '(inline TRACE-NEW-ASSUMPTION-INSERTION-BEGIN))
(proclaim '(inline TRACE-ASSUMPTION-INSERTION))
(proclaim '(inline TRACE-NEW-VALUE-INSERTION-END))
(proclaim '(inline TRACE-NEW-VALUE-INSERTION-BEGIN))
(proclaim '(inline TRACE-VALUE-INSERTION))
(proclaim '(inline TRACE-ASSERT-END))
(proclaim '(inline TRACE-ASSERT-BEGIN))
(proclaim '(inline TRACE-QUERY-END))
(proclaim '(inline TRACE-QUERY-BEGIN))
(proclaim '(inline TRACE-NEW-PRED))
(proclaim '(inline END-TRACE))
(proclaim '(inline INIT-TRACE))
(proclaim '(inline TRACE-FPUT))
(proclaim '(inline TRACE-DELETION))
(proclaim '(inline TRACE-FRAME-SEARCH-END))
(proclaim '(inline TRACE-FRAME-SEARCH-BEGIN))
(proclaim '(inline TRACE-FRAME-CREATION))
(proclaim '(inline TRACE-NAME-RECORDING))
(proclaim '(inline TRACE-NAME-RETRIEVAL))

; Dot tracing:
(defparameter *dot-trace* t)

; Parameter to remember if we are tracing:
(defparameter *tracing* t "True if some level of tracing is active.")
(defparameter *trace-name* 'default "Type of tracing currently being done")

; Parameters to control output format:
(defparameter *indent-level* 0)
(defparameter *white-space* 2)
(defparameter *trace-add-blank-lines* nil)


; **** AMATCH ****
;    - currently prints no messages -- might want to trace attempts and successes ??


; **** ANAMES ****

(defparameter *trace-names* nil)

(defun trace-name-retrieval (name values)
   (cond (*trace-names*
	  (tab-over)
	  (format t "From name ~a retrieving object ~a." name values)))
   name
)


(defun trace-name-recording (object name)
  (cond (*trace-names*
	 (tab-over)
	 (format t "For name ~a recording the object ~a." name object)))
  object)

; **** AFRAMES ****

(defparameter *trace-frames* nil)
(defparameter *trace-frame-accesses* nil)
(defparameter *trace-frame-creation* nil)


(defun trace-frame-creation (name)
  (cond (*dot-trace*
	 (dot-present "+" name))
	(*trace-frame-creation*
	 (tab-over)
	 (format t "Creating new frame: ~(~a~)." name)))
  name)


(defun trace-frame-search-begin (frame slot facet)
  (cond (*trace-frame-accesses*
	 (tab-over)
	 (format t "Looking for a value for frame ~a slot ~a facet ~a." frame slot facet)
	 (set-tab-in)))
  frame)

(defun trace-frame-search-end (value)
  (cond (*trace-frame-accesses*
	 (tab-over)
	 (format t "Found value ~a." value)
	 (set-tab-out)))
  value)

; trace-frame-access made into macro.

;(defun trace-frame-access (frame slot facet value)
;  (cond (*trace-frame-accesses*
;	 (tab-over)
;	 (format t "Accessing value ~a from frame ~a slot ~a facet ~a." value frame slot facet)))
;  value
;)

(defun trace-deletion (frame slot facet value)
  (cond (*trace-frames*
	 (tab-over)
	 (format t "Deleteing value ~a from frame ~a slot ~a facet ~a." value frame slot facet)))
  value
)

(defun trace-fput (frame slot facet value)
  (cond (*trace-frames*
	 (tab-over)
	 (format t "Adding value ~a to frame ~a slot ~a facet ~a." value frame slot facet)))
  value
)


; **** ALOGIC ****

(defparameter *trace-alogic* nil)
(defparameter *trace-alists* nil)

(defun init-trace ()
  (reset-tab)
  (reset-presenter)
  (if *tracing* (format t "~&~%  ** Beginning ~(~a~) Trace **" *trace-name*)))

(defun end-trace ()
  (if *tracing* (format t "~%  ** End ~(~a~) Trace **~%" *trace-name*)))

(defun trace-new-pred (pred)
  (cond ((and *dot-trace* *top-level*)
	 (dot-terpri)
	 (dot-present "p" pred))
	((and *trace-alogic* *top-level*)
	 (terpri)
	 (tab-over)
	 (pp-list *indent-level* 80 *indent-level* t pred)))
  pred)

(defun trace-query-begin (pred result)
  (cond ((and *dot-trace* *top-level*)
	 (dot-present "?" (substitute-bindings pred (aresult-sub result))))
	(*trace-alogic*
	 (tab-over)
	 (format t "Querying: ")
	 (pp-list (+ 9 *indent-level*) 80 (+ 9 *indent-level*) t
                  (substitute-bindings pred (aresult-sub result)))
	 (format t ".")
	 (set-tab-in)))
  pred)

(defun trace-query-end (pred stream)
  (cond (*trace-alogic*
	 (set-tab-out)
	 (tab-over)
	 (cond (stream
		(format t "Query succeeded: ")
		(pp-list (+ 17 *indent-level*) 80 (+ 17 *indent-level*) nil
			 (mapcar #'(lambda (result) 
                                     (substitute-bindings pred (aresult-sub result)))
				 stream))
		(format t "."))
	       (t
		(format t "Query failed.")))))
  stream)


(defun trace-assert-begin (pred result)
  (cond ((and *dot-trace* *top-level*)
	 (dot-present "!" (substitute-bindings pred (aresult-sub result))))
	(*trace-alogic*
	 (tab-over)
	 (format t "Asserting: ")
	 (pp-list (+ 11 *indent-level*) 80 (+ 11 *indent-level*) t
                  (substitute-bindings pred (aresult-sub result)))
	 (format t ".")
	 (set-tab-in)))
  pred)

(defun trace-assert-end (pred stream)
  (cond (*trace-alogic*
	 (set-tab-out)
	 (tab-over)
	 (cond (stream
		(format t "Assert succeeded: ")
		(pp-list (+ 18 *indent-level*) 80 (+ 18 *indent-level*) nil
			 (mapcar #'(lambda (result)
                                     (substitute-bindings pred (aresult-sub result)))
				 stream))
		(format t "."))
	       (t
		(format t "Assert failed.")))))
  stream)

; **** AIDG ****

(defparameter *trace-values* nil)
(defparameter *trace-new-values* nil)
(defparameter *trace-assumptions* nil)
(defparameter *trace-retrieval* nil)

(defun trace-value-insertion (pred result)
  (cond (*trace-values*
	 (tab-over)
	 (let* ((outstr (format nil "Inserting old value ~(~a~). " pred))
		(outcol (+ *indent-level* (length outstr))))
	   (format t outstr)
	   (pp-result (+ *indent-level* 4) 80 outcol result))))
  pred)

(defun trace-new-value-insertion-begin (pred result)
  (cond ((and (eql *search-strategy* 'depth-first) *dot-trace*)
	 (dot-present "!" pred))
	(*trace-new-values*
	 (tab-over)
	 (let* ((outstr (format nil "Inserting new value ~(~a~). " pred))
		(outcol (+ *indent-level* (length outstr))))
	   (format t outstr)
	   (pp-result (+ *indent-level* 4) 80 outcol result))
	 (set-tab-in)))
  pred)

(defun trace-new-value-insertion-end (pred success)
  (cond (*trace-new-values*
	 (cond ((not success)
		(tab-over)
		(format t "New value ~(~a~) inconsistent." pred)))
	 (set-tab-out)))
  success)

(defun trace-assumption-insertion (pred result)
  (cond (*trace-assumptions*
	 (tab-over)
	 (let* ((outstr (format nil "Inserting old assumption ~(~a~)." pred))
		(outcol (+ *indent-level* (length outstr))))
	   (format t outstr)
           (tab-over)
	   (pp-result (+ *indent-level* 4) 80 outcol result))))
  pred)

(defun trace-new-assumption-insertion-begin (pred result)
  (cond (*dot-trace*
	 (dot-present "a" pred))
	(*trace-assumptions*
	 (tab-over)
	 (let* ((outstr (format nil "Inserting new assumption ~(~a~)." pred))
		(outcol (+ *indent-level* (length outstr))))
	   (format t outstr)
           (tab-over)
	   (pp-result (+ *indent-level* 4) 80 outcol result))
	 (set-tab-in)))
  pred)

(defun trace-new-assumption-insertion-end (pred success)
  (cond (*trace-assumptions*
	 (cond ((not success)
		(tab-over)
		(format t "New assumption ~(~a~) inconsistent." pred)))
	 (set-tab-out)))
  success)

(defun trace-value-deletion (pred)
  (cond (*trace-values*
	 (tab-over)
	 (format t "Deleting value ~(~a~)." pred)))
  pred)

(defun trace-value-get (pred)
  (cond ((and *trace-values* *trace-retrieval*)
	 (tab-over)
	 (format t "Getting value ~(~a~)." pred)))
  pred)


; **** ACONTRA ****
;

(defparameter *trace-contra* nil)
(defparameter *trace-contradiction-resolvers* nil)

(defun trace-contradiction (pred neg-pred assump-ll not-assump-ll)
  (cond (*dot-trace*
	 (dot-present " ->" assump-ll)
	 (dot-present "c" pred)
	 (dot-present "<- " not-assump-ll))
	(*trace-contra*
         (pp-labeled-list *indent-level* "Contradiction:" (list pred neg-pred) t)
         (set-tab-in)
         (terpri)

	 (tab-over)
         (format t "~(~a~) supported by assumptions: " pred)
         (set-tab-in)
         (tab-over)
         (pp-list *indent-level* 80 *indent-level* nil assump-ll)
         (set-tab-out)

	 (tab-over)
         (format t "~(~a~) supported by assumptions: " (negate pred))
         (set-tab-in)
         (tab-over)
         (pp-list *indent-level* 80 *indent-level* nil not-assump-ll)
         (set-tab-out)

         (set-tab-out))))
	 

(defun trace-drop-an-assump (assumps)
  (cond (*trace-contra*
	 (tab-over)
	 (format t "Dropping an assumption from:")
	 (pp-list (+ 3 *indent-level*) 80 (+ 30 *indent-level*) nil assumps))))

(defun trace-assumption-drop (assump)
  (cond (*dot-trace*
	 (dot-present "d" assump))
	(*trace-contra*
         (terpri)
         (terpri)
	 (tab-over)
	 (format t "Dropping assumption: ~(~a~)." assump)
         (tab-over)
         (format t "Asserting its negation: ~(~a~)." (negate assump)))))

(defun trace-contradiction-resolution (resolver assumps)
  (cond (*trace-contradiction-resolvers*
         (tab-over)
         (format t "~@(~a~) recommends dropping assump~p: ~(~a~)."
                 resolver (length assumps) assumps))))


; **** ARULES ****
;    - traces rule application.
;    - traces automatic 'completion' of if-added rules.

(defparameter *trace-rules* nil)
(defparameter *trace-iterations* nil)
(defparameter *trace-rule-completion* nil)
(defparameter *trace-automatic-rule-addition* nil)

(defun trace-iterations-start (pred)
  (cond (*trace-iterations*
	 (tab-over)
	 (format t "Applying rules for ~(~a~)." pred)
	 (set-tab-in)))
  pred)

(defun trace-iteration (as-list qu-list)
  (cond (*dot-trace*
	 (dot-present (format nil "~a" *iteration-count*)
		      `((Assertions --- ,(pred-list-2-list as-list))
			(Queries    --- ,(pred-list-2-list qu-list)))))
	(*trace-iterations*
	 (tab-over)
	 (let* ((label (format nil "*Iteration ~a* " *iteration-count*))
		(indent-col (+ (length label) *indent-level*)))
	   (format t label)
	   (if as-list (pp-list indent-col 80 indent-col nil (cons "as-list =" as-list))
	               (format t "as-list = nil"))
	   (format t "~%~VT" indent-col)
	   (if qu-list (pp-list indent-col 80 indent-col nil (cons "qu-list =" qu-list))
	               (format t "qu-list = nil"))))))

;;; pred-list-2-list -- Inefficient but simple function to turn a pred list into
;;; a normal list (see arules).
;;;
(defun pred-list-2-list (pred-list)
  (let (list)
    (map-over-preds #'(lambda (pred) (push pred list)) pred-list)
    list))

(defun trace-iterations-end ()
  (cond (*trace-iterations*
	 (set-tab-out)
	 (tab-over)
	 (format t "All rules applied."))))

(defun trace-rule-start (rule result propagation)
  (cond (*dot-trace*
         (setq rule (substitute-bindings rule (aresult-sub result)))
	 (if propagation
	     (dot-present "," rule)
	     (dot-present "." rule)))
	(*trace-rules*
	 (tab-over)
	 (format t "Applying: ")
	 (let* ((alist (aresult-sub result))
                (sub-rule (if (eql alist 'failed)
			     rule
			     (substitute-bindings rule alist))))
	   (pp-rule (+ *indent-level* 10) (cons sub-rule result)))
	 (format t ".")
	 (set-tab-in)))
  result)

(defun trace-rule-end (stream)
  (cond (*trace-rules*
	 (set-tab-out)
	 (tab-over)
	 (cond (*trace-alists*
		(format t "Rule left the bindings: ")
		(pp-list (+ *indent-level* 2) 80 (+ *indent-level* 24) t stream))
	       (t
		(if
		  stream
		  (format t "Rule applied.")
		  (format t "Rule failed."))))))
  stream)

(defun trace-rule-addition (new-rule-pair)
  (cond (*dot-trace*
	 (dot-present ">" new-rule-pair))
	(*trace-automatic-rule-addition*
	 (tab-over)
	 (format t "Added rule: ")
	 (pp-rule (+ *indent-level* 12) new-rule-pair))))


(defun trace-rule-completion (rule-pair)
  (cond (*trace-rule-completion*
	 (tab-over)
	 (format t "Completing rule: ")
	 (pp-rule (+ *indent-level* 17) rule-pair))))


(defun trace-sets-iars-completion-wrt-frame (frame set)
  (cond (*trace-rule-completion*
	 (tab-over)
	 (format t "Completing if-added rules of ~(~a~) wrt ~(~a~)." set frame))))
	 


; **** ADEPNET ****

(defparameter *trace-depnet* nil)
(defparameter *trace-top-level-propagate* nil)

(defun trace-propagation-begin (pred)
  (cond ((or *trace-depnet* (and *top-level* *trace-top-level-propagate*))
	 (tab-over)
	 (format t "Propagating addition of ~(~a~)." pred)
	 (set-tab-in))))

(defun trace-propagation-end ()
  (cond ((or *trace-depnet* (and *top-level* *trace-top-level-propagate*))
	 (set-tab-out))))


; **** APART ****

(defparameter *trace-partitions* nil)
(defparameter *trace-partition-updates* nil)

(defun trace-partition-retrieval (partitions frame slot)
  (cond ((and *trace-partitions* partitions)
	 (tab-over)
	 (let* ((label (format nil "Partitions for frame-slot <~(~a~),~(~a~)>: " frame slot))
		(indent-col (+ (length label) *indent-level*)))
	   (format t label)
	   (pp-list indent-col 80 indent-col nil partitions))))
  partitions)

(defun trace-partition-updates (partitions)
  (cond (*dot-trace*
	 (dot-present "[" partitions))
	(*trace-partition-updates*
	 (tab-over)
	 (let* ((label (format nil "Updating partition~P: " (length partitions)))
		(indent-col (+ (length label) *indent-level*)))
	   (format t label)
	   (pp-list indent-col 80 indent-col nil partitions))))
  partitions)

(defun trace-partition-updates-end (all-inserted-values old-inserted-values)
  (cond (*dot-trace*
	 (dot-present "]" (set-difference all-inserted-values old-inserted-values :test #'equal)))
	(*trace-partition-updates*
	 (tab-over)
	 (let* ((new-preds (set-difference all-inserted-values old-inserted-values :test #'equal))
		(label (format nil "New pred~P: " (length new-preds)))
		(indent-col (+ (length label) *indent-level*)))
	   (format t label)
	   (pp-list indent-col 80 indent-col nil new-preds)))))

; **** APREP ****

(defparameter *trace-prep* nil)

(defun trace-prep-path (out-path)
  (cond (*trace-prep*
	 (format t "~% Pre-processor output:~&     ")
	 (pp-list 5 80 5 t out-path)))
  out-path)

; Indenting utilities to make it all readable

(defun reset-tab ()
  (setq *indent-level* 1))

(defun set-tab-in ()
  (setq *indent-level* (+ *indent-level* *white-space*)))

(defun set-tab-out ()
  (setq *indent-level* (- *indent-level* *white-space*)))

(defun tab-over ()
  (format t "~&~VT" *indent-level*))


; Finally some short cuts for setting the flags:

; DEFAULT TRACE
;
; Tracing as set when Algernon is reset.  Should agree initial values given
; by defparameters above.
;
(defun default-trace ()
  (setq *tracing* nil)
  (setq *trace-name* 'default)
  (setq *dot-trace* (find 'dot-trace *algy-features*))
  (setq *trace-names* nil)
  (setq *trace-frames* nil)
  (setq *trace-frame-accesses* nil)
  (setq *trace-frame-creation* nil)
  (setq *trace-alogic* nil)
  (setq *trace-alists* nil)
  (setq *trace-rules* nil)
  (setq *trace-iterations* nil)
  (setq *trace-automatic-rule-addition* nil)
  (setq *trace-rule-completion* nil)
  (setq *trace-values* nil)
  (setq *trace-new-values* nil)
  (setq *trace-assumptions* nil)
  (setq *trace-contra* nil)
  (setq *trace-retrieval* nil)
  (setq *trace-depnet* nil)
  (setq *trace-top-level-propagate* nil)
  (setq *trace-partitions* nil)
  (setq *trace-partition-updates* nil)
  (setq *trace-prep* nil)
)
  

(defun trace-all ()
  (setq *tracing* t)
  (setq *trace-name* 'universal)
  (setq *dot-trace* nil)
  (setq *trace-names* t)
  (setq *trace-frames* t)
  (setq *trace-frame-accesses* t)
  (setq *trace-frame-creation* t)
  (setq *trace-alogic* t)
  (setq *trace-alists* t)
  (setq *trace-rules* t)
  (setq *trace-iterations* t)
  (setq *trace-automatic-rule-addition* t)
  (setq *trace-rule-completion* t)
  (setq *trace-values* t)
  (setq *trace-new-values* t)
  (setq *trace-assumptions* t)
  (setq *trace-contra* t)
  (setq *trace-retrieval* t)
  (setq *trace-depnet* t)
  (setq *trace-top-level-propagate* t)
  (setq *trace-partitions* t)
  (setq *trace-partition-updates* t)
  (setq *trace-prep* t)
)

(defun trace-off ()
  (setq *tracing* nil)
  (setq *trace-name* nil)
  (setq *dot-trace* nil)
  (setq *trace-names* nil)
  (setq *trace-frames* nil)
  (setq *trace-frame-accesses* nil)
  (setq *trace-frame-creation* nil)
  (setq *trace-alogic* nil)
  (setq *trace-alists* nil)
  (setq *trace-rules* nil)
  (setq *trace-iterations* nil)
  (setq *trace-automatic-rule-addition* nil)
  (setq *trace-rule-completion* nil)
  (setq *trace-values* nil)
  (setq *trace-new-values* nil)
  (setq *trace-assumptions* nil)
  (setq *trace-contra* nil)
  (setq *trace-retrieval* nil)
  (setq *trace-depnet* nil)
  (setq *trace-top-level-propagate* nil)
  (setq *trace-partitions* nil)
  (setq *trace-partition-updates* nil)
  (setq *trace-prep* nil)
)

(defun trace-logic ()
  (setq *tracing* t)
  (setq *trace-name* 'logic)
  (setq *dot-trace* nil)
  (setq *trace-names* nil)
  (setq *trace-frames* nil)
  (setq *trace-frame-accesses* nil)
  (setq *trace-frame-creation* t)
  (setq *trace-alogic* t)
  (setq *trace-alists* nil)
  (setq *trace-rules* t)
  (setq *trace-iterations* t)
  (setq *trace-automatic-rule-addition* t)
  (setq *trace-rule-completion* t)
  (setq *trace-values* t)
  (setq *trace-new-values* t)
  (setq *trace-assumptions* t)
  (setq *trace-contra* t)
  (setq *trace-retrieval* nil)
  (setq *trace-depnet* t)
  (setq *trace-top-level-propagate* t)
  (setq *trace-partitions* nil)
  (setq *trace-partition-updates* t)
  (setq *trace-prep* nil)
)

(defun trace-interest ()
  (setq *tracing* t)
  (setq *trace-name* 'basic)
  (setq *dot-trace* nil)
  (setq *trace-names* nil)
  (setq *trace-frames* nil)
  (setq *trace-frame-accesses* nil)
  (setq *trace-frame-creation* t)
  (setq *trace-alogic* nil)
  (setq *trace-alists* nil)
  (setq *trace-rules* nil)
  (setq *trace-iterations* t)
  (setq *trace-automatic-rule-addition* nil)
  (setq *trace-rule-completion* nil)
  (setq *trace-values* nil)
  (setq *trace-new-values* t)
  (setq *trace-assumptions* t)
  (setq *trace-contra* t)
  (setq *trace-retrieval* nil)
  (setq *trace-depnet* nil)
  (setq *trace-top-level-propagate* nil)
  (setq *trace-partitions* nil)
  (setq *trace-partition-updates* nil)
  (setq *trace-prep* nil)
)

(defun trace-contra ()
  ;; Avoid duplicate output:
  ;; (this variable belongs to app so this reference violates modularity ...)
  (setq *show-other-kb-changes* nil)

  (setq *tracing* t)
  (setq *trace-name* 'RAA)
  (setq *dot-trace* nil)
  (setq *trace-names* nil)
  (setq *trace-frames* nil)
  (setq *trace-frame-accesses* nil)
  (setq *trace-frame-creation* nil)
  (setq *trace-alogic* nil)
  (setq *trace-alists* nil)
  (setq *trace-rules* nil)
  (setq *trace-iterations* nil)
  (setq *trace-automatic-rule-addition* nil)
  (setq *trace-rule-completion* nil)
  (setq *trace-values* nil)
  (setq *trace-new-values* nil)
  (setq *trace-assumptions* t)
  (setq *trace-contra* t)
  (setq *trace-retrieval* nil)
  (setq *trace-depnet* nil)
  (setq *trace-top-level-propagate* nil)
  (setq *trace-partitions* nil)
  (setq *trace-partition-updates* nil)
  (setq *trace-prep* nil)
)

; Code for macro with-trace-off appears in aglobals.

