;;; -*- Mode:Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-

;;; Copyright (c) 1990 by James Crawford and Benjamin Kuipers.
;;;  $Id: arules.lisp,v 1.1 92/04/16 09:30:20 clancy Exp $

;;;                        ****** ARULES ******

;;; Currently assumes that frames, slots, facets, and variables can all be compared using
;;; eq.

;;; First a vars to keep track of rules:
(defvar *num-rules* 0 "Number of rules in knowledge-base.")
(defvar *rule-application-id* 0 "Number of times a rule applied.")

;;; Then some variables to do chaining.  These vars are only used in breadth-first
;;; search.

;;; (Note: *as-list* and *qu-list* are currently referenced by adepnet.)
(defvar *as-list* nil "Assertions made durring current iteration.")
(defvar *qu-list* nil "Queries made durring current iteration.")
(defvar *old-as-list* nil "Assertions made durring last iteration for which rules have not yet been fired.")
(defvar *old-qu-list* nil "Queries made durring last iteration for which rules have not yet been fired.")

;;; Hack as of 3/7/90 so that *as-list* and *qu-list* are kept as assoc lists
;;; (or frames and then slots).  This is for efficiency in large problems.  If
;;; we get to really big problems we should probably go to hash tables.

;;; FIRE-IF-ADDED-RULES
;;;
(defun FIRE-IF-ADDED-RULES (pred)
  (fire-if-added-rules-for-preds (list pred)))


;;; FIRE-IF-ADDED-RULES-FOR-PREDS applies all applicable if-added rules for preds
;;; in pred-list.
;;;
(defun FIRE-IF-ADDED-RULES-FOR-PREDS (pred-list)
  (case *search-strategy*
    ('depth-first
     (let ((*top-level* nil))
       (dolist (pred pred-list)
         (mapc #'(lambda (rule-pair)
                   (make-rule-current rule-pair)
                   (apply-rule rule-pair nil)
                   (update-if-added-rule-pair-wrt-pred rule-pair pred))
               (if (negated pred) (get-rules pred @n-if-added)
                   (get-rules pred @if-added))))))
    (t
     (cond (*top-level*
            (let ((*top-level* nil)
                  (*qu-list* nil)
                  (*as-list* nil))
              (dolist (pred pred-list)
                (setq *as-list* (add-pred pred *as-list*)))
              (trace-iterations-start pred-list)
              (fire-rules)
              (trace-iterations-end)))
           (t
            (dolist (pred pred-list)
              (setq *as-list* (add-pred pred *as-list*))))))))


;;; FIRE-IF-NEEDED-RULES applies all applicable if-needed rules for pred.
;;;
(defun FIRE-IF-NEEDED-RULES (pred)
  (if (visit-frame-slot pred)                             ; Don't visit if visited already.
    (case *search-strategy*
      ('depth-first
       (let ((*top-level* nil))
         (mapc #'(lambda (rule-pair)
                   (make-rule-current rule-pair)
                   (apply-rule rule-pair nil))
               (if (negated pred) (get-rules pred @n-if-needed)
                   (get-rules pred @if-needed)))))
      (t
       (cond (*top-level*
              (let ((*top-level* nil)
                    (*as-list* nil)
                    (*qu-list* nil))
                (setq *qu-list* (add-pred pred *qu-list*))
                (trace-iterations-start pred)
                (fire-rules)
                (trace-iterations-end)))
             (t
              (setq *qu-list* (add-pred pred *qu-list*))))))))

;;; FIRE-SELECTED-RULES applies the rules in rule-pairs for pred.
;;;
(defun fire-selected-rules (rule-pairs)
  (cond (*top-level*
	 (let ((*top-level* nil)
	       (*as-list* nil)
	       (*qu-list* nil))
	   (dolist (rule-pair rule-pairs)
	     (make-rule-current rule-pair)
	     (apply-rule rule-pair nil))
	   (when (or *as-list* *qu-list*)
	     (fire-rules))))
	(t
	 (dolist (rule-pair rule-pairs)
	   (make-rule-current rule-pair)
	   (apply-rule rule-pair nil)))))

;;; As of 11/18/88 get-rules returns pair (rule . assump-ll), and
;;; assumptions passed to apply-rule.
;;;
;;; As of 1/14/90 *as-list* is just a list of predicates.
;;;
;;; Modified 2/5/90 so that *old-as-list* and *old-qu-list* change to always
;;; hold the predicates which have not yet been processed.
;;;
(defun fire-rules ()
  (let ((*iteration-count* 0)
        (*old-as-list* nil)
        (*old-qu-list* nil))
    (loop
      (incf *iteration-count*)
      (trace-iteration *as-list* *qu-list*)
      ;;(format t " ~a" (max (length *as-list*) (length *qu-list*)))
      (setq *old-as-list* *as-list*)
      (setq *old-qu-list* *qu-list*)
      (setq *as-list* nil)
      (setq *qu-list* nil)
      
      (loop
        (if (null *old-as-list*) (return))
        (let ((pred (first-pred-list *old-as-list*)))
          (setq *old-as-list* (nrest-pred-list *old-as-list*))
          (mapc #'(lambda (rule-pair)
                    (make-rule-current rule-pair)
                    (apply-rule rule-pair nil)
                    (update-if-added-rule-pair-wrt-pred rule-pair pred))
                (if (negated pred) (get-rules pred @n-if-added)
                    (get-rules pred @if-added)))))
      
      (loop
        (if (null *old-qu-list*) (return))
        (let ((pred (first-pred-list *old-qu-list*)))
          (setq *old-qu-list* (nrest-pred-list *old-qu-list*))
          (mapc #'(lambda (rule-pair)
                    (make-rule-current rule-pair)
                    (apply-rule rule-pair nil))
                (if (negated pred) (get-rules pred @n-if-needed)
                    (get-rules pred @if-needed)))))
      
      (when (and (not *as-list*) (not *qu-list*))
        (setq *max-iterations* (max *iteration-count* *max-iterations*))
        (return)))))

;;; STORE-RULES stores rules in frame.
;;;
(defun STORE-RULES (frame rules class)
  (store-rules-with-results frame
			    (mapcar #'(lambda (rule) (cons rule (new-aresult)))
                                    rules)
			    class))

;;; store-rules-with-results.
;;;
;;; Hacked 1/15/90 so pairs are of form (rule . result).
;;;
;;; Distructively modifies the list structure of rule-pairs
;;; (to remove pairs in which the rule is nil).
;;;
(defun STORE-RULES-WITH-RESULTS (frame rule-pairs class)
  (if (not (framep frame))
    (throw 'error (format nil "~%Attempt to add rules to non-frame ~a." frame)))
  (setq rule-pairs (delete-if #'null rule-pairs :key #'car))
  (if *contra-positive*
    (setq rule-pairs
          (mapcan #'(lambda (pair)
                      (let ((cp (contra-positive pair)))
                        (if cp
                          (list pair cp)
                          (list pair))))
                  rule-pairs)))
  (setq *num-rules* (+ *num-rules* (length rule-pairs)))
  (mapc #'(lambda (pair)
            (let* ((slot (rule-slot (car pair) (cdr pair)))
                   (type (rule-type (car pair)))
                   (rule-facet (rule-facet type class)))
              (fput frame slot rule-facet pair)
              (if (and (or (eql type @if-added) (eql type @n-if-added))    ; For if-addded rules
                       (eql class 'frame))                                 ; which are associated with a set.
                (complete-set-wrt-rule-pair frame pair))))
        rule-pairs))

;;; DEL-RULES deletes all rules for slot of frame.
;;;
(defun DEL-RULES (frame slot)
  (fclear-facet frame slot @if-needed)
  (fclear-facet frame slot @if-added)
  (fclear-facet frame slot @n-if-needed)
  (fclear-facet frame slot @n-if-added))

(defun DEL-SRULES (frame slot)
  (fclear-facet frame slot @sif-needed)
  (fclear-facet frame slot @sif-added)
  (fclear-facet frame slot @sn-if-needed)
  (fclear-facet frame slot @sn-if-added))

(defun DEL-RULE (frame rule)
  (fremove frame (rule-slot rule) (rule-facet (rule-type rule) 'frame) rule))

(defun DEL-SRULE (frame rule)
  (fremove frame (rule-slot rule) (rule-facet (rule-type rule) 'slot) rule))

(defun RESET-RULES ()
  (setq *as-list* nil)
  (setq *qu-list* nil)
  (setq *old-as-list* nil)
  (setq *old-qu-list* nil)
  (setq *num-rules* 0)
  (setq *rule-application-id* 0))


;;; apply-rule: Applies rule.
;;;   To apply a rule:
;;;     - query the antecedent and build alist
;;;     - assert consequent.
;;;
;;; Modified 12/6/89 to record no dependencies for the antecedents of if-added rules.
;;; (Not needed because if-added rules are completed.)
;;; Propagate is true only when this rule is being re-fired as part of the propigation
;;; of a new value --- it is used only in formatting the tracing output.
;;;
(defun apply-rule (pair propagate)
  (let* ((rule (car pair))
         (initial-result (cdr pair))
         (rule-type (rule-type rule)))
    
    (setq *rule-count* (+ 1 *rule-count*))
    (rule-accountant rule initial-result)
    
    (trace-rule-start rule initial-result propagate)

    ;; Enable depnet:
    (with-depnet
	(let* ((antecedent (antecedent rule))
	       (antecedent-result
		;; Some search strategies require forward-chaining rules to record dependencies:
		(if (or (eql rule-type @n-if-added) (eql rule-type @if-added))
		    (if (not (eq *search-strategy* 'depth-first))
			(with-no-depnet (internal-query antecedent (list initial-result)))
			(internal-query
			 (cdr antecedent)
			 (with-no-depnet
			     (with-no-back-chaining
				 (internal-query (list (car antecedent))
						 (list initial-result))))))
		    (internal-query antecedent (list initial-result))))
	       (consequent-result
		(if antecedent-result
		    (internal-assert (consequent rule) antecedent-result))))
	  (end-current-rule)
	  (trace-rule-end antecedent-result)
	  consequent-result))))

#| OLD VERSION
(defun apply-rule (pair propagate)
  (let* ((rule (car pair))
         (initial-result (cdr pair))
         (rule-type (rule-type rule)))
    
    (setq *rule-count* (+ 1 *rule-count*))
    (rule-accountant rule initial-result)
    
    (trace-rule-start rule initial-result propagate)

    ;; Enable depnet:
    (with-depnet
      (let* ((antecedent-result
	       ;; Some search strategies require forward-chaining rules to record dependencies:
	       (if (and (or (eql rule-type @n-if-added) (eql rule-type @if-added))
			(not (eql *search-strategy* 'depth-first)))
		   (with-no-depnet (internal-query (antecedent rule) (list initial-result)))
		   (internal-query (antecedent rule) (list initial-result))))
	     (consequent-result
	       (if antecedent-result
		   (internal-assert (consequent rule) antecedent-result))))
	(end-current-rule)
	(trace-rule-end antecedent-result)
	consequent-result))))
|#

;;; COUNT-RULES: Counts the total number of rules in the system.
;;;
(defun COUNT-RULES ()
  *num-rules*)


;;; Utility routines.

;;; Fullp checks number restriction for slot to see if slot is full.
;;; Slot only full if full of values without assumptions.
;;;
(defun fullp (frame slot)
  (let* ((num (car (fget slot @slot-props @num-res))))
    (if num
        (<= num (length (fs-get-values-woa frame slot @value)))
	nil)))

(defun overfullp (frame slot)
  (let* ((num (car (fget slot @slot-props @num-res))))
    (if num
        (< num (length (fs-get-values-woa frame slot @value)))
	nil)))


;;; Rename-variables-in.
;;; The variables in a rule have to be renamed to be unique, to avoid 
;;; accidental collisions with existing bindings.  Thus, rules are copied
;;; for each application with new variable-names (actually new structures with
;;; the same name).
;;;
;;; Modified 6/25/90 to take a result and apply substitution first.
;;;
(defun rename-variables-in (exp result)
  (let ((new-exp (substitute-bindings exp (aresult-sub result))))
    (rename-vars new-exp)
    new-exp))

;;; Rename-vars.
;;; Distructuvely replace all variables in l with new structures.
;;;
(defun rename-vars (l &optional alist)
  (if (consp (car l))
      (setq alist (rename-vars (car l) alist))
      (if (variable? (car l))
	  (let ((l-struct (cdr (assoc (car l) alist :test #'eq))))
	    (when (not l-struct)
	      (setq l-struct (copy-algy-variable (car l)))
	      (setq alist (acons (car l) l-struct alist)))
	    (rplaca l l-struct))))
  (if (consp (cdr l))
      (setq alist (rename-vars (cdr l) alist))
      (if (variable? (cdr l))
	  (let ((l-struct (cdr (assoc (cdr l) alist :test #'eq))))
	    (when (not l-struct)
	      (setq l-struct (copy-algy-variable (cdr l)))
	      (setq alist (acons (cdr l) l-struct alist)))
	    (rplacd l l-struct))))
  alist)



;;; Collect all rules of type for pred, and unifies them with pred.
;;; Collects rules by following isa links and rules stored with the relation in pred.
;;;
;;; Modified 12/7/89 to not rename variables in if-added rules (not needed since
;;; there are no vars in pred which they could be the same as).
;;;
;;; Modified 1/15/90 to return list of pairs (rule . result).
;;;
;;; Modified 6/25/90 to include switch always-unify.  In general get-rules matches
;;; against the keys of if-added rules to save time.  The switch forces get-rules
;;; to unify instead.  This should only be needed when listing all possible if-added
;;; rules for debugging purposes.
;;;
;;; Modified 6/12/91 to get rules from sets frame is member of, then rules
;;; for slot, and finally rules for sets slot is a member of.
;;;
(defun get-rules (pred type &optional always-unify)
  (let ((unify (or always-unify (eql type @if-needed) (eql type @n-if-needed))))
    (nconc 
     ;; Get values from sets frame is a member of:
     (get-rules-from-frames pred (rule-facet type 'frame) unify
                            (if (slotp 'isa)
                              ;; At this point should assert a dependence of any current
                              ;; rule on isa (but don't know how in breadth first search).
                              (fs-get-values (frame pred) 'isa @value)))
     ;; And from slot and sets slot is a member of:
     (get-rules-from-frames pred (rule-facet type 'slot) unify
                            (cons (cons (slot pred) '(nil))
                                  (if (slotp 'isa)
                                    ;; (again should assert depnet info)
                                    (fs-get-values (slot pred) 'isa @value)))))))

;;; Collect all rules of type for pred from the frames in alist.
;;; alist is list of pairs (value . assump-ll).
;;;
;;; Modified 1/17/90 to distructively modify results in alist.
;;;
;;; Each pair returned has a unique (and new) result.
;;;
(defun get-rules-from-frames (pred facet unify alist)
  (let ((slot (slot pred)))
    (mapcan #'(lambda (pair)
                (let ((set (car pair))
                      (assump-ll (cdr pair)))
                  (nconc
                   (mapcan #'(lambda (rule-pair)
                               (process-rule rule-pair pred unify assump-ll
                                             set @generic-rules facet))
                           (fget set @generic-rules facet))
                   (mapcan #'(lambda (rule-pair)
                               (process-rule rule-pair pred unify assump-ll
                                             set slot facet))
                           (fget set slot facet)))))
            alist)))

;;; Process-Rule: Takes a rule-pair and a predicate and unifies them to return nil or
;;; a list of a pair (rule . new-result).  assump-ll is additional assumptions to be conjuncted
;;; into new-result.  unify is used to decide to match or unify (see amatch).
;;;
;;; As of 6/23/90 calls check-value to verify the assumptions in rule-pair.
;;; frame, slot, and facet passed to that rule can be deleted if necessary.
;;;
(defun process-rule (rule-pair pred unify assump-ll frame slot facet)
  (let ((current-value (rule-check-value rule-pair nil)))
    (cond (current-value
           (let ((rule (car current-value))
                 (rule-result (cdr current-value)))
             ;; When unify we first verify might-unify before doing the
             ;; expensive variable renaming:
             (unless (and unify
                          (not (might-unify pred (car rule))))
               (if unify (setq rule (rename-variables-in rule rule-result)))
               (let ((bindings (if unify
                                 (unify (car rule) pred (aresult-sub rule-result))
                                 (match (car rule) pred (aresult-sub rule-result)))))
                 (unless (eql bindings 'failed)
                   (list (cons rule
                               (make-aresult :assump-ll
                                             (conjunct-assump-ll assump-ll
                                                                 (aresult-assump-ll rule-result))
                                             :sub bindings))))))))
          ((null (aresult-assump-ll (cdr rule-pair)))
           (fremove frame slot facet rule-pair)
           nil))))

;;; Rule-Check-Value: Translates from rule pairs, which are pairs (rule . result), and the
;;; type of pairs expected (and returned) by check-value which are of form (value . assump-ll).
;;;
;;; Returns pair (rule . new-result) or nil.
;;;
(defun rule-check-value (rule-pair visited)
  (let* ((rule (car rule-pair))
         (old-result (cdr rule-pair))
         (assump-ll (aresult-assump-ll old-result)))
    ;; Optimization:
    (if (equal assump-ll '(nil))
      rule-pair
      (let* ((check-value-pair (cons rule assump-ll))
             (checked-value (check-value check-value-pair visited))
             (new-result (copy-aresult old-result)))
        ;; check-value may have modified the cdr of check-value-pair so we copy it into rule-pair
        (setf (aresult-assump-ll (cdr rule-pair)) (cdr check-value-pair))
        (when checked-value
          (setf (aresult-assump-ll new-result) (cdr checked-value))
          (cons rule new-result))))))

;;; Contra-Positive: Routine to translate a rule into its contra-positive.
;;; In the abstract this just means that p <- q becomes (not p) -> (not q).
;;;
;;; In more detail:
;;;   p1,...,pn <- q1,...,qn
;;; becomes:
;;;   (not p1),q1,...,qn-1 -> (not qn)
;;; and:
;;;   q1,...,qn -> p1,...,pn
;;; becomes:
;;;   (not q1) <- q2,...,qn,(not p1)
;;;
;;; Notes:
;;;  1. If-added rules become if-needed rules.
;;;  2. Currently p2,...,pn ignored (due access limitations).  Perhaps there is some
;;;  analogy of rule completion that could be applied here ???
;;;
;;; Returns a new rule pair or nil if the contra-positive cannot be taken
;;; (because there is a special form in the wrong place or because the consequent
;;; of the rule is of length greater than 1).

(defun contra-positive (rule-pair)
  (let* ((text (car rule-pair))
         (ant (antecedent text))
         (conseq (consequent text))
         (new-result (copy-aresult (cdr rule-pair)))
         (new-text 
          (if (eql (length conseq) 1)   ; If consequent longer than 1 then don't take cp.
            (if (member '<- text :test #'eq)
              (let ((p1 (car conseq))
                    (qn (car (last ant))))
                (if (and (not (member (car p1) *special-forms* :test #'eq))
                         (not (member (car qn) *special-forms* :test #'eq)))
                  (append (cons (negate p1) (butlast ant))
                          (list '->)
                          (list (negate qn)))))
              (let ((p1 (car conseq))
                    (q1 (first ant)))
                (if (and (not (member (car p1) *special-forms* :test #'eq))
                         (not (member (car q1) *special-forms* :test #'eq)))
                  (append (list (negate q1))
                          (list '<-)
                          (rest ant)
                          (list (negate p1)))))))))
    (if new-text (cons new-text new-result))))


;;; Code to solve problem of if-added incompleteness.

;;; Complete-Set-Wrt-Rule-Pair -- "Completes" a new if-added rule for set by fireing and updating it for every
;;; fact which could have accessed the rule and which matches the key of its antecedent.
;;;
;;; There is a problem with elements known to be isa to the set when the
;;; rule is added, but not yet backlinked to member --- the rule is never fired or
;;; completed for them.  Eventually there should be some sort of dependency on
;;; member put in the depnet.
;;;
(defun complete-set-wrt-rule-pair (set rule-pair)
  (trace-rule-completion rule-pair)
  (let* ((frame-pairs (with-no-depnet
                        (with-no-back-chaining
                          (get-values (list 'member set)))))	; Known member of the set
	 (rule (car rule-pair))
	 (slot (rule-slot rule (cdr rule-pair)))
	 (negated (negated (car rule)))
	 ;; First collect and save a copy of all the values (since they might change as rules are fired).
	 (value-lists (mapcar #'(lambda (frame-pair) (get-value-list (car frame-pair) slot negated))
			      frame-pairs)))
    (dolist (frame-pair frame-pairs)
      (complete-frame-pair-wrt-rule-pair frame-pair rule-pair
					 (pop value-lists)))))

;;; Complete-Set-Wrt-Frame -- Completes the if-added rules associated with
;;; set with respect to frame (which is a new member of set).
;;;
(defun complete-set-wrt-frame (set frame)
  (let ((result (with-no-depnet (with-no-back-chaining (cdr (known (list 'isa frame set)))))))
    (if (not result)
	(error "Attempt to complete set ~(~a~) wrt frame not in set: ~(~a~)." set frame))
    (complete-set-wrt-frame-pair
      set
      (cons frame result))))

;;; Complete-Set-Wrt-Frame-Pair -- Completes the if-added rules associated with
;;; set with respect to frame-pair (which is a pair (frame . result)).
;;; Used when it is learned that frame is a member of set under result.
;;;
;;; Modified 1/15/90 so that pairs are (frame . result) instead of (frame . assump-ll).
;;;
;;; Modified 1/17/90 to distructively modify (cdr frame-pair).
;;;
(defun complete-set-wrt-frame-pair (set frame-pair)
  (let ((frame (car frame-pair))
	(value-lists nil)
	(negated-value-lists nil)
	(rules (get-iar-of-set set)))
    (trace-sets-iars-completion-wrt-frame frame set)

    ;; First collect and save a copy of all the values (since they might change as rules are fired).
    (dolist (rule-pair rules)
      (let* ((rule (car rule-pair))
	     (slot (rule-slot rule (cdr rule-pair)))
	     (negated (negated (car rule)))
	     (value-list (if negated 
			     (assoc slot negated-value-lists :test #'eq)
			     (assoc slot value-lists :test #'eq))))
	(unless value-list
	  (if negated
	      (push (cons slot (get-value-list frame slot negated)) negated-value-lists)
	      (push (cons slot (get-value-list frame slot negated)) value-lists)))))

    (dolist (rule-pair rules)
      (let* ((rule (car rule-pair))
	     (slot (rule-slot rule (cdr rule-pair)))
	     (negated (negated (car rule)))
	     (value-list (if negated 
			     (cdr (assoc slot negated-value-lists :test #'eq))
			     (cdr (assoc slot value-lists :test #'eq)))))
	;; Hack out the pred (isa (car frame-pair) set):
	(if (eql slot 'isa)
	    (setq value-list (delete set value-list :key #'car)))
	(complete-frame-pair-wrt-rule-pair frame-pair rule-pair value-list)))))

(defun get-value-list (frame slot negated)
  (with-no-depnet
    (with-no-back-chaining
      (get-values (if negated
		      (negate (list slot frame))
		      (list slot frame))))))

;;; Complete-Frame-Pair-Wrt-Rule-Pair -- Completes a the rule in rule-pair
;;; with respect to the frame in frame-pair.  The cdr's of rule-pair and
;;; frame-pair are their respective results.
;;;
;;; Hacked 12/6/89 to look in *old-as-list* and not fire or complete rules wrt preds
;;; which are in *old-as-list* (as they will be fired and updated anyway).
;;;
;;; Modified 1/17/90 to distructively modify (cdr frame-pair).
;;;
;;; Hacked 2/5/90 to also not fire or complete for preds in *as-list*.
;;;
;;; Modified 2/13/90 to look at current partitions and put a rule not in them
;;; in rule-queues.
;;;
(defun complete-frame-pair-wrt-rule-pair (frame-pair rule-pair value-list)
  (let* ((rule (car rule-pair))
         (frame (car frame-pair))
         (slot (rule-slot rule (cdr rule-pair)))
	 (rule-value (rule-value rule (cdr rule-pair)))
         (negated (negated (car rule))))
    (unless (eql slot @generic-rules)
      (dolist (value-pair
		(if (or (consp rule-value) (variable? rule-value))
		    value-list
		    ((lambda (x) (if x (list x))) (find rule-value value-list :key #'car))))
        (let ((pred (if negated
                      (negate (list slot frame (car value-pair)))
                      (list slot frame (car value-pair))))
              (sub (aresult-sub (cdr rule-pair))))
          (unless (or (pred-list-member pred *old-as-list*)
                      (pred-list-member pred *as-list*))
            (let ((unify-result (match (car rule) pred sub)))
              (unless (eql unify-result 'failed)
                ;;(format t "~%~%~s ~s" pred rule)
                (let ((result (nconjunct-results (cdr rule-pair)
                                                 (nconjunct-results (cdr frame-pair)
                                                                    (cdr value-pair)))))
                  (unless (null (aresult-assump-ll result))
                    (let* ((new-result (copy-aresult-with-sub result unify-result))
                           (unified-rule-pair (cons rule new-result)))
                      ;; Apply unified-rule ...
                      (if *forward-chain*
                        (if (in-partitions (frame pred) (slot pred))
                          (fire-selected-rules (list unified-rule-pair))
                          (queue-rule unified-rule-pair pred)))
                      ;; ... and complete it.
                      (update-if-added-rule-wrt-result rule new-result))))))))))))

(defun copy-aresult-with-sub (result sub)
  (let ((new-result (copy-aresult result)))
    (setf (aresult-sub new-result) sub)
    new-result))

;;; Update-If-Added-Rule-Pair-Wrt-Pred --  Updates the rule in rule-pair to reflect
;;; the fact that pred has been added.
;;;
;;; Modified 3/5/90 to halt quickly if rule is short.
;;;
(defun update-if-added-rule-pair-wrt-pred (rule-pair pred)
  (when (> (length (antecedent (car rule-pair)))
           1)
    (let ((result (with-no-depnet (with-no-back-chaining (cdr (known pred))))))
      ;; Error taken out since sometimes pred has been removed by a contradiction ...
      ;;(if (not result)
      ;;	(error "Attempt to complete rule ~(~a~) wrt unknown pred: ~(~a~)." rule-pair pred))
      (if result
        (update-if-added-rule-wrt-result
         (car rule-pair)
         (nconjunct-results (cdr rule-pair) result))))))

;;; Update-If-Added-Rule-Wrt-Result -- Updates rule under assumptions in result.
;;;
;;; Modified 11/21/89 to work even when second term on ant of rule is
;;; a :retrieve.  :lisp added 11/27/89.
;;;
;;; Modified 1/17/90 to distructively modify (cdr pred-pair).
;;;
;;; Modified 2/5/90 to complete through name, :unp, and :ask.
;;;
;;; :rules added 2/16/90.
;;;
;;; :neq added 10/14/90.
;;;
;;; :no-completion added 3/6/91.
;;;
;;; :test added 5/16/91.
;;;
(defun update-if-added-rule-wrt-result (rule result)
  (when result
    (if (eql *search-strategy* 'depth-first)
	;; In depth first search we do all completions at the end:
	(push (cons (cdr rule) result) *rules-to-complete*)
	(update-iar-now (cdr rule) result))))
             

(defun update-iar-now (new-rule result)
  (let ((first-ant (car new-rule)))
    (unless (eql (car new-rule) '->)
      (cond ((member (car first-ant) *special-forms* :test #'eq)
             (case (car first-ant) 
               (:retrieve
		 (update-iar-now (cons (second first-ant) (cdr new-rule)) result))
               (:no-completion
		 (let ((new-results (with-no-depnet
				      (with-no-back-chaining
					(internal-query (cdr first-ant) (list result))))))
		   (dolist (new-result new-results)
		     (update-iar-now (cdr new-rule) new-result))))
               (:ask
		 (update-iar-now (cons (second first-ant) (cdr new-rule)) result))
               (:unp
		 (if (with-no-depnet (not (internal-query (cdr first-ant) (list result)))) ; should we backchain here ?
		     (update-iar-now (cdr new-rule) result)))
               (:test
		 (if (eval (substitute-bindings (second first-ant)
						(aresult-sub result)))
                   (update-iar-now (cdr new-rule) result)))
	       (:lisp
		 (update-iar-now (cdr new-rule) result))
               (:neq
                (let ((frames (substitute-bindings (cdr first-ant) (aresult-sub result))))
                  (unless (eql (first frames) (second frames))
                    (update-iar-now (cdr new-rule) result))))
               (:rules
		 (update-iar-now (cdr new-rule) result))
               (t
		 (algy-warning
		   (format nil "Can't complete rule containing the form ~a." first-ant)))))
            (t
             (setq first-ant (substitute-bindings first-ant (aresult-sub result)))
	     (let ((frame (frame first-ant))
		   (slot (slot first-ant))
		   (value (value first-ant)))
	       (cond ((and (eql slot 'NAME)
			   (variable? frame)
			   (not (variable? value))
			   (not (negated first-ant)))
		      (dolist
			(new-result (extend-with-values result frame
							(objects-from-name value)))
			(update-iar-now (cdr new-rule) new-result)))
		     ((fullp frame slot)	; skip full frame-slots.
		      (dolist
			(new-result
			  (if (has-variables value)
			      (extend-with-values result value
						  (with-no-back-chaining (get-values first-ant)))
			      (extend-result result
					     (cdr (with-no-back-chaining (known first-ant))))))
			(update-iar-now (cdr new-rule) new-result)))
		     (t
		      (add-if-added-rule first-ant new-rule result)))))))))

(defun add-if-added-rule (key rule result)
  (let ((frame (frame key))
	(slot (slot  key)))
    (if (and (framep frame) (slotp slot))
	(let* ((selfset  (find-or-create-selfset frame))
	       (new-rule-pair (cons rule result)))
          (with-no-forward-chaining
	    (store-rules-with-results selfset (list new-rule-pair) 'frame))
	  (trace-rule-addition new-rule-pair)))))



;;; Retrieves or creates the selfset of frame.  The selfset of a frame is the
;;; set consisting of only the frame.
;;;
(defun find-or-create-selfset (frame)
  (let ((selfset (caar (fs-get-values frame 'selfset @value))))
    (cond (selfset)
	  (t
	   (setq selfset (make-new-frame (intern (format nil "~a-~a" frame 'selfset))))
	   (insert-value (list 'selfset frame selfset) (new-aresult))
	   ; Immediately link frame to its selfset:
	   (insert-value (list 'isa frame selfset) (new-aresult))
	   (insert-value (list 'member selfset frame) (new-aresult))
	   selfset))))


(defun get-iar-of-set (set)
  (mapcan #'(lambda (slot)
	      (mapcan #'(lambda (type) (copy-list (cdr (assoc type (cdr slot) :test #'eq))))
		      (list @if-added @n-if-added)))
	  (cdr (get set 'frame))))


;;; might-unify: Detects the cases in which key could not possibly unify with
;;; pred and returns false.  Used only to speed up the code since renaming the
;;; variables in a rule is expensive.
;;;
(defun might-unify (key pred)
  (let ((key-slot (slot key))
	(pred-slot (slot pred))
	(key-frame (frame key))
	(pred-frame (frame pred))
	(key-value (value key))
	(pred-value (value pred)))
    (and (eq (negated key) (negated pred))
         (or (eql key-slot pred-slot) (variable? key-slot) (variable? pred-slot))
	 (or (eql key-frame pred-frame) (variable? key-frame) (variable? pred-frame))
	 (or (consp key-value) (consp pred-value)
	     (eql key-value pred-value) (variable? key-value) (variable? pred-value)))))


;;; Code to efficiently maintain lists of predicates.

(defun add-pred (pred list)
  (fs-add-pred (frame pred) (slot pred) (value pred) (negated pred) list))

(defun fs-add-pred (frame slot value negated list)
  (if negated (setq value (cons '$not value)))
  (let ((cur-slots (assoc frame list :test #'eq)))
    (if cur-slots
      (let ((cur-values (assoc slot (cdr cur-slots) :test #'eq)))
        (if cur-values
          (pushnew value (cdr cur-values) :test #'equal)
          (push (list slot value) (cdr cur-slots)))
        list)
      (cons (list frame (list slot value)) list))))

(defun pred-list-member (pred list)
  (fs-pred-list-member (frame pred) (slot pred) (value pred) (negated pred) list))

(defun fs-pred-list-member (frame slot value negated list)
  (if negated (setq value (cons '$not value)))
  (member value
          (cdr (assoc slot (cdr (assoc frame list :test #'eq)) :test #'eq))
          :test #'equal))


;;; Maps a function taking one argument (a predicate) over a list of preds.
;;;
(defun map-over-preds (function list)
  (dolist (frame-list list)
    (dolist (slot-list (cdr frame-list))
      (dolist (value (cdr slot-list))
        (if (and (consp value) (eql (car value) '$not))
          (apply function (list (negate (list (car slot-list) (car frame-list) (cdr value)))))
          (apply function (list (list (car slot-list) (car frame-list) value))))))))

;;; Maps a function taking four arguments (frame slot value negated) over a list of preds.
;;;
(defun fs-map-over-preds (function list)
  (dolist (frame-list list)
    (dolist (slot-list (cdr frame-list))
      (dolist (value (cdr slot-list))
        (let ((negated nil))
          (if (and (consp value) (eql (car value) '$not))
            (setq value (cdr value) negated t))
          (apply function (list (car slot-list) (car frame-list) value negated)))))))


(defun first-pred-list (list)
  (unless (null list)
    (let* ((frame-list (first list))
           (slot-list (first (cdr frame-list)))
           (first-value (second slot-list)))
      (if (and (consp first-value) (eql (car first-value) '$not))
        (negate (list (car slot-list) (car frame-list) (cdr first-value)))
        (list (car slot-list) (car frame-list) first-value)))))

(defun nrest-pred-list (list)
  (unless (null list)
    (let ((frame-list (first list)))
      (let ((slot-list (first (cdr frame-list))))
        (rplacd slot-list (cddr slot-list))
        (if (eql (length slot-list) 1)
          (rplacd frame-list (cddr frame-list))))
      (if (eql (length frame-list) 1)
        (cdr list)
        list))))