;;; -*- Mode:Common-Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-

;;; Copyright (c) 1990 by James Crawford.
;;;  $Id: apart.lisp,v 1.1 92/04/16 09:30:16 clancy Exp $

;;;                        ****** APART ******

; Routines to manage partitions.
;
; Could be speed up by recording the current partition in the network.


; A local variable to keep track of the current partitions:
(defvar *cur-partitions* nil
              "List of the partitions of current primitive operation.")

; And the frame-slot which was used to set the current partition:
(defvar *cur-frame* nil "Current frame")
(defvar *cur-slot* nil "Currentslot")

; Set-Current-Partitions -- sets the current partitions to be those of pred.
;   Returns the partitions of pred.
;
(defun set-current-partitions (frame slot)
  (or (and (eql frame *cur-frame*)
           (eql slot  *cur-slot*))
      (let ((partitions (partitions frame slot)))
        (setq *cur-partitions* partitions
              *cur-frame* frame
              *cur-slot* slot)
        (update-partitions partitions)
        (if (not partitions)
          (algy-warning (format nil "No partitions for <~(~a~),~(~a~)>." frame slot)))
        partitions)))


; In-Partitions -- returns nil if pred is none of the current partitions.
;                  Otherwise returns T.
;
(defun in-partitions (frame slot)
  (or (and (eql frame *cur-frame*)
           (eql slot  *cur-slot*))
      (intersectionp *cur-partitions* (partitions frame slot))))


; partitions -- returns partitions of pred.  If we ever get LOTS of partitions
; this should do something more clever than a union.
;
(defun partitions (frame slot)
  (trace-partition-retrieval
      (intersect-complete (union (if (and (not (null slot)) (slotp 'slot-partition))
				     (fs-get-values-ignoring-assumptions slot 'slot-partition @value))
				 (if (and (not (null frame)) (slotp 'frame-partition))
				     (fs-get-values-ignoring-assumptions frame 'frame-partition @value))))
      frame slot))

; intersect-complete -- (distructively) completes a list of partitions under intersection.
;   (i.e. if (p1 <intersect> p2 <subset> p) and if p1 and p2 are in `list' then p will
;   be added to list).  Intersections are represented by putting (p2 p)
;   in the `intersect-sub' slot of p1 (and (p1 p) in the intersect-sub slot of p2).
;
;   This routine uses an undocumented property of dolist that the interated list
;   can be lengthened durring evaluation of the dolist without harm (and the body
;   will be evaluated on the added elements).
;
(defun intersect-complete (list)
  (let ((end (last list)))
    (dolist (partition list)
      (mapc #'(lambda (pair)
		(cond ((member (car pair) list)
		       (rplacd end (list (cdr pair)))
		       (setq end (cdr end)))))
	    (if (slotp 'intersect-sub)
		(fs-get-values-woa partition 'intersect-sub @value)))))
  list)


; Queue-Assertion -- Puts pred in the `assert-queue' of its partitions.
;
(defun queue-assertion (pred)
  (queue 'assert-queue pred (partitions (frame pred) (slot pred))))

; Queue-Query -- Puts `pred' in the `query-queue' of its partitions.
;
;   The query-queue may be used later in deciding what questions to ask
;   to solve hard problems (? when should query-queue be cleared ?).
;
(defun queue-query (pred)
  (queue 'query-queue pred (partitions (frame pred) (slot pred))))

; Queue-Rule -- Puts rule-pair in the `rule-queue's of the partitions of its pred.
;
(defun queue-rule (rule-pair pred)
  (queue 'rule-queue rule-pair (partitions (frame pred) (slot pred))))

(defun queue (queue object partitions)
  (if (slotp queue)
      (mapc #'(lambda (partition) 
                (with-no-forward-chaining
                  (insert-value (list queue partition object) (new-aresult))))
	    partitions)
      (algy-warning (format nil "Queue failed (~(~a~) not a declared slot):" queue)
		    (list object))))

; Update-Current-Partitions -- Added July 23, 1990 to make it easy to call from
; outside file and update the current partitions.
;
(defun Update-Current-Partitions ()
  (update-partitions *cur-partitions*))

; Update-Partitions -- Updates the partitions in `partitions'.  Updating currently
;                      consists in applying if-added rules for everything in the assert-queue,
;                      and applying the rules in the rule-queue.
;
; As noted above the rule-queue actually consists of pairs (rule . result).
; Such objects may actually
; end up in the rule-queues of several partitions. To avoid applying the rules
; multiple times we distructively replace the car with nil after
; applying the rule.
;
; It is also not clear whether or not we should check *forward-chain* and
; *back-chain* here.  My current thinking is that we are fireing
; rules that were already queued for other predicates, so the settings now of
; *forward-chain* and *back-chain* are irrelevant.
;
;
; Modified 12/4/89 to loop until all assert-queues in partitions are gone.
;
; Modified 1/8/91 to copy rule-pair before calling fire-selected-rules (this
; is necessary because this routine later distructively modifies rule-pair).
;
(defun update-partitions (partitions)
  (when partitions
    (let ((assert-path (union-values partitions 'assert-queue))
          (rule-objects (union-values partitions 'rule-queue))
          (old-inserted-values *last-inserted-values* ))
      (dolist (frame partitions)
        (fclear-slot frame 'assert-queue)
        (fclear-slot frame 'query-queue)
        (fclear-slot frame 'rule-queue))
      (trace-partition-updates partitions)
      (let ((*cur-partitions* partitions)
            (*cur-frame* nil)
            (*cur-slot* nil))
        (fire-if-added-rules-for-preds assert-path)
        (dolist (rule-pair rule-objects)
          (if (car rule-pair)
	      (fire-selected-rules (list (cons (car rule-pair) (cdr rule-pair)))))
          (rplaca rule-pair nil))
      (trace-partition-updates-end *last-inserted-values* old-inserted-values)))))

; Update-Par-Wrt-Frame -- Updates par to reflect the fact that frame has been
;                         added to it.  This requires fireing rules for all predicates in
;                         its assert-queue which match frame.  Also, if frame = *cur-frame*
;                         then we update the current partitions.
;
; Taken out since does not work in all cases (there may be preds in the assert-queues
; of other partitions that we now need to fire rules for, but it is not at all clear
; how to find them all ...).
;
;(defun update-par-wrt-frame (par frame)
;  (let ((assert-path (union-values (list par) 'assert-queue)))
;    (format t "~%Assert queue: ~a" assert-path)
;    (setq assert-path (delete-if-not
;			#'(lambda (pair) (eql (frame (car pair)) frame))
;			assert-path))
;    (format t "~%Assert queue for frame ~a: ~a" frame assert-path)
;    (mapc #'(lambda (pred-pair) 
;	      (fire-if-added-rules (car pred-pair) (cdr pred-pair))
;	      (delete-value `(assert-queue ,par ,pred-pair)))
;	    assert-path))
;  (if (eql frame *cur-frame*) (set-current-partitions *cur-frame* *cur-slot*)))

; union-values -- Takes the values in `slot' in all frames in `list' and returns
;                 their union.  Assumes that the values are not lists.  If we ever get LOTS
;                 of values then this should do something more clever than a union.
;
(defun union-values (list slot)
  (if (slotp slot) (union-values-rec list slot)))

(defun union-values-rec (list slot)
  (if (consp list)
      (union (fs-get-values-woa (car list) slot @value)
	     (union-values-rec (cdr list) slot))))


; Intersectionp -- Returns T if two lists share a common member.  If the lists ever
; get long this algorythm should be made more clever.
;
(defun intersectionp (l1 l2)
  (dolist (x l1)
    (if (dolist (y l2) (if (eql x y) (return t)))
      (return t))))
