;;; -*- Syntax: Common-lisp; Package: USER -*-

;;; Copyright (c) 1990 by James Crawford and Benjamin Kuipers.
;;;  $Id: anames.lisp,v 1.1 92/04/16 09:30:15 clancy Exp $

;;;                        ****** ANAMES ******

; A name is a list of atoms.  Any frame may have one or more names, stored in its
; NAME slot.  Each atom in a name points back to frames whose names include
; it, through its IN-NAME property.
;
;       <name>  ::=  list of atoms
;
;       frame:
;          name.value:  set of names
;
;       atom:
;          in-name:  Association list of frames and assump-ll's.

; Keep track of all names so we can reset them:
;
(defvar *all-names* nil)

; In alogic, a name predicate is (NAME <object> <name>)
; Candidates are retrieved by taking the union of frames pointed to.
;
(defun OBJECTS-FROM-NAME (words)
  (do ((L words (cdr L))
       (candidates nil))
      ((null L)
       (trace-name-retrieval words candidates)
       candidates)
    (setq candidates
	  (union-results (mapcan #'(lambda (object) (match-name object words))
				 (get (car L) 'in-name))
			 candidates))))

; UNION-RESULTS: Takes two alists (whose values are results)
; and does union.  Union may distructively modify assoc pairs in r1 and r2,
; and results in r2.  If the lists ever get long this should be made more clever.
;
(defun union-results (r1 r2)
  (dolist (item r1)
    (setq r2 (add-result item r2)))
  r2)


; ADD-RESULT: Add object to alist doing subsumption checking if object is old
; and adding it to the end if new.  May distructively modify assoc pairs in r1 and r2,
; and results in r2.
;
(defun add-result (object alist)
  (let ((old-pair (assoc (car object) alist)))
    (cond (old-pair
	   (rplacd old-pair (ndisjunct-results (cdr object) (cdr old-pair)))
	   alist)
	  (t
	   (cons object alist)))))


; name matches known names for object if it is a subsequence of
; some known name.
;
; If name matches then returns list of pair (object result) otherwise returns nil.
;
; In some cases name might match several values in the name slot of object
; (e.g. a pathologic object might have names 'John Smith' and 'John Jones' each with
; its own assump-ll and name might be 'John').
; In such cases the disjunction of the assump-ll's are returned.
;
(defun MATCH-NAME (object name)
  (let ((result (new-aresult)))
    (setf (aresult-assump-ll result) nil)
    (dolist (pair (with-no-back-chaining (get-values (list 'name object))))
      (if (subsetp name (car pair))
	  (setf (aresult-assump-ll result)
		(disjunct-assump-ll (aresult-assump-ll (cdr pair))
				    (aresult-assump-ll result)))))
    (list (cons object result))))

; Associate with all words in name the object it is the name of.
;
(defun GIVE-NAME (object name)
  (mapc #'(lambda (word) (if (null (get word 'in-name))
                           (push word *all-names*))
	                 (pushnew object (get word 'in-name)))
	name)
  name)


; Delete association of name with object -- object should be a symbol.
;
(defun DELETE-NAME (object name)
  (mapc #'(lambda (word) (delete object (get word 'in-name)))
	name)
  name)

; Reset all names.
;
(defun RESET-NAMES ()
  (dolist (name *all-names*)
    (remprop name 'in-name))
  (setq *all-names* nil))
