;;; -*- Mode:Common-Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-

;;; Copyright (c) 1990 by James Crawford.
;;;  $Id: amac.lisp,v 1.1 92/04/16 09:30:12 clancy Exp $

;;;                        ****** AMAC ******

;;; Mac (Allegro CL) versions of device dependent routines.


;;; Don't know how to do this on a mac yet.
(defun clear-window ()
  (format t "~& Don't know how to clear window on this system.~%"))

;;; Dot tracing.  In common lisp we can output trace but it is not mouse sensitive:
;;; Don't know how to do this on a mac yet.

(defparameter *dot-count* 0)
(defparameter *dot-limit* 100)
(defvar *dot-output* *standard-output*)

(defparameter *dot-trace-present* t)

(defun dot-present (symbol text)
  (declare (ignore text))
  (when *dot-trace-present* 
    (princ symbol *dot-output*)
    (dot-count)))

(defun dot-count ()
  (incf *dot-count*)
  (when (> *dot-count* *dot-limit*)
    (terpri *dot-output*)
    (princ "  ")
    (setq *dot-count* 2)))

(defun dot-terpri ()
  (terpri *dot-output*)
  (setq *dot-count* 0))

(defun reset-presenter ()
  (setq *dot-count* 0))

(defun view-dot-trace ()
  (format t "~& Can't view dot trace on your machine."))


;;; Utilities for overviewing the knowledge-base.

(defparameter *relations* nil)
(defparameter *roots* nil "Roots of current graphs.")
(defparameter *collected-relations* nil)
(defparameter *graph-windows* nil "Windows for most recent set of graphs.")

(defun kb-overview (&key (relation 'subset) (root 'things) stream depth border (rtl t) (rl nil))
  (declare (ignore stream depth border))
  (setq *relations* (if (consp relation) relation (list relation)))
  (setq *roots* (if (consp root) root (list root)))

  (collect-relations)
  (handle-nary-relations)
  (if rl
      (setq *collected-relations*
	    (mapcar #'(lambda (rel) (list (second rel) (first rel)))
		    *collected-relations*)))
  (remove-loops *roots*)
  (if rtl (remove-trans-links))
  (pp-labeled-list 0 "Showing relations:" *relations* nil)
  (if rtl
      (format t "(transitive links suppressed)~%~%")
      (terpri))
  (setq *graph-windows* (mapcar #'mac-graph *roots*)))

(defun close-graph-windows ()
  (dolist (window *graph-windows*) (ask window (window-close))))

(defun graph-without-frame (frame)
  (setq *collected-relations*
        (remove-if #'(lambda (x) (find frame x)) *collected-relations*))
  (close-graph-windows)
  (setq *graph-windows* (mapcar #'mac-graph *roots*)))


(defun collect-relations ()
  (setq *collected-relations* nil)
  (flet ((my-push (item)
	   (push item *collected-relations*)))
    (dolist (relation *relations*)
      (with-no-output (dolist (f (all-frames))
			(a-query nil `((:retrieve (,relation (:quote ,f) ?x))
				       (:apply ,#'my-push ((,relation ,f ?x))))))))))

(defun handle-nary-relations ()
  (let ((counter 1))
    (setq *collected-relations*
          (mapcan #'(lambda (triple)
                      (let ((relation (first triple))
                            (arg1 (second triple))
                            (arg2 (third triple)))
                        (cond
                         ((consp arg2)
                          (setq relation (make-symbol (format nil "~a~a" relation (incf counter))))
                          (cons `(,arg1 ,relation)
                                (mapcar #'(lambda (x) `(,relation ,x))
                                        arg2)))
                         (t
                          (list (cdr triple))))))
                  *collected-relations*))
    t))

;; Quick and dirty function to remove most of the transitive links.
;; Should be replaced with a beter algorythm.
;;
(defun remove-trans-links ()
  (let ((hit-list nil))
    (dolist (rel1 *collected-relations*)
      (dolist (rel2 *collected-relations*)
	(if (eql (second rel1) (first rel2))
	    (push (list (first rel1) (second rel2))
		  hit-list))))
    (setq *collected-relations*
	  (set-difference *collected-relations* hit-list :test #'equal))))

(defun remove-loops (roots)
  (when (consp roots)
    (let* ((root (car roots))
	   (accessable-subtree (accessables (list root))))
      ;;(format t "~% root = ~a   subtree = ~a" root accessable-subtree)
      (when (find root accessable-subtree)	; When there is a loop.
	(setq *collected-relations*
	      (delete-if #'(lambda (rel)
			     (and (eql (cadr rel) root)
				  (find (car rel) accessable-subtree)))
			 *collected-relations*)))
      ;; Then search breadth first.
      (remove-loops (append (cdr roots) (children root))))))


(defun accessables (nodes)
  (let (new-nodes)
    (loop
      (setq new-nodes (mapcan #'children nodes))
      (if (subsetp new-nodes nodes) (return nodes))
      (setq nodes (union nodes new-nodes)))))

(defun children (frame)
  ;;(sort 
  (mapcan #'(lambda (rel) (if (equal frame (first rel)) (list (second rel))))
          *collected-relations*)
  ;;#'alphalessp)
  )

(defun parents (frame)
  (mapcan #'(lambda (rel) (if (equal frame (second rel)) (list (first rel))))
          *collected-relations*))


;; Then the code to collect the sets in the kb and show their members:


(defun show-sets ()
  (setq *relations* '(isa))
  (collect-relations)
  (let ((sets nil))
    (dolist (rel *collected-relations*)
      (if (not (end-search (string 'selfset) (string (second rel))))
	  (pushnew (second rel) sets)))
    ;; Explicitly take things out:
    (setq sets (delete 'things sets))
    (dolist (set (sort sets #'alphalessp))
      (pp-labeled-list 2
		       (format nil "~@(~a~):" set)
		       (sort (mapcan #'(lambda (rel)
					 (if (eql set (second rel))
					     (list (first rel))))
				     *collected-relations*)
			     #'alphalessp)
		       nil))))


(defun end-search (x y)
  (if (not (< (length y) (length x)))
      (string-equal x y :start2 (- (length y) (length x)))))


;;; Interface to grapher:

(defobject *algy-node* *node*)

;;; Given a frame in the kb, finds or creates an *algy-node* for it.
;;;
(defun make-algy-node (frame)
  (let ((node (get frame 'node)))
    (unless node
      (setq node (oneof *algy-node*))
      (setf (get frame 'node) node)
      (ask node 
        (have 'frame frame)
        (have 'node-size-iv
              (make-point (string-width (format nil "~a" frame)
                                        *grapher-font*)
                          20))))))

(defobfun (algy-node-frame *algy-node*) ()
  (objvar frame))
          
;;; Convert to frames, get children, and convert back to nodes ...
;;;
(defobfun (node-children *algy-node*) ()
  (mapcar #'(lambda (x) (get x 'node))
          (children (algy-node-frame))))

(defobfun (node-parents *algy-node*) ()
  (mapcar #'(lambda (x) (get x 'node))
          (parents (algy-node-frame))))

(eval-when (eval compile load)

(defmacro with-clip-rect-intersect (rect &rest body)
    (let ((old (gensym))
          (new (gensym)))
      `(let ((,old (_NewRgn :ptr))
             (,new (_NewRgn :ptr)))
         (_getclip :ptr ,old)
         (_rectrgn :ptr ,new :ptr ,rect)
         (_SectRgn :ptr ,old :ptr ,new :ptr ,new)
         (_SetClip :ptr ,new)
         (unwind-protect
           (progn ,@body)
           (_SetClip :ptr ,old)
           (_DisposRgn :ptr ,old)
           (_DisposRgn :ptr ,new)))))

) ;end eval-when

(defobfun (node-draw *algy-node*) ()
  (when (usual-node-draw)
    (let* ((topleft (node-position))
           (left (point-h topleft))
           (bottomright (add-points topleft (node-size)))
           (bottom (point-v bottomright)))
      (rlet ((r :rect
                :topleft topleft
                :bottomright bottomright))
        (_eraserect :ptr r)
        (_framerect :ptr r)
        (_moveto :word (+ left 3) :word (- bottom 5))
        (_insetrect :ptr r :long #@(2 2))
        (without-interrupts
         (with-clip-rect-intersect r
           (with-pstrs ((str (format nil "~(~a~)" (algy-node-frame))))
             (_drawstring :ptr str))))))))

(defun mac-graph (root)
  (dolist (rel *collected-relations*)
    (dolist (x rel)
      (make-algy-node x)))
  (oneof *grapher-window*
         :root-node (get root 'node)
         :window-title "Graph"))

(defobfun (node-click-event-handler *algy-node*) (where)
  (declare (ignore where))
  (pp-frame (algy-node-frame)))

#|
(setq *collected-relations* '((A B) (A C) (B D) (B E)))
(mac-graph 'a)
|#


;;; Graph-Path: Graphs the most recent access path (in a query or assertion).  One
;;; current limitations:
;;;
;;; 1. Special forms are ignored.
;;;

(defun graph-path (&key stream depth border)
  (declare (ignore stream depth border))

  (let ((algy-preds (preprocess *last-predicates*))
        (counter 0))
    (setq *roots* (list (frame (car algy-preds))))
    (setq *collected-relations* nil)
    
    (let* ((paths (mapcar #'(lambda (result)
                              (gp-substitute-bindings algy-preds (aresult-sub result)))
                          *last-results*))
           (big-path (list-union paths)))
      (dolist (pred big-path)
        ;; For the moment igmore special forms:
        (if (not (member (car pred) *special-forms* :test #'eq))
          (let ((relation (make-symbol (format nil "~a~a" (slot pred) (incf counter)))))
            (push (list (frame pred) relation) *collected-relations*)
            (let* ((pred-value (value pred))
                   (value-list (if (consp pred-value) pred-value (list pred-value))))
              (dolist (term value-list)
                (push (list relation term) *collected-relations*)))))))
    
    (mac-graph (car *roots*))))


(defun gp-substitute-bindings (exp alist)
  (cond ((consp exp)
	 (cons (gp-substitute-bindings (car exp) alist)
	       (gp-substitute-bindings (cdr exp) alist)))
	((variable? exp)
	 (let ((b (assoc (algy-variable-name exp) alist
                         :key #'algy-variable-name)))
	   (if b
	       (cadr b)
	       exp)))
	(t exp)))

(defun list-union (list)
  (if (consp list)
    (union (car list) (list-union (cdr list)) :test #'equal)))

