;;; -*- Mode:Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-
(in-package :user)

;;;  $Id: alogic.lisp,v 1.1 92/04/16 09:30:10 clancy Exp $
;;; Copyright (c) 1990, 1992 by James Crawford and Benjamin Kuipers.

;;;                        ****** ALOGIC ******


;;; QUERY -- returns a stream of the results that successfully retrieve or match
;;;          known information all the way through.
;;; ASSERT -- looks for known information when a predicate includes variables,
;;;           and asserts predicates with no variables.  Returns a stream of all
;;;           results making it all the way through.
;;;
;;; All routines return explicit NIL on failure.

;;; Algernon queries and assertions return "streams".  A stream is a list of "aresults".
;;; An aresult is a structure.  Its defstruct appears in aglobals:
;;;
;;; (defstruct (aresult
;;;	     (:constructor new-aresult (&optional alist)))
;;;  (sub 'nil)
;;;  (assump-ll '(()))
;;;  alist)
;;;
;;; A substitution (sometimes called a binding or an alist) is a list of pairs (?x v) binding
;;; ?x to the value v (e.g. ((?x1 john) (?y3 mary) ...)).  nil is the empty substitution binding
;;; no variables.
;;;
;;; An assumption list is a list of variable free predicates
;;; (e.g. ((father john tom) (flies tweety true) ...)).  An assumption list list
;;; (assump-ll) is a list of assumption lists.  If a value depends on an assump-ll
;;; this means that the value follows from the conjunct of any assumption list
;;; in it.  For example if a value follows with no assumptions then its assump-ll
;;; is (()) while a value which is always false has assump-ll nil.
;;;
;;; The structure slot alist is for other properties which may be needed
;;; (e.g. certainty factors, justifications, ...).
;;;
;;;
;;; alogic modified 2/28/90 to be lazy about applying substitutions.
;;;
;;; 12-20-91 (BJK)  Modifying Lisp interface.


;;; One variable to disable :Retrieve for testing:
(defparameter *disable-retrieve* nil)

(defun INTERNAL-QUERY (sequence initial-stream &optional user-call)
  (do ((L sequence (cdr L))
       (S initial-stream
	  (mapcan #'(lambda (result)
		      (query-predicate (car L) result))
		  S)))
      ((null L) S)
    (cond ((null S)
	   (return S)))
    (when user-call (trace-new-pred (setq *last-predicate* (car L))))))

(defun INTERNAL-ASSERT (sequence initial-stream &optional user-call)
  (do ((L sequence (cdr L))
       (S initial-stream
	  (mapcan #'(lambda (result)
		      (assert-predicate (car L) result))
		  S)))
      ((null L) S)
    (cond ((null S)
	   (return S)))
    (when user-call (trace-new-pred (setq *last-predicate* (car L))))))


;;; Query-predicate creates all the consistent extensions of a given
;;; result by matching a given predicate against stored values.
;;;
(defun query-predicate (pred result)	; => stream
  (if @debug
    (if (not (aresult-p result))
      (error "Algernon Bug -- Invalid result passed to query-predicate: ~(~a~)." result)))
  (trace-query-begin pred result)
  (query-accountant pred)
  (let ((stream (if (member (car pred) *special-forms* :test #'eq)
                  (query-special-forms pred result)
                  (let* ((pred (substitute-bindings pred (aresult-sub result)))
                         (frame (frame pred))
                         (slot (slot pred))
                         (value (value pred)))
                    (cond
                     ((and (eql slot 'NAME)
                           (variable? frame)
                           (not (variable? value))
                           (not (negated pred)))
                      (extend-with-values result frame
                                          (objects-from-name value)))
                     ((variable? slot)
                      (if *top-level*
                        (algy-error
                         (format nil
                                 "Illegal predicate ~(~a~) -- ~
                                  slot cannot be the variable (~(~a~))."
				 pred slot))))
                     ((variable? frame)
                      (if *top-level*
                        (algy-error
                         (format nil
                                 "Illegal predicate ~(~a~) -- ~
                                  frame cannot be the variable (~(~a~))."
				 pred frame))))
                     (t
                      (if (and *top-level* *back-chain*)
                        (set-current-partitions frame slot))
                      (if (not (has-variables value))
                        (extend-result result (cdr (known pred)))
                        (extend-with-values result value (get-values pred)))))))))
    (trace-query-end pred stream)))

;;; Assert predicate asserts a single predicate given a result.
;;;
(defun assert-predicate (pred result)
  (if @debug
    (if (not (aresult-p result))
      (error "Algernon Bug -- Invalid result passed to assert-predicate: ~(~a~)." result)))
  (trace-assert-begin pred result)
  (let ((stream (if (member (car pred) *special-forms* :test #'eq)
                  (assert-special-forms pred result)
                  (let ((pred (substitute-bindings pred (aresult-sub result))))
                    (cond
                     ((has-variables pred)
                      (query-predicate pred result))
                     (t
                      (if (and *top-level* *forward-chain*)
                        (set-current-partitions (frame pred) (slot pred)))
                      (if (insert-value pred result)
                        (list result))))))))
    (trace-assert-end pred stream)))


;;; Special forms.
;;;
(defun query-special-forms (pred result)
  (case (car pred)
    (:RETRIEVE (let ((*back-chain* *disable-retrieve*))
		 (query-predicate (cadr pred) result)))
    (:NO-COMPLETION (query-predicate (cadr pred) result))
    (:WO-CONTRA-POSITIVE (let ((*contra-positive* nil))
                           (internal-query (cdr pred) (list result))))
    (:W-CONTRA-POSITIVE (let ((*contra-positive* t))
			  (internal-query (cdr pred) (list result))))
    (:IN-OWN-PARTITION
     (let* ((pred (substitute-bindings (cadr pred) (aresult-sub result)))
	    (result (with-partitions (frame pred) (slot pred)
				     (query-predicate pred result))))
       (update-current-partitions)
       result))
    (:ALL-PATHS
     (let ((new-result (copy-aresult result)))
       (setf (aresult-assump-ll new-result)
	     (all-paths (third pred) (internal-query (second pred) (list result))))
       (if (aresult-assump-ll new-result) (list new-result))))
    (t (handle-special-forms pred result))))

(defun assert-special-forms (pred result)
  (case (car pred)
    (:RETRIEVE (let ((*back-chain* *disable-retrieve*))
                 (assert-predicate (cadr pred) result)))
    (:NO-COMPLETION (assert-predicate (cadr pred) result))
    (:WO-CONTRA-POSITIVE (let ((*contra-positive* nil))
                           (internal-assert (cdr pred) (list result))))
    (:W-CONTRA-POSITIVE (let ((*contra-positive* t))
			  (internal-assert (cdr pred) (list result))))
    (:IN-OWN-PARTITION
     (let* ((pred (substitute-bindings (cadr pred) (aresult-sub result)))
	    (result (with-partitions (frame pred) (slot pred)
				     (assert-predicate pred result))))
       (update-current-partitions)
       result))
    (:ALL-PATHS
     (let ((new-result (copy-aresult result)))
       (setf (aresult-assump-ll new-result)
	     (assert-all-paths (third pred) (internal-query (second pred) (list result))))
       (if (aresult-assump-ll new-result) (list new-result))))
    (t (handle-special-forms pred result))))

(defun handle-special-forms (pred result)
  (case (car pred)

    ;; Controlled retrieval.

    (:CREATE
      (setq pred (substitute-bindings pred (aresult-sub result)))
      (if (variable? (second pred))
	  (let ((new-frame (make-new-frame (third pred))))
	    (if (and (third pred)
		     (not (eql (third pred) new-frame)))
		(algy-warning
		  (format nil "A frame with name ~(~a~) exists -- ~
                               created frame will have name ~(~a~)."
			  (third pred) new-frame)))
	    (if (third pred) (internal-assert `((name ,new-frame (,new-frame)))
					      (list (new-aresult))))
	    (extend-with-values result (second pred)
				(list (cons new-frame (new-aresult)))))
	  (throw 'error (format nil "Second argument to ~(~a~) must be a variable." pred))))
	  
    (:THE (let* ((vars (second pred))
		 (path (cddr pred))
		 (query-result (with-no-depnet (internal-query path (list result)))))
	    (if query-result
		(if (single-binding vars query-result)
		    query-result
		    (throw 'error (format nil "~& Ambiguous definite description ~a.~%" path)))
		(create-and-assert vars path result))))
    
    (:FORC (let* ((vars (second pred))
		  (path (cddr pred))
		  (query-result (with-no-depnet (internal-query path (list result)))))
	     (if query-result
		 query-result
		 (create-and-assert vars path result))))

    (:A    (let* ((vars (second pred))	;  (:A <vars> . <path>)
		  (path (cddr pred)))
	     (create-and-assert vars path result)))

    (:ANY  (let* ((path (cdr pred))	;  (:ANY  . <path>)
		  (query-result (with-no-depnet (internal-query path (list result)))))
	     (last query-result)))

    ;; Modified 12/7/89 to update partitions when rules are added.
    ;; We update the partitions of isa since frame rules are accessed
    ;; by following isa links.

    (:RULES (setq pred (substitute-bindings pred (aresult-sub result)))
            (when *top-level*
	      (update-partitions (partitions (second pred) nil))
	      (update-partitions (partitions nil 'isa)))					
            (store-rules (second pred) (eval-lisp-in-rules (cddr pred)) 'frame)
            (list result))

    (:SRULES (setq pred (substitute-bindings pred (aresult-sub result)))
             (when *top-level*
	       (update-partitions (partitions (second pred) nil)))
             (store-rules (second pred) (eval-lisp-in-rules (cddr pred)) 'slot)
             (list result))

    (:DECL-SLOTS (setq pred (substitute-bindings pred (aresult-sub result)))
                 (dolist (slot (cdr pred)) (decl-slot slot))
		 (list result))

    ;; User interface

    (:ASK (setq pred (substitute-bindings pred (aresult-sub result)))
          (algy-ask (second pred) result))

    (:SHOW (setq pred (substitute-bindings pred (aresult-sub result)))
           (cond ((not (variable? (second pred))) 
		  (pp-frame (second pred))))
	   (list result))

    ;; Non-monotonic reasoning

    (:UNP (if (not (with-no-depnet (internal-query (cdr pred) (list result))))
            (list result)))

    (:ASSUME (let ((p (substitute-bindings (second pred) (aresult-sub result))))
	       (if (and *top-level* *forward-chain*)
                 (set-current-partitions (frame p) (slot p)))
	       (insert-assumption p result)
	       (extend-result result (cdr (with-no-back-chaining (known p))))))


    ;; Deleting things
    
    (:DEL-RULES (setq pred (substitute-bindings pred (aresult-sub result)))
                (del-rules (second pred) (third pred))
                (list result))

    (:DEL-SRULES (setq pred (substitute-bindings pred (aresult-sub result)))
                 (del-srules (second pred) (third pred))
                 (list result))
    
    (:DEL-RULE (setq pred (substitute-bindings pred (aresult-sub result)))
               (del-rule (second pred) (third pred))
	       (list result))
    
    (:DEL-SRULE (setq pred (substitute-bindings pred (aresult-sub result)))
                (del-srule (second pred) (third pred))
	        (list result))

    (:DELETE (setq pred (substitute-bindings pred (aresult-sub result)))
             (if (delete-value (second pred))
	         (list result)))
    
    (:CLEAR-SLOT (setq pred (substitute-bindings pred (aresult-sub result)))
                 (let ((frame (second pred))
		       (slot (third pred)))
		   (fclear-facet frame slot @value)
		   (fclear-facet frame slot @n-value)
		   (list result)))

    ; Here are the Lisp <-> Algernon commands

    ((:EVAL :LISP)
           (setq pred (substitute-bindings pred (aresult-sub result)))
           (eval (second pred))
           (list result))
    
    (:TEST (setq pred (substitute-bindings pred (aresult-sub result)))
           (if (eval (second pred))
             (list result)))
    
    (:NEQ (setq pred (substitute-bindings pred (aresult-sub result)))
          (unless (eql (second pred) (third pred))
            (list result)))
    
    (:FUNCALL (setq pred (substitute-bindings pred (aresult-sub result)))
	      (lisp-form pred (aresult-assump-ll result))
	      (list result))

    ;      Extended to do destructuring of the value returned.
    (:BIND (setq pred (substitute-bindings pred (aresult-sub result)))
           (if (has-variables (second pred))
	       (multiple-value-bind (val assumps)
		   (lisp-form (third pred) (aresult-assump-ll result))
		 (let ((nres (if assumps
				 (make-aresult :assump-ll assumps)
				 (new-aresult))))
		   (extend-with-values result (second pred) (list (cons val nres)))))
	       (throw 'error (format nil "Special form ~a requires variable(s) in first arg" pred))))

    ;      (:BRANCH <pat> <form>) assumes that <form> evaluates to a list of values,
    ;      and branches on matching <pat> against each of those values.
    (:BRANCH (setq pred (substitute-bindings pred (aresult-sub result)))
	     (extend-with-values result
				 (second pred)
				 (mapcar #'(lambda (v) (cons v (new-aresult)))
					 (lisp-form (third pred)
						    (aresult-assump-ll result)))))

    ; The following Lisp/Algernon commands are OBSOLETE.

    (:APPLY ; (cerror ":APPLY is obsolete.  Use :FUNCALL.")
	    (setq pred (substitute-bindings pred (aresult-sub result)))
            (cond ((apply (second pred) (third pred))
		   (list result))
		  (t nil)))
    
    (:BIND-TO-VALUES
;     (cerror ":BIND-TO-VALUES is obsolete.  Use (:BIND <var> (:VALUES <frame> <slot>))")
     (setq pred (substitute-bindings pred (aresult-sub result)))
     (when (variable? (second pred))
       (let ((variable (second pred))
             (frame (third pred))
             (slot (fourth pred)))
         (if (and *top-level* *back-chain*)
           (set-current-partitions frame slot))
         (let ((values (mapcan #'(lambda (pair)
                                   ;; Filter out results, keeping only values known w/o assumptions:
                                   (if (member nil (aresult-assump-ll (cdr pair)) :test #'eq)
                                     (list (car pair))))
                               (get-values (list slot frame variable)))))
           (extend-with-values result variable
                               (list (cons values (new-aresult))))))))
    
    (:BRANCH-ON-VALUES
;     (cerror ":BRANCH-ON-VALUES is obsolete.  Use :BRANCH.")
     (setq pred (substitute-bindings pred (aresult-sub result)))
     (when (variable? (second pred))
       (extend-with-values result (second pred)
                           (mapcar #'(lambda (value) (cons value (new-aresult)))
                                   (eval (third pred))))))
    
    (t (throw 'error (format nil "Special form ~a has bad syntax." pred)))))


;;; Evaluate a form in Lisp:
;;;    (:FUNCALL  <function>  <arg1> ... <argn>)
;;;    (:EVAL  <form> )
;;;    <form>
;;;
;;; An argument can be:
;;;    <form>
;;;    :ASSUMPTIONS                  -->  current assump-ll
;;;    (:VALUES <frame> <slot>)      -->  list of values in the slot
;;;    (:NON-VALUES <frame> <slot>)  -->  list of non-values in the slot
;;; The values returned are those currently believed, but the appended assumption
;;; information is discarded before the values are returned.

(defun lisp-form (pred assump-ll)			; assumes all substitutions done
  (if (has-variables pred) 
      (algy-error (format nil "All variables in ~a should be bound." pred)))
  (if (atom pred)
      (if (eql pred :ASSUMPTIONS) assump-ll (eval pred))
      (case (car pred)
	(:FUNCALL    (apply (eval (second pred))
			    (mapcar #'(lambda (arg) (lisp-form arg assump-ll))
				    (cddr pred))))
	(:EVAL       (eval (second pred)))
	(:VALUES     (mapcar #'car
			     (fs-get-values (second pred) (third pred) @value)))
	(:NON-VALUES (mapcar #'car
			     (fs-get-values (second pred) (third pred) @n-value)))
	(t           (eval pred)))))

;;; Eval-List-In-Rules -- Hack to support :lisp in a list of rules.
;;;    If an element in :RULES or :SRULES is of the form (:lisp exp), 
;;;    then it is replaced by (eval exp).
;;; Called in HANDLE-SPECIAL-FORMS to support features in AKBASE.
;;;
(defun eval-lisp-in-rules (rules)
  (mapcan #'(lambda (rule)
	      (if (eql (car rule) :lisp)
		  (list (eval (second rule)))
		  (list rule)))
	  rules))
  
;;; Utilities for all-paths.

(defun all-paths (path results)
  (let ((assump-ll (list nil)))
    (dolist (result results)
        (setq assump-ll
              (conjunct-assump-ll assump-ll
                                  (disjunct-stream (internal-query path (list result)))))
      (if (null assump-ll) (return)))
    assump-ll))

(defun assert-all-paths (path results)
  (let ((assump-ll (list nil)))
    (dolist (result results)
        (setq assump-ll
              (conjunct-assump-ll assump-ll
                                  (disjunct-stream (internal-assert path (list result)))))
      (if (null assump-ll) (return)))
    assump-ll))


;;; Steps toward clarifying definite descriptions.

;;; Single-binding: Returns t iff results contains only one binding for vars.
;;;
(defun single-binding (vars results)
  (if (eql (length results) 1)
      t
      (single-binding-rec vars
			  (cdr results)
			  (substitute-bindings vars (aresult-sub (car results))))))

(defun single-binding-rec (vars results bindings)
  (if results
      (let ((new-bindings (substitute-bindings vars (aresult-sub (car results)))))
	(and (equal bindings new-bindings)
	     (single-binding-rec vars (cdr results) new-bindings)))))


;;; Routines to support new version of find or create.

;;; CREATE-AND-ASSERT
;;;  Create new frames and bind them to variables in vars, then assert path.
;;;  vars must be a list.
;;;
(defun create-and-assert (vars path result)
  (internal-assert path (list (add-new-frames result vars path))))

;;; Add-new-Frames: Creates new frames for the variables in vars
;;; (which must be a list) and returns result (with additional bindings).
;;; "Addresses" for the new frames are chosen to reflect the fact that path is
;;; true of the new vars.
;;;
(defun add-new-frames (result vars path)
  (if (not (consp vars))
      result
      (let ((first-var (substitute-bindings (car vars) (aresult-sub result))))
	(if (variable? first-var)
	    (car (extend-with-values (add-new-frames result (cdr vars) path)
				     first-var
				     (list (cons (make-new-frame (find-address first-var path result))
						 (new-aresult)))))
	    (algy-error (format nil "Illegal argument to :forc -- ~(~a~) is not a variable." first-var))))))

;;; Find-address
;;;  Tries to find a numonic "address" for the frame for var.
;;;  Path is path about to be asserted about var (under substitutions in result).
;;;  The address of a frame is the symbol on whose property list the frame lives.
;;;
(defun find-address (var path result)
  (setq path (substitute-bindings path (aresult-sub result)))
  (or
    ;; Try to base address on the name of the new frame:
    (funcall #'(lambda (pred) (if pred (intern (string (car (value pred))))))
	     (car (member-if #'(lambda (pred) (and (eql (slot pred) 'name)
						   (eql var (frame pred))
						   (not (variable? (value pred)))))
			     path)))
    ;; Try to base address on binary relation to other frames:
    (funcall #'(lambda (pred)
		 (if pred (intern (format nil "~a-~a" (frame pred) (slot pred)))))
	     (car (member-if #'(lambda (pred) (and (eql var (value pred))
						   (not (variable? (frame pred)))
						   (not (variable? (slot pred)))))
			     path)))
    ;; Try to base address on non-binary relation to other frames:
    (funcall #'(lambda (pred)
		 (if pred (intern (format nil "~a-~a" (frame pred) (slot pred)))))
	     (car (member-if #'(lambda (pred) (and (listp (value pred))
						   (member var (value pred))
						   (not (variable? (frame pred)))
						   (not (variable? (slot pred)))))
			     path)))))

;;; Utility routines

;;; Extend-With-Values:
;;; Extends an aresult by binding match-form to the values in alist.
;;; alist is an alist of values and results.  The results in alist
;;; are distructively modified.
;;;
;;; Returns a list of new results.
;;;
(defun extend-with-values (result match-form alist)
  (mapcan #'(lambda (entry)
	      (let ((new-sub (match match-form (car entry) (aresult-sub result))))
		(unless (eql new-sub 'failed)
		  (let ((new-result (nconjunct-results result (cdr entry))))
		    (unless (null (aresult-assump-ll new-result))
		      (setf (aresult-sub new-result) new-sub)
		      (list new-result))))))
	  alist))

;;; Extend-Result --- Distructively extends r2 to include r1 and returns (r2).
;;; Unless r2 = nil, in which case it returns nil. Currently the sub and alist
;;; fields of r2 are ignored (and those of r1 are returned).
;;; 
(defun extend-result (r1 r2)
  (if r2
      (let ((r (nconjunct-results r1 r2)))
        (if (not (null (aresult-assump-ll r)))
	  (list r)))))

;;; Nconjunct-Results -- Distructively returns the conjunction of r1 and r2
;;; (i.e. r2 = r1 and r2). Currently the sub and alist fields of r2 are ignored
;;; (and those of r1 are returned).
;;;
(defun nconjunct-results (r1 r2)
  (setf (aresult-assump-ll r2) (conjunct-assump-ll (aresult-assump-ll r1)
                                                   (aresult-assump-ll r2))
        (aresult-sub r2) (aresult-sub r1)
        (aresult-alist r2) (aresult-alist r1))
  r2)

;;; Ndisjunct-Results -- Distructively returns the disjunction of r1 and r2.
;;; (i.e. r2 = r1 or r2). Currently the sub and alist fields or r2 are ignored
;;; (and those of r1 are returned).
;;;
(defun ndisjunct-results (r1 r2)
  (setf (aresult-assump-ll r2) (disjunct-assump-ll (aresult-assump-ll r1)
						   (aresult-assump-ll r2))
        (aresult-sub r2) (aresult-sub r1)
        (aresult-alist r2) (aresult-alist r1))
  r2)

(defun conjunct-results (r1 r2)
  (let ((r (copy-aresult r1)))
    (setf (aresult-assump-ll r) (conjunct-assump-ll (aresult-assump-ll r1)
                                                    (aresult-assump-ll r2)))
    r))

;;; Takes a stream (i.e. list of results) and returns a single assump-ll
;;;  which is the disjunction of assump-ll's in stream.
;;;
(defun disjunct-stream (stream)
  (unless (null stream)
    (disjunct-assump-ll (aresult-assump-ll (car stream))
			(disjunct-stream (cdr stream)))))

;;; decl-slot declares a slot and its number restriction (if any).
;;;
(defun decl-slot (slot)
  (let ((slot-name (car slot))
	(num       (cadr slot)))
    (cond (slot-name
	   (cond ((eql slot-name 'quote)
		  (throw 'error "Sorry, can't have a slot named QUOTE.")))
	   (cond ((not (framep slot-name))
		  (make-new-frame slot-name)
		  (make-into-slot slot-name)
		  (if num (fput slot-name @slot-props @num-res num)))
		 ((and (slotp slot-name)
		       (eql num (car (fget slot-name @slot-props @num-res)))))
		 (t (throw 'error
		      (format nil "Illegal slot declaration --- ~
                                  Incompatible frame ~a already exists."
			      slot-name))))))))
