(in-package :user)

;;; Copyright (c) 1990 by James Crawford.
;;;  $Id: akbase.lisp,v 1.1 92/04/16 09:30:06 clancy Exp $

;;;                        ****** AKBASE ******

;; This file is a collection of "common sense" knowledge common to many
;; Algernon examples.

;; The common-sense knoweldge-base should be loaded before any examples are
;; run.

;; The Algernon code in this file should not in general be used as a
;; model.  There are two reasons for this:
;; 
;; 1. The initial a-asserts are used to "bootstrap" the knowledge-base
;; and so have unusual forms.
;; 
;; 2. Several potentially dangerous techniques are used to increase
;; effiecency.  First, unbound variables sometimes occur in the
;; consequents of if-added rules.  This is dangerous since the rules
;; cannot be completed (but is used in those cases in which I do not want
;; the rules to be completed).  Second, :RETRIEVE is used in several
;; rules.  When used in the consequent of rules :RETRIEVE prevents
;; forward chaining (just as in the antecedent it prevents backward
;; chaining). However, this use is fairly dangerous as any if-added rules
;; for the fact asserted within the :RETRIEVE are NEVER fired.  In
;; general :RETRIEVE should only be used with care to increase the
;; effieciency of expensive operations (such as computing the
;; transitive closure of relations).


;; This file is organized as follows:
;;
;;   Bootstrapping:
;;      Machinery to define slots
;;      Machinery to define taxonomies
;;   Taxonomy
;;   Slots and Rules (organized by set).

(defun common-sense-facts ()
  (akb-basic-properties)
  (akb-set-theory)
  (akb-common-things)
  (akb-order+equivalence)
  )



(defun akb-basic-properties ()

  ;; Bootstrapping:
  ;;
  ;;   If we could use :slot at this point we would define the basic slots as:
  ;;
  ;;  (:slot isa (things sets)
  ;;            :partition main-partition
  ;;            :comment "(isa ?x ?s) = ?x is a member of the set ?s.")
  ;;
  ;;  (:slot name (things nil)
  ;;            :comment "Label used to refer to frame.")
  ;;
  ;;  (:slot type-slot (slots nil)
  ;;            :partition slot-info-partition
  ;;            :comment "(type-slot slot s_1 s_2 ... s_n) = i-th argument of slot is a member of set s_i.")
  ;;
  ;;  (:slot slot-partition (slots partitions)
  ;;            :partition partition-partition
  ;;            :comment "(slot-partition ?s ?p) = ?s is in partition ?p.)
  ;;
  (a-assert "Bootstrap"
	    '(;; Basic slots:
	      (:decl-slots (isa nil) (name nil) (type-slot 1) (slot-partition nil))
	      
	      ;; Then the partitions:
	      (:create ?mp main-partition)
	      (:create ?sp set-partition)
	      (:create ?sip slot-info-partition)
	      (:create ?pp partition-partition)

	      ;; And partitions for the basic slots:
	      (slot-partition (:slot isa) ?mp)
	      (slot-partition (:slot name) ?mp)
	      (slot-partition (:slot type-slot) ?sip)
	      (slot-partition (:slot slot-partition) ?pp)

	      ;; And the sets at the top of the hierarchy:
	      (:create ?things things)
	      (:create ?sets sets)
	      (:create ?slots slots)
	      (:create ?partitions partitions)
	      
	      ;; Now the rule to let us type slots. The rule for type-slot generates an if-added rule
	      ;; which adds the necessary isa relations. If any s_i is nil then no restriction is placed
	      ;; on that argument position.  The code for type-slot-rule is at the end of akbase.
	      ;;
	      (:srules (:slot type-slot)
	               ((type-slot ?s ?typing)
			->
			(:srules ?s (:lisp (type-slot-rule '?s '?typing)))))

	      ;; Now we can type the basic slots:
	      (type-slot (:slot type-slot) ?slots nil) ; Can't type other args since arity is not fixed.
	      (isa (:slot type-slot) ?slots)           ; Have to add explicitly b/c rule comes in too late.
	      (type-slot (:slot isa) ?things ?sets)
	      (type-slot (:slot name) ?things nil)
	      (type-slot (:slot slot-partition) ?slots ?partitions)

	      ;; We can now use :slot (with some descriptions) to add more slots:
	      ;;
	      (:slot comment (?things nil)
		   :partition ?mp
		   :comment "English description of the frame or slot.")
	      ;;
	      (:slot backlink (?slots ?slots)
		   :cardinality 1
		   :partition ?sip
		   :comment "(backlink s1 s2)  = s1 is backlinked to s2.")
	      ;;
	      (:slot inverse (?slots ?slots)
		   :cardinality 1
		   :partition ?sip
		   :comment "(inverse s1 s2) = s1 is the inverse of s2 ~
                             (equivalent to bi-directional backlinks).")

	      ;; Then the rules for backlink and inverse:
	      ;;
	      ;; Backlink rule adds forward chaining rules which actually generate the
	      ;; backlink:
	      ;;
	      (:srules (:slot backlink)
		       ((backlink ?s1 ?s2) ->
			(:srules ?s1
				 ((?s1 ?f1 ?f2) -> (?s2 ?f2 ?f1))
				 ((not (?s1 ?f1 ?f2)) -> (not (?s2 ?f2 ?f1))))))
	      ;;
	      ;; Inverse rule adds bi-directional backlinks:
	      ;;
	      (:srules (:slot inverse)
	               ((inverse ?s1 ?s2) -> (backlink ?s1 ?s2) (backlink ?s2 ?s1)))

	      ;; Then we can complete the slot declarations for the basic slots:
	      (comment (:slot isa) (:quote "(isa ?x ?s) = ?x is a member of the set ?s."))
	      (comment (:slot name) (:quote "Label used to refer to frame."))
	      (comment (:slot type-slot)
		          (:quote "(type-slot slot s_1 s_2 ... s_n) = i-th argument of slot ~
                                  is a member of set s_i."))
	      (comment (:slot slot-partition) (:quote "(slot-partition s p) => <f,s> in p."))

	      ;; :slot is now completely functional, and basic slots completely defined.

;;;	      
;;;
;;;  a-assert "Fundamental slots underlying set theory"
;;;	    
	    
	      ;; Slots to enable us to use :taxonomy.
	      (:slot member (?sets ?things)
		   :partition ?sp
		   :backlink isa
		   :comment "(member ?s ?x) = A member of ?s is ?x.")
	      ;;
	      (:slot subset (?sets ?sets)
		   :partition ?sp
	           :comment "(subset ?s1 ?s2) = A subset of ?s1 is ?s2.")
	      ;;
	      (:slot superset (?sets ?sets)
		   :partition ?sp
		   :inverse subset
		   :comment "(superset ?s1 ?s2) = A superset of ?s1 is ?s2.")
	      ;;
	      (:slot imp-superset (?sets ?sets)
		   :partition ?sp
		   :backlink subset
		   :comment "(imp-superset ?s1 ?s2) = An important superset of ?s1 is ?s2.")

	      ;; The key rule for imp-superset is that the isa links up are added immediately:	     
	      (:srules (:slot imp-superset)
	       ((imp-superset ?set1 ?set2) ->
		(:rules ?set1
		 ((isa ?x ?set1) -> (isa ?x ?set2)))))

	      ;; Finally, some slots needed for Algernon's machinery:
	      (:slot selfset (?things ?sets)
	           :cardinality 1
		   :partition ?mp
		   :backlink member
		   :comment "The selfset of x is the set consisting exactly of {x}.")
	      ;;
	      (:slot assert-queue (?partitions nil)
		   :partition ?pp
		   :comment "Assertions waiting in partition.")
	      ;;
	      (:slot query-queue (?partitions nil)
		   :partition ?pp
		   :comment "Queries to partition which were not preformed.")

	      (:slot rule-queue (?partitions nil)
		   :partition ?pp
		   :comment "Rules waiting in partition.")))

  )


;;;  Utility function for type-slot.


;;; Type-Slot-Rule -- restriction is of form (s1 s2 ... sn).
;;; Function returns rule using Algernon's internal syntax:
;;;
;;; (s ?x1 (?x2 ... ?xn)) -> (isa ?x1 s1) ... (isa ?xn sn)
;;;
;;; except for the si = nil.
;;;
(defun type-slot-rule (slot restriction)
  ;; hack to force restriction to be a list:
  (if (not (consp restriction)) (setq restriction (list restriction)))
  (let ((restricted-vars nil)
        (consequents nil)
        (counter 0))
    (dolist (set (reverse restriction))
      (let ((var (new-variable (concatenate 'string "?x" (prin1-to-string (incf counter))))))
        (push var restricted-vars)
        (if set
          (push `(isa ,var ,set) consequents))))
    (when consequents
      (if (> (length restricted-vars) 2)
        `((,slot ,(car restricted-vars) (,@(cdr restricted-vars))) -> ,@consequents)
        `((,slot ,@restricted-vars) -> ,@consequents)))))

;;; Inorder to avoid a lot of unnecessary queries and asserts,
;;; replace with rules of the form:
;;;  ((R ?x ?y) (:unp (:retrieve (isa ?x S1))) -> (isa ?x S1))
;;;  ((R ?x ?y) (:unp (:retrieve (isa ?y S1))) -> (isa ?y S1))

	    



(defun akb-set-theory ()

  (a-assert "Taxonomy."
            '((:taxonomy (things
                          (objects
                           (sets things objects sets slots partitions
                                 (partitions main-partition set-partition
                                             slot-info-partition partition-partition))
                           (booleans true false)
                           (physical-attributes
                            (colors)
                            (genders male female))
                           (physical-objects
                            (people))
                           (contexts global-context))
                          (slots
                           (order-relations
                            (tc-order-relations
                             (equivalence-relations))))))))

  (a-assert "Bootstrap patch"
	    '(;; The imp-superset links came in "too late" for frames created in
	      ;; bootstrapping so we have to add isa links explicitly:
	      ;;
	      (isa things objects) (isa objects objects) (isa sets objects)
	      (isa slots objects) (isa partitions objects)
	      (isa things things) (isa objects things) (isa sets things)
	      (isa slots things) (isa partitions things)
	      ;;
	      (isa main-partition sets) (isa set-partition sets)
	      (isa slot-info-partition sets) (isa partition-partition sets)
	      (isa main-partition objects) (isa set-partition objects)
	      (isa slot-info-partition objects) (isa partition-partition objects)
	      (isa main-partition things) (isa set-partition things)
	      (isa slot-info-partition things) (isa partition-partition things)
	      ;;
	      (isa (:slot isa) things) (isa (:slot name) things) (isa (:slot type-slot) things)
	      (isa (:slot slot-partition) things) (isa (:slot comment) things) (isa (:slot backlink) things)
	      (isa (:slot inverse) things) (isa (:slot member) things) (isa (:slot subset) things)
	      (isa (:slot superset) things) (isa (:slot imp-superset) things)))



  ;; From here we define slots and rules for each set.

  ;; SETS.
  ;;
  ;; Several important slots for sets were actually defined above:
  ;;   (isa x s) = x is a member of s.
  ;;   (member s x) = A member of s is x.
  ;;   (subset s1 s2)      = A subset of s1 is s2. 
  ;;   (superset s1 s2)    = A superset of s1 is s2.
  ;;   (imp-superset s1 s2) = Important superset.
  ;;
  ;;
  (a-assert "Sets."
	    '(;; Subset and superset:
	      (isa (:slot subset) order-relations) (isa (:slot superset) order-relations)

	      (:slot union-of (sets sets)
		   :partition set-partition
		   :cardinality 2
		   :backlink superset
		   :comment "(union-of s1 s2)    = s1 is union of s2 with other set in slot.")
	      ;;
	      (:slot intersection-of (sets sets)
		   :partition set-partition
		   :cardinality 2
		   :backlink subset
		   :comment "(intersection-of s1 s2)    = s1 is intersection of s2 with other set in slot.")

	      ;; Note: Need to unify `intersect-sub' with rest of representation.
	      (:slot intersect-sub (sets sets sets)
		   :partition set-partition
		   :comment "(intersect-sub s1 s2 s3) = (s1 <intersect> s2) <subset> s3")
	      ;; intersect-sub has a strange sort of "backlink":
	      (:SRULES (:slot intersect-sub)
	        ((intersect-sub ?s1 ?s2 ?s3) -> (intersect-sub ?s2 ?s1 ?s3)))

	      (:slot total (sets booleans)
		   :partition set-partition
		   :backlink total
		   :comment "(total s1 s2)       = s1 union s2 contains all things.")
	      ;;
	      (:slot disjoint (sets sets)
		   :partition set-partition
		   :backlink disjoint
		   :comment "(disjoint s1 s2)    = s1 and s2 are disjoint.")
	      ;;
	      (:slot complement (sets sets)
		   :partition set-partition
		   :cardinality 1
		   :backlink complement
		   :comment "(complement s1 s2)  = The complement of s1 is s2.")

	       (:slot complete (sets booleans)
		   :partition set-partition
		   :comment "(complete s1 true) = All members of s1 are known.")

	      (:slot cf-member (things sets)
		   :partition set-partition
		   :comment "(cf-member x s)     = x is coreferent with some member of s.")
	      ;;
	      (:slot one-to-one-into (sets sets)
		   :partition set-partition
		   :comment "(one-to-one-onto s1 s2) = All members of s1 are coreferent with some member of s2.")
	      ;;
	      (:slot one-to-one (sets sets)
		   :partition set-partition
		   :comment "Bi-directional one-to-one-into.")
	      ;; 
	      (:SRULES (:slot one-to-one)
	       ((one-to-one ?s1 ?s2) -> (one-to-one-into ?s1 ?s2)
				        (one-to-one-into ?s2 ?s1)))
              ;;
	      ))

  (a-assert "Slots"
	   '((disjoint slots objects)
	      ;;
	      ;; Internally slotp is checked using the slotp property (this should probably change):
	      (:rules slots
	        ((isa ?slot slots) -> (:lisp (make-into-slot '?slot))))))
  
  ;; Partitions
  ;;
  ;; Partitions are sets of frame-slots.  Rather than trying to represent this
  ;; directly we use the slots slot-partition and frame-partition.  These are
  ;; interpreted as follows:
  ;;
  ;;   (slot-partition s p) => <f,s> in p
  ;;   (frame-partition f p) => <f,s> in p
  ;;
  ;; slot-partition was defined above so here we just define frame-partition.
  ;;
  (a-assert "Partitions"
	    '((:slot frame-partition (things partitions)
		   :partition partition-partition
		   :comment "(frame-partition f p) => <f,s> in p.")))


  (a-assert "Rules for sets"
	    '((:RULES sets
                      ;; Rules for deducing relationships between sets
                      ;;   complement = disjoint and total.
                      ((complement ?s1 ?s2) <- (disjoint ?s1 ?s2) (total ?s1 ?s2))
                      ((complement ?s1 ?s2) -> (disjoint ?s1 ?s2) (total ?s1 ?s2))
                      ;;
                      ;; Then ways for deducing membership
                      ;; (this first rule is actually subsumed by subset rule below)
                      ;;   In the union if in one of the unioned sets:
                      ;;((member ?s0 ?x) <- (union-of ?s0 ?s1) (member ?s1 ?x))
                      ;;   Must be in both sets to be in the intersection.
                      ((member ?s0 ?x) <- (intersection-of ?s0 ?s1) (intersection-of ?s0 ?s2) (:NEQ ?s1 ?s2)
                       (member ?s1 ?x) (member ?s2 ?x))
                      ;;   If two sets are total then have to be in one or the other.
                      ((member ?s0 ?x) <- (total ?s0 ?s2) (not (member ?s2 ?x)))
                      ;;   In a set if in one of its subsets.
                      ((member ?s0 ?x) <- (subset ?s0 ?s2) (member ?s2 ?x))
                      ;;
                      ;; And the corresponding ways of deducing non-membership (see comments
                      ;; above).
                      ;; (again first rule subsumed)
                      ;;((not (member ?s0 ?x)) <- (intersection-of ?s0 ?s1) (not (member ?s1 ?x)))
                      ((not (member ?s0 ?x)) <- (union-of ?s0 ?s1) (union-of ?s0 ?s2) (:NEQ ?s1 ?s2)
                       (not (member ?s1 ?x)) (not (member ?s2 ?x)))
                      ((not (member ?s0 ?x)) <- (disjoint ?s0 ?s2) (member ?s2 ?x))
                      ((not (member ?s0 ?x)) <- (superset ?s0 ?s2) (not (member ?s2 ?x)))
                      ;;
                      ;; An important property of complete sets:
                      ;;   If a set is complete and it is consistent to assume x is
                      ;;   not a member then assume x is a non-member.
                      ;; (an alternative here would be to assume ?x not coreferent with
                      ;;  all members of the set).
                      ((not (member ?s0 ?x)) <- (complete ?s0 true)
                       (:ASSUME (not (member ?s0 ?x))))
                      ;;
                      ;; Can deduce isa by following superset:
                      ((isa ?x ?s2) <- (isa ?x ?s1) (superset ?s1 ?s2))
                      ;;
                      ;; Can deduce one-to-one:
                      ((one-to-one ?s1 ?s2) <- (one-to-one-into ?s1 ?s2) (one-to-one-into ?s2 ?s1)))

              (:RULES objects
                      ;; Can deduce cf-member:
                      ((cf-member ?x ?s1) <- (isa ?x ?s2) (one-to-one-into ?s2 ?s1)))))
  )

;;; Common objects that should be in the basic KB.

(defun akb-common-things ()

  (a-assert "Booleans"
	    '((complete booleans true)))

  (a-assert "Genders"
	    '((complete genders true)))

  (a-assert "Physical-Objects."
	    '((:slot color (physical-objects colors)
		   :cardinality 1
		   :comment "(color x c) = The color of x is c.")
	      ;;
	      (:slot gender (physical-objects genders)
		   :cardinality 1
		   :comment "(gender x g) = The gender of x is g.")
	      ;;
	      ;; Don't define this here so we can use it in QPC:
	      ;;(:slot temperature (physical-objects nil)
	      ;;   :cardinality 1
	      ;;   :comment "(temperature x temp) = The temperature of x is temp.")
	      ))
  
  (a-assert "People."
	    '((:slot spouse (people people)
		   :cardinality 1
		   :backlink spouse
		   :comment "(spouse a b) = The spouse of a is b.")
	      ;;
	      (:slot wife (people people)
		   :cardinality 1
		   :backlink spouse
		   :comment "(wife a b) = The wife of a is b.")
	      ;;
	      (:slot husband (people people)
		   :cardinality 1
		   :inverse wife
		   :comment "(husband a b) = The husband of a is b.")
	      ;;
	      (:slot friend (people people)
		   :comment "(friend a b) = A friend of a is b.")

	      ;; Husband and wife also imply genders:
	      (:RULES people
	       ((wife ?x ?p1) -> (gender ?p1 female)))
	      (:RULES people
	       ((husband ?x ?p1) -> (gender ?p1 male)))))

  (a-assert "Contexts."
	    '((:slot super-context (contexts contexts)
		   :comment "(super-context c1 c2) = A super-context of c1 is c2.")
	      ;;
	      (:slot sub-context  (contexts contexts)
		   :inverse super-context
		   :comment "(sub-context c1 c2) = A sub-context of c1 is c2.")

	      (:slot current-context (contexts contexts)
		   :cardinality 1
		   :backlink super-context
		   :comment "(current-context c1 c2) = The current sub-context of c1 is c2.")

	      (:slot speaker (contexts people)
		   :cardinality 1
		   :comment "(speaker c s) = The speaker in c is s.")
	      ;;
	      (:slot recent (contexts objects)
		   :comment "(recent c r) = A recently mentioned thing in c is r.")))

  )

  ;; Order and Equivalence Relations.
  ;;
  ;; The set of equiv-relations is the set of relations under which things can
  ;; be equivivalent.  The set of order-relations is currently taken to be just the
  ;; set of transitive relations.  However, only those in the set tc-order-relations
  ;; are transitively closed with if-added rules.

(defun akb-order+equivalence ()
  
  (a-assert "Order and equivalence relations."
            '((:SRULES tc-order-relations
                       ;;Transitivity with a carefully coded if-added rule
                       ((?or1 ?x ?y) (:NEQ ?x ?y) (:RETRIEVE (?or1 ?y ?z))
                        ->
                        (?or1 ?x ?z))
                       ;;
                       ;;Actually the above rule suffices to define transitivity, but
                       ;;computationally the following contrapositives of transitivity
                       ;;avoid lots of proofs by contradiction.
                       ;; ? Should these be associated with ?or or ?tcor ?
                       ;; 
                       ((not (?or1 ?x ?y))
                        <-
                        (:RETRIEVE (not (?or1 ?x ?z)))
                        (inverse ?or1 ?or1i)
                        (:RETRIEVE (?or1i ?z ?y)))
                       ((not (?or1 ?y ?z))
                        <-
                        (inverse ?or1 ?or1i) 
                        (:RETRIEVE (?or1i ?y ?x))
                        (:RETRIEVE (not (?or1 ?x ?z)))))
              (:RULES equivalence-relations
                      ;;Symmetry
                      ((isa ?er1 ?er) -> (inverse ?er1 ?er1)))
              (:SRULES equivalence-relations
                       ;;Reflexivity
                       ((?er1 ?x ?x) <- (isa ?x things)))))
  
  ;; Coreference -- a particularly important equivalence relation.  However several of the
  ;; rules for equivalence relations are useless for coreference so I do not currently link
  ;; coreference to the set of equivalence relations.  Eventually may want to special case
  ;; coreference in the code itself.
  ;;
  (a-assert "Coreference"
            '((:slot coreferent (things things)
                     :inverse coreferent
                     :comment "(coreferent x y) = x and y are frames for the same thing.")
              
              (:RULES objects
                      ;;inheritance through coreferent link
                      ((?p ?x ?v) <- (:RETRIEVE (coreferent ?x ?y)) (:NEQ ?x ?y)
                       (:RETRIEVE (?p ?y ?v)))
                      ((not (?p ?x ?v)) <- (:RETRIEVE (coreferent ?x ?y)) (:NEQ ?x ?y)
                       (:RETRIEVE (not (?p ?y ?v))))
                      
                      ;; From cf-member (see sets above) can sometimes deduce coreference:
                      ((coreferent ?x ?y)
                       <-
                       (cf-member ?x ?s1) (complete ?s1 true) (member ?s1 ?y)
                       (:ALL-PATHS ((member ?s1 ?m1) (:neq ?y ?m1))
                                   ((not (coreferent ?x ?m1)))))
                      
                      ;;anything coreferent with itself
                      ((isa ?x objects) -> (coreferent ?x ?x)))
              
              (:SRULES (:slot coreferent)
                       ((coreferent ?x ?y) (:NEQ ?x ?y) (:RETRIEVE (coreferent ?y ?z))
                        ->
                        (coreferent ?x ?z))
                       ;; 
                       ((not (coreferent ?x ?y))
                        <-
                        (:RETRIEVE (not (coreferent ?x ?z)))
                        (:RETRIEVE (coreferent ?z ?y))))
              
              (:SRULES equivalence-relations
                       ;;Equivalent under any relation if coreferent
                       ;;(this rule not strictly needed but avoids obscure queries).
                       ((?er1 ?x ?y) <- (:RETRIEVE (coreferent ?x ?y))))))
  
  (a-assert "Partial orders."
            '((:slot less (objects objects)
                     :comment "(less ?x ?y) = ?x less than ?y.")
              ;;
              (:slot greater (objects objects)
                     :inverse less
                     :comment "(greater ?x ?y) = ?x greater than ?y.")
              
              (:slot equal (objects objects)
                     :inverse equal
                     :comment "(equal ?x ?y) = ?x is equal to ?y.")
              
              (:slot least (objects sets)
                     :comment "(least ?x ?s) = ?x is the least member of ?s.")
              ;;
              (:slot greatest (objects sets)
                     :comment "(greatest ?x ?s) = ?x is the greatest member of ?s.")
              
              ;; propagate through equal with if-added rules:
              ;; (carefully crafted to not backchain or complete)
              ;; (Since less and greater are inverses don't need rules for both.)
              (:srules (:slot less)
                       ((less ?x ?y) -> (:RETRIEVE (equal ?x ?z)) (:neq ?x ?z) (less ?z ?y))
                       ((less ?x ?y) -> (:RETRIEVE (equal ?y ?z)) (:neq ?x ?z) (less ?x ?z)))
              (:srules (:slot equal)
                       ((equal ?x ?y) -> (:RETRIEVE (less ?x ?z)) (:neq ?x ?z) (less ?y ?z)))
              
              ;; ways to prove negation of relations:
              (:srules (:slot equal)
                       ((not (equal ?x ?y)) <- (less ?x ?y))
                       ((not (equal ?x ?y)) <- (greater ?x ?y)))
              (:srules (:slot less)
                       ((not (less ?x ?y))  <- (equal ?x ?y))
                       ((not (less ?x ?y))  <- (greater ?x ?y)))
              (:srules (:slot greater)
                       ((not (greater ?x ?y))  <- (equal ?x ?y))
                       ((not (greater ?x ?y))  <- (less ?x ?y)))
              
              ;; deductions from least and greatest:
              (:srules (:slot less)
                       ((less ?x ?y) <- (least ?x ?s1) (member ?s1 ?y)
                        (:NEQ ?x ?y)
                        (:ASSUME (not (coreferent ?x ?y)))))
              ;;
              (isa (:slot less) tc-order-relations)
              (isa (:slot greater) order-relations)
              (isa (:slot equal) equivalence-relations))))
