;;; -*- Mode:Common-Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-

;;; Copyright (c) 1990 by James Crawford.
;;;  $Id: aidg.lisp,v 1.1 92/04/16 09:30:04 clancy Exp $

;;;                        ****** AIDG ******

; Provides an abstract interface to the knowledge-base.
; All insertions, deletions, and retrievals of VALUES should be done
; through this module (one exception: browser routines, which need
; to examine the knowledge-base without changing it in any way,
; should go directly through the aframes level).  All external calls
; are currently at the predicate level (i.e. not the fs level),
; except for a call to fs-get-values-woa in fullp.
;
; Basic data structures:
;
; It is an interesting design question wheither there should be a
; structure "value tag" which is sometimes passed around instead
; of the current "result" structure, and which would be stored
; in the k-base with values.  The current philosophy is to pass
; around result structures in alogic and arules, and to translate
; in aidg to the form used in the k-base itself.  This means that
; if we add anything to the result structures then aidg has to
; be re-written.
;
; Currently values are stored in the knowledge-base with a
; lists of lists of assumptions (assump-ll's) making them true.
;
; Example: Suppose V1 is always true, V2 is true only under
; the assumption A1, and V3 is true under the assump-ll A2 or
; A3 and A4.  Then these values are associated with assump-ll's
; as follows:
;
;         V1 -- (nil)
;         V2 -- ((A1))
;         V3 -- ((A2) (A3 A4))
;
;
; Externally callable routines:
;
;  INSERT-VALUE -- pred result
;    Inserts pred in the knowledge-base under the assumptions
;    in result.  If *forward-chain* then if-added rules are fired (if
;    anything new is actually added).
;
;    Returns t if value added without contradiction.
;
;  INSERT-ASSUMPTION -- pred result
;    Just like insert-value except that pred is inserted as an
;    assumption.
;
;    Returns t if value added without contradiction.
;
;  DELETE-VALUE -- pred
;    pred is deleted from the knowledge-base.
;
;    Returns t.
;
;  GET-VALUES -- pred
;    Gets currently believed values for pred.  To be believed a
;    value must be in the knowledge-base with assumptions all of which are
;    currently believed in the knowledge-base.
;    If *back-chain* then if-needed-rules are fired before values returned.
;
;    Returns an alist of values and results.
;
;  KNOWN  -- pred
;
;    Like get-values but only examines a single value (thus (value pred) must
;    not be a variable).
;
;    Returns nil or pair ((value pred) result).


; INSERT-VALUE: Inserts pred under assump-ll in result.  Lots of details to
; keep track of when a new value is added:
;
; * Adds pred with subsumption checking of assumptions.
; * Checks to make sure pred is consistent.
; * If pred exceeds a number-restriction then asserts a coreference link.
;   Currently this only happens if the frame-slot is full of values without assumptions,
;   the number-restriction is 1, and pred is binary.
; * If pred a new value then makes call to give-name.
; * Makes calls to tracing utilities and to set up output.
; * Calls frame-slot-overfull if slot now too full (of values without assumptions).
; * Propagates value through dependency net.
; * If *forward-chain* then applies if-added rules for pred.
;
; Returns t if value added without contradiction.
;
(defun insert-value (pred result)
  (let ((frame (frame pred))
	(slot (slot pred))
	(facet (facet pred))
	(value (value pred))
	(negated (negated pred))
	(assump-ll (aresult-assump-ll result)))
    (if (and @debug 
	     (or (not (aresult-p result)) (null assump-ll)
		 (variable? frame) (variable? slot) (variable? value)))
	(error "Algernon bug -- Bad value insertion of ~(~a~) under result ~(~a~)."
	       pred result))
    (cond
      ((fs-put frame slot facet value assump-ll)
       (trace-new-value-insertion-begin pred result)
       (push (list pred assump-ll) *last-inserted-values*)
       (trace-new-value-insertion-end
	 pred
	 (when (with-no-back-chaining (consistent pred))
	   (if (and (not (negated pred)) (overfullp frame slot))
	       (frame-slot-overfull frame slot value))
	   (propagate pred)
	   (when *forward-chain*
	     (if (and (not negated) (eql slot 'isa))
		 (complete-set-wrt-frame value frame))
	     (if (in-partitions frame slot)
		 (fire-if-added-rules pred)
		 (queue-assertion pred)))	; (queue so we can fire rules later)
	   t)))
      (t					; pred previously known.
       (trace-value-insertion pred result)
       t))))

; INSERT-ASSUMPTION: Insert a new assumption.  An assumption is inserted as
; a value with itself as one of its assumptions.  The same details are
; taken care of as for insert-value except that the consistency check is
; more thorough (see acontra.l).
;
; Returns t if value is legal assumption and does not cause contradiction.
;
(defun insert-assumption (pred result)
  (let ((frame (frame pred))
	(slot (slot pred))
	(facet (facet pred))
	(value (value pred))
	(assump-ll (aresult-assump-ll result)))
    (if (and @debug 
	     (or (not (aresult-p result)) (null assump-ll)
		 (variable? frame) (variable? slot) (variable? value)))
	(error "Algernon bug -- Bad assumption insertion of ~(~a~) under result ~(~a~)."
	       pred result))
    (cond
      ((fs-put frame slot facet value
	       (conjunct-assumps (list pred) assump-ll))
       (trace-new-assumption-insertion-begin pred result)
       (push (list pred assump-ll) *last-inserted-assumptions*)
       (trace-new-assumption-insertion-end
	   pred
           (cond ((consistent pred)
		  (propagate pred)
		  (if *forward-chain*
		      (if (in-partitions frame slot)
			  (fire-if-added-rules pred)
			  (queue-assertion pred)))    ;    (queue so we can fire rules later)
		  t))))
      (t
       (trace-assumption-insertion pred result)
       t))))

; DELETE-VALUE: Removes a value from knowledge-base.  If slot = name then name is retracted.
; Calls also made to tracing and output routines.  It is not clear what should be done to
; the dependency net -- currently nothing is done.
;
; Returns t if fs-delete succeeds (which it currently always does).
;
(defun delete-value (pred)
  (let ((frame (frame pred))
	(slot (slot pred))
	(facet (facet pred))
	(value (value pred)))
    (if (and @debug 
	     (or (variable? frame) (variable? slot) (variable? value)))
	(error "Algernon bug -- Bad deletion of ~(~a~)." pred))
    (cond ((fs-delete frame slot facet value)
	   (trace-value-deletion pred)
	   (push pred *last-deleted-values*)
	   ; (propagate frame slot) ?????
	   t))))

; GET-VALUES: Get currently believed values for pred.  To be believed a
; value must be in the knowledge-base with assumptions all of which are
; currently in the knowledge-base.  Call also made to tracing function,
; and to dependency function.
;
; If *back-chain* then if-needed-rules are fired before values returned.
;
; If a value is believed under a set of assumptions and the negation of one
; of these assumptions is currently known (without assumptions) then the entire
; set of assumptions is deleted.  If all sets of assumptions for a value are ever
; deleted then the entire value is deleted (by calling delete-values above).
;
; Returns an alist of values and results.
;
(defun get-values (pred)
  (let ((frame (frame pred))
	(slot (slot pred))
	(facet (facet pred)))
    (if (and @debug 
	     (or (variable? frame) (variable? slot)))
	(error "Algernon bug -- Bad get-values of ~(~a~)." pred))
    (trace-value-get pred)
    (if *back-chain* 
	(if (in-partitions frame slot)
	    (fire-if-needed-rules pred)
	    (queue-query pred)))
    (depends pred)
    (mapcar #'(lambda (pair)
		(let ((new-result (new-aresult)))
		  (setf (aresult-assump-ll new-result) (cdr pair))
		  (cons (car pair) new-result)))
	    (fs-get-values frame slot facet))))

; KNOWN: Like get-values but only examines a single value (thus (value pred) must
; not be a variable).
;
; It is not clear wheither back-chaining should occur on values believed
; under some assumption -- currently it does not but maybe such back-chaining
; should be put near the end of some sort of agenda.  Back-chaining could also
; suppresed if the negation of the predicate is known without assumptions
; but this is not currently done as it would make it difficult (at best) to do proofs
; by contradiction.
;
; Returns nil or pair ((value pred) result).
;
(defun known (pred)
  (let ((frame (frame pred))
	(slot (slot pred))
	(facet (facet pred))
	(value (value pred)))
    (if (and @debug 
	     (or (variable? frame) (variable? slot) (variable? value)))
	(error "Algernon bug -- Bad call to known of ~(~a~)." pred))
    (trace-value-get pred)
    (let ((assump-ll (cdr (fs-known frame slot facet value pred nil))))
      (when (and *back-chain*
		 ;;(not (known-woa (negate pred)))
		 (not (member nil assump-ll :test #'eq)))  ; ?? Fire rules when known under some assumption ??
	(if (in-partitions frame slot)
	    (fire-if-needed-rules pred)
	    (queue-query pred))
	(setq assump-ll (cdr (fs-known frame slot facet value pred nil))))
      (if (not (member nil assump-ll :test #'eq))
	  (depends pred))
      (if assump-ll
	  (let ((new-result (new-aresult)))
	    (setf (aresult-assump-ll new-result) assump-ll)
	    (cons value new-result))))))



; FS routines.

; Frame-Slot routines.  A slightly lower level interface to the
; knowledge-base.  These routines access the knowledge-base at the
; frame-slot-facet level and provide a level of abstraction just
; above that in aframes.  They handle the management of assump-ll's
; and calls to anames.
;
; In the value (and non-value) facet of the knoweldge-base are
; kept association lists of values and assump-ll.  Thus if
; V1 is always true, V2 is true only under the assumption A1,
; and V3 is true under the assump-ll A2 or A3 and A4 then the
; value facet would hold:
;
;        ((V1 nil) (V2 (A1)) (V3 (A2) (A3 A4)))

; Fs-Put: merge assump with existant assumptions for value.
; * If old value then does subsumption checking with
;   previous assumptions for value.
; * If new value then inserts and makes call to give-name.
;
; Returns true iff a new value or assumptions actually inserted.
;
(defun fs-put (frame slot facet value assump-ll)
  (if @debug (check-frame frame slot facet value))
  (let ((old-assumps (assoc value (fget frame slot facet) :test #'equal)))
    (cond (old-assumps
	   (cond ((assump-ll-implies assump-ll (cdr old-assumps))
		  nil)
	         (t
		  (dolist (assumps assump-ll) (setf (cdr old-assumps)
						    (ndisjunct-assumps assumps (cdr old-assumps))))
		  t)))
          ((fput frame slot facet (cons value assump-ll))
	   (if (and (eql slot 'name) (eql facet @value))
	       (give-name frame value))
	   t))))

; Fs-Delete: Delete value from facet of slot of frame.  Also updates naming info.
; Returns t if fremove succeeds (which it currently always does).

(defun fs-delete (frame slot facet value)
  (if @debug (check-frame frame slot facet value))
  (cond ((fremove frame slot facet value
		  #'(lambda (x) (equal (car x) value)))
	 (if (and (eql slot 'name) (eql facet @value))
	   ; clear out all old names
	   (delete-name frame value))
	 t)))

; Fs-Get-Values: Get currently believed values in frame slot facet.  To be believed a
; value must be in the knowledge-base with assumptions all of which are
; currently in the knowledge-base.
;
; If a value is believed under a set of assumptions and the negation of one
; of these assumptions is currently known (without assumptions) then the entire
; set of assumptions is deleted.  If all sets of assumptions for a value are ever
; deleted then the entire value is deleted (if value is deleted naming info is
; updated but no tracing is currently done).
;
; Returns an alist of values and their assumptions.
;
(defun fs-get-values (frame slot facet)
  (if @debug (check-frame frame slot facet 'all-values))
  (let ((kb-value-list (fget frame slot facet)))
    (if kb-value-list
      (mapcan #'(lambda (kb-value)
                  (let ((current-value (check-value kb-value nil)))
                    (cond (current-value
                           (list current-value))
                          ((null (cdr kb-value))
                           ; back up to pred level for deletion
                           (delete-value (predicate frame slot facet (car kb-value)))
                           nil))))
              kb-value-list))))

; Fs-Known: Like Fs-Get-Values but only looks at one value.  Keeps list of preds 'visited'
; to avoid infinite loops.
;
; Returns nil or pair (value . list of list of assumptions).  One exception: just returns
; t if pred already visited.
;
(defun fs-known (frame slot facet value pred &optional visited)
  (if @debug (check-frame frame slot facet value))
  (if (member pred visited :test #'eq) ; Here we explicitly assume structure sharing.
    t
    (let ((kb-value (assoc value (fget frame slot facet) :test #'equal)))
      (if kb-value
        (let ((current-value (check-value kb-value (cons pred visited))))
          (cond (current-value)
                ((null (cdr kb-value))
                 ; back up to pred level for deletion
                 (delete-value (predicate frame slot facet (car kb-value)))
                 nil)))))))

(defun fs-get-values-woa (frame slot facet)
  (mapcan #'(lambda (x) (if (member nil (cdr x) :test #'eq) 
                          (list (car x))))
	  (fget frame slot facet)))

(defun fs-get-values-ignoring-assumptions (frame slot facet)
  (mapcan #'(lambda (x) (list (car x)))
	  (fget frame slot facet)))



; Utility Routines

; Check-Value: Check if value is currently believed.  To be believed a
; value must be in the knowledge-base with assumptions all of which are
; currently in the knowledge-base.
;
; If a value is believed under a set of assumptions and the negation of one
; of these assumptions is currently known (without assumptions) then the entire
; set of assumptions is deleted.  As an efficiency hack, an association list
; is built of list of assumptions and "tags":
;
;   0 -- Every assump in assumps is currently believed and the negation of none is known (without assumption).
;   1 -- Some assump is not believed but the negation of none is known (without assumption).
;   2 -- The negation of some assump is known.
;
; Returns pair (value . list of list of believed assumptions ) or nil.
;
(defun check-value (value visited)
  (let ((assump-ll (cdr value)))
    (if (equal assump-ll '(nil))
      value
      (let ((taged-assumps (mapcar #'(lambda (assumps)
                                       (cons assumps (tag-assumps assumps visited)))
                                   assump-ll)))
        (setf (cdr value) (delete-if #'(lambda (assumps)
                                         ; (Assume structure sharing so use :test #'eq)
                                         (eq (cdr (assoc assumps taged-assumps :test #'eq)) 2))
                                     (cdr value)))
        (let ((believed-assump-ll (mapcan #'(lambda (assumps)
                                              (if (eq (cdr assumps) 0) (list (car assumps))))
                                          taged-assumps)))
          (if believed-assump-ll
            (cons (car value) believed-assump-ll)))))))

; Tag-Assumps: Returns tag for assumps.
;
; If an assumption is believed then we do not check to see if its negation
; is known (i.e. we assume that the knoweldge-base is consistent).
;
(defun tag-assumps (assumps visited)
  (if (consp assumps)
      (let ((tag (tag-assump (car assumps) visited)))
	(case tag
	  (0 (tag-assumps (cdr assumps) visited))
	  (1 (max 1 (tag-assumps (cdr assumps) visited)))
	  (2 2)))
      0))

; Tag-Assump: Returns tag for assump.
;
(defun tag-assump (assump visited)
  (if (fs-known (frame assump) (slot assump) (facet assump) (value assump) assump visited)
      0
      (if (known-woa (negate assump)) 2 1)))


; KNOWN-WOA: Like get-values-woa but returns t iff pred believed without assumption.
; Currently only used by known.
;
; Returns t or nil.
;
(defun known-woa (pred)
  (let ((frame (frame pred))
	(slot (slot pred))
	(facet (facet pred))
	(value (value pred)))
    (if (and @debug 
	     (or (variable? frame) (variable? slot) (variable? value)))
	(error "Algernon bug -- Bad call to known-woa of ~(~a~)." pred))
    (member nil (cdr (assoc value (fget frame slot facet) :test #'equal)))))


; Frame-Slot-Overfull: Called when new-value is added to a full frame-slot.
;  If frame-slot holds a single value then new-value is asserted to
;  be coreferent with it.  There are lots of hard problems if the
;  number restriction is more than one, or if some values are
;  believed under assumptions.  For now we put these off.
;
; Also, if new-value is a list then should assert coreference for all elements
; in it --- this is not currently done.
;
(defun frame-slot-overfull (frame slot new-value)
  (if (equal (fget slot @slot-props @num-res) '(1))
      (let ((new-var (new-variable (gen-new-input-var))))
	(if (and (slotp 'coreferent)
		 (framep new-value))
	    (if (consp new-value)
		nil				; Needs to be filled in.
		(with-no-depnet (internal-assert `((:retrieve (,slot ,frame ,new-var))
                                                   (coreferent ,new-value ,new-var))
                                                 (list (new-aresult)))))))))

