;;; -*- Mode:Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-

;;; Copyright (c) 1990 by James Crawford
;;;  $Id: adump.lisp,v 1.1 92/04/16 09:29:59 clancy Exp $
;;; (with much help from Adam Farquhar).

;;;                        ****** ADUMP ******


;;; Supports saving and restoring states of the Algernon knowledge-base.
;;; Knowledge-bases can be stored in memory and/or dumped to a file on disk.

;;; Assumes frames are comparable using eq.

;;; Interface:
;;;
;;;   KB-SNAPSHOT name
;;;  
;;;   Take a "snapshot" of the knowledge-base and call it name.
;;;   Returns the stored kb.
;;; 
;;;   WRITE-SNAPSHOT name &optional (path *user-kb-path*)
;;;
;;;   Write snapshot name to disk in a file called name.kb.
;;;   Returns t if write successful.
;;;
;;;   LOAD-KB name &optional (path *user-kb-path*)
;;;   DUMP-KB name &optional (path *user-kb-path*)
;;;
;;;   Read and write knowledge-bases.  load-kb loads either the snapshot name
;;;   or the file name.kb, whichever is more recent.  load-kb returns t if 
;;;   load was successful.  dump-kb stores the current
;;;   knowledge-base both in memory and on disk.  Returns t if write successful.
;;;
;;;   KB-EXISTS? name &optional (path *user-kb-path*)
;;;
;;;   Returns t if kb known (either in memory or on disk).
;;;
;;;   DELETE-SNAPSHOT name
;;;
;;;   Delete snapshot name.  Returns t if name existed previously.

(defparameter @algy-properties
	      '(in-name framep slotp facetp frame)
  "Symbol-plist properties used by Algernon")

(defstruct (kb
	     (:print-function print-kb))
  name
  write-date
  file						; the actual file object
  frames
  max-frame
  num-values
  all-names
  num-rules
  rule-application-id
  frame-definitions)


;; KB-SNAPSHOT
;;
(defun kb-snapshot (name)
;; dump to a kb and return it
  (let ((kb (find-kb name :create t)))
    (setf (kb-write-date kb) (get-universal-time)
	  (kb-frames kb) *all-frames*
	  (kb-max-frame kb) *max-frame*
	  (kb-num-values kb) *num-values*
	  (kb-all-names kb) *all-names*
	  (kb-num-rules kb) *num-rules*
	  (kb-rule-application-id kb) *rule-application-id*
	  (kb-frame-definitions kb) (frame-definitions))
    kb))

;; WRITE-SNAPSHOT
;;
(defun write-snapshot (name &optional (path *user-kb-path*))
  (let ((kb (find-kb name)))
    (if kb
	(write-kb kb path))))

;; LOAD-KB
;;
(defun load-kb (kb-name &optional (path *user-kb-path*))
  (let ((kb (find-kb kb-name)))
    (cond ((null kb)
	   (when (probe-file (format nil "~a~(~a~)~a"
				     path kb-name @kb-file-extension))
	     (install-kb (read-kb (find-kb kb-name :create t) path))))
	  ((null (kb-file kb))
	   ;(format t "~&Installing in-core kb, ~a." (kb-name kb))
	   (install-kb kb))
	  ((probe-file (pathname (kb-file kb)))
	   (when  (> (file-write-date (kb-file kb))
		     (kb-write-date kb))
	     (format t "~&A more recent version of the knowledge base ~a ~
                         has been written to disk.~@
                         Reading in the newest version.~%" kb-name)
	     (read-kb kb path))
	   (install-kb kb))
	  (T NIL))))

;; DUMP-KB
;;
(defun dump-kb (name &optional (path *user-kb-path*))
  (write-kb (kb-snapshot name) path))

;; KB-EXISTS?
;;
(defun kb-exists? (kb-name &optional (path *user-kb-path*))
  (or (member kb-name *kbs* :key #'kb-name)
      (probe-file (format nil "~a~(~a~)~a"
			  path kb-name @kb-file-extension))))

;;; DELTE-snapshot
;;;
(defun delete-snapshot (name)
  (let ((kb (find-kb name)))
    (if kb
	(setq *kbs* (delete kb *kbs*)))))


;;; Utility functions.

(defun print-kb (kb str depth)
  (declare (ignore depth))
  (format str "#<Kb: ~a>" (kb-name kb)))

;; WRITE-KB
;;  Writes the snapshot kb to disk.
;;
(defun write-kb (kb path)
  (let ((file-path (format nil "~a~(~a~)~a" path (kb-name kb) @kb-file-extension)))
    ;; CLtL does not define a *print-structure* variable.  It is called
    ;; different things on different systems.  See algy.lisp for the
    ;; parameter definition.  Hopefully ANSI is better.  If we don't do
    ;; this, then the aresults can get printed out unreadably, making a
    ;; kb unreadable.
    (progv (list *sys-print-structure-var*) '(t)
      (with-open-file (out-file file-path :direction :output :if-exists :supersede)
	(print (kb-frames kb) out-file)
	(print (kb-max-frame kb) out-file)
	(print (kb-num-values kb) out-file)
	(print (kb-all-names kb) out-file)
	(print (kb-num-rules kb) out-file)
	(print (kb-rule-application-id kb) out-file)
	(dolist (thing (kb-frame-definitions kb))
	  (print thing out-file)))
      (setf (kb-write-date kb) (file-write-date file-path)	; change time in kb so will load from kb not file.
	    (kb-file kb) file-path)
      t)))


(defun find-kb (name &key create)
  (or (find name *kbs* :key #'kb-name)
      (and create
	   (car (push (make-kb :name name) *kbs*)))))

; return a list <frame-name> <algy-plist>
;
(defun frame-definitions ()
  (let (definitions)
    (dolist (symbol (union *all-frames* *all-names*))
      (push symbol definitions)
      (push (keep-properties @algy-properties (symbol-plist symbol)
			     :copy t)
	    definitions))
    (nreverse definitions)))

;; READ-KB
;;  Reads from the file (kb-name kb) and stores the result in *kbs*.
;;
(defun read-kb (kb path)
  (let ((path
	  (format nil "~a~(~a~)~a" path (kb-name kb) @kb-file-extension)))
    (with-open-file (in-file path :direction :input)
      (setf 
	(kb-write-date kb) (get-universal-time)
        ;; Used to be "(kb-file kb) in-file" (changed since breaks mac).
	(kb-file kb) path
	(kb-frames kb)     (read in-file)
	(kb-max-frame kb)  (read in-file)
	(kb-num-values kb) (read in-file)
	(kb-all-names kb)  (read in-file)
	(kb-num-rules kb)  (read in-file)
	(kb-rule-application-id kb) (read in-file)
	(kb-frame-definitions kb)
	(let (definitions 
	      frame-name)
	  (loop (if (eql (setq frame-name (read in-file nil 'EOF))
			 'EOF)
		    (return definitions)
		    (setq definitions
			  (cons frame-name
				(cons (let ((frame-def (read in-file)))
					(prep-vars frame-def)
					frame-def)
				      definitions)))))))))
  kb)

;; INSTALL-KB
;;  Installs a snapshot into Algernon.
;;
(defun install-kb (kb)
  (setq *all-frames* (union (kb-frames kb) *all-frames*)
	*max-frame*  (max (kb-max-frame kb) *max-frame*)
	*num-values* (kb-num-values kb)		;bugged, conservative estimate
	*all-names*  (union (kb-all-names kb) *all-names*)	
	*num-rules*  (kb-num-rules kb)		; conservative estimate
	*rule-application-id* (max (kb-rule-application-id kb)
				   *rule-application-id*))
  (do ((frames (kb-frame-definitions kb))
       (name)
       (slots))
      ((null frames) T)
    (setq name (first frames)
	  slots (second frames))
    (if (and @debug
	     (framep name)
	     (not (member name @system-slots :test #'eq))) ; Every loaded kb redefines system-slots.
	(algy-warning (format nil "Knowledge-base ~(~a~) redefining the frame ~@(~s~)." kb name)))
    ;;(format t "~&Installing ~a~%" name)
    (install-properties name slots)
    (setq frames (cddr frames)))
  T)

(defun install-properties (symbol plist)
  (cond ((null plist))
	((setf (get symbol (first plist))
	       (copy-tree (second plist)))
	 (install-properties symbol
			     (cddr plist)))))

(defun keep-properties (properties plist &key copy)
  (cond ((null plist) NIL)
	((member (car plist) properties :test #'eq)
	 (cons (car plist)
	       (cons (if copy
			 (copy-tree (cadr plist))
			 (cadr plist))
		     (keep-properties properties (cddr plist) :copy copy))))
	(T
	 (keep-properties properties (cddr plist) :copy copy))))

(defun wipe-clean (&optional verbose)
  (do-symbols (symbol)
    (when (framep symbol)
	  (when verbose
	    (pp-frame symbol))
	 (dolist (prop @algy-properties)
	   (remprop symbol prop)))))