;;; -*- Mode:Lisp; Package:USER; Syntax:COMMON-LISP; Base:10 -*-

;;; Copyright (c) 1990 by James Crawford.
;;;  $Id: adepnet.lisp,v 1.1 92/04/16 09:29:59 clancy Exp $

;;;                        ****** ADEPNET ******

; Routines to manage a dependency network amoung the rules and
; slots.
;
; A frame-slot is a cons pair (frame . slot).
;
; Modified 3/5/90 so that *dn-frame-slots* is an assoc list (frames then slots)
; rather than a list of pairs.
;
; Assumes frames are comparable using eq.

(defvar *dn-frame-slots* nil)
(defvar *dn-cur-rules* nil)

; WITH-NO-DEPNET
; A macro to temporarily suppress recording of dependencies.
;
; (Code in aglobals).


; Reset-Dnet: Clear out dependency network.
;
(defun Reset-Dnet ()

  ;; (trace-off)
  
  ;; First complete post-poned rules (if required by search strategy):
  (if (eql *search-strategy* 'depth-first)
      (loop
	(if (not *rules-to-complete*) (return))
	(let ((rules *rules-to-complete*))
	  (setq *rules-to-complete* nil)
	  (dolist (pair rules)
	    (let ((rule (car pair))
		  (result (cdr pair)))
	      (update-iar-now rule result))))))
  
  ;; Then the dependency net itself.
  (dolist (slot-list *dn-frame-slots*)
    (let ((frame (car slot-list)))
      (dolist (slot (cdr slot-list))
        (when (and (framep frame) (slotp slot))
          ;;(format t "~% <~a,~a>" frame slot)
          (fclear-facet frame slot @preds)
          (fclear-facet frame slot @rules-dep)))))
  (setq *dn-frame-slots* nil)
  (if (and @debug *dn-cur-rules*)
    (format t "~% A rule is still current (possibly because of an unexpected abort): ~a."
            *dn-cur-rules*))
  (setq *dn-cur-rules* nil))


; Visit-frame-slot: Returns true if if-needed rules for pred should be applied.
;
(defun Visit-Frame-Slot (pred)
  (let* ((frame (frame pred))
	 (slot  (slot pred)))
    (cond ((and (not (negated pred)) (fullp frame slot)) ; Don't visit if full.
	   nil)
          ((member pred (fget frame slot @preds) :test #'less-general) ; Don't visit if visited already.
	   nil)
          (t
	   (add-to-dn-frame-slots frame slot)
	   (fput frame slot @preds pred)
	   t))))

; Make-Rule-Current: Makes rule-pair the current rule.
;
(defun Make-Rule-Current (rule-pair)
  (push rule-pair *dn-cur-rules*))

(defun current-rule ()
  (car *dn-cur-rules*))

; End-Current-Rule.
;
(defun End-Current-Rule ()
  (if (null *dn-cur-rules*)
      (error "Algernon Bug -- No current rule for end-current-rule to end."))
  (pop *dn-cur-rules*))

; Depends: Declares that the current rule depends on pred.
;
(defun Depends (pred)
  (when (and (current-rule) (not *no-depnet*))
    (let ((frame (frame pred))
          (slot (slot pred)))
      (add-to-dn-frame-slots frame slot)
      (fput frame slot @rules-dep (cons pred (current-rule)))
      ;(break (format nil "Rule = ~a   Pred = ~a" (current-rule) pred))
      )))

; Propagate: Propagates addition of pred to knowledge-base.
;
; Propagate accesses *as-list* and *qu-list* which should be internal to arules,
; but I don't know of a cleaner way to write this (can't use fire-selected-rules
; since need to check each pair to see if pred less general).
;
; It is also not clear whether or not we should check *forward-chain* and
; *back-chain* here.  My current thinking is that we are propigating through
; rules that were fired already for other predicates, so the settings now of
; *forward-chain* and *back-chain* are irrelevant.
;
; Modified 7/2/91 to backchain while propagating.
;
(defun Propagate (pred)
  (trace-propagation-begin pred)
  (cond (*top-level*
	 (let ((*top-level* nil)
	       (*as-list* nil)
	       (*qu-list* nil)
	       (*back-chain* t))
	   (apply-depnet-rules pred)
	   (if (or *as-list* *qu-list*) (fire-rules))))
	(t
	 (apply-depnet-rules pred)))
  (trace-propagation-end))

(defun apply-depnet-rules (pred)
  (mapc #'(lambda (pair)
	    (let* ((queried-pred (car pair))
		   (rule-pair (cdr pair))
		   (result (cdr rule-pair))
		   (new-subst (match queried-pred pred (aresult-sub result))))
            (when (not (eql new-subst 'failed))
	      (let* ((new-result (copy-aresult result))
		     (new-rule-pair (cons (car rule-pair) new-result)))
		(setf (aresult-sub new-result) new-subst)
                (make-rule-current new-rule-pair)
                (apply-rule new-rule-pair t)))))
        (fget (frame pred) (slot pred) @rules-dep)))


(defun add-to-dn-frame-slots (frame slot)
  (let ((cur-slots (assoc frame *dn-frame-slots* :test #'eq)))
    (if cur-slots
      (pushnew slot (cdr cur-slots) :test #'eq)
      (push (list frame slot) *dn-frame-slots*))))
      
