;;;   -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-    ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; PEANO
;;;
;;; Encode a part of Peano arithmetic using frames.  The main purpose of this example is
;;; to test Algernon on large knoweldge-bases.

(defun facts-about-numbers ()
  (a-assert "Numbers" '((:taxonomy (objects (numbers zero)))))
  (a-assert "Slots" '((:slot succ (numbers numbers)
                             :cardinality 1)
                      (:slot prev (numbers numbers)
                             :inverse succ
                             :cardinality 1)
                      (:slot value (numbers nil)
                             :comment "Equivalent in base 10")
                      (:slot sum (numbers numbers numbers)
                             :comment "(sum x y z) = x + y = z")
                      (:slot diff (numbers numbers numbers)
                             :comment "(diff x y z) = x - y = z")
                      (:slot mult (numbers numbers numbers)
                             :comment "(mult x y z) = x * y = z")))
  (a-assert "Arithmetic"
            '((:rules numbers
                      ;; 0 + x = x
                      ((sum zero ?x ?x) <-)

                      ;; x + y = z <- (x-1) + y = w and w + 1 = z
                      ((sum ?x ?y ?z) <- (sum (prev ?x) ?y ?w) (succ ?w ?z))

                      ;; x - y = z <- (x-1) - y = w and w + 1 = z
                      ((diff ?x ?x zero) <-)
                      ((diff ?x ?y ?z) <- (diff (prev ?x) ?y ?w) (succ ?w ?z))

                      ;; x * y = z <- (x-1) * y = w and w + y = z
                      ((mult zero ?x zero) <-)
                      ((mult ?x ?y ?z) <- (mult (prev ?x) ?y ?w) (sum ?y ?w ?z)))))
  (a-assert "Zero"
            '((value zero 0))))

;;; Make frames for numbers less than n:
;;;
(defun make-numbers (n)
  (let ((last-frame 'zero))
    (flet ((my-setq (x) (setq last-frame x)))
      (dotimes (x n)
        (let ((value (+ 1 x)))
          (a-assert nil `((:create ?x)
                          (succ (:quote ,last-frame) ?x)
                          (value ?x ,value)
                          (:apply ,#'my-setq (?x)))))))))

(defun test-numbers (max-num interations)
  (let ((total-time 0)
	(total-rules 0))
    (kb-snapshot 'numbers)
    (dotimes (x interations)
      (let (x1 x1-frame x1-value
	    x2 x2-frame x2-value
	    sum
	    start-time end-time seconds)
	;; First find two numbers:
	(loop
	  (setq x1 (random (round (/ max-num 2.0))))
	  (setq x1-frame  (intern (format nil "FRAME~a" x1)))
	  (if (and (framep x1-frame)
		   (silently (a-query nil `((isa (:quote ,x1-frame) numbers)))))
	      (return)))
	(loop
	  (setq x2 (random (round (/ max-num 2.0))))
	  (setq x2-frame  (intern (format nil "FRAME~a" x2)))
	  (if (and (framep x2-frame)
		   (silently (a-query nil `((isa (:quote ,x2-frame) numbers)))))
	      (return)))
	;; And get their values:
	(flet ((set-x1-value (x) (setq x1-value x))
	       (set-x2-value (x) (setq x2-value x)))
	  (silently (a-query nil `((value (:quote ,x1-frame) ?value)
				   (:apply ,#'set-x1-value (?value)))))
	  (silently (a-query nil `((value (:quote ,x2-frame) ?value)
				   (:apply ,#'set-x2-value (?value))))))
      
	(format t "~%~% Adding ~a and ~a (sum should be ~a)."
		x1-value x2-value (+ x1-value x2-value))
      
	;; Do addition:
	(flet ((set-sum (x) (setq sum x)))
	  (setq start-time (get-internal-run-time))
	  (silently (a-query nil `((sum (:quote ,x1-frame) (:quote ,x2-frame) ?x)
				   (value ?x ?value)
				   (:apply ,#'set-sum (?value)))))
	  (setq end-time (get-internal-run-time)))
	(setq seconds (/ (- end-time start-time) internal-time-units-per-second))
      
	;; And print results:
	(format t "~%   Algernon says sum is: ~a" sum)
	(format t "~%   Time (seconds): ~,2f          Rules: ~a          Rules per second: ~,2f"
		seconds *rule-count* (/ *rule-count* seconds))

	(incf total-time seconds)
	(incf total-rules *rule-count*)
      
	(reset-algy)
	(load-kb 'numbers)))

    (format t "~%~% Adverage rules per second: ~,2f" (/ total-rules total-time))
    (/ total-rules total-time)))

(defun test-it (n interations &optional (tests 10))
  (format t "~%~%Resetting Algernon.")
  (reset-algy)
  (load-common-sense-kb)
  (silently (facts-about-numbers))
  (kb-snapshot 'test-it-snapshot)

  (let (results)
    (unwind-protect
	(dotimes (x interations)
	  (format t "~%Creating frames for numbers from 1 to ~a." n)
	  (silently (make-numbers n))
	  (push (cons n (test-numbers n tests)) results)
	  (format t "~%Resetting Algernon:")
	  (reset-algy)
	  (load-kb 'test-it-snapshot)
	  (setq n (* n 10)))

      (format t "~%~%RESULTS")
      (dolist (result results)
	(format t "~% Numbers: ~a          Rules per second: ~,2f" (car result) (cdr result))))))

;;; Old versions:
#|
(defun build-frames (n)
  (a-assert "Link slot" '((:slot link (objects objects))
                          (:slot first-frame (contexts objects))))
  (let (current-frame
        last-frame)
    (dotimes (x n)
      (setq current-frame (intern (format nil "BF~a" x)))
      (a-assert nil `((:create ?x ,current-frame)))
      (if last-frame
        (a-assert nil `((link ,last-frame ,current-frame)))
        (a-assert nil `((first-frame global-context ,current-frame))))
      (setq last-frame current-frame))))

(defun bf (n)
  (a-assert "Link slot" '((:slot link (objects objects))
                          (:slot first-frame (contexts objects))))
  (let (last-frame)
    (flet ((my-setq (x) (setq last-frame x)))
      (dotimes (x n)
        (if last-frame
          (a-assert nil `((:create ?x) (link (:quote ,last-frame) ?x)
                          (:apply ,#'my-setq (?x))))
          (a-assert nil `((:create ?x) (first-frame global-context ?x)
                          (:apply ,#'my-setq (?x)))))))))
|#
