;;;   -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-    ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Simple expert system.
;;;
;;; This is an example of a very simple expert system built on top
;;; of the finite-state machine.  There are only two modifications to
;;; the basic fsm (in fsm.lisp):
;;;
;;;  1. The current time of the fsm is non-monotonically stored in the current context
;;;  (this allows the rules to retrieve the "current state" of the fsm).
;;;
;;;  2. There are no "inputs" here and the transitions are triggered by
;;;  rules in the expert system asserting the next state explicitly.
;;;
;;; The states are as follows:
;;;
;;;   Diagnosing -> Testing -> Prescribing
;;;       ^            |
;;;       |            |
;;;       --------------
;;;
;;; In this simple example, each state fires rules to fill a given slot
;;; and then passes control to the next state.

(defun facts-about-expert ()
  (a-assert "Taxonomy"
	    '((:taxonomy
		(objects
		  (times fsm-time)
		  (states (non-final-states diagnosing testing Prescribing)
			  (final-states final-state))
		  (symptoms low-fever high-fever cough tiredness)
		  (diseases flu mono pneumonia)
		  (tests mono-test xray)
		  (results positive negative)
		  (prescriptions rest lots-of-rest penicillin)))))

  (a-assert "New slots"
	    '((:slot next-time (times times) :cardinality 1)
	      (:slot current-time (contexts times) :cardinality 1)
	      (:slot state (times states) :cardinality 1)
	      (:slot next-state (states times states))

              (:slot temperature (physical-objects nil)
                     :cardinality 1)
	      (:slot has-symptom (people symptoms))
	      (:slot diagnosis (people diseases))
	      (:slot probable-diagnosis (people diseases))
	      (:slot patient (contexts people) :cardinality 1)
	      (:slot test (diseases tests))
	      (:slot result (tests people results))
	      (:slot prescription (people prescriptions))))

  (a-assert "Times"
	    '((:rules contexts
	       ;; There's a first time for everything:
	       ;; (this saves us from having to assert a first time
	       ;; in each new context).
	       ((current-time ?cc fsm-time)
		<-
		(:unp (current-time ?cc ?t))))
	      (:rules times
	       ;; And there's always next time:
	       ((next-time ?t1 ?t2)
		<-
		(:forc ?t2 (name ?t2 (:quote (fsm-time)))
		       (next-time ?t1 ?t2))))))

  ;; The transition rule is just like before except that:
  ;;  1. It changes the current-time in the current-context.
  ;;  2. It will not fire to create a new state for any state except for the
  ;;  state at the current time (this avoids some strange bugs
  ;;  involving old states executing transitions ...).
  ;;
  (a-assert "Transitions"
	    '((:rules times
	       ;; Transitions:
	       ((state ?t1 ?s1)
		(isa ?s1 non-final-states)
		(next-time ?t1 ?t2)
		(next-state ?s1 ?t1 ?s2)
		(current-context global-context ?cc)
		(current-time ?cc ?t1)
		->
		(:clear-slot ?cc current-time)
		(current-time ?cc ?t2)
                ;; Output
                (:lisp (format t "~% State is now ~(~a~).~%" '?s2))
		(state ?t2 ?s2)))))


  (a-assert "Symptoms"
	    '((:rules people
	       ((temperature ?x ?t)
		<-
		(:ask (temperature ?x ?t)))
	       ((has-symptom ?x low-fever)
		<-
		(temperature ?x ?t)
		(:test (and (> ?t 99) (< ?t 102))))
	       ((has-symptom ?x high-fever)
		<-
		(temperature ?x ?t)
		(:test (or (= ?t 102) (> ?t 102))))
	       ;;
	       ((has-symptom ?x tiredness)
		<-
		(:ask (has-symptom ?x tiredness)))
	       ((has-symptom ?x cough)
		<-
		(:ask (has-symptom ?x cough))))))

  (a-assert "Tests"
	    '((test mono mono-test)
	      (test pneumonia xray)
	      ;;
	      ;; Performing a test:
	      (:rules tests
	       ((result ?test ?patient ?result)
		<-
		;; Only instruct user to perform test if no value
		;; is known at all:
		(:unp (:retrieve (result ?test ?patient ?any-result)))
		(:lisp (format t "~% Apply test ~(~a~).~%" '?test))
		(:ask (result ?test ?patient ?result)))
	       
	       ;; These last two rules should be handled in a more
	       ;; general way:
	       ((result ?test ?patient positive)
		->
		(not (result ?test ?patient negative)))
	       ((result ?test ?patient negative)
		->
		(not (result ?test ?patient positive))))))


  ;; For each state in the fsm we have three types of rules:
  ;;   <back-chaining rules to fill the slot for the state>
  ;;   <forward-chaining rules to print some tracing information>
  ;;   <transition rules to find the next state>
  ;;
  (a-assert "Diagnosing"
	    '((:rules people
	       ;;
	       ;; Finding the probable-diagnosis:
	       ((probable-diagnosis ?p flu)
		<-
		(state (current-time (current-context global-context)) diagnosing)
		(has-symptom ?p low-fever))
	       ((probable-diagnosis ?p flu)
		<-
		(state (current-time (current-context global-context)) diagnosing)
		(not (diagnosis ?p pneumonia))
		(has-symptom ?p high-fever))
	       ((probable-diagnosis ?p mono)
		<-
		(state (current-time (current-context global-context)) diagnosing)
		(has-symptom ?p low-fever)
		(has-symptom ?p tiredness))
	       ((probable-diagnosis ?p pneumonia)
		<-
		(state (current-time (current-context global-context)) diagnosing)
		(has-symptom ?p high-fever)
		(has-symptom ?p cough))
	       ;;
	       ;; Output
	       ((probable-diagnosis ?p ?x)
		->
		(:lisp (format t "~% ~(~a~) may have ~(~a~).~%" '?p '?x))))
	      ;;
	      ;; Transition rule:
	      (:rules times
	       ((state ?t diagnosing)
		(probable-diagnosis (patient (current-context global-context)) ?d)
		->
		(next-state diagnosing ?t testing)))))


  (a-assert "Testing"
	    '((:rules people
	       ;;
	       ;; Finding the diagnosis:
	       ((diagnosis ?p ?d)
		<-
		(state (current-time (current-context global-context)) testing)
		(probable-diagnosis ?p ?d)
		(result (test ?d) ?p positive))
	       ((not (diagnosis ?p ?d))
		<-
		(state (current-time (current-context global-context)) testing)
		(probable-diagnosis ?p ?d)
		(result (test ?d) ?p negative))
	       ;; No test exists:
	       ((diagnosis ?p ?d)
		<-
		(state (current-time (current-context global-context)) testing)
		(probable-diagnosis ?p ?d)
		(:unp (test ?d ?test))
		;; And everything else has been rules out:
		(:all-paths ((probable-diagnosis ?p ?d2) (:neq ?d ?d2))
		            ((not (diagnosis ?p ?d2)))))
	       ;;
	       ;; Output
	       ((diagnosis ?p ?x)
		->
		(:lisp (format t "~% ~(~a~) has ~(~a~).~%" '?p '?x))))
	      ;;
	      ;; Transition rules:
	      (:rules times
	       ((state ?t testing)
		(diagnosis (patient (current-context global-context)) ?d)
		->
		(next-state testing ?t Prescribing))
	       ((state ?t testing)
		(patient (current-context global-context) ?patient)
		(not (diagnosis ?patient ?d))
		;; If all probable-diagnosis's fail then its back to
		;; the drawing board:
		(:all-paths ((probable-diagnosis ?patient ?diag))
		            ((not (diagnosis ?patient ?diag))))
		->
		(next-state testing ?t diagnosing)))))


  (a-assert "Prescribing"
	    '((:rules people
	       ;;
	       ;; Finding the prescription:
	       ((prescription ?p rest)
		<-
		(state (current-time (current-context global-context)) Prescribing)
		(diagnosis ?p flu))
	       ((prescription ?p lots-of-rest)
		<-
		(state (current-time (current-context global-context)) Prescribing)
		(diagnosis ?p mono))
	       ((prescription ?p  penicillin)
		<-
		(state (current-time (current-context global-context)) Prescribing)
		(diagnosis ?p pneumonia))
	       ;;
	       ;; Output
	       ((prescription ?p ?x)
		->
		(:lisp (format t "~% The prescription for ~(~a~) is ~(~a~).~%" '?p '?x))))
	      ;;
	      ;; Transition rule:
	      (:rules times
	       ((state ?t Prescribing)
		(prescription (patient (current-context global-context)) ?p)
		->
		(next-state Prescribing ?t final-state))))))


(defun queries-about-expert ()
  (a-assert "Diagnosing John."
	    '((:create ?con context)
	      (:create ?j John)
	      (:clear-slot global-context current-context)
	      (current-context global-context ?con)
	      (patient ?con ?j)
	      
	      (state (current-time ?con) diagnosing))))
