/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

/*
 * Qu-Prolog Interpreter
 */
 
/*
 * Repetitively read in a query from standard input, solve it.
 */

interpreter:-
	repeat,
	write('?- '),
	set_indent(0),
	readR1Term(Query, VarList),
	(
		/* exit */
		(	Query == end_of_file
		;	Query == exit
		;	Query == bye
		)
	;
		'$solve_query'(Query, VarList)
	).

/* The recursive version for interactive implicit parameter use. */
ri :-
    name(A, "Running recursive interpreter"),
    writeln(A),
    '$ri2'.
'$ri2' :-
    write('?- '),
    set_indent(0),
    readR1Term(Query, VarList),
    (
	/* exit */
	(       Query == end_of_file
	;       Query == exit
	;       Query == bye
	)
    ;
	'$rsolve'(Query, VarList)
    ).

'$rsolve'(Query, VarList) :-
    '$rsolve_query'(Query, VarList),
    !,
    '$ri2'.
'$rsolve'(Query, VarList) :- '$ri2'.


/*
 * Solve Query, print the solution and ask whether more solution is needed.
 */
'$solve_query'(Query, VarList):-
	call(Query),
	retry_delay_problems,
	'$solve_query1'(Query, VarList),
	!,
	fail.

'$solve_query'(_, _):-
	indent,
	writeln('no (more) solutions'),
	fail.

'$solve_query1'(Query, VarList) :-
	(VarList == [] ->
	    indent, write(yes),  
            write_delayed_problems_as_constraints,
	    nl
	;
	    '$write_solutions'(VarList),
            write_delayed_problems_as_constraints,
	    getl([C|L]),
	    (C == 0'; -> fail ; true), % enter ';' for more solutions
	    (C == 0', ->  % enter ',' to extend query
		'$next_query_layer'(Query, VarList)
	    ;
		true
	    )
	).

'$next_query_layer'(Query, VarList) :-
	indent,
	readR1Term(MoreQuery, MoreVarList),
	list_union(VarList, MoreVarList, AllVarList),
	inc_indent,
	'$solve_query'(MoreQuery, AllVarList).
'$next_query_layer'(Query, VarList) :-
	indent,
	write('level '),
	ipLookup('$indent', Level),
	write(Level),
	writeln(' query answer'),
	'$solve_query1'(Query, VarList).

	
'$rsolve_query'(Query, VarList) :-
	call(Query),
	retry_delay_problems,
	(VarList == [] ->
	    indent, write(yes),  
            write_delayed_problems_as_constraints,
	    nl
	;
	    '$write_solutions'(VarList),
            write_delayed_problems_as_constraints,
	    getl([C|L]),
	    (C == 0'; -> fail ; true) % enter ';' for more solutions
	).
	
list_union([H|T], [], [H|T]).
list_union([], L, L) :- !.
list_union(L1, [H2|T2], L) :-
    member(H2, L1),
    !,
    list_union(L1, T2, L).
list_union(L1, [H|T2], [H|L]) :-
    list_union(L1, T2, L).

/*
 * Write solution to standard output.
 */
'$write_solutions'([Var]):-
    !,
    indent,
    writeR(Var,_),
    write(' ').
'$write_solutions'([Var|Vars]):-
    indent,
    writeR(Var,_),
    writeln(','),
    '$write_solutions'(Vars).

