/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

/*
 * Qu-Prolog Debugger
 */

:- '$setFlag'(debugging,off).
:- '$setFlag'(tmpdebugging,unset).
:- '$setFlag'(debugpause,off).
:- '$setFlag'(call_number,0).
:- '$setFlag'(debugpermision,on).

%:- dynamic '$debugging'/1.
%:- dynamic '$leash'/1.
%:- dynamic '$spied'/2.
%:- dynamic '$tmpunsetafters'/1.
%:- dynamic '$tmpunsetons'/1.

%:- assert('$debugging'(off)).
:- assert('$leash'([call, exit, redo, fail])).
:- assert('$tmpunsetafters'([])).
:- assert('$tmpunsetons'([])).


enable_debugging :-
    '$setFlag'(debugpermision, on).
disable_debugging :-
    '$setFlag'(debugpermision, off).
pause_debugging(C) :-
    '$getFlag'(debugpermision, Old),
    '$setFlag'(debugpermision, off),
    (C -> '$setFlag'(debugpermision, Old)
       ;  '$setFlag'(debugpermision, Old), fail
    ).

debugpause(C) :-
    '$getFlag'(debugpause, Old),
    '$setFlag'(debugpause, on),
    (C -> '$setFlag'(debugpause, Old)
       ;  '$setFlag'(debugpause, Old), fail
    ).

/*----------------------------------------------------------------------------
spy(Predicate) :-
    Place a spy point on the Predicate F/N or just F, if it has not
    already got a spy point. 

    if the flag '$debugging' is not debug then set it

    Fail on redo.
----------------------------------------------------------------------------*/
spy Predicates :-
    debugpause((
	'$spy'(Predicates),
	debug)).

'$spy'([]) :- !.
'$spy'([C|Cs]) :- !, '$spy'(C), '$spy'(Cs).
'$spy'(F/N) :- !, (\+ clause('$spied'(F, N),_)  -> assert('$spied'(F, N)) ; true).
'$spy'(F) :- atom(F), (\+ clause('$spied'(F, _),_) -> assert('$spied'(F, _)) ; true).

/*----------------------------------------------------------------------------
nospy Predicate:-
    Remove spypoint on Predicate.
    Fails if Predicate is F/N and there is no spy point on F/N.
    True always if Predicate is just F.
----------------------------------------------------------------------------*/
nospy P :- debugpause(nospy2(P)).
nospy2([]) :- !.
nospy2([C|Cs]) :- !, nospy(C), nospy(Cs).
nospy2(F/N) :- !, retractall('$spied'(F, N)).
nospy2(F) :- once(((retractall('$spied'(F, _N)), fail) ; true)).

/*----------------------------------------------------------------------------
nospyall :-
    True, always.  Remove all spypoints.

    Fail on redo.
----------------------------------------------------------------------------*/
nospyall :- debugpause(abolish('$spied', 2)).

/*
 * Set the flag '$debugging' to debug
 */
debug :-
	'$setFlag'(debugging,debug),
	'$setFlag'(tmpdebugging, unset).

'$tmpdebug' :-
	'$setFlag'(tmpdebugging,debug).
	
/*
 * Set the flag '$debugging' to off
 */
nodebug :-
	'$setFlag'(debugging,off),
	'$setFlag'(tmpdebugging, unset).
'$tmpnodebug' :-
	'$setFlag'(tmpdebugging,off).

/*
 * Set the flag '$debugging' to trace
 */
trace :-
	'$setFlag'(debugging,trace),
	'$setFlag'(call_number,0),
	'$setFlag'(tmpdebugging, unset).
'$tmptrace' :-
	'$setFlag'(tmpdebugging,trace).
	
/*
 * Set the flag '$debugging' to off
 */
notrace :-
	'$setFlag'(debugging,off),
	'$setFlag'(tmpdebugging, unset),
	'$setFlag'(call_number,0).
'$tmpnotrace' :-
	'$setFlag'(tmpdebugging,off).

/*
'$tmpunset'(Mode) :- debugpause('$tmpunset2'(Mode)).
*/
'$tmpunset'(Mode) :-
	once(clause('$tmpunsetons'([r(N)]), _)), !,
	Mode = r(M),
	M =< N,
	'$tmpunseton'([]).
'$tmpunset'(r(_)) :- !, fail.
'$tmpunset'(Mode) :-
	(
	once(clause('$tmpunsetons'(Unsets), true)),
	member(Mode, Unsets)
	 -> '$setFlag'(tmpdebugging, unset),
	    '$setFlag'('$tmpunsetpending', off),
	    '$tmpunseton'([])
	 ;  (once(clause('$tmpunsetafters'(UnsetsA), true)),
	     member(Mode, UnsetsA)
	     -> '$setFlag'('$tmpunsetpending', on)
	     ;
		true)
	), !.

'$tmpunseton'(Unsets) :- 
    /*(\+ (clause('$tmpunsetafters'(Unsets), true), !) ->*/
    debugpause((
	retractall('$tmpunsetons'(_)),
	assert('$tmpunsetons'(Unsets))
    )).

'$tmpunsetafter'(Unsets) :- 
    /*(\+ (clause('$tmpunsetafters'(Unsets), true), !) ->*/
    debugpause((
	retractall('$tmpunsetafters'(_)),
	assert('$tmpunsetafters'(Unsets))
    )).

/*
 * Display the debugging status and all the spypoints.
 */
debugging :-
    debugpause((
    '$getdebuggingFlag'(Mode),
    writeln(Mode),
    writeln('There are spypoints on'),
    (
	clause('$spied'(F, N), _),
        write(F),
	(nonvar(N) ->
	    write('/'), write(N)
	;
	    true
	),
	nl,
	fail
    ;
	true
    ))).

/*
 * Control which port to pause.
 * Add the following clause when separate from NU-Prolog
 */
leash(P) :- debugpause(leash2(P)).
leash2(all) :- !, leash2([call, exit, redo, fail]).
leash2(Ports) :-
    (\+ (clause('$leash'(Ports), _), !) ->
	retractall('$leash'(_)),
	assert('$leash'(Ports))
    ;
	true
    ).

spyCondition(Goal, Port, Condition) :-
    debugpause(assert('$spyhook'(Goal, Port, Condition))).

leashCondition(Goal, Port, Condition) :-
    debugpause(assert('$leashhook'(Goal, Port, Condition))).

/*----------------------------------------------------------------------------
'$debug_call'(P) :- 
    All predicates go through this module, if flag '$debugging' is switched
    on, then messages (call; fail; exit; redo) will be displayed.

    Called once only.

    True, if call the predicate P succeeds
    False, otherwise.

    Fail on redo, side-affect see redo message.
----------------------------------------------------------------------------*/
'$debug_call'(P) :- 
    '$getFlag'(call_number, N), 
    Nplus1 is N + 1,
    '$setFlag'(call_number, Nplus1),
    '$debug_call'(P, N).

'$debug_call'(P, N) :-
    '$tmpunset'(call),
    '$tmpunset'(call(N)),
    '$debug_port'(call, P, N),
    fail.
'$debug_call'(P, N) :-
    '$real_call'(P),
    (
	'$tmpunset'(exit),
	'$tmpunset'(exit(N)),
	'$debug_port'(exit, P, N)
    ;
	'$tmpunset'(redo),
	'$tmpunset'(redo(N)),
	'$debug_port'(redo, P, N),
	fail
    ).
'$debug_call'(P, N) :-
    '$tmpunset'(fail),
    '$tmpunset'(fail(N)),
    '$debug_port'(fail, P, N),
    fail.

'$debug_call'(P, N) :-
    '$tmpunset'(r(N)),
    '$debug_call'(P, N).

/*
 * Different debugging mode control.
 */
'$debug_port'(Port, Predicate, CallNumber) :-
    '$getFlag'(debugpermision, on)
    ->
    (is_spied(Port, Predicate) -> '$tmpunset'(spy) ; true),
    ('$getdebuggingFlag'(trace) ->
	'$debug_message'(Port, Predicate, CallNumber)
    ;
        ('$getdebuggingFlag'(debug)
	 ->
         (is_spied(Port, Predicate)
	  -> 
	  '$debug_message'(Port, Predicate, CallNumber) ; true)
    	;
	   true
	)
    )
    ;
    true
    .

/*----------------------------------------------------------------------------
'$debug_message'(Port, Term, CallNumber) :-
    display the (call; fail; exit; redo) message if the flag '$debugging' is on.
    True, always.
    Fail on redo.
----------------------------------------------------------------------------*/
'$debug_message'(Port, Term, CallNumber) :-
    debugpause((
    '$getFlag'(debugpermision, on),
    (is_spied(Port, Term) ->
	write('*')
    ;
	write(' ')
    ),
    write(' '),		% skipped character '>' or ' '
    write('('),
    write(CallNumber),
    write(')'),
    write(' '),
    write(Port),
    write(' : '),
    write(Term),
    (is_leashed(Port, Term)
      -> write(' ? '), '$debuginst'(Port, Term, CallNumber)
      ;  nl))), !.

'$debug_message'(_, _, _).

is_spied(Port, Term) :-
    debugpause((
    functor(Term, F, N),
    clause('$spied'(F, N), _),
    (clause('$spyhook'(Term, Port, Condition), _) -> Condition ; true), 
    !)).

is_leashed(Port, Term) :-
    clause('$leash'(Ports), true), 
    member(Port, Ports),
    (clause('$leashhook'(Term, Port, Condition), _) -> Condition ; true),
    !.

'$debuginst'(Port, Term, CallNumber) :-
	getl([C|_]),
	nonvar(C),
	'$process_debug_char'(C, Port, Term, CallNumber),
	!.

'$process_debug_char'(10, _, _, _) :-      % newline
    '$tmptrace', 
    '$tmpunsetafter'([call, exit, redo, fail]),
    !.
'$process_debug_char'(0'+, P, T, _) :- functor(T, F, N), debugpause('$spy'(F/N)), !.
'$process_debug_char'(0'-, P, T, _) :- functor(T, F, N), nospy F/N, !.
'$process_debug_char'(0'@, P, T, Call) :- 
    debugpause((
	read1Term(L, VarList),
	('$static_predicate'(legal_debug_query, 1, Address)
	 -> (legal_debug_query(L)
	     ; errorln('illegal debugging query'), fail
	    )
    	;
	   true
	),!,
	'$process_legal_query'(L, VarList))),
    '$debug_message'(P, T, Call).
'$process_debug_char'(0'=, _, _, _) :- debugging, !.
'$process_debug_char'(0'<, _, _, _) :- 
    debugpause(errorln('not implemented')), !.
'$process_debug_char'(0'|, _, _, _) :- 
    debugpause(errorln('not implemented')), !.
'$process_debug_char'(0'?, _, _, _) :- 
    debugpause(errorln('not implemented')), !.
'$process_debug_char'(0'a, _, _, _) :- 
    debugpause(errorln('not implemented')), !.
'$process_debug_char'(0'b, _, _, _) :- 
    debugpause(errorln('not implemented')), !.
'$process_debug_char'(0'c, _, _, _) :- 
    '$tmptrace', 
    '$tmpunsetafter'([call, exit, redo, fail]), 
    !.
'$process_debug_char'(0'd, _, _, _) :- 
    debugpause(errorln('not implemented')), !.
'$process_debug_char'(0'f, _, _, _) :- 
    debugpause(errorln('not implemented')), !.
'$process_debug_char'(0'g, _, _, _) :- 
    debugpause(errorln('not implemented')), !.
'$process_debug_char'(0'h, _, _, _) :- 
    debugpause(errorln('not implemented')), !.
'$process_debug_char'(0'l, _, _, _) :- 
    '$tmpdebug', 
    '$tmpunsetafter'([spy]),
    !.
'$process_debug_char'(0'n, _, _, _) :- nodebug, !.
'$process_debug_char'(0'p, _, _, _) :- 
    debugpause(errorln('not implemented')), !.
'$process_debug_char'(0'r, P, T, Call) :- 
    '$tmpunseton'([r(Call)]),
    !.
'$process_debug_char'(0's, P, T, Call) :- 
    (P == call; P == redo), !,
    '$tmpnodebug',
    '$tmpunseton'([exit(Call), fail(Call)]),
    !.
'$process_debug_char'(0's, _, T, Call) :- 
    '$tmptrace', 
    '$tmpunsetafter'([call, redo, exit, fail]),
    !.
'$process_debug_char'(0'w, _, _, _) :- 
    debugpause(errorln('not implemented')), !.
'$process_debug_char'(_, P, T, Call) :- 
    '$debug_message'(P, T, Call).

'$getdebuggingFlag'(Mode) :-
    '$getFlag'(tmpdebugging, SysMode),
    (SysMode == unset
     -> '$getFlag'(debugging, Mode)
     ;  Mode = SysMode
    ),
    ('$getFlag'('$tmpunsetpending', on) 
     -> '$setFlag'(tmpdebugging, unset),
        '$setFlag'('$tmpunsetpending', off),
	'$tmpunsetafter'([])
    ;
	true
    ).

'$process_legal_query'(L, V) :-
    call(L),
    retry_delay_problems,
    (V \== [] ->
	'$write_results'(V),
	write_delayed_problems_as_constraints,
	\+ getl([0'; , 10])             % enter ';' for more solutions
    ;
	writeln(yes),
	write_delayed_problems_as_constraints
    ),
    !.
'$process_legal_query'(_, _):-
    write('no (more) solutions'),
    nl.

'$write_results'([Var]) :-
    !,
    write(Var),
    write(' ').
'$write_results'([Var|Vars]):-
    write(Var),
    writeln(','),
    '$write_results'(Vars).
