/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#include "bind.h"
#include "cells.h"
#include "data_area.h"
#include "delayed_problems.h"
#include "dereference.h"
#include "examine_term.h"
#include "name_table.h"
#include "persistent.h"
#include "string_table.h"
#include "substitution.h"
#include "system.h"
#include "unify.h"
#include "x_registers.h"


/*----------------------------------------------------------------------------
unify(t1, t2)
----------------------------------------------------------------------------*/

int unify_apply (VALUE *t1, VALUE *t2);
int unify_pair (VALUE *t1, VALUE *t2);
int unify_quantifier (VALUE *t1, VALUE *t2);
int unify_frozen_variable (VALUE *t1, VALUE *t2);
int unify_variable (VALUE *t1, VALUE *t2);
int unify_frozen_object_variable (VALUE *t1, VALUE *t2);
int occurs_check (VALUE *variable, VALUE *term);
int perform_occurs_check (int type, VALUE *variable, VALUE *term);
int unify_variable_variable (VALUE *t1, VALUE *t2);
int contain_local_object_variable (cell *objvar, cell *sub);
int generate_distinction (cell objvar, cell sub);
int unify_objvar_objvar (cell objvar, VALUE *t);
int unify_frozen_frozen_object_variable (VALUE *t1, VALUE *t2);
int perform_occurs_check_parts (int type, VALUE *variable, VALUE *t1, VALUE *t2);

global	boolean	
unify(VALUE *t1, VALUE *t2)
{
	dereference(t1);

	switch (Tag(t1->term))
	{
	when CONSTANT:
		return(unify_constant(t1->term, t2));
	when APPLY:
		return(unify_apply(t1, t2));
	when PAIR:
		return(unify_pair(t1, t2));
	when QUANTIFIER:
		return(unify_quantifier(t1, t2));
	when REFERENCE:
		if (Frozen(t1->term))
			return(unify_frozen_variable(t1, t2));
		else
			return(unify_variable(t1, t2));
	when OBJECT_REFERENCE:
		if (Frozen(t1->term))
			return(unify_frozen_object_variable(t1, t2));
		else
			return(unify_object_variable(t1, t2));
	}
	return(FALSE);
}

/*----------------------------------------------------------------------------
unify_constant(constant, t2)
----------------------------------------------------------------------------*/
global	boolean
unify_constant(cell constant, VALUE *t2)
{
	dereference(t2);
	switch (Tag(t2->term))
	{
	when CONSTANT:
		return(constant == t2->term);
	when REFERENCE:
		if (Frozen(t2->term))
			return(FALSE);
		else if (yield_constant(t2->sub, constant))
		{
			delay(UNIFY, Location(t2->term),
			      make_substitution(t2), constant);
                }
		else
			Bind(t2->term, constant);
	when OBJECT_REFERENCE:
		if (yield_constant(t2->sub, constant))
		{
			delay(UNIFY, Location(t2->term),
			      make_substitution(t2), constant);
                }
		else
			return(FALSE);
	otherwise:
		return(FALSE);
	}
	return(TRUE);
}

/*----------------------------------------------------------------------------
unify_apply(t1, t2)
----------------------------------------------------------------------------*/
local	boolean
unify_apply(VALUE *t1, VALUE *t2)
{
	VALUE	t1fn;
	VALUE	t2fn;
	VALUE	t1arg;
	VALUE	t2arg;

	dereference(t2);
	switch (Tag(t2->term))
	{
	when APPLY:
		t1fn.sub = t1arg.sub = t1->sub;
		t2fn.sub = t2arg.sub = t2->sub;
		t1fn.term = Reference(&Functor(t1->term));
		t1arg.term = Reference(&Argument(t1->term));
		t2fn.term = Reference(&Functor(t2->term));
		t2arg.term = Reference(&Argument(t2->term));
		return(unify(&t1fn, &t2fn) && unify(&t1arg, &t2arg));
	when REFERENCE:
		if (Frozen(t2->term))
			return(FALSE);
		else if (Invertible(t2->sub))
			return(!occurs_check(t2, t1));
		else if (yield_tag(t2->sub, APPLY) ||
			 perform_occurs_check(DIRECT, t2, t1) == DIRECT_OCCURS)
                {
			delay(UNIFY, Location(t2->term),
			      make_substitution(t2), make_substitution(t1));
                } 
		else
		{
			Bind(t2->term, Apply());
			return(unify_apply(t1, t2));
		}
	when OBJECT_REFERENCE:
		if (yield_tag(t2->sub, APPLY))
		{
			delay(UNIFY, Location(t2->term),
			      make_substitution(t2), make_substitution(t1));
                }
		else
			return(FALSE);
	otherwise:
		return(FALSE);
	}
	return(TRUE);
}

/*----------------------------------------------------------------------------
unify_pair(t1, t2)
----------------------------------------------------------------------------*/
local	boolean
unify_pair(VALUE *t1, VALUE *t2)
{
	VALUE	t1left;
	VALUE	t2left;
	VALUE	t1right;
	VALUE	t2right;

	dereference(t2);
	switch (Tag(t2->term))
	{
	when PAIR:
		t1left.sub = t1right.sub = t1->sub;
		t2left.sub = t2right.sub = t2->sub;
		t1left.term = Reference(&Left(t1->term));
		t1right.term = Reference(&Right(t1->term));
		t2left.term = Reference(&Left(t2->term));
		t2right.term = Reference(&Right(t2->term));
		return(unify(&t1left, &t2left) && unify(&t1right, &t2right));
	when REFERENCE:
		if (Frozen(t2->term))
			return(FALSE);
		else if (Invertible(t2->sub))
			return(!occurs_check(t2, t1));
		else if (yield_tag(t2->sub, PAIR) ||
			 perform_occurs_check(DIRECT, t2, t1) == DIRECT_OCCURS)
                {
			delay(UNIFY, Location(t2->term),
			      make_substitution(t2), make_substitution(t1));
                }
		else
		{
			Bind(t2->term, Pair());
			return(unify_pair(t1, t2));
		}
	when OBJECT_REFERENCE:
		if (yield_tag(t2->sub, PAIR))
		{
			delay(UNIFY, Location(t2->term),
			      make_substitution(t2), make_substitution(t1));
                }
		else
			return(FALSE);
	otherwise:
		return(FALSE);
	}
	return(TRUE);
}

/*----------------------------------------------------------------------------
unify_quantifier(t1, t2)
----------------------------------------------------------------------------*/
local	boolean
unify_quantifier(VALUE *t1, VALUE *t2)
{
	cell	objvar;
	VALUE	t1body;
	VALUE	t2body;

	dereference(t2);
	switch(Tag(t2->term))
	{
	when QUANTIFIER:
		objvar = NewLocalObjectVariable();
		t1body.sub = RenameSub(BoundVar(t1->term), objvar, t1->sub);
		t1body.term = Reference(&Body(t1->term));
		t2body.sub = RenameSub(BoundVar(t2->term), objvar, t2->sub);
		t2body.term = Reference(&Body(t2->term));
		return(unify(&t1body, &t2body));
	when REFERENCE:
		if (Frozen(t2->term))
			return(FALSE);
		else if (Invertible(t2->sub))
			return(!occurs_check(t2, t1));
		else if (yield_tag(t2->sub, QUANTIFIER) ||
			 perform_occurs_check(DIRECT, t2, t1) == DIRECT_OCCURS)
                {
			delay(UNIFY, Location(t2->term),
			      make_substitution(t2), make_substitution(t1));
                }
		else
		{
			Bind(t2->term, Quantifier());
			BoundVar(Value(t2->term)) = NewObjectVariable();
			return(unify_quantifier(t1, t2));
		}
	when OBJECT_REFERENCE:
		if (yield_tag(t2->sub, QUANTIFIER))
		{
			delay(UNIFY, Location(t2->term),
			      make_substitution(t2), make_substitution(t1));
                } 
		else
			return(FALSE);
	otherwise:
		return(FALSE);
	}
	return(TRUE);
}

/*----------------------------------------------------------------------------
unify_variable(t1, t2)
----------------------------------------------------------------------------*/
local	boolean
unify_variable(VALUE *t1, VALUE *t2)
{
	dereference(t2);
	switch (Tag(t2->term))
	{
	when CONSTANT:
		return(unify_constant(t2->term, t1));
	when APPLY:
		return(unify_apply(t2, t1));
	when PAIR:
		return(unify_pair(t2, t1));
	when QUANTIFIER:
		return(unify_quantifier(t2, t1));
	when REFERENCE:
		if (Frozen(t2->term))
		{
			return(unify_frozen_variable(t2, t1));
		}
		else if (t1->term == t2->term)
		{
			if (!equal(t1, t2))
			{
				delay(UNIFY, Location(t1->term),
				      make_substitution(t1),
				      make_substitution(t2));
                        }
                }
		else if (t1->sub == EMPTY_SUB && t2->sub == EMPTY_SUB)
			BindVariable(t1->term, t2->term);
		else
		{
			return(Location(t1->term) ==
					Junior(Location(t1->term),
					       Location(t2->term)) ?
			       unify_variable_variable(t1, t2) :
			       unify_variable_variable(t2, t1));
		}
	when OBJECT_REFERENCE:
		return(unify_object_variable(t2, t1));
	}
	return(TRUE);
}

/*----------------------------------------------------------------------------
unify_object_variable(t1, t2)
----------------------------------------------------------------------------*/
global	boolean
unify_object_variable(VALUE *t1, VALUE *t2)
{
	dereference(t2);
	switch (Tag(t2->term))
	{
	when CONSTANT:
		return(unify_constant(t2->term, t1));
	when APPLY:
		return(unify_apply(t2, t1));
	when PAIR:
		return(unify_pair(t2, t1));
	when QUANTIFIER:
		return(unify_quantifier(t2, t1));
	when REFERENCE:
		if (Frozen(t2->term))
			return(unify_frozen_variable(t2, t1));
		else if (Invertible(t2->sub))
			return(!occurs_check(t2, t1));
		else if (IsLocalObjectVariable(t1->term))
			if (!contain_local_object_variable(&(t1->term),
							   &(t2->sub)))
				return(FALSE);
			else
			{
				object_dereference(&(t1->term));
				return(unify_object_variable(t1, t2));
			}
		else if (t1->sub == EMPTY_SUB &&
			 !yield_object_variable(t2->sub, t1->term))
		{
			Bind(t2->term, t1->term);
			return(generate_distinction(t1->term, t2->sub));
		}
		else
		{
			delay(UNIFY, Location(t2->term),
			      make_substitution(t2), make_substitution(t1));
                }
	when OBJECT_REFERENCE:
		if (Frozen(t2->term))
			return(unify_frozen_object_variable(t2, t1));
		else if (t1->term == t2->term)
		{
			if (!equal(t1, t2))
			{
				delay(UNIFY, Location(t1->term),
				      make_substitution(t1),
				      make_substitution(t2));
                        }
                }
		else if (t1->sub == EMPTY_SUB && t2->sub == EMPTY_SUB)
			if (distinct_from(t1->term, t2->term))
				return(FALSE);
			else
				BindObjectVariable(t1->term, t2->term);
		else if (IsLocalObjectVariable(t1->term))
			if (!contain_local_object_variable(&(t1->term),
							   &(t2->sub)))
				return(FALSE);
			else
			{
				object_dereference(&(t1->term));
				return(unify_object_variable(t1, t2));
			}
		else if (IsLocalObjectVariable(t2->term))
			if (!contain_local_object_variable(&(t2->term),
							   &(t1->sub)))
				return(FALSE);
			else
			{
				object_dereference(&(t2->term));
				return(unify_object_variable(t2, t1));
			}
		else if (Invertible(t1->sub))
			if (invert(t1->sub, t2))
				return(unify_objvar_objvar(t1->term, t2));
			else
				return(FALSE);
		else if (Invertible(t2->sub))
			if (invert(t2->sub, t1))
				return(unify_objvar_objvar(t2->term, t1));
			else
				return(FALSE);
		else
		{
			delay(UNIFY, Location(t1->term),
			      make_substitution(t1), make_substitution(t2));
                }
	otherwise:
		return(FALSE);
	}
	return(TRUE);
}

/*
 *  unify_variable_variable - t1->term must contain the younger variable
 *			      than t2->term
 */
local	boolean
unify_variable_variable(VALUE *t1, VALUE *t2)
{
        cell    *h;

	if (Invertible(t1->sub) &&
	    perform_occurs_check(ALL_CHECKS, t1, t2) != STATIC_OCCURS)
	{
		if (invert(t1->sub, t2))
		{
			Bind(t1->term, make_substitution(t2));
		}
		else
			return(FALSE);
	}
	else if (Invertible(t2->sub) &&
		 perform_occurs_check(ALL_CHECKS, t2, t1) != STATIC_OCCURS)
		if (invert(t2->sub, t1))
			Bind(t2->term, make_substitution(t1));
		else
			return(FALSE);
	else  
	{
		delay(UNIFY, Location(t1->term), make_substitution(t1),
		      make_substitution(t2));
        }
	return(TRUE);
}

local	boolean
contain_local_object_variable(cell *objvar, cell *sub)
{
reg	cell	s;
	cell	end;
	boolean	found;

	end = EMPTY_SUB;
	found = FALSE;
	for (s = *sub; s != EMPTY_SUB; s = NextSub(s))
	{
		if (Domain(s, 1) == *objvar)
			found = FALSE;
		else if (Range(s, 1) == *objvar)
		{
			found = TRUE;
			end = s;
		}
	}
	if (! found)
		return(FALSE);
	else
	{
		*objvar = Domain(end, 1);
		*sub = copy_substitution(*sub, end);
	}
	return(TRUE);
}

local	boolean
unify_objvar_objvar(cell objvar, VALUE *t)
{
        VALUE   tnew; 

	dereference(t);
	if (objvar == t->term)
	{
	        tnew.sub = EMPTY_SUB;
	        tnew.term = objvar;
		if (!equal(&tnew, t))
		{
			delay(UNIFY, Location(objvar), objvar,
			      make_substitution(t));
                } 
	}
	else if (yield_object_variable(t->sub, objvar))
	{
		delay(UNIFY, Location(objvar), objvar, make_substitution(t));
        }
	else if (distinct_from(objvar, t->term))
		return(FALSE);
	else
	{
		BindObjectVariable(objvar, t->term);
		return(generate_distinction(objvar, t->sub));
	}
	return(TRUE);
}

/*----------------------------------------------------------------------------
unify_frozen_variable(t1, t2)
 * t1 is guaranteed to be frozen.
----------------------------------------------------------------------------*/
local	boolean
unify_frozen_variable(VALUE *t1, VALUE *t2)
{
	dereference(t2);
	switch (Tag(t2->term))
	{
	when REFERENCE:
		if (Frozen(t2->term))
		{
			if (t1->term != t2->term)
				return(FALSE);
			else
			{
				if (! equal(t1, t2))
				{
					delay(UNIFY, Location(t1->term),
					      make_substitution(t1),
					      make_substitution(t2));
                                }
			}
		}
		else if (t1->sub == EMPTY_SUB && t2->sub == EMPTY_SUB)
			BindVariable(t2->term, t1->term);
		else if (Invertible(t2->sub) &&
			 perform_occurs_check(ALL_CHECKS, t2, t1) !=
							STATIC_OCCURS)
			if (invert(t2->sub, t1))
				Bind(t2->term, make_substitution(t1));
			else
				return(FALSE);
		else
		{
			delay(UNIFY, Location(t2->term),
			      make_substitution(t2), make_substitution(t1));
                }
	when OBJECT_REFERENCE:
		if (t2->sub == EMPTY_SUB)
			return(FALSE);
		else
		{
			delay(UNIFY, Location(t2->term), make_substitution(t2),
			      make_substitution(t1));
                } 
	otherwise:
		return(FALSE);
	}
	return(TRUE);
}

/*----------------------------------------------------------------------------
unify_frozen_object_variable(t1, t2)
 * t1 is guaranteed to be frozen.
----------------------------------------------------------------------------*/
local	boolean
unify_frozen_object_variable(VALUE *t1, VALUE *t2)
{

	dereference(t2);
	switch (Tag(t2->term))
	{
	when CONSTANT:
		return(unify_constant(t2->term, t1));
	when APPLY:
		return(unify_apply(t2, t1));
	when PAIR:
		return(unify_pair(t2, t1));
	when QUANTIFIER:
		return(unify_quantifier(t2, t1));
	when REFERENCE:
		if (Frozen(t2->term))
		      if (t1->sub == EMPTY_SUB)
			      return(FALSE);
		      else
		      {
			delay(UNIFY, Location(t1->term), make_substitution(t1),
			      make_substitution(t2));
                      }
		else
			return(unify_object_variable(t1, t2));
	when OBJECT_REFERENCE:
		if (Frozen(t2->term))
			return(unify_frozen_frozen_object_variable(t1, t2));
		else if (t1->sub == EMPTY_SUB && t2->sub == EMPTY_SUB){
			if (distinct_from(t1->term, t2->term))
				return(FALSE);
			else{
				BindObjectVariable(t2->term, t1->term);
			}
		}
		else if (IsLocalObjectVariable(t2->term))
			if (!contain_local_object_variable(&(t2->term),
							   &(t1->sub)))
				return(FALSE);
			else
				return(unify_frozen_object_variable(t1, t2));
		else if (Invertible(t2->sub))
			if (invert(t2->sub, t1))
			{
				dereference(t1);
				if (yield_object_variable(t1->sub, t2->term))
				{
					delay(UNIFY, Location(t2->term),
					      t2->term, make_substitution(t1));
                                }
				else if (distinct_from(t2->term, t1->term))
					return(FALSE);
				else
				{
					BindObjectVariable(t2->term,
								 t1->term);
					return(generate_distinction(t2->term,
								    t1->sub));
				}
			}
			else
				return(FALSE);
		else
		{
			delay(UNIFY, Location(t2->term),
			      make_substitution(t2), make_substitution(t1));
                }
	otherwise:
		return(FALSE);
	}
	return(TRUE);
}

/*----------------------------------------------------------------------------
unify_frozen_frozen_object_variable(t1, t2)
 * t1 and t2 are guaranteed to be frozen.
----------------------------------------------------------------------------*/
local	boolean
unify_frozen_frozen_object_variable(VALUE *t1, VALUE *t2)
{
	if (t1->term == t2->term)
	{
		if (!equal(t1, t2))
		{
			delay(UNIFY, Location(t1->term),
			      make_substitution(t1),
			      make_substitution(t2));
                } 
	}
	else if (t1->sub == EMPTY_SUB && t2->sub == EMPTY_SUB)
		return(FALSE);
	else
	{
		delay(UNIFY, Location(t1->term),
		      make_substitution(t1), make_substitution(t2));
        }
	return(TRUE);
}

/*
 * Check whether the two object variables are different.
 * objvar1, objvar2 = OBJECT_REFERENCE data cell
 */
global	boolean
distinct_from(cell objvar1, cell objvar2)
{
local	boolean	distinct_from_list(cell objvar, cell distinfo);
	VALUE	obj1;
	VALUE	obj2;

	DereferenceTerm(obj1, objvar1);
	DereferenceTerm(obj2, objvar2);

	if (obj1.term == obj2.term) 
		return(FALSE);
	else if (AllDistinct(obj1.term) || AllDistinct(obj2.term) ||
		 /* (Frozen(obj1.term) && Frozen(obj2.term)) || */
		 (InPersistentStack(obj1.term) &&
		  InPersistentStack(obj2.term)))
		return(TRUE);
	else    return(distinct_from_list(obj1.term, obj2.term)
		       ||
		       distinct_from_list(obj2.term, obj1.term));
}

local	boolean
distinct_from_list(cell objvar, cell distinfo)
{
	cell	*d;
	VALUE	val;

	/*
	for (d = (cell *)Distinction(distinfo); d != NULL; d = NextDistinction(d))
	*/
	for (d = Location(Distinction(distinfo)); d != NULL; d = NextDistinction(d))
	{
		DereferenceTerm(val, DistinctObjectVar(d));
		if (Location(val.term) == Location(objvar))
			return(TRUE);
	}
	return(FALSE);
}

global	void
generate_distinction_information(cell objvar, cell sub)
{
	natural	i, j;

	for (; sub != EMPTY_SUB; sub = NextSub(sub))
		for (i = Size(sub), j = 1; j <= i; j++)
			SetDistinct(objvar, Domain(sub, j));
}

local	boolean
generate_distinction(cell objvar, cell sub)
{
	natural	i, j;
	cell	v;

	for (; sub != EMPTY_SUB; sub = NextSub(sub))
		for (i = Size(sub), j = 1; j <= i; j++)
		{
			v = Domain(sub, j);
			object_dereference(&v);
			if (objvar == v)
				return(FALSE);
			else if (!distinct_from(objvar, v))
				SetDistinct(objvar, v);
		}
	return(TRUE);
}

/*
 * type = ALL_CHECKS, DIRECT
 */
local	boolean
occurs_check(VALUE *variable, VALUE *term)
{
	switch (perform_occurs_check(ALL_CHECKS, variable, term))
	{
	when DIRECT_OCCURS:
		return(TRUE);
	when STATIC_OCCURS:
		delay(UNIFY, Location(variable->term),
		      make_substitution(variable), make_substitution(term));
	when NONE:
		if (invert(variable->sub, term))
			Bind(variable->term, make_substitution(term));
		else
			return(TRUE);
	}
	return(FALSE);
}

local	int
perform_occurs_check(int type, VALUE *variable, VALUE *term)
{
static	cell	unique = Atom(DOLLAR);
	VALUE	t1;
	VALUE	t2;

	dereference(term);
	switch (Tag(term->term))
	{
	when CONSTANT:
		return(NONE);
	when APPLY:
		t1.sub = t2.sub = term->sub;
		t1.term = Reference(&Functor(term->term));
		t2.term = Reference(&Argument(term->term));
		return(perform_occurs_check_parts(type, variable, &t1, &t2));
	when PAIR:
		t1.sub = t2.sub = term->sub;
		t1.term = Reference(&Left(term->term));
		t2.term = Reference(&Right(term->term));
		return(perform_occurs_check_parts(type, variable, &t1, &t2));
	when QUANTIFIER:
		t1.sub = RenameSub(BoundVar(term->term), unique, term->sub);
		t1.term = Reference(&Body(term->term));
		return(perform_occurs_check(type, variable, &t1));
	when REFERENCE:
		if (variable->term == term->term)
			return(DIRECT_OCCURS);
		else if (type == DIRECT)
			return(NONE);
		else
		{
			return(perform_occurs_check_sub(type, variable,
							term->sub));
		}
	when OBJECT_REFERENCE:
		if (IsLocalObjectVariable(term->term) &&
		    flag_value(Atom(add_name_string_offset("localfix", ATOM_W)))
			 == Atom(add_name_string_offset("off", ATOM_W)) &&
		    !in_range(term->term, variable->sub))
			return(DIRECT_OCCURS);
		else if (type == DIRECT)
			return(NONE);
		else
			return(perform_occurs_check_sub(type, variable,
							term->sub));
	otherwise:
		return(NONE);
	}
}

local	int
perform_occurs_check_parts(int type, VALUE *variable, VALUE *t1, VALUE *t2)
{
	switch (perform_occurs_check(type, variable, t1))
	{
	when DIRECT_OCCURS:
		return(DIRECT_OCCURS);
	when STATIC_OCCURS:
		switch (perform_occurs_check(type, variable, t2))
		{
		when DIRECT_OCCURS:
			return(DIRECT_OCCURS);
		when STATIC_OCCURS or NONE:
			return(STATIC_OCCURS);
		}
	when NONE:
		return(perform_occurs_check(type, variable, t2));
	}
	return(NONE);
}

global	int
perform_occurs_check_sub(int type, VALUE *variable, cell sub)
{
	int	i, j;
	VALUE	t;

	for (; sub != EMPTY_SUB; sub = NextSub(sub))
	{
		for (i = Size(sub), j = 1; j <= i; j++)
		{
			t.sub = NextSub(sub);
			t.term = Reference(&Range(sub, j));

			if (perform_occurs_check(type, variable, &t) != NONE)
				return(STATIC_OCCURS);
		}
	}
	return(NONE);
}
