/*--------------------------------------------------------------------
  simplify.c   -     simplify the term and store it on a new stack -  
                     - scratch_pad.

 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.

  Revision History:
  =================

  Version  DD/MM/YY   Author   Description
  -------  --------   ------   ----------------------------------------
   1.0     20/08/92     SZ      simplify the term  

--------------------------------------------------------------------*/

#include "data_area.h"
#include "delayed_problems.h"
#include "dereference.h"
#include "examine_term.h"
#include "name_table.h"
#include "simplify.h"
#include "substitution.h"
#include "system.h"
#include "unify.h"
#include "x_registers.h"

cell        *scratch_pad;
cell        *end_scratch_pad;
cell        *top_scratch_pad;
natural     scratch_pad_size = DEFAULT_SCRATCH_PAD_SIZE;
local	boolean	is_first;
local	boolean	should_swap;

/*
 * Initialise a stack - scratch_pad.The stack sratch_pad  
 * is a stack where the term being simplified is stored.
 */


extern void create_incomplete_choice_point (void);
extern int create_del_problem (cell var, cell objvar);

void
initialise_scratch_pad(void)
{
        if ((scratch_pad = (cell *)malloc(Kwords_to_chars(scratch_pad_size)))
             == NULL)
        {
             fatal("not enough memory for the scratch pad %d K",  
                    scratch_pad_size);
        } 
        end_scratch_pad = (cell *)(((char *)scratch_pad) +
                                   Kwords_to_chars(scratch_pad_size));

        top_scratch_pad = scratch_pad;
}

/*
 * Simplify the term, branching on data tags of the term.
 */

cell
simp(VALUE *term)
{
local   cell         simp_apply(VALUE *term);
local   cell         simp_var_term(VALUE *term);
local   cell         simp_obvar_term(VALUE *term);
    
     

	dereference(term);
        switch (Tag(term->term))
        {
        case CONSTANT:
                return(term->term); 
                break;
        case APPLY:
                return(simp_apply(term)); 
                break;
        case REFERENCE:
	        return(simp_var_term(term));  	
                break;
        case OBJECT_REFERENCE:
                if (IsLocalObjectVariable(term->term) &&
                    flag_value(Atom(add_name_string_offset("localfix", ATOM_W)))
                    == Atom(add_name_string_offset("off", ATOM_W)))
                {
                        fatal("local found in simplify");
                        /*
                        return(term->term); 
                        */
                }
                else
                {
                        return(simp_obvar_term(term));
                }
                break;
        default:
                return(FALSE);
        }
}

/*
 * The simplified term is separated from the original one. The new
 * simplified term is built on the stack scratch_pad. Before building
 * the term the scratch_pad is cleared.
 */

boolean
esc_simplify(void)
{
	VALUE	     simplified, s;

	simplified.sub = XS(0);
	simplified.term = X(0);
        top_scratch_pad = scratch_pad;
        s.sub = EMPTY_SUB; 
	s.term = simp(&simplified);
	return(unify(XV(1), &s));
}

/*-----------------------------------------------------------------------------
copy_swap_sub_sp(sub)

    [r1/x1]* ...*[rn/xn] -> [rn/xn]*...*[r1/x1]

    Copy substitution onto the scratch pad and swap pointers at the same time.
    New proper properties are put.
    It returns pointer to the new substitution.
-----------------------------------------------------------------------------*/
local cell
copy_swap_sub_sp(cell sub)
{
        cell    previous;
        cell    copied;
        cell    s;
        cell    property;
        cell    prop;
        cell    prev_prop;
        int     i;

        previous = EMPTY_SUB;
        prev_prop = INVERTIBLE;

        for (s = sub; s != EMPTY_SUB; s = NextSub(s))
        {
                i = determine_property(s);

                prop =  ContainLocal(s, 1) ? INVERTIBLE :
                                             i;
                property = prop > prev_prop ? prop : prev_prop;

                copied = NewSubSP(Table(s), previous, property);

                previous = (cell)copied;

                prev_prop = property;

        }
        return(previous);
}
/*
 * Repeat the following until no more reductions:
 * {
 *         If ( z is apart from x and  
 *              y is apart from x and 
 *              z and y are not locals)
 *         { 
 *                 Replace the term of the form: 
 *                         s * [z/y] * @(q, x ^ t)  
 *                 with: 
 *                         s * @(q, x [z/y] * t) 
 *         } 
 *         Replace the term of the form: 
 *                 s * [u/x] * @(q, x ^ t)  
 *         with:
 *                 s * @(q, x ^ t) 
 * }
 */

local   void
apply_reductions(cell *bodysub, cell *termsub, cell *objvar)
{
        cell        range, temp_termsub;
        int         property;
        VALUE       domain;

        if (*termsub != EMPTY_SUB && Size(*termsub) == 1)
        {
                range = Range(*termsub, 1);
                variable_dereference(&range);
                DereferenceTerm(domain, Domain(*termsub, 1));
                if (IsObjectReference(range) &&
                    IsObjectVariable(ObjectValue(range)) &&
                    !IsLocalObjectVariable(range) &&
                    !IsLocalObjectVariable(domain.term) && 
                    distinct_from(range, *objvar) &&
                    distinct_from(domain.term, *objvar))  
                {
/*
 *                      s * [z/y] * @(q, x ^ t)  =>  s * @(q, x [z/y] * t)
 */ 
			if (is_first && *bodysub != EMPTY_SUB && 
				NextSub(*bodysub) != EMPTY_SUB)
			{
				*bodysub = copy_swap_sub_sp(*bodysub);
			}
			is_first = FALSE;
			should_swap = TRUE;
                        property = determine_property(*termsub);
			*bodysub = NewSubSP(Table(*termsub),
                                            *bodysub,
                                   (Property(*bodysub) > property ?
                                    Property(*bodysub) : property));
                        *termsub = NextSub(*termsub);
                        apply_reductions(bodysub, termsub, objvar);
                }
                else if (domain.term == *objvar)
                {
/*
 *                      s * [u/x] * @(q, x ^ t) => s * @(q, x ^ t)
 */
                        *termsub = NextSub(*termsub);
                        apply_reductions(bodysub, termsub, objvar);
                        
                }
        }
}

/*
 * While term is of the form: s * @(q, x ^ [x/n] * s2 * [n/y] * t) 
 *         (In this example : 
 *                  bodysub     = [x/n] * s2 * [n/y], 
 *                  quantsub    = s,  
 *                  objvar      = x,
 *                  lobjvar     = n,    
 *                  newquantsub = new substitution which will                 
 *                                be on the place of s
 *                  newsub      = new substitution which will 
 *                                be on the place of  [x/n] * s2 * [n/y]
 *         ) 
 *
 * {
 *         Apply rule 1: 
 *                 Replace the term with: s * s2 * @(q, y ^ t) 
 * }
 */

local	int
apply_rules(cell bodysub, cell *quantsub, cell *objvar, cell *lobjvar, cell *newquantsub, cell *newsub)
{
	VALUE	    range;
	int	    property;

	if (NextSub(bodysub) == EMPTY_SUB)
	{
		range.sub = EMPTY_SUB;
		range.term = Reference(&Range(bodysub, 1));
		dereference(&range);
		if (range.term == *objvar &&
		    IsLocalObjectVariable(Domain(bodysub, 1)))
		{	
/*
 *                      Rule 1 applies. 
 */ 
			*lobjvar = Domain(bodysub, 1);
                        *newquantsub = *quantsub;
			return(RULE_1);
		}
		else
		{
/*
 *                      No rule is applied - get out.
 *                      No need to copy.
 */
			*newsub = bodysub;
			return(NO_RULE);
		}
	}
	else switch (apply_rules(NextSub(bodysub), quantsub, objvar, lobjvar,
				 newquantsub, newsub))
	{
	when NO_RULE:
/*
 *              Continue to recure out. 
 */ 
		*newsub = bodysub;
		return(NO_RULE);
	when RULE_1_TERMINATE:
/*
 *              Rule 1 was applied in the last recursion.
 *              Repeat to try to apply again.
 */
                *quantsub = *newquantsub;
		range.sub = EMPTY_SUB;
		range.term = Reference(&Range(bodysub, 1));
		dereference(&range);
		if (range.term == *objvar &&
		    IsLocalObjectVariable(Domain(bodysub, 1)))
		{		/* rule 1 applies */
/*
 *                      Rule 1 applies. 
 */ 
                        *newsub = EMPTY_SUB;
			*lobjvar = Domain(bodysub, 1);
			return(RULE_1);
		}
		else
		{
/*
 *                      No rule applies.
 *                      Copy the remaining substitutions.
 */
			*newsub = NewSubSP(Table(bodysub), EMPTY_SUB,
					determine_property(bodysub));
			return(COPY_SUB);
		}
	when RULE_1:
		if (*lobjvar == Range(bodysub, 1))
                {
		
/*
 *                      Rule 1 applies. 
 */ 
			*objvar = Domain(bodysub, 1);
			object_dereference(objvar);
			return(RULE_1_TERMINATE); 
		}
		else
	        {	
/*
 *                      Move substitution to outside quantifier. 
 */
			property = determine_property(bodysub);
			*newquantsub = NewSubSP(Table(bodysub),
                                                       *newquantsub, 
					(Property(*newquantsub) > property ?
					    Property(*newquantsub) : property));
			return(RULE_1);
		}
	when COPY_SUB:
/*
 *              No rule applies.
 *              Copy the remaining substitutions.
 */
		property = determine_property(bodysub);
		*newsub = NewSubSP(Table(bodysub), *newsub,
			     (Property(NextSub(*newsub)) > property ?
				Property(NextSub(*newsub)) :
				property));
		return(COPY_SUB);
	}
}

/*
 * Apply the quantify rules, calling apply_rules, where
 * the rules are applied and terminate rule 1 (below).
 */

local	cell
apply_quantify_rules(cell bodysub, cell *quantsub, cell *objvar)
{
	cell	newsub, newquantsub, lobjvar;

	if (bodysub == EMPTY_SUB)
        {
		return(EMPTY_SUB);
        }
	else
	{
		lobjvar = NULL;
		newquantsub = EMPTY_SUB;
		newsub = EMPTY_SUB;
                if (apply_rules(bodysub, quantsub, objvar, &lobjvar,
                    &newquantsub, &newsub) == RULE_1_TERMINATE)
                {
                        *quantsub = newquantsub;
                }
		return(newsub);
	}
}

/*
 * Incomplete backtrack.
 */   

local   void
do_simp_backtrack(void)
{

        top_of_heap = last_choice_point->top_of_heap;
        top_delayed_stack = last_choice_point->top_delayed_stack;
        while (top_of_trail > last_choice_point->top_of_trail)
        {
                top_of_trail--;
                if (top_of_trail->address != NULL)
                {
                        * (top_of_trail->address) =
                                top_of_trail->previous_value;
                }
        }
        last_choice_point = last_choice_point->last_choice_point;
}

/*
 * Count elements in a substitution.
 */

local	natural
sub_elem(cell sub)
{

	if (sub == EMPTY_SUB)
        {
		return((int)NULL);
        }
	else 
        {
                return(Size(sub) + sub_elem(NextSub(sub)));
        }
}

/*
 * Check if the variable - var is in the variable table. 
 * Until not found:
 *         Set var to be distinct from the variables from the variables table.
 */

local boolean
is_var_used(cell *vars_used, cell *top_vars_used, cell var)
{
        cell        *z;

        for (z = vars_used; z < top_vars_used; z++) 
        {
                if (var == *z) 
                {  
                        return(TRUE);
                }
                else
                {
                        SetDistinct(var, *z);
                }
        }
        return(FALSE);
}

/*
 * Check if the variable - var is in the variable table. 
 */

local boolean
is_range_used(cell *vars_used, cell *top_vars_used, cell var)
{
        cell        *z;

        for (z = vars_used; z < top_vars_used; z++) 
        {
                if (var == *z) 
                {  
                        return(TRUE);
                }
        }
        return(FALSE);
}
       
/*
 * For a given parallel substitution - par_sub, 
 * push a new substitution on the scratch_pad.
 */

local cell
new_sub_sp(cell par_sub, unsigned int subsize)
{
        cell        new_sub;

        SizeSP(par_sub) = subsize;
        new_sub = NewSubSP((cell)par_sub, EMPTY_SUB, INVERTIBLE);     
        new_sub = NewProperty(new_sub, determine_property(new_sub));     
        return(new_sub);
}

/*
 * dom_size = the number of domain elements in the substitution - sub
 * 
 * create a block to take a parallel substitution of size at most dom_size
 * create a block of size  (dom_size + 1) to hold used variables 
 * 
 * set the size of parallel substitution - subsize to 0
 * store object variable - objvar in vars_used
 * set var_used_size to 1 
 *
 * for each domain of substitution sub (skip locals) 
 * {
 *         create incomplete choice point
 *         if variable is not used
 *         {       
 *                 add domain to vars_used 
 *                 if retry_delay 
 *                 {
 *                         create entry (simp(NextSub(s) * Range(s)), domain)
 *                         increment subsize by 1 
 *                 }
 *         }       
 *         backtrack - remove incomplete choice point
 * }
 *
 * set the size of parallel substitution to subsize
 * return pointer to parallel substitution 
 *
 */

local	cell  
simp_sub(cell sub, cell objvar)
{
        VALUE       val;
        natural     dom_size, subsize, var_used_size, l, l1;
        cell        *vars_used, *top_vars_used, *par_sub;
        cell        domain, range, s;

        dom_size = sub_elem(sub);

        if (dom_size == 0)
        {
                return(EMPTY_SUB); 
        }

        par_sub = AllocateSP(dom_size * 2 + 1);
        vars_used = AllocateSP(dom_size + 1);
        top_vars_used = vars_used;
        subsize = 0;
        *top_vars_used++ = objvar;
        var_used_size = 1;

        for (s = sub; s != EMPTY_SUB; s = NextSub(s))
        {
                for (l = 1, l1 = Size(s); l <= l1; l++)   
                {
                        ObjectDereference(domain, Domain(s, l));
                        if (!IsLocalObjectVariable(domain)) 
                        {
                                create_incomplete_choice_point();
                                if (is_var_used(vars_used,
                                                top_vars_used, domain) == FALSE)
                                {
                                        *top_vars_used++ = domain;
                                        var_used_size++;
                                        if ((retry_nfi_delay() != FALSE))
                                        {
                                                val.sub = NextSub(s);
                                                val.term = Range(s, l);
                                                subsize++;
                                                DomainSP(par_sub, subsize) = 
                                                            domain;
                                                RangeSP(par_sub, subsize) = 
                                                           simp(&val);
                                        }
                                }
                                do_simp_backtrack();
                        }
                        else
                        {
                                ObjectDereference(range, Range(s, l));
                                if (is_range_used(vars_used,
                                                top_vars_used, range) == FALSE)
                                {
                                        *top_vars_used++ = range;
                                        var_used_size++;
 
                                 }
                        }
                }
        }
        if (subsize > 0) 
        {
                return(new_sub_sp((cell)par_sub, subsize));     
        }
        else
        {
                return(EMPTY_SUB);     
        }
}


/*
 * If apply node is genuine       / t = @(u, v) / 
 * {
 *         set c-local variables to simp(s*u) and simp(s*v)
 *         create apply node on scratch_pad with arguments 
 *         set to these C-variables
 *         return pointer to apply node created
 * }
 * else  / quantifier t = @(q, x ^ t) /  
 * {
 *         apply quantifier rules:  
 *                 s * @(q, x ^ [x/n] * s2 * [n/y] * t) => s * s2 * @(q, y ^ t)
 *         apply reductions:
 *                 s * [z/y] * @(q, x ^ t)  =>  s * @(q, x ^ [z/y] * t) 
 *                 s * [u/x] * @(q, x ^ t)  =>  s * @(q, x ^ t)
 *         / assume that after these reductions term is s3 * @(q, x3 ^ t3) / 
 *         assign local C-variables:
 *                 l_simp_sub  to simp_sub(s3), 
 *                 l_simp_term  to simp(s3)
 *         build the term l_simp_sub * @(q, x3 ^ l_simp_term) 
 *         return the pointer to this term
 * }
 */ 

local	cell
simp_apply(VALUE *term)
{
	VALUE	     fn, arg, val, body; 
        cell         l_arg, l_fn, app, l_simp_sub, l_simp_term, newterm, objvar;
        cell         term_newterm;

	if (!IsQuantifier(DerefTerm(val, Reference(&Argument(term->term)))))
        {
                arg.sub = term->sub;
                arg.term = Reference(&Argument(term->term)); 
                fn.sub = term->sub;
                fn.term = Reference(&Functor(term->term)); 
                l_arg = simp(&arg);
                l_fn = simp(&fn);
                app = ApplySP(); 
                Argument(app) = l_arg;
                Functor(app) = l_fn;
                return(app); 
        }
        else
        {
                body.sub = EMPTY_SUB;
                body.term = Reference(&Body(val.term));
                dereference(&body);
                objvar = BoundVar(val.term);
                object_dereference(&objvar);
                body.sub = apply_quantify_rules(body.sub, &(term->sub),
                                                &objvar);
		is_first = TRUE;
		should_swap = FALSE;
                apply_reductions(&(body.sub), &(term->sub), &objvar);
		if (should_swap)
		{
			body.sub = copy_swap_sub_sp(body.sub);
		}
                l_simp_sub = simp_sub(term->sub, objvar);
                l_simp_term = simp(&body);
                newterm = SubOperatorSP();
                Substitution(newterm) = l_simp_sub;
                term_newterm = ApplySP();
                Functor(term_newterm) = Functor(term->term);
                Argument(term_newterm) = QuantifierSP();  
                BoundVar(Argument(term_newterm)) = objvar;
                Body(Argument(term_newterm)) = l_simp_term;
                Term(newterm) = term_newterm;
                return(newterm);
        }
}

/*
 * dom_size = the number of domain elements in the substitution - sub
 * 
 * create a block - par_sub to take a parallel substitution of size
 * at most dom_size 
 * create a block - vars_used of size dom_size to hold used variables 
 * 
 * set the size of parallel substitution - subsize to 0
 * set the size of the block of used variables - var_used_size to 0 
 *
 * create incomplete choice point
 * 
 * for each domain of substitution term->sub (skip locals) 
 * {
 *         create incomplete choice point
 *         if variable is not used
 *         {       
 *                 add domain to vars_used 
 *                 increment vars_used_size by 1 
 *                 for each delayed problem of the form:
 *                         z not_free_in s1 * X
 *                         create
 *                         z not_free_in s1 * domain
 *                 if retry_delay 
 *                 {
 *                         create entry (simp(NextSub(s) * Range(s)), domain)
 *                         increment subsize by 1 
 *                 }
 *         }       
 *         backtrack - remove incomplete choice point
 * }
 * backtrack - remove incomplete choice point
 * set the size of the parallel substitution to subsize
 * return pointer to the term: (parallel substitution) * term->term   
 */ 

local	cell
simp_var_term(VALUE *term)
{

        VALUE        val;
        natural      dom_size, subsize, var_used_size, l, l1;
        cell         *vars_used, *top_vars_used, *par_sub, *d;
        cell         domain, s, newterm, new_sub;

        dom_size = sub_elem(term->sub);
        if (dom_size == 0)
        {
                return(term->term);
        }
        newterm = SubOperatorSP();
        Term(newterm) = term->term;  

        par_sub = AllocateSP(dom_size * 2 + 1);
        vars_used = AllocateSP(dom_size);
        top_vars_used = vars_used;
        subsize = 0;
        var_used_size = 0;
        create_incomplete_choice_point();
 
        for (s = term->sub; s != EMPTY_SUB; s = NextSub(s)) 
        {
                for (l = 1, l1 = Size(s); l <= l1; l++)   
                {
                        ObjectDereference(domain, Domain(s, l));
                        if (!IsLocalObjectVariable(domain)) 
                        {
                                create_incomplete_choice_point();
                                if (is_var_used(vars_used,
                                                top_vars_used, domain) == FALSE)
                                {
                                        *top_vars_used++ = domain;
                                        var_used_size++;
                                        if ((create_del_problem(term->term,
                                                   domain) != FALSE) &&
                                            (retry_nfi_delay() != FALSE))
                                        {
                                                val.sub = NextSub(s);
                                                val.term = Range(s, l);
                                                subsize++;
                                                DomainSP(par_sub, subsize) = 
                                                                     domain;
                                                RangeSP(par_sub, subsize ) = 
                                                                     simp(&val);
                                        }   
                                }
                                do_simp_backtrack();
                                delay(NOT_FREE, Location(term->term),
                                      term->term, domain);
                        }
                }
        }
        do_simp_backtrack();
        if (subsize > 0)
        {
                Substitution(newterm) = new_sub_sp((cell)par_sub, subsize);  
        }
        else
        {
                Substitution(newterm) = EMPTY_SUB;
        }
        return(newterm); 
}

/*
 * dom_size = the number of domain elements in the substitution - sub
 * 
 * create a block - par_sub to take a parallel substitution of size
 * at most dom_size 
 * create a block - vars_used of size dom_size to hold used variables 
 * 
 * set the size of parallel substitution - subsize to 0
 * set the size of the block of used variables - var_used_size to 0 
 *
 * create incomplete choice point
 * 
 * for each domain of substitution term->sub (skip locals) 
 * {
 *         create incomplete choice point
 *         if variable is not used
 *         {       
 *                 if (domain != term->term)
 *                 {
 *                         add domain to vars_used 
 *                         increment vars_used_size by 1 
 *                 }
 *                 if retry_delay 
 *                 {
 *                         create entry (simp(NextSub(s) * Range(s)), domain)
 *                         increment subsize by 1 
 *                 }
 *         }       
 *         backtrack - remove incomplete choice point
 *         if (domain == term->term)
 *         {
 *                 break
 *         }
 *         else
 *         {
 *                 make domain apart from term->term
 *         }
 * }
 * backtrack - remove incomplete choice point
 * set the size of the parallel substitution to subsize
 * return pointer to the term: (parallel substitution) * term->term   
 */ 

local	cell
simp_obvar_term(VALUE *term)
{
        VALUE        val;
        natural      dom_size, subsize, var_used_size, l, l1;
        cell         *vars_used, *top_vars_used, *par_sub;
        cell         s, domain, newterm, new_sub;

        dom_size = sub_elem(term->sub); 
        if (dom_size == 0)
        {
                return(term->term);
        }
        newterm = SubOperatorSP();
        Term(newterm) = term->term;  
        
        par_sub = AllocateSP(dom_size * 2 + 1);
        vars_used = AllocateSP(dom_size);
        top_vars_used = vars_used;
        subsize = 0;
        var_used_size = 0;
        create_incomplete_choice_point();
 
        for (s = term->sub; s != EMPTY_SUB; s = NextSub(s)) 
        {
                for (l = 1, l1 = Size(s); l <= l1; l++)   
                {
                        ObjectDereference(domain, Domain(s, l));
                        if (!IsLocalObjectVariable(domain)) 
                        {
                                create_incomplete_choice_point();
                                if (is_var_used(vars_used,
                                                top_vars_used, domain) == FALSE)
                                {
                                        if (domain != term->term)
                                        {
                                                *top_vars_used++ = domain; 
                                                var_used_size++;
                                        }
                                        if (retry_nfi_delay())
                                        {
                                                val.sub = NextSub(s);
                                                val.term = Range(s, l);
                                                subsize++;
                                                DomainSP(par_sub, 
                                                         subsize) = domain;
                                                RangeSP(par_sub,
                                                        subsize) = simp(&val);
                                        }
                                }
                                do_simp_backtrack();
                                if (domain == term->term)
                                {
                                        break;
                                }
                                else
                                {
                                        SetDistinct(domain, term->term);
                                }
                        }
                }
        }
        do_simp_backtrack();
        if (subsize > 0)
        {
                Substitution(newterm) = new_sub_sp((cell)par_sub, subsize); 
        }
        else
        {
                Substitution(newterm) = EMPTY_SUB;
        }
        return(newterm); 
}

